// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48 nobackup
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2015 Freescale Semiconductor, Inc. 
// All rights reserved
//
// This is unpublished proprietary source code of Freescale Semiconductor.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           reset.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR                DESCRIPTION
// 1.0      2015-06-30  Gary Milliorn         New (from LS2085ARDB)
// -----------------------------------------------------------------------------
// KEYWORDS:            RESET SEQUENCER RSEQ
// -----------------------------------------------------------------------------
// PURPOSE:  "reset" manages reset assertion to the T1040, from several sources:
//              * reset switch
//              * power up
//              * RESET_REQ_B assertion.
//              * JTAG/CCS RST_B assertion.
//
//            "reset" is a simplified version of the T1040QDS reset sequencer, 
//            with no support for I2C clock programming or specialized test features.
//            Because of this, the FSM has a few non-optimal issues, such as
//            an 8-bit FSM state register.  I'm assuming the compiler will optimize
//            this away for you.
// -----------------------------------------------------------------------------
// PARAMETERS       
//                      CLK_FREQ            clock frequency (in kHz).
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b              asynchronous, active low
// Clock Domains:        clk                system main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module reset #(
   parameter CLK_FREQ = 66666,
   parameter SIM_MODE = 0
)
( 
    // Reset sources.
    //
    input   wire            jtag_rst_b, 
    input   wire            pwr_is_off, 
    input   wire            rst_from_switch_b, 
    input   wire            rst_from_wdog_b,
    input   wire            rst_rst, 
    input   wire            rcfg_go,
    input	wire			rst_qspi_b, 

    // Config.
    //
    input   wire            sw_bypass_b,

    input   wire    [7:0]   rst_force1,
    input   wire    [7:0]   rst_force2,
    input   wire    [7:0]   rst_force3,
    input   wire    [7:0]   rst_mask1, 
    input   wire    [7:0]   rst_mask2, 
    input   wire    [7:0]   rst_mask3, 
    input   wire            cfg_rstreq_md, 
    input   wire            cfg_rst_noddr, 

    input   wire            present_proc_b,

    // DUT Reset Signals
    //
    output  wire            dut_poreset_b, 
    output  wire            int_poreset_b, 
    inout   wire            dut_hreset_b, 
    output  wire            int_hreset_b, 
    input   wire            int_reset_req_b, 
    output  wire            dut_trst_b, 

    // System Reset outputs.
    //
    output  wire            rst_clocks_b,
    output  wire            rst_emmc_b, 
    output  wire            rst_i2c_b, 
    output  wire            rst_ieee_b,
    output  wire            rst_mem_b, 
    output  wire            rst_phy1_b, 
    output  wire            rst_phy2_b, 
    output  wire            rst_phy3_b, 
    output  wire            rst_retimer_b,
    output  wire            rst_slot1_b,
    output  wire            rst_slot2_b,
    output  wire            rst_slot3_b,
    output  wire            rst_tdm_b,

    // Config control
    //
    output  wire            cfg_drv,                // Asserted for external config
    output  wire            int_cfg_drv_b,          // Asserted for internal config
    output  reg             rst_cfg_regs_b, 
    output  reg             rst_ctl_regs_b, 

    // Status output.
    //
    output  reg             rst_in_rst, 
    output  wire    [7:0]   rst_state,
    output  reg     [7:0]   rst_reason,

	output	wire	[3:0]	rst_mon,

    // General inputs.
    //
    input   wire            disable_io_b, 
    input   wire            rst_b, 
    input   wire            clk_10kHz_ena,
    input   wire            clk
);


//===========================================================================
// Include standardized definitions and values.
//
`include <qixis_defs.h>


//---------------------------------------------------------------------------
// RESET_REQ_B handling.
//      Mask if SW_BYPASS_B switch is set (global), or if
//      SW_RSTMD == 0 (ignore).
//
    wire    rst_from_rstreq_b;

    assign  rst_from_rstreq_b   = (!sw_bypass_b)    ? 1'b1
                                : (present_proc_b)  ? 1'b1              // Ignore if no CPU
                                : (!cfg_rstreq_md)  ? 1'b1
                                :                     int_reset_req_b;


//---------------------------------------------------------------------------
// FSM State codes.
//    States partially encode RSEQ_STATE as exported to the main level.
//    We have only 4 STAT LEDs, and the MSB==1 in reset, so encoding
//    bits 6-4 to be informative helps.
//
/*
*/
    localparam [7:0]    RSEQ_IDLE               = 8'h00,			// 1000_
                        RSEQ_NORMAL             = 8'h01,
                        RSEQ_RST_BY_RSTREQ      = 8'h90,			// 1001_
                        RSEQ_RST_BY_JTAG        = 8'h92,			// 1001_
                        RSEQ_RST_BY_RCFG        = 8'h94,			// 1001_
                        RSEQ_RST_BY_RCFG_DLY    = 8'h95,			// 1010_
                        RSEQ_RST_BY_RRST        = 8'h96,			// 1001_
                        RSEQ_RST_BY_RRST_DLY    = 8'h97,			// 1010_
                        RSEQ_RST_BY_SWITCH      = 8'h98,			// 1001_
                        RSEQ_RST_BY_WATCHDOG    = 8'h9A,			// 1001_
                        RSEQ_RST_BY_POWER       = 8'h9C,			// 1001_
                        RSEQ_POST_RST           = 8'hB0,			// 1011_
                        RSEQ_CLK_LOCK           = 8'hC0,			// 1100_
                        RSEQ_CLK_WAIT           = 8'hD0,			// 1101_
                        RSEQ_RELEASE_NONDUT     = 8'h60,			// 1110_
                        RSEQ_RELEASE_DUTPRE     = 8'h61,			// 1110_
                        RSEQ_RELEASE_DUT        = 8'h70,			// 1111_
                        RSEQ_RELEASE_DUTEND     = 8'h79,			// 1111_
                        RSEQ_WAIT_RESET         = 8'h80;			// 1000_		OK


//---------------------------------------------------------------------------
// Timers:
//		Units=100us
//		The timers count down to zero, and stay there when expired.  The
//		time-out trigger is at t=1, which gives us a 1-cycle window to
//		cascade timer events without danger of a fall through.
//
//	Validation (via pwr_mon):
//		regdly		: 500ms		OK
//		clk_lock	: 250ms		OK
//		cpu_rel		: 500ms		OK
//		deinit		:   1ms		OK
//
    wire [15:0]  rseq_timer_deinit,		rseq_timer_clk_lock;
    wire [15:0]  rseq_timer_regdly,		rseq_timer_cpu_rel;

	assign  rseq_timer_cpu_rel  =  (SIM_MODE) ? 16'd1	:	16'd_500_0;		//  500 ms  OK
	assign  rseq_timer_deinit   =  (SIM_MODE) ? 16'd1	:	16'd___1_0;     //    1 ms
	assign  rseq_timer_clk_lock =  (SIM_MODE) ? 16'd1	:	16'd_250_0;     //  250 ms  nominal
	assign  rseq_timer_regdly   =  (SIM_MODE) ? 16'd1	:	16'd_500_0;     //  500 ms  CCS/I2C access delay


//===========================================================================
// Collect reset inputs.  Latch events until the reset FSM has handled them.  This
// is necessary as resets may cause source signals to disappear early (register bits,
// CPU reset signals, etc).
//
    reg [2:0]   rstctl_sam,     rcfg_sam,       jtagrst_sam,      rstqspi_sam;
    reg         rstctl_evt_b,   rcfg_evt_b,     jtag_evt_b,       qspi_evt_b;
    reg         rst_evt_reg_b,  rst_evt_rcfg_b, rst_evt_jtag_b,   rst_evt_qspi_b;


// Watch for 0->1 transition on RSTCTL[RST].
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b) begin
            rstctl_sam      <= 3'b000;
            rstctl_evt_b    <= 1'b1;
        end else begin
            rstctl_sam      <= { rstctl_sam[1:0], rst_rst };
            rstctl_evt_b    <= ~(rstctl_sam == 3'b011);
        end

    always @(negedge rst_b  or  negedge rseq_handled_b  or  posedge clk)
        if (!rst_b | !rseq_handled_b)   rst_evt_reg_b  <= 1'b1;
        else                            rst_evt_reg_b  <= rst_evt_reg_b & rstctl_evt_b;


// Watch for 0->1 transition on RCFG[GO].
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b) begin
            rcfg_sam        <= 3'b000;
            rcfg_evt_b      <= 1'b1;
        end else begin
            rcfg_sam        <= { rcfg_sam[1:0], rcfg_go };
            rcfg_evt_b      <= ~(rcfg_sam == 3'b011);
        end

    always @(negedge rst_b  or  negedge rseq_handled_b  or  posedge clk)
        if (!rst_b | !rseq_handled_b)   rst_evt_rcfg_b  <= 1'b1;
        else                            rst_evt_rcfg_b  <= rst_evt_rcfg_b & rcfg_evt_b;


// Watch for 1->0 transition on JTAG_RST_B
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b) begin
            jtagrst_sam     <= 3'b111;
            jtag_evt_b      <= 1'b1;
        end else begin
            jtagrst_sam     <= { jtagrst_sam[1:0], jtag_rst_b };
            jtag_evt_b      <= ~(jtagrst_sam == 3'b100);
        end

    always @(negedge rst_b  or  negedge rseq_handled_b  or  posedge clk)
        if (!rst_b | !rseq_handled_b)   rst_evt_jtag_b  <= 1'b1;
        else                            rst_evt_jtag_b  <= rst_evt_jtag_b & jtag_evt_b;


// Watch for 1->0 transition on JTAG_RST_B
//
    always @(negedge rst_b  or  posedge clk)
        if (!rst_b) begin
            rstqspi_sam     <= 3'b111;
            qspi_evt_b      <= 1'b1;
        end else begin
            rstqspi_sam     <= { rstqspi_sam[1:0], rst_qspi_b };
            qspi_evt_b      <= ~(rstqspi_sam == 3'b100);
        end

    always @(negedge rst_b  or  negedge rseq_handled_b  or  posedge clk)
        if (!rst_b | !rseq_handled_b)   rst_evt_qspi_b  <= 1'b1;
        else                            rst_evt_qspi_b  <= rst_evt_qspi_b & qspi_evt_b;


//===========================================================================
// FSM synchronous elements.
//
    reg [7:0]   rseq_state,         next_rseq_state;
    reg                             next_rst_in_rst;
    reg                             next_rst_cfg_regs_b;
    reg                             next_rst_ctl_regs_b;
    reg [7:0]                       next_rst_reason;

    reg         dut_porst_b,        next_dut_porst_b;
    reg         dut_trst_drv_b,     next_dut_trst_drv_b;
    reg         rst_devices_b,      next_rst_devices_b;

    reg         rseq_handled_b,     next_rseq_handled_b;

    reg [15:0]  rseq_timer,			next_rseq_timer;
    reg         rseq_timer_sig;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rseq_state       <= RSEQ_IDLE;
        else if (clk_10kHz_ena)     rseq_state       <= next_rseq_state;

   always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rst_in_rst		<= 1'b0;
        else if (clk_10kHz_ena)     rst_in_rst		<= next_rst_in_rst;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rst_cfg_regs_b   <= 1'b0;
        else if (clk_10kHz_ena)     rst_cfg_regs_b   <= next_rst_cfg_regs_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rst_ctl_regs_b   <= 1'b0;
        else if (clk_10kHz_ena)     rst_ctl_regs_b   <= next_rst_ctl_regs_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rst_reason      <= { RSTEVT_NONE, RSTEVT_NONE };
        else if (clk_10kHz_ena)     rst_reason      <= next_rst_reason;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 dut_porst_b      <= 1'b0;
        else if (clk_10kHz_ena)     dut_porst_b      <= next_dut_porst_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 dut_trst_drv_b   <= 1'b0;
        else if (clk_10kHz_ena)     dut_trst_drv_b   <= next_dut_trst_drv_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rst_devices_b    <= 1'b0;
        else if (clk_10kHz_ena)     rst_devices_b    <= next_rst_devices_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rseq_handled_b   <= 1'b1;
        else if (clk_10kHz_ena)     rseq_handled_b   <= next_rseq_handled_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                 rseq_timer		<= 16'd0;
        else if (clk_10kHz_ena)     rseq_timer		<= next_rseq_timer;


// Timer-timeout (non-registered).
//
	always @*
		rseq_timer_sig	<= (rseq_timer == 16'd0);


//---------------------------------------------------------------------------
// Async next-state/next-output.
//
    always @* begin
        next_rseq_state         <= rseq_state;
        next_rseq_timer         <= (|rseq_timer) ? rseq_timer - 16'd1 : rseq_timer;
        next_rseq_handled_b     <= 1'b1;

        next_rst_in_rst		    <= rst_in_rst;
        next_rst_cfg_regs_b     <= 1'b1;
        next_rst_ctl_regs_b     <= 1'b1;

        next_rst_reason         <= rst_reason;

        next_dut_porst_b        <= dut_porst_b;
        next_dut_trst_drv_b     <= 1'b1;
        next_rst_devices_b      <= rst_devices_b;


        case (rseq_state)               // pragma parallel_case   

//---------------------------------------------------------------------------
// STEP 5: DONE
//

// WAIT_RESET   -- reset sequence completed, go idle and wait for an event.
//
        RSEQ_WAIT_RESET: begin
            next_rst_cfg_regs_b     <= 1'b1;

			if (pwr_is_off) begin
				next_rseq_state     <= RSEQ_RST_BY_POWER;
				next_rst_reason		<= { rst_reason[3:0], RSTEVT_POWER };

			end else if (!rst_evt_jtag_b  ||  !rst_evt_qspi_b) begin
				next_rseq_state     <= RSEQ_RST_BY_JTAG;
				next_rst_reason     <= { rst_reason[3:0], RSTEVT_JTAG };

			end else if (!rst_evt_rcfg_b) begin
				next_rseq_state     <= RSEQ_RST_BY_RCFG_DLY;
				next_rseq_timer     <= rseq_timer_regdly;
				next_rst_reason     <= { rst_reason[3:0], RSTEVT_RECONFIG };

			end else if (!rst_evt_reg_b) begin
				next_rseq_state     <= RSEQ_RST_BY_RRST_DLY;
				next_rseq_timer     <= rseq_timer_regdly;
				next_rst_reason		<= { rst_reason[3:0], RSTEVT_REG };

			end else if (!rst_from_switch_b) begin
				next_rseq_state     <= RSEQ_RST_BY_SWITCH;
				next_rst_reason		<= { rst_reason[3:0], RSTEVT_SWITCH };

			end else if (!rst_from_wdog_b) begin
				next_rseq_state     <= RSEQ_RST_BY_WATCHDOG;
				next_rst_reason		<= { rst_reason[3:0], RSTEVT_WATCHDOG };

			end else if (!rst_from_rstreq_b) begin
				next_rseq_state     <= RSEQ_RST_BY_RSTREQ;
				next_rst_reason		<= { rst_reason[3:0], RSTEVT_RSTREQ };
            end
        end


// RELEASE_DUTEND -- release the DUT.  External logic controls CFG_DRV_B/etc.
//
        RSEQ_RELEASE_DUTEND: begin
            if (rseq_timer_sig)
				next_rseq_state     <= RSEQ_WAIT_RESET;
		end


// RELEASE_DUT -- release the DUT.  External logic controls CFG_DRV_B and
//                manages pin-sampled hardware.   Stay in this state at least
//                100 clocks and ignore all reset events to help prevent
//                hysteresis due to RESET_REQ_B feedback.
//
        RSEQ_RELEASE_DUT: begin
            next_dut_porst_b        <= 1'b1;
            next_rst_in_rst			<= 1'b0;

			next_rseq_state			<= RSEQ_RELEASE_DUTEND;
			next_rseq_timer			<= rseq_timer_deinit;
        end


// RELEASE_DUTPRE -- wait for non-CPU devices to complete.
//
        RSEQ_RELEASE_DUTPRE: begin
            if (rseq_timer_sig)
				next_rseq_state     <= RSEQ_RELEASE_DUT;
        end


// RELEASE_NONDUT -- clock complete, release all devices except DUT.
//
        RSEQ_RELEASE_NONDUT: begin
            next_rst_devices_b		<= 1'b1;

			next_rseq_state			<= RSEQ_RELEASE_DUTPRE;
			next_rseq_timer			<= rseq_timer_cpu_rel;
        end


//---------------------------------------------------------------------------
// STEP 2: CLOCK PROGRAMMING
//
// CLK_LOCK -- the clock was reset.  Wait for the clock PLL's to stabilize before 
//             releasing the DUT and other dependent hardware.
//             Since this is slow, look for (re)reset events.
//
        RSEQ_CLK_LOCK: begin
			next_rseq_timer			<= rseq_timer_clk_lock;
			next_rseq_state			<= RSEQ_CLK_WAIT;
		end

        RSEQ_CLK_WAIT: begin
            if (rseq_timer_sig)
				next_rseq_state     <= RSEQ_RELEASE_NONDUT;

            else if (pwr_is_off)
                next_rseq_state     <= RSEQ_RST_BY_POWER;

            else if (!jtag_rst_b)
                next_rseq_state     <= RSEQ_RST_BY_JTAG;
		end


//---------------------------------------------------------------------------
// PHASE 1: MONITOR RESET SOURCES AND SYNC
//
// POST_RST -- reset was triggered; wait for all assertions to clear
//             before starting a reset cycle.  
//
        RSEQ_POST_RST: begin
            next_rst_cfg_regs_b     <= 1'b1;
            next_rst_ctl_regs_b     <= 1'b1;

            if (pwr_is_off)
                next_rseq_state     <= RSEQ_RST_BY_POWER;

            else if (jtag_rst_b          
                 &&  rst_from_switch_b   
                 &&  rst_from_wdog_b   
                 &&  rst_from_rstreq_b)
                next_rseq_state     <= RSEQ_CLK_LOCK;
        end


// RST_BY_JTAG -- reset due to "jtag_rst_b" assertion.
//                Considered equivalent to a switch: reset CTL CFG GEN
//
        RSEQ_RST_BY_JTAG: begin
            next_rseq_handled_b     <= 1'b0;
            
            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

			next_rseq_state			<= RSEQ_POST_RST;
        end


// RST_BY_RCFG_DLY -- wait a bit after a register reset to allow the caller
//                    (typically CCS) to complete.
//                    On entry, the timer was set to <REGDLY>
//
        RSEQ_RST_BY_RCFG_DLY: begin
            next_rseq_handled_b     <= 1'b0;

            if (rseq_timer_sig)
				next_rseq_state		<= RSEQ_RST_BY_RCFG;
        end


// RST_BY_RCFG -- reset due to "RCFG[GO]" assertion.
//
        RSEQ_RST_BY_RCFG: begin
            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b1;                    // Don't reset registers
            next_rst_ctl_regs_b     <= 1'b1;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

			next_rseq_state			<= RSEQ_POST_RST;
        end


// RST_BY_RRST_DLY -- (see RST_BY_RCFG_DLY).
//                    On entry, the timer was set to <REGDLY>
//
        RSEQ_RST_BY_RRST_DLY: begin
            next_rseq_handled_b     <= 1'b0;

            if (rseq_timer_sig)
				next_rseq_state		<= RSEQ_RST_BY_RRST;
        end


// RST_BY_RRST -- reset due to "RST[RST]" assertion.
//
        RSEQ_RST_BY_RRST: begin
            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

			next_rseq_state			<= RSEQ_POST_RST;
        end


// RST_BY_SWITCH -- reset due to: push the button Max!
//                  User-initiated reset: switch: reset CTL CFG GEN
//                  State time: 300cyc
//
        RSEQ_RST_BY_SWITCH: begin
            next_rseq_handled_b     <= 1'b0;

            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

			next_rseq_state			<= RSEQ_POST_RST;
        end


// RST_BY_WATCHDOG -- reset due to: watchdog event
//                  WDog-initiated reset: switch: reset CTL CFG GEN
//
        RSEQ_RST_BY_WATCHDOG: begin
            next_rseq_handled_b     <= 1'b0;

            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

			next_rseq_state			<= RSEQ_POST_RST;
        end


// RST_BY_RSTREQ -- reset due to RESET_REQ_B assertion from the DUT.
//                  Generally an RCW/HW config issue: reset CFG GEN
//
        RSEQ_RST_BY_RSTREQ: begin
            next_rseq_handled_b     <= 1'b0;

            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

			next_rseq_state			<= RSEQ_POST_RST;
        end


// RST_BY_POWER  -- reset due to power loss.  Also applies to initial power-up.
//                  Very similar to PBSW etc. except that TRST_B is asserted,
//                  which is otherwise not to be touched.
//
//	Note: RDBs do not support power-off register access, so registers are
//		  reset continually, as compared to QDS.
//
        RSEQ_RST_BY_POWER: begin
            next_rseq_handled_b     <= 1'b0;

            next_dut_trst_drv_b     <= 1'b0;        // Drive TRST_B once, only now.

            next_dut_porst_b        <= 1'b0;
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;
            next_rst_in_rst		    <= 1'b1;

            if (pwr_is_off)
                next_rseq_state     <= RSEQ_RST_BY_POWER;
			else
                next_rseq_state     <= RSEQ_POST_RST;
        end

// IDLE     - Reset everything.
//
        RSEQ_IDLE: begin
            next_rst_reason         <= { RSTEVT_NONE, RSTEVT_NONE };
            next_rseq_handled_b     <= 1'b1;
            next_rst_in_rst		    <= 1'b1;
            next_dut_porst_b        <= 1'b0;
            next_dut_trst_drv_b     <= 1'b0; 
            next_rst_cfg_regs_b     <= 1'b0;                    // Clear all
            next_rst_ctl_regs_b     <= 1'b0;                    // "
            next_rst_devices_b      <= 1'b0;

            next_rseq_state         <= RSEQ_RST_BY_POWER;
        end

// (recovery)   
//
        default: begin
            next_rseq_state         <= RSEQ_IDLE;
        end
        endcase
    end


//---------------------------------------------------------------------------
// Monitoring.
//
    assign  rst_state	= rseq_state;

	assign	rst_mon[3]	= (rseq_state == RSEQ_RELEASE_DUTPRE);
	assign	rst_mon[2]	= (rseq_state == RSEQ_POST_RST);
	assign	rst_mon[1]	= (rseq_state == RSEQ_CLK_WAIT);
	assign	rst_mon[0]	= (rseq_state == RSEQ_RST_BY_RCFG_DLY)
						| (rseq_state == RSEQ_RST_BY_RRST_DLY);


//---------------------------------------------------------------------------
// CFG_DRV - assert on PORESET, keep asserted for 3*SYSCLK after release.
//           Since SYSCLK (125MHz) is faster than the FPGA clock, we keep it
//           asserted for 2 clocks.
//
    reg   [2:0] cfg_time;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)         cfg_time <= 3'b111;
        else                cfg_time <= { cfg_time[1:0], dut_porst_b };

    assign int_cfg_drv_b    = (!dut_porst_b)        ? 1'b0      // FSM driving PORESET_B
                            : (cfg_time != 3'b111)	? 1'b0      // FSM released PORESET_B
                            :                         1'b1;

    assign cfg_drv          = (!disable_io_b)       ? 1'b0
                            :                         ~int_cfg_drv_b;


//---------------------------------------------------------------------------
// General output assignments
//

// R1.7 - unused
     
// R1.6 - unused

// R1.5 - unused

// R1.4 - I2C
//
    assign rst_i2c_b        = (!disable_io_b)   ? 1'bZ
                            : (rst_mask1[4])    ? 1'b1
                            : (rst_force1[4])   ? 1'b0
                            :                     rst_devices_b;  

// R1.3 - PHY3
//
    assign rst_phy3_b       = (!disable_io_b)   ? 1'bZ
                            : (rst_mask1[3])    ? 1'b1
                            : (rst_force1[3])   ? 1'b0
                            :                     rst_devices_b;  

// R1.2 - PHY2
//
    assign rst_phy2_b       = (!disable_io_b)   ? 1'bZ
                            : (rst_mask1[2])    ? 1'b1
                            : (rst_force1[2])   ? 1'b0
                            :                     rst_devices_b;  

// R1.1 - PHY2
//
    assign rst_phy1_b       = (!disable_io_b)   ? 1'bZ
                            : (rst_mask1[1])    ? 1'b1
                            : (rst_force1[1])   ? 1'b0
                            :                     rst_devices_b;  
// R1.0 - MEM
//
    assign rst_mem_b        = (!disable_io_b)   ? 1'bZ
                            : (cfg_rst_noddr)   ? 1'b1
                            : (rst_mask1[0])    ? 1'b1
                            : (rst_force1[0])   ? 1'b0
                            :                     rst_devices_b;

// R2.7 - IEEE
//
    assign rst_ieee_b       = (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[7])    ? 1'b1
                            : (rst_force2[7])   ? 1'b0
                            :                     rst_devices_b;  

// R2.6 - RETIMER
//
    assign rst_retimer_b    = (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[6])    ? 1'b1
                            : (rst_force2[6])   ? 1'b0
                            :                     rst_devices_b;  

// R2.5 - CLOCKS
//
	assign rst_clocks_b		= (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[5])    ? 1'b1
                            : (rst_force2[5])   ? 1'b0
                            :					  rst_devices_b;

// R2.4 - TDM
//
    assign rst_tdm_b        = (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[4])    ? 1'b1
                            : (rst_force2[4])   ? 1'b0
                            :                     rst_devices_b;  
// R2.3 - EMMC
//
    assign rst_emmc_b       = (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[3])    ? 1'b1
                            : (rst_force2[3])   ? 1'b0
                            :                     rst_devices_b;  

// R2.2 - TRST
//
    assign dut_trst_b       = (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[2])    ? 1'b1
                            : (!dut_trst_drv_b) ? 1'b0
                            : (rst_force2[2])   ? 1'b0
                            :                     1'bZ;

// R2.1 - HRESET
//
    assign dut_hreset_b     = (!disable_io_b)   ? 1'bZ                  // Drive only to 0
                            : (rst_mask2[1])    ? 1'b1
                            : (rst_force2[1])   ? 1'b0
                            :                     1'bZ;                 // Valid when powered
    assign int_hreset_b     = (!disable_io_b)   ? 1'b1
                            :                     dut_hreset_b;         // Always valid. 

// R2.0 - PORESET
//
    assign dut_poreset_b    = (!disable_io_b)   ? 1'bZ
                            : (rst_mask2[0])    ? 1'b1
                            : (rst_force2[0])   ? 1'b0
                            :                     dut_porst_b;          // Valid when powered
    assign int_poreset_b    =                     dut_porst_b;          // Always valid.


// R3.7 - SLOT1
//
    assign rst_slot1_b      = (!disable_io_b)   ? 1'bZ
                            : (rst_mask3[7])    ? 1'b1
                            : (rst_force1[3])   ? 1'b0                  // All slots
                            : (rst_force3[7])   ? 1'b0                  // Just this one
                            :                     rst_devices_b;  

// R3.6 - SLOT2
//
    assign rst_slot2_b      = (!disable_io_b)   ? 1'bZ
                            : (rst_mask3[6])    ? 1'b1
                            : (rst_force1[3])   ? 1'b0                  // All slots
                            : (rst_force3[6])   ? 1'b0                  // Just this one.
                            :                     rst_devices_b;  

// R3.5 - SLOT3
//
    assign rst_slot3_b      = (!disable_io_b)   ? 1'bZ
                            : (rst_mask3[5])    ? 1'b1
                            : (rst_force1[3])   ? 1'b0                  // All slots
                            : (rst_force3[5])   ? 1'b0                  // Just this one
                            :                     rst_devices_b;  

endmodule
