// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48 nobackup
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2015 Freescale Semiconductor, Inc. 
// All rights reserved
//
// This is unpublished proprietary source code of Freescale Semiconductor.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           top.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR DESCRIPTION
// 1.0      2015-06-29  Gary Milliorn         Original
// 2.0      2018-0703   Gary Milliorn         Changes to support -PB variant.
// -----------------------------------------------------------------------------
// KEYWORDS:            TOP
// -----------------------------------------------------------------------------
// PURPOSE:             Top-level interconnect diagram for LS1088ARDB
//						and LS1088ARDB-PB.
// -----------------------------------------------------------------------------
// PARAMETERS
//                      CLK_FREQ            core clock in kHz.
//                      SIM_MODE            set to 1 to accelerate simulation
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:      HOT_RST_B:          asynchronous, active low
// Clock Domains:       HOT_CLK:            25 MHz system main clock
// Critical Timing:     <none>
// Test Features:       <none>
// Asynchronous I/F:    <none>
// Scan Methodology:    <none>
// Instantiations:      <none>
// Synthesizable (y/n): Yes
// Other:
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module top #( 
    parameter   CLK_FREQ = 25000,
    parameter   SIM_MODE = 0
) (
    input   wire             HOT_CLK,					// From IDT clkgen (low-cost option)
    input   wire             HOT_RST_B, 
    input   wire             ALT_HOT_CLK,				// Y11 (backup option).

    // IFC Port
    //
    inout   wire    [15:0]   IFC_AD, 
    inout   wire             IFC_ALE, 
    input   wire             IFC_BCTL, 
    inout   wire             IFC_CLE, 
    input   wire             IFC_CLK, 
    input   wire    [2:0]    IFC_CS_B, 
    inout   wire             IFC_OE_B, 
    output  wire             IFC_RB1_B, 
    inout   wire             IFC_TE, 
    inout   wire             IFC_WE0_B, 
    inout   wire             IFC_WP_B, 

    output  wire             CFG_IFC_A5,                // External cfg_drv_b-controlled
    output  wire             CFG_IFC_A1,                // "
    output  wire             CFG_IFC_A0,                // "

    output  wire             NAND_CS_B,

    // Switch inputs/outputs
    //
    inout   wire             SW_PWR_B, 
    input   wire             SW_RST_B, 
    input   wire    [1:8]    SW1,
    input   wire    [1:8]    SW2,
    input   wire    [1:8]    SW3,
    input   wire    [1:8]    SW4,
    input   wire    [1:8]    SW5,
    input   wire    [1:0]    CPUID,
    output  wire             SW_PWR_LEDGRN_B,
    output  wire             SW_PWR_LEDYEL_B,

    // Config controls
    //
    output  wire             CFG_CFG_WP, 
    output  wire             CFG_EVDD_VSEL, 
    output  wire             CFG_IEEE_CLK, 
    output  wire             CFG_MUX_I2C2, 
    output  wire             CFG_MUX_I2C3, 
    output  wire             CFG_MUX_SPI, 
    output  wire             CFG_MUX_SD, 
    output  wire    [4:0]    CFG_QSPI_MAP, 
    output  wire             CFG_SFP_TXEN_B,
    output  wire             CFG_SPREAD, 
    output  reg              CFG_XTEST,

    // DUT Control/Status
    //
    inout   wire             DUT_ASLEEP, 
    output  wire             DUT_PORESET_B, 
    inout   wire             DUT_HRESET_B, 
    inout   wire             DUT_RESET_REQ_B, 
    output  wire             DUT_TBSCAN_EN_B, 
    output  wire             DUT_TESTSEL_B, 

    // Status, Monitoring, Etc.
    //
    output  wire    [3:0]    LEDS, 
    output  wire             LED_PASS, 
    output  wire             LED_FAIL, 
    output  wire             LED_ASLEEP, 
    output  wire             LED_PORST, 
    output  wire             LED_RSTREQ, 
    output  wire             LED_THERM, 
    input   wire             PRESENT_PROC_B, 
    input   wire             PRESENT_SLOT3_B, 
    input   wire             PRESENT_TDM_B, 
    input   wire             LOS_RETIMER_B, 
    input   wire             RETIMER_DONE_B, 
    input   wire             ROTATION_ERR_B, 
    input   wire             SFP_TX_FAULT,
    input   wire             SFP_RX_LOS,					// New for Rev D.
    input   wire             SMB_ALERT_B,
    input   wire             THERM_FAULT_B, 
    input   wire             THERM_WARN_B, 
    input   wire    [2:0]    PCB_REV, 

    // SDHC Handling
    //
    input   wire             SDSLOT_CD_B,
    input   wire             SDSLOT_WP,
    output  wire             DUT_CD_B,
    output  wire             DUT_WP,
    input   wire             SDHC_VS,

    // Power
    //
    output  wire             ATX_ON_B, 
    input   wire             ATX_PWRGD, 
    output  wire             PRELOAD_ON, 

    output  wire             PS_VDD_EN, 
    input   wire             PS_VDD_PG, 
    output  wire             PS_GVDD_EN, 
    input   wire             PS_GVDD_PG, 
    output  wire             PS_VTT_EN, 
    input   wire             PS_VTT_PG_B, 
    output  wire             PS_VPP_EN, 
    input   wire             PS_VPP_PG, 
    output  wire             PS_OVDD_EN, 
    input   wire             PS_OVDD_PG, 
    output  wire             PS_XVDD_EN, 
    input   wire             PS_XVDD_PG_B, 
    output  wire             PS_SVDD_SEL,
    output  wire             PS_2V5_EN, 
    input   wire             PS_2V5_PG_B, 
    output  wire             PS_1V2_EN, 
    input   wire             PS_1V2_PG_B, 
    output  wire             PS_0V85_EN, 
    input   wire             PS_0V85_PG_B, 
    output  wire             PS_EVDD_EN, 
    output  wire             PS_DVDD_EN, 
    input   wire             PS_TA_BB_PG_B,
    output  wire             PS_TA_BB_SEL,

    // Resets
    //
    input   wire             JTAG_RST_B, 
    input   wire             RST_QSPI_EMU_B,			// No longer connected on Rev D.  Pin has pullup

    output  wire             DUT_TRST_B, 
    output  wire             CFG_DRV,
    output  wire             RST_EMMC_B, 
    output  wire             RST_I2C_B, 
    output  wire             RST_IEEE_B, 
    output  wire             RST_MEM_B, 
    output  wire             RST_PHY1_B, 
    output  wire             RST_PHY2_B, 
    output  wire             RST_PHY3_B, 
    output  wire             RST_RETIMER_B, 
    output  wire             RST_SLOT1_B, 
    output  wire             RST_SLOT2_B, 
    output  wire             RST_SLOT3_B, 
    output  wire             RST_TDM_B, 

    // Clocks
    //
    output  wire             CLK_SI5341B_FINC,
    output  wire             CLK_SI5341B_FDEC,
    output  wire             CLK_SI5341B_RST_B,

    // Interrupts
    //
    output  wire    [0:2]    IRQ_B, 
    output  wire             EVT9_B,
    input   wire    [2:1]    IRQ_TDM_B,
    input   wire             IRQ_QSPHY1_B,
    input   wire             IRQ_QSPHY2_B,
    input   wire             IRQ_AQR105_B,
    input   wire             IRQ_RTC_B,

    // Misc
    //
    inout   wire             I2C1_SCL,
    inout   wire             I2C1_SDA,

	// LS1088ARDB-PB signals
	//
	input	wire			 BRD_ID,
	output	wire			 CFG_PCI1_WGIG_EN,
	output	wire			 CFG_PCI1_DIS_B,
	output	wire			 CFG_PCI2_WGIG_EN,
	output	wire			 CFG_PCI2_DIS_B,
	input	wire			 IRQ_PCIE1WAKE_B,
	input	wire			 IRQ_PCIE2WAKE_B,
	input	wire			 PS_5V0_PG,
	input	wire			 PS_3V3_PG,
	output	wire			 PS_PCIE3V3_EN,
	input 	wire			 PS_PCIE3V3_PG,
	input	wire			 PS_PWR_IRQ_B,
	output	wire			 PS_PWR_OFF_B
);


//===========================================================================
// Include standardized definitions and values.
//
`include <qixis_defs.h>


//--------------------------------------------------------------
// Internal nets and busses.
//
    wire    [7:0]   alarm,				alarm_mask;
    wire            boot_act_b;

    wire    [15:0]  cfg_ifc_ad;
    wire            cfg_ifc_a5,			cfg_ifc_a1,			cfg_ifc_a0;
    wire            cfg_ifc_cle,		cfg_ifc_oe_b,		cfg_ifc_te;
    wire            cfg_ifc_we0_b,		cfg_ifc_wp_b,		cfg_rsp_dis;

    wire            cfg_evdd_force,		cfg_rstreq_md,		cfg_rst_noddr;
	wire			cfg_xtest;

    wire            ctl_led,			ctl_passfail;
	wire	[1:0]	ctl_pwrled;

    wire            disable_io_b;           // Low when power is off.  All IO tristates.
    wire            force_off_b;

    wire    [7:0]   i2crmt_addr,		i2crmt_wdata;
    wire            i2crmt_cs_b,		i2crmt_we_b;

    wire    [7:0]   ifcreg_addr,		ifcreg_wdata;
    wire            ifcreg_cs_b,		ifcreg_oe_b,		ifcreg_we_b;

    wire    [7:0]   irq_stat0,			irq_stat1;

    wire            in_alarm_b;
	wire			is_pb_variant;
	wire			is_revb;

	wire		    int_asleep;
    wire            int_cfg_drv_b,		int_cfg_mux_sd;
    wire            int_poreset_b,		int_hreset_b,		int_reset_req_b;

    wire    [3:0]   leds_user;
    wire    [1:0]   led_test;

    wire    [7:0]   pwr_state;
    wire            pwr_is_off,			ps_faulted,			ps_starterr;
    wire    [7:0]   pwr_mstat,			pwr_stat1,			pwr_stat2;
    wire    [7:0]   pwr_reason,			pwr_tier_rpt;
	wire	[3:0]	pwr_mon;

	wire			pwr_ctl2_pwr;

    wire    [7:0]   reg_addr,			reg_wdata,			reg_rdata;
    wire            reg_cs_b,			reg_oe_b,			reg_we_b;
	wire			reg_drv_b;

    wire            rcfg_go,			rcfg_wden; 
    wire    [7:0]   rcfg_watch;

    wire    [7:0]   rst_force1,			rst_force2,			rst_force3;
    wire    [7:0]   rst_mask1,			rst_mask2,			rst_mask3;
    wire    [7:0]   rst_reason,			rst_state;
	wire	[3:0]	rst_mon;

    wire            rst_cfg_regs_b,		rst_ctl_regs_b;
    wire            rst_rst;
    wire            rst_in_rst;

	wire	[1:0]	sps;
    wire            sw_auto_on,			sw_bypass_b;
	wire	[2:0]	sw_debug;

    wire            wdog_evt_b;


//--------------------------------------------------------------
// Board ID.
//		Detect -PB variant.
//
	assign	is_pb_variant	= (BRD_ID == 1'b0);
	assign	is_revb			= (PCB_REV == 3'b001);

	
//--------------------------------------------------------------
// Synchronous reset.
//
	reg     rst_b, rff;

    always @(negedge HOT_RST_B  or  posedge HOT_CLK)
        if (!HOT_RST_B)     { rst_b, rff } <= 2'b00;
        else                { rst_b, rff } <= { rff, 1'b1 };


//---------------------------------------------------------------------------
// Divided clock enable tree.  
//      Outputs are not clocks but single-core-clock (HOT_CLK) enable pulses.
//
    wire    clk_10kHz_ena;
    wire    clk_100Hz_ena;
    wire    clk_4Hz_ena;

    clocks #(
        .CLK_FREQ       (CLK_FREQ),
        .SIM_MODE       (SIM_MODE)
    ) CLKTREE (
        .clk            (HOT_CLK), 
        .rst_b          (rst_b), 

        .clk_10kHz_ena  (clk_10kHz_ena),
        .clk_100Hz_ena  (clk_100Hz_ena),
        .clk_4Hz_ena    (clk_4Hz_ena)
    );


//---------------------------------------------------------------------------
// SW_PWR_B is ... interesting.
// SW_PWR_B is on pin R13 which is NC on -PB.
//
	wire	sw_in_pwr_b;

	assign	sw_in_pwr_b	= (!is_pb_variant)	? SW_PWR_B			// -PA: true power switch.
						:					  1'b1;				// -PB: not an input.

	assign	SW_PWR_B	= (!is_pb_variant)	? 1'bZ 				// -PA: input only.
						: (!sw_auto_on)		? 1'bZ				// -PB: ignore if not auto-on
						: (auto_on_pulse)	? 1'b0				//		pulse at auto-on time.
						:					  1'bZ;				// Z for wire-or


//---------------------------------------------------------------------------
// Switch debouncers.
//      Clean up the noisy signals.
//      SW_PWR is ignored on -PB
//
    wire    sw_pwr_db_b;
    wire    sw_rst_db_b;

    debouncer #(
        .CLK_FREQ       (CLK_FREQ),
        .SIM_MODE       (SIM_MODE)
    ) DB0 ( 
        .clk            (HOT_CLK), 
        .clk_db_ena     (clk_100Hz_ena),
        .rst_b          (rst_b), 

        .sw_b           (sw_in_pwr_b),			// See above
        .out_b          (sw_pwr_db_b)
    ); 

    debouncer #(
        .CLK_FREQ       (CLK_FREQ),
        .SIM_MODE       (SIM_MODE)
    ) DB1 ( 
        .clk            (HOT_CLK), 
        .clk_db_ena     (clk_100Hz_ena),
        .rst_b          (rst_b), 

        .sw_b           (SW_RST_B), 
        .out_b          (sw_rst_db_b)
    ); 


//---------------------------------------------------------------------------
// Fault detection + system monitoring.   
//      If sw_bypass_b is set (0), ignore most faults.
//
    fault FAULT (
        .dut_present_b      (PRESENT_PROC_B), 
        .alarm_mask         (alarm_mask),

        .vdd_therm_b        (SMB_ALERT_B    | ~sw_bypass_b), 
        .therm_warn_b       (THERM_WARN_B   | ~sw_bypass_b), 
        .therm_fault_b      (THERM_FAULT_B  | ~sw_bypass_b), 
        .ptempoff_fault     (ps_faulted),
        .force_off_b        (force_off_b),
        .ctl_fail           (ctl_passfail),

        .alarm_stat         (alarm), 
        .in_alarm_b         (in_alarm_b)
    );

	// Manage LEDS.
	//
    monitor #(
        .CLK_FREQ           (CLK_FREQ),
        .SIM_MODE           (SIM_MODE)
    ) MON (
        .alarm_code         (alarm),
        .in_alarm_b         (in_alarm_b),
        .therm_fault_b      (THERM_FAULT_B),
        .therm_warn_b       (THERM_WARN_B),

        .ctl_led            (ctl_led),
        .ctl_passfail       (ctl_passfail),
		.ctl_pwrled			(ctl_pwrled),
        .leds_user          (leds_user),

		.sw_debug			(sw_debug),

        .pwr_is_off         (pwr_is_off),
        .pwr_state			(pwr_state),
		.pwr_tier_rpt		(pwr_tier_rpt),
		.sps				(sps),
        .ps_faulted         (ps_faulted),

        .rst_in_rst		    (rst_in_rst),
        .rst_state			(rst_state),

        .poreset_b          (int_poreset_b),
        .reset_req_b        (int_reset_req_b),
        .asleep             (int_asleep),

        .mon_act0_b         (I2C1_SCL),

        .leds               (LEDS),
        .led_pass           (LED_PASS),
        .led_fail           (LED_FAIL),
        .led_asleep         (LED_ASLEEP),
        .led_porst          (LED_PORST),
        .led_rstreq         (LED_RSTREQ),
        .led_therm          (LED_THERM),
        .led_pwrsw_grn_b    (SW_PWR_LEDGRN_B),
        .led_pwrsw_yel_b    (SW_PWR_LEDYEL_B),

        .rst_b              (rst_b),
        .clk_100Hz_ena      (clk_100Hz_ena),
        .clk_4Hz_ena        (clk_4Hz_ena),
        .clk                (HOT_CLK)
    );

// The watchdog will assert reset if enabled and not periodically reset.  It is overly
// simple but mostly present for boot-box support.
//
    WatchDog #(
        .CLK_FREQ           (CLK_FREQ)
    ) WDOG (
        .watch_time         (rcfg_watch),
        .watch_enable       (rcfg_wden),
        .sys_enable_b       (~rst_ctl_regs_b),		// If general reset, abort

        .wdog_evt_b         (wdog_evt_b),
        .wdog_mon           (),						// Monitoring (unused)
        .wdog_tick          (),						// "

        .rst_b              (rst_b),
        .clk                (HOT_CLK)
    );


//---------------------------------------------------------------------------
// SDHC Card voltage control
//      Using card detect + operating mode to set correct EVDD voltage.
//
    sdhc_voltage SDVOLT (
        .sdslot_cd_b        (SDSLOT_CD_B),
        .sdslot_wp          (SDSLOT_WP),
        .sdhc_vs            (SDHC_VS),
        .cfg_evdd_force     (cfg_evdd_force),
        .cfg_mux_sd         (int_cfg_mux_sd),
        .disable_io_b       (disable_io_b),

        .dut_cd_b           (DUT_CD_B),
        .dut_wp             (DUT_WP),
        .cfg_evdd_vsel      (CFG_EVDD_VSEL)
    );


//---------------------------------------------------------------------------
// Power Control
//    Sequence the power supplies up/down as needed based on various inputs
//    (switches, registers, etc).


// The LS1088ARDB-PB replaces the ATX PSU with a brick in the false belief that 
// doing so makes the system smaller.  Thus, it does not have full control of the 
// system power, but can only power-off.
// Additionally, it must co-operate with LS1088RDB which has full control.
// 
	assign	PS_PWR_OFF_B	= (!is_pb_variant)			? 1'bZ			// Not -PB, float
//							: (!sw_bypass_b)			? 1'bZ
							: (sps == SPS_DEPOWERING)	? 1'b0
							: (sps == SPS_OFF)			? 1'b0
							:							  1'bZ;			// Float if not asserted.


// Normal power sequencer:
//		LS1088ARDB		: SW_PWR powers up or powers down.
//		LS1088ARDB-PB	: No SW_PWR, PS_PWR_IRQ_B is an equivalent for
//						  power-down so power-down on either.
//						  If 5V/3V is up (-PB unique) then the system was
//						  powered up via the switch, so follow along.
//
// Note: -PB cannot force a power-on, so SW_AUTO_ON will not work.
//
	wire	atx_pwrgd;
	wire	ltc_pwr_off,	ltc_pwr_on;

	assign	ltc_pwr_off	= (!is_pb_variant)	?	1'b0		// n/a for -PA
						:						~PS_PWR_IRQ_B;

	assign	ltc_pwr_on	= (!is_pb_variant)	?	1'b0		// n/a for -PA
						:						PS_3V3_PG;

	assign	atx_pwrgd	= (!is_pb_variant)	?	ATX_PWRGD
						:						PS_3V3_PG;


// Force AUTO_ON via CFG_XTEST->PWR_SW
//
	reg	[10:0]	pb_auto_pulse;
	wire		auto_on_pulse;

	// Pulse after ATX_ON_B is driven low.
	//
	always @(negedge rst_b  or  posedge HOT_CLK)
		if (!rst_b)				pb_auto_pulse	<= 10'b11_1111_1111;
		else if (clk_4Hz_ena)	pb_auto_pulse	<= { pb_auto_pulse[9:0], ATX_ON_B };

	assign	auto_on_pulse	= (pb_auto_pulse[10:9] == 2'b10);


// Common power sequencer.
//
    power #(
        .CLK_FREQ           (CLK_FREQ),
        .SIM_MODE           (SIM_MODE)
    ) PWR (
        // General controls
        //
        .sw_pwr_b           (sw_pwr_db_b),					// Power cycle (or down) on either.
        .sw_auto_on         (sw_auto_on), //~is_pb_variant & sw_auto_on),	// Not allowed on -PB
		.sw_bypass_b		(sw_bypass_b),
        .force_off_b        (force_off_b),

		// LS1088ARDB-PB: LTC power controller has powered system on or off.
		// Follow along.
		//
		.is_pb				(is_pb_variant),
		.ltc_pwr_off		(ltc_pwr_off),
		.ltc_pwr_on			(ltc_pwr_on),

		// Register controls
		//
		.pwr_ctl2_pwr		(pwr_ctl2_pwr),
        .pwr_reason         (pwr_reason),
        .pwr_mstat          (pwr_mstat),
        .pwr_stat1          (pwr_stat1),
        .pwr_stat2          (pwr_stat2),

        // ATX Power Controls
        //
        .ps_on_b            (ATX_ON_B), 
        .pwrgd              (atx_pwrgd),
        .pwr_preload        (PRELOAD_ON),

        // PSU Power Controls
        //
        .ps_vdd_en          (PS_VDD_EN),
        .ps_vdd_pg          (PS_VDD_PG),
        .ps_gvdd_en         (PS_GVDD_EN),
        .ps_gvdd_pg         (PS_GVDD_PG),
        .ps_vtt_en          (PS_VTT_EN),
        .ps_vtt_pg_b        (PS_VTT_PG_B),
        .ps_vpp_en          (PS_VPP_EN),
        .ps_vpp_pg          (PS_VPP_PG),
        .ps_ovdd_en         (PS_OVDD_EN),
        .ps_ovdd_pg         (PS_OVDD_PG),
        .ps_xvdd_en         (PS_XVDD_EN),
        .ps_xvdd_pg_b       (PS_XVDD_PG_B),
        .ps_2v5_en          (PS_2V5_EN),
        .ps_2v5_pg_b        (PS_2V5_PG_B),
        .ps_1v2_en          (PS_1V2_EN),
        .ps_1v2_pg_b        (PS_1V2_PG_B),
        .ps_0v85_en         (PS_0V85_EN),
        .ps_0v85_pg_b       (PS_0V85_PG_B),
        .ps_evdd_en         (PS_EVDD_EN),
        .ps_dvdd_en         (PS_DVDD_EN),
        .ps_ta_bb_pg_b      (PS_TA_BB_PG_B),

		// LS1088ARDB-PB Monitoring (unsequenced)
		//
		.ps_5v0_pg			(PS_5V0_PG			|  ~is_pb_variant),
		.ps_3v3_pg			(PS_3V3_PG			|  ~is_pb_variant),
		.ps_pcie3v3_en		(PS_PCIE3V3_EN),
		.ps_pcie3v3_pg		(PS_PCIE3V3_PG		|  ~is_pb_variant),

        // Status outputs.
        //
        .disable_io_b       (disable_io_b),
		.sps				(sps),
        .pwr_is_off         (pwr_is_off),
        .pwr_state			(pwr_state),
		.tier_report		(pwr_tier_rpt),
        .ps_faulted         (ps_faulted),
		.pwr_mon			(pwr_mon),

        .rst_b              (rst_b),
        .clk_10kHz_ena      (clk_10kHz_ena),
        .clk                (HOT_CLK)
    );
		

//---------------------------------------------------------------------------
// Reset Control + Reconfig
//
    reset #(
        .CLK_FREQ           (CLK_FREQ),
        .SIM_MODE			(SIM_MODE)
    ) RST (
        // Reset sources.
        //
        .jtag_rst_b         (JTAG_RST_B), 
        .pwr_is_off         (pwr_is_off),
        .rst_from_switch_b  (sw_rst_db_b),
        .rst_from_wdog_b    (wdog_evt_b),						// From WatchDog
        .rst_rst            (rst_rst),							// RSTCTL[RST]
        .rcfg_go            (rcfg_go),
        .rst_qspi_b         (RST_QSPI_EMU_B), 

        // Config.
        //
		.sw_bypass_b		(sw_bypass_b),

        .rst_force1         (rst_force1),
        .rst_force2         (rst_force2),
        .rst_force3         (rst_force3),
        .rst_mask1          (rst_mask1),
        .rst_mask2          (rst_mask2),
        .rst_mask3          (rst_mask3),

        .cfg_rst_noddr      (cfg_rst_noddr),
        .cfg_rstreq_md      (cfg_rstreq_md),

		.present_proc_b		(PRESENT_PROC_B),

        // DUT Reset outputs.
        //
        .dut_poreset_b      (DUT_PORESET_B),
        .int_poreset_b      (int_poreset_b),            // Internal version of above.
        .dut_hreset_b       (DUT_HRESET_B),
        .int_hreset_b       (int_hreset_b),
        .int_reset_req_b    (int_reset_req_b),
        .dut_trst_b         (DUT_TRST_B),

        // System Reset outputs.
        //
		.rst_clocks_b		(CLK_SI5341B_RST_B),
        .rst_emmc_b         (RST_EMMC_B), 
        .rst_i2c_b          (RST_I2C_B), 
        .rst_ieee_b         (RST_IEEE_B),
        .rst_mem_b          (RST_MEM_B), 
        .rst_phy1_b         (RST_PHY1_B), 
        .rst_phy2_b         (RST_PHY2_B), 
        .rst_phy3_b         (RST_PHY3_B), 
        .rst_retimer_b      (RST_RETIMER_B),
        .rst_slot1_b        (RST_SLOT1_B),
        .rst_slot2_b        (RST_SLOT2_B),
        .rst_slot3_b        (RST_SLOT3_B),
        .rst_tdm_b          (RST_TDM_B),

		// Config control
		//
        .cfg_drv            (CFG_DRV),
        .int_cfg_drv_b      (int_cfg_drv_b),
        .rst_cfg_regs_b     (rst_cfg_regs_b),
        .rst_ctl_regs_b     (rst_ctl_regs_b), 

        // Status
        //
        .rst_in_rst		    (rst_in_rst), 
        .rst_state			(rst_state), 
        .rst_reason         (rst_reason),
		.rst_mon			(rst_mon),

        .disable_io_b       (disable_io_b),
        .rst_b              (rst_b), 
        .clk_10kHz_ena      (clk_10kHz_ena),
        .clk                (HOT_CLK)
    );


//---------------------------------------------------------------------------
// INPUT SOURCE 1:
//		IFC Interface
//
    ifc IFC0 ( 
        // IFC interface
        //
        .ifc_ad             (IFC_AD),
        .ifc_ale            (IFC_ALE),
        .ifc_bctl           (IFC_BCTL),
        .ifc_cle            (IFC_CLE),
        .ifc_clk            (IFC_CLK),
        .ifc_cs_b           (IFC_CS_B),
        .ifc_oe_b           (IFC_OE_B),
        .ifc_rb1_b          (IFC_RB1_B),
        .ifc_te             (IFC_TE),
        .ifc_we0_b          (IFC_WE0_B),
        .ifc_wp_b           (IFC_WP_B),

        .drv_ifc_a5         (CFG_IFC_A5),
        .drv_ifc_a1         (CFG_IFC_A1),
        .drv_ifc_a0         (CFG_IFC_A0),

        .nand_cs_b          (NAND_CS_B),

        // IFC register interface
        //
        .ifcreg_addr        (ifcreg_addr),
        .ifcreg_cs_b        (ifcreg_cs_b),
        .ifcreg_oe_b        (ifcreg_oe_b),
        .ifcreg_we_b        (ifcreg_we_b),
        .ifcreg_rdata       (reg_rdata),
        .ifcreg_wdata       (ifcreg_wdata),
		.ifcreg_drv_b		(reg_drv_b),

        // System configuration
        //
        .cpu_id             (CPUID),

        // Reset handling (due to config use).
        //
        .dut_reset_req_b    (DUT_RESET_REQ_B), 
        .int_reset_req_b    (int_reset_req_b), 
        .dut_asleep         (DUT_ASLEEP), 
        .int_asleep         (int_asleep), 

        // Config control inputs
        //
        .cfg_drv_b          (int_cfg_drv_b),
		.cfg_ifc_ad			(cfg_ifc_ad),
		.cfg_ifc_a5			(cfg_ifc_a5),
		.cfg_ifc_a1			(cfg_ifc_a1),
		.cfg_ifc_a0			(cfg_ifc_a0),
		.cfg_ifc_cle		(cfg_ifc_cle),
		.cfg_ifc_oe_b		(cfg_ifc_oe_b),
		.cfg_ifc_te			(cfg_ifc_te),
		.cfg_ifc_we0_b		(cfg_ifc_we0_b),
		.cfg_ifc_wp_b		(cfg_ifc_wp_b),
		.cfg_rsp_dis		(cfg_rsp_dis),

		// Monitoring
		//
		.mon_boot_b			(),

        // General
        //
        .disable_io_b       (disable_io_b),
        .rst_b              (rst_b),
        .clk                (HOT_CLK)
    );


//---------------------------------------------------------------------------
// INPUT SOURCE 2:
//---------------------------------------------------------------------------
// i2c_remote -- provide access to BCSRs over I2C, since IFC is not longer
//               available.
//               Uses the dual-address QX_I2CSlave IP used on most QDS systems.
//
    wire    [7:0]   i2c_addr, i2c_rdata, i2c_wdata;
    wire            i2c_rd,   i2c_rd_b;
    wire            i2c_wr,   i2c_we_b;
    wire            i2c_which_slv;
    wire            sda_drv,  sda_rcv,   scl_drv, scl_rcv;

    assign i2c_rd = ~i2c_rd_b;
    assign i2c_wr = ~i2c_we_b;

    i2c_remote I2CRMT (
        .i2c_addr       (i2c_addr),
        .is_i2c_ctl     (i2c_which_slv), 
        .i2c_rd         (i2c_rd), 
        .i2c_wr         (i2c_wr), 
        .i2c_rdata      (i2c_rdata), 
        .i2c_wdata      (i2c_wdata), 

        .reg_addr       (i2crmt_addr),
        .reg_cs_b       (i2crmt_cs_b),
        .reg_we_b       (i2crmt_we_b),
        .reg_rdata      (reg_rdata),
        .reg_wdata      (i2crmt_wdata),

        .activity_b     (),						// Changed to just monitor I2C1_SCL now.
        .rst_b          (rst_b), 
        .clk            (HOT_CLK)
    );

    QX_I2CSlave #(
        .CLK_FREQ       (CLK_FREQ),
        .EXT_ADDR       (0),
        .DUAL_ADDR      (1),
        .SIM_MODE       (SIM_MODE)
    ) I2CSLV (
        .clk            (HOT_CLK), 
        .rst_b          (rst_b), 
        .enable         (1'b1),                 // Always enabled.
        .slv_addr       (7'b110_0110),	    	// Pri: 0x66 + 0x67
        .slv_stat       (),

        .addr_sel       (i2c_which_slv),
        .ext_addr       (),
        .reg_addr       (i2c_addr),
        .pause_b        (1'b1),

        .reg_rdata      (i2c_rdata),
        .reg_rd_b       (i2c_rd_b),
        .reg_wdata      (i2c_wdata),
        .reg_we_b       (i2c_we_b),

        .sda_drv        (sda_drv),
        .sda_rcv        (sda_rcv),
        .scl_drv        (scl_drv),
        .scl_rcv        (scl_rcv)
    );

    assign  I2C1_SDA    = (sda_drv)     ? 1'b0 : 1'bZ;                 
    assign  sda_rcv     = I2C1_SDA;
    assign  I2C1_SCL    = (scl_drv)     ? 1'b0 : 1'bZ;                 
    assign  scl_rcv     = I2C1_SCL;


//###########################################################################
//								MUX
//###########################################################################
// BCSR access can be via IFC or I2C.  Only the DUT drives these, and it is
// up to SW not to do both at the same time (theoretically possible).
//
    wire	i2crmt_oe_b;

// If activity on IFC (rare), switch there else ignore it.
//
    assign  i2crmt_oe_b = ~(!i2crmt_cs_b  &&  i2crmt_we_b);

    assign  reg_addr    = (!ifcreg_cs_b)  ? ifcreg_addr[7:0] : i2crmt_addr;
    assign  reg_cs_b    = (!ifcreg_cs_b)  ? ifcreg_cs_b		 : i2crmt_cs_b;
    assign  reg_oe_b    = (!ifcreg_cs_b)  ? ifcreg_oe_b      : i2crmt_oe_b;
    assign  reg_we_b    = (!ifcreg_cs_b)  ? ifcreg_we_b      : i2crmt_we_b;
    assign  reg_wdata   = (!ifcreg_cs_b)  ? ifcreg_wdata     : i2crmt_wdata;

    assign  reg_drv_b   = !reg_cs_b  &&  !reg_oe_b  ? 1'b0 : 1'b1;


//###########################################################################
// OUTPUT 1:
//		BCSRs
//
    bcsrs BCSRs ( 
        // Register port
        //
        .reg_addr           (reg_addr),
        .reg_cs_b           (reg_cs_b),
        .reg_we_b           (reg_we_b),
        .reg_rdata          (reg_rdata),
        .reg_wdata          (reg_wdata),

        // Switch IO
        //
        .sw1                (SW1),
        .sw2                (SW2),
        .sw3                (SW3),
        .sw4                (SW4),
        .sw5                (SW5),
        .sw_auto_on         (sw_auto_on),           // Out to rest of system
        .sw_bypass_b        (sw_bypass_b),          // "
		.sw_debug			(sw_debug),

        // Reset monitoring.
        //
        .int_poreset_b      (int_poreset_b),
        .int_hreset_b       (int_hreset_b),
        .int_reset_req_b    (int_reset_req_b), 
        .int_asleep         (int_asleep), 
        .rst_reason         (rst_reason),

        // DUT config outputs
        //
		.cfg_testsel_b		(DUT_TESTSEL_B), 
		.dut_tbscan_en_b	(DUT_TBSCAN_EN_B),

        .cfg_drv_b          (int_cfg_drv_b),
		.cfg_ifc_ad			(cfg_ifc_ad),
		.cfg_ifc_a5			(cfg_ifc_a5),
		.cfg_ifc_a1			(cfg_ifc_a1),
		.cfg_ifc_a0			(cfg_ifc_a0),
		.cfg_ifc_cle		(cfg_ifc_cle),
		.cfg_ifc_oe_b		(cfg_ifc_oe_b),
		.cfg_ifc_te			(cfg_ifc_te),
		.cfg_ifc_we0_b		(cfg_ifc_we0_b),
		.cfg_ifc_wp_b		(cfg_ifc_wp_b),
		.cfg_rsp_dis		(cfg_rsp_dis),

		// System config
		//
		.cfg_cfg_wp			(CFG_CFG_WP),
        .cfg_ieee_clk       (CFG_IEEE_CLK), 
        .cfg_mux_i2c2       (CFG_MUX_I2C2), 
        .cfg_mux_i2c3       (CFG_MUX_I2C3), 
        .cfg_mux_sd         (CFG_MUX_SD), 
        .cfg_mux_spi        (CFG_MUX_SPI), 
		.cfg_pci1_wgig_en	(CFG_PCI1_WGIG_EN),
		.cfg_pci1_dis_b		(CFG_PCI1_DIS_B),
		.cfg_pci2_wgig_en	(CFG_PCI2_WGIG_EN),
		.cfg_pci2_dis_b		(CFG_PCI2_DIS_B),
		.cfg_qspi_map       (CFG_QSPI_MAP), 
		.cfg_rstreq_md      (cfg_rstreq_md), 
		.cfg_rst_noddr      (cfg_rst_noddr), 
		.cfg_sfp_txen_b     (CFG_SFP_TXEN_B), 
		.cfg_spread			(CFG_SPREAD), 
		.cfg_svdd_sel		(PS_SVDD_SEL), 
		.cfg_tabb_sel		(PS_TA_BB_SEL), 

		.int_cfg_mux_sd		(int_cfg_mux_sd),
        .cfg_evdd_force     (cfg_evdd_force),

        // Fault, Power.
        //
        .alarm              (alarm),
        .alarm_mask         (alarm_mask),
        .ctl_led            (ctl_led),
        .ctl_passfail       (ctl_passfail),
        .ctl_pwrled         (ctl_pwrled),
        .ctl_xtest          (cfg_xtest),

        .irq_stat0          (irq_stat0),
        .irq_stat1          (irq_stat1),

        .leds_user          (leds_user),

        .pwr_ctl2_pwr       (pwr_ctl2_pwr),
        .pwr_reason         (pwr_reason),
        .pwr_mstat          (pwr_mstat),
        .pwr_stat1          (pwr_stat1),
        .pwr_stat2          (pwr_stat2),

        // Reset forcing
        //
        .rst_rst            (rst_rst),					// RSTCTL[RST]
        .rst_force1         (rst_force1),
        .rst_force2         (rst_force2),
        .rst_force3         (rst_force3),
        .rst_mask1          (rst_mask1),
        .rst_mask2          (rst_mask2),
        .rst_mask3          (rst_mask3),

        // Config Controls
        //
        .rst_cfg_regs_b     (rst_cfg_regs_b),
        .rst_ctl_regs_b     (rst_ctl_regs_b), 
        .rcfg_go            (rcfg_go),
        .rcfg_wden          (rcfg_wden),
        .rcfg_watch         (rcfg_watch),

        // Monitoring
        //
        .present_proc_b     (PRESENT_PROC_B),
        .present_slot3_b    (PRESENT_SLOT3_B),
        .present_tdm_b      (PRESENT_TDM_B),
        .cpu_id             (CPUID),
		.brd_id				(BRD_ID),
		.los_retimer_b		(LOS_RETIMER_B),
		.retimer_done_b		(RETIMER_DONE_B),
		.sdslot_cd_b		(SDSLOT_CD_B),
		.sdslot_wp			(SDSLOT_WP),
		.sfp_tx_fault		(SFP_TX_FAULT),
		.sfp_rx_los			(SFP_RX_LOS),
        .pcb_rev            (PCB_REV), 

        // General
        //
        .disable_io_b       (disable_io_b),
        .rst_b              (rst_b),
        .clk                (HOT_CLK)
    );


//---------------------------------------------------------------------------
// Interrupts.
//
    interrupts IRQ ( 
        // Interrupt inputs.
        //
        .irq_tdm_b			(IRQ_TDM_B),
        .irq_aqr105_b		(IRQ_AQR105_B),
        .irq_qsphy1_b		(IRQ_QSPHY1_B),
        .irq_qsphy2_b		(IRQ_QSPHY2_B),
        .irq_rtc_b			(IRQ_RTC_B),
		.irq_pcie1wake_b	(IRQ_PCIE1WAKE_B),			// -PB only
		.irq_pcie2wake_b	(IRQ_PCIE2WAKE_B),			// "

		// Monitoring.
		//
		.irq_stat0			(irq_stat0),
		.irq_stat1			(irq_stat1),

        // Interrupt outputs to DUT.
        //
        .irq_b				(IRQ_B),
        .evt9_b				(EVT9_B),

        .disable_io_b		(disable_io_b)
    );


//---------------------------------------------------------------------------
// Unused/Debug/Test outputs.
//
	reg 	sel_xtest;
	
	always @*
		case (leds_user)
		default:	sel_xtest	<= cfg_xtest;						// TP177 (white)	0 = DEFAULT
		4'h1:		sel_xtest	<= ALT_HOT_CLK | IFC_CLK | IFC_BCTL | IFC_CS_B[1] 
								 | ROTATION_ERR_B;					// Junk
		4'd2:		sel_xtest	<= clk_10kHz_ena;
		4'd3:		sel_xtest	<= clk_100Hz_ena;
		4'h4:		sel_xtest	<= sps[0];
		4'h5:		sel_xtest	<= sps[1];

		4'h8:		sel_xtest	<= rst_mon[0];
		4'h9:		sel_xtest	<= rst_mon[1];
		4'hA:		sel_xtest	<= rst_mon[2];
		4'hB:		sel_xtest	<= rst_mon[3];
		4'hC:		sel_xtest	<= pwr_mon[0];
		4'hD:		sel_xtest	<= pwr_mon[1];
		4'hE:		sel_xtest	<= pwr_mon[2];
		4'hF:		sel_xtest	<= pwr_mon[3];
		endcase

	always @*
		CFG_XTEST	<=  (!is_pb_variant)	?	sel_xtest
					 :  (is_revb)			?	sel_xtest		// -PB Rev B: back to XTEST
					 :	(!sw_auto_on)		?	1'bZ
					 :	(auto_on_pulse)		?	1'b0
					 :							1'bZ;			// Z for wire-or

	assign	CLK_SI5341B_FDEC	= 1'b0;
	assign	CLK_SI5341B_FINC	= 1'b0;

endmodule
