/******************************************************************************
*
*       COPYRIGHT (c) 2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
* Filename:     freq.c
* Author:       Mark Jonas
* Revision:     1.0
*
* History:      2003-May-30: First release.
*
* Description:  Supplies functions to retrieve the clock speeds used in
*               MPC5200 and to the clock ration between XLB, IPBI and PCI.
*
* Notes:        
*
******************************************************************************/

#include "ppctypes.h"

#include "mpc5200.h"
#include "cdm.h"
#include "mm.h"

#include "freq.h"
#include "core.h"

/*
 * Frequency of the external quartz.
 */
#define QUARTZ_FREQ	33000000

/*
 * Global variables to store frequency status
 */
static unsigned int fvco = 0;
static unsigned int fcore = 0;
static unsigned int fxlb = 0;
static unsigned int fipbi = 0;
static unsigned int fpci = 0;
static unsigned int fusb = 0;
static unsigned int firda = 0;

/*
 * Clocks in each phase of the fractional dividers
 */
static const unsigned int fracdiv[] = {
	8, 9, 10, 11, 11, 11, 6, 7
};

/*
 * 2 x Multipy factors for the CPU clock
 * Note - divide by 2 to get the actual multiply factor
 */
static const unsigned cpumult2[] = {
	3, 2, 2, 2, 4, 4, 5, 9, 6, 11, 8, 10, 3, 12, 7, 0,
	6, 5, 13, 2, 14, 4, 15, 9, 0, 11, 8, 10, 16, 12, 7, 0
};

/*--------------------------------------------------------------------------
   Function    : void freqCalculate (void)
   Description : Calculate clock speeds. This function must be called every
                 time the settings of the Clock Distribution Module (CDM)
                 have been changed. Otherwise values returned by other
                 functions of this module might not reflect reality.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
void freqCalculate (void)
{
	cdm_regs *cdm;
	uint8 phase[4], phase_buf, i;

	cdm = (cdm_regs *) (readMBAR () + MBAR_CDM);

	/*
	 * Calculate VCO frequency
	 */
	if (cdm->rstcfg & 0x40) {
		fvco = QUARTZ_FREQ * 12;
	} else {
		fvco = QUARTZ_FREQ * 16;
	}	

	/*
	 * Calculate USB and IrDA frequencies
	 */	
	if (cdm->fd_enable == 0) {
		if (cdm->ext_48mhz_en & 0x01) {
			for(i = 0; i < 4; i++)
			{
				phase_buf = (uint8)((cdm->fd_counters & (0x7 << (i * 4))) >> (i * 4));
				if(phase_buf < 4)
					phase[i] = (uint8)(phase_buf + 0x8);
				else
					if(phase_buf < 6)	
						phase[i] = 11;
					else
						phase[i] = phase_buf;
			}
			firda = (fvco * 4) / (phase[0] + phase[1] + phase[2] + phase[3]);
		} else {
			firda = 0;
		}
		
		if (cdm->ext_48mhz_en & 0x02) {
			for(i = 0; i < 4; i++)
			{
				phase_buf = (uint8)((cdm->fd_counters & (0x7 << (i * 4))) >> (i * 4));
				if(phase_buf < 4)
					phase[i] = (uint8)(phase_buf + 8);
				else
					if(phase_buf < 6)	
						phase[i] = 11;
					else
						phase[i] = phase_buf;
			}
			fusb = (fvco * 4) / (phase[0] + phase[1] + phase[2] + phase[3]);			
		} else {
			fusb = 0;
		}
	} else {
		fusb = (4 * fvco) /
			(fracdiv[((cdm->fd_counters >> 12) & 7)] +
			fracdiv[((cdm->fd_counters >> 8) & 7)] + fracdiv[((cdm->fd_counters >> 4) & 7)] + fracdiv[((cdm->fd_counters) & 7)]);
		firda = fusb;		
	}

	/*
	 * Calculate XLB bus frequency
	 */
	if (cdm->rstcfg & 0x20) {
		fxlb = fvco / 8;
	} else {
		fxlb = fvco / 4;
	}

	/*
	 * Calculate Core frequency
	 */
	fcore = (fxlb * cpumult2[cdm->rstcfg & 0x1F]) / 2;

	/*
	 * Calculate IPBI bus frequency
	 */
	if (cdm->ipg_clk_sel) {
		fipbi = fxlb / 2;
	} else {
		fipbi = fxlb;
	}

	/*
	 * Calculate PCI bus frequency
	 */
	switch (cdm->pci_clk_sel) {
		case 0:
			fpci = fipbi;
			break;
			
		case 1:
			fpci = fipbi / 2;
			break;
		
		case 2:
			fpci = fipbi / 4;
			break;
			
		default:
			fpci = fipbi;
			break;
	}
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqQuartz (void)
   Description : Return frequency of the external quartz.
   Parameter(s): none
   Returns     : Quartz frequency.
  --------------------------------------------------------------------------*/
unsigned int freqQuartz (void)
{
	return QUARTZ_FREQ;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqVCO (void)
   Description : Return frequency of the system VCO.
   Parameter(s): none
   Returns     : System VCO frequency.
  --------------------------------------------------------------------------*/
unsigned int freqVCO (void)
{
	if (fvco == 0) {
		freqCalculate ();
	}

	return fvco;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqCore (void)
   Description : Return frequency of the core.
   Parameter(s): none
   Returns     : Core frequency.
  --------------------------------------------------------------------------*/
unsigned int freqCore (void)
{
	if (fcore == 0) {
		freqCalculate ();
	}
	
	return fcore;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqXLB (void)
   Description : Return frequency of the XLB bus.
   Parameter(s): none
   Returns     : XLB bus frequency.
  --------------------------------------------------------------------------*/
unsigned int freqXLB (void)
{
	if (fxlb == 0) {
		freqCalculate ();
	}
	
	return fxlb;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqIPBI (void)
   Description : Return frequency of the IPBI bus.
   Parameter(s): none
   Returns     : IPBI bus frequency.
  --------------------------------------------------------------------------*/
unsigned int freqIPBI (void)
{
	if (fipbi == 0) {
		freqCalculate ();
	}
	
	return fipbi;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqPCI (void)
   Description : Return frequency of the PCI bus.
   Parameter(s): none
   Returns     : PCI bus frequency.
  --------------------------------------------------------------------------*/
unsigned int freqPCI (void)
{
	if (fpci == 0) {
		freqCalculate ();
	}
	
	return fpci;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqUSB (void)
   Description : Return frequency of the USB module.
   Parameter(s): none
   Returns     : USB module frequency.
  --------------------------------------------------------------------------*/
unsigned int freqUSB (void)
{
	if (fusb == 0) {
		freqCalculate ();
	}
	
	return fusb;
}

/*--------------------------------------------------------------------------
   Function    : unsigned int freqIrDA (void)
   Description : Return frequency of the IrDA module.
   Parameter(s): none
   Returns     : IrDA module frequency.
  --------------------------------------------------------------------------*/
unsigned int freqIrDA (void)
{
	if (firda == 0) {
		freqCalculate ();
	}
	
	return firda;
}

/*--------------------------------------------------------------------------
   Function    : void freqSetRatio (int ratio)
   Description : Set frequency ratio XLB:IPBI:PCI.
   Parameter(s): ratio - Either FREQ_RATIO_442, FREQ_RATIO_441,
                   FREQ_RATIO_422 or FREQ_RATIO_421.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void freqSetRatio (int ratio)
{
	cdm_regs *cdm;
	mm_regs *mm;

	cdm = (cdm_regs *) (readMBAR () + MBAR_CDM);
	mm = (mm_regs *) (readMBAR () + MBAR_MM);

	switch (ratio) {
		case FREQ_RATIO_421:
	    	cdm->pci_clk_sel = 0x01;
			EIEIO;
		    cdm->ipg_clk_sel = 0x01;
			EIEIO;
			clrbit (mm->ipbi_wait_state_en, 0x0001);
			EIEIO;
			break;
			
		case FREQ_RATIO_422:
	    	cdm->pci_clk_sel = 0x00;
			EIEIO;
		    cdm->ipg_clk_sel = 0x01;
			EIEIO;
			clrbit (mm->ipbi_wait_state_en, 0x0001);
			EIEIO;
			break;
			
		case FREQ_RATIO_441:
			setbit (mm->ipbi_wait_state_en, 0x0001);
			EIEIO;
	    	cdm->pci_clk_sel = 0x02;
			EIEIO;
		    cdm->ipg_clk_sel = 0x00;
			EIEIO;
			break;
			
		case FREQ_RATIO_442:
			setbit (mm->ipbi_wait_state_en, 0x0001);
			EIEIO;
	    	cdm->pci_clk_sel = 0x01;
			EIEIO;
		    cdm->ipg_clk_sel = 0x00;
			EIEIO;
			break;
			
		default:
			break;
	}

	freqCalculate ();
}
