/******************************************************************************
*
*       COPYRIGHT (c) 2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
* Filename:     main.c
* Author:       Mark Jonas
* Revision:     1.0
*
* History:      2003-May-30: First release.
*
* Description:  Main program. Computes primes using the Sieve of Erastostenes
*               algorithm.
*
* Notes:        
*
******************************************************************************/

#include "ppctypes.h"

#include "stdlib.h"
#include "pscuart.h"
#include "time.h"

void initializeSieve (int primes[], int length);
void makeSieveOne (int primes[], int length, int prime);
void makeSieve (int primes[], int length);
void printPrimes (int primes[], int length);

#define NUM_PRIMES	100000

/*--------------------------------------------------------------------------
   Function    : int main (void)
   Description : Calculate and print all primes between 0 and NUM_PRIMES.
   Parameter(s): none
   Returns     : always 0
  --------------------------------------------------------------------------*/
int main (void)
{
	int primes[NUM_PRIMES];
	uint32 start, fini;

	pscuartInit (115200);	
	timeInit ();

	printf ("\n\nCalculating primes up to %d using the Sieve of Erastostenes algorithm ...\n\n", NUM_PRIMES);

	start = timeRead ();
	makeSieve (primes, NUM_PRIMES);
	fini = timeRead ();

	printPrimes (primes, NUM_PRIMES);
	
	printf ("\nTime elapsed: %d ms\n", fini-start);

	return 0;
}

/*--------------------------------------------------------------------------
   Function    : void initializeSieve (int primes[], int length)
   Description : Initialize the sieve array so the algorithm can start.
   Parameter(s): primes[] - Array which will hold the primes found. A prime
                   is marked with a 1, a non-prime with a 0.
                 length - Size of the array.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void initializeSieve (int primes[], int length)
{
	int i;
	
	primes[0] = 0;
	primes[1] = 0;
	
	for (i=2; i<length; i++) {
		primes[i] = 1;
	}
}

/*--------------------------------------------------------------------------
   Function    : void makeSieveOne (int primes[], int length, int prime)
   Description : Mark multiples of a number as non-primes with a 0.
   Parameter(s): primes[] - Array which will hold the primes found. A prime
                   is marked with a 1, a non-prime with a 0.
                 length - Size of the array.
                 prime - A known prime.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void makeSieveOne (int primes[], int length, int prime)
{
	int i;
	
	for (i=2*prime; i<length; i=i+prime) {
		primes[i] = 0;
	}
}

/*--------------------------------------------------------------------------
   Function    : void makeSieve (int primes[], int length)
   Description : Mark all non-primes with a 0.
   Parameter(s): primes[] - Array which will hold the primes found. A prime
                   is marked with a 1, a non-prime with a 0.
                 length - Size of the array.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void makeSieve (int primes[], int length)
{
	int i;
	
	initializeSieve (primes, length);
	
	for (i=2; i*i<length; i++) {
		if (primes[i] == 1) {
			makeSieveOne (primes, length, i);
		}
	}
}

/*--------------------------------------------------------------------------
   Function    : void printPrimes (int primes[], int length)
   Description : Print all primes found.
   Parameter(s): primes[] - Array which will hold the primes found. A prime
                   is marked with a 1, a non-prime with a 0.
                 length - Size of the array.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void printPrimes (int primes[], int length)
{
	int i;
	int count;
	
	count = 0;
	
	for (i=0; i<length; i++) {
		if (primes[i] == 1) {		
			printf ("%6d ", i);
			count++;
			if (count%10 == 0) {
				printf ("\n");
			}
		}
	}
	printf ("\n");
	printf ("\n %d primes found.\n", count);
}