/******************************************************************************
*
*       COPYRIGHT (c) 2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
* Filename:     pscuart.c
* Author:       Mark Jonas
* Revision:     1.0
*
* History:      2003-May-30: First release.
*
* Description:  Functions to use MPC5200 PSC1 in UART mode. Code can be easily
*               adjusted to work with any other PSC as well.
*
* Notes:        
*
******************************************************************************/

#include "ppctypes.h"

#include "mpc5200.h"
#include "gps.h"
#include "psc.h"

#include "freq.h"
#include "core.h"

#define NULL	((void *) 0)

/*
#define HARDWARE_FLOW_CTRL
#define RCV_ECHO
#define RCV_CR_TO_LF
#define SND_LF_TO_CRLF
*/

static psc_regs *psc = NULL;

/*--------------------------------------------------------------------------
   Function    : void pscuartInit (int baudRate)
   Description : Initialize PSC1 to UART mode.
   Parameter(s): baudrate - Baudrate to set.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void pscuartInit (int baudRate)
{
	gps_regs *gps;
	uint16 divider;

	gps = (gps_regs *) (readMBAR () + MBAR_GPS);
	psc = (psc_regs *) (readMBAR () + MBAR_PSC1);

	/*
	 * Set GPIO port mapping for PSC1 to UART
	 */
	gps->port_config = (gps->port_config & 0xFFFFFFF7UL) | 0x00000004UL;
	
	/*
	 * Calculate baud rate divider setting
	 */
	divider = (uint16) (((freqIPBI () / baudRate) + 16) / 32);

	psc->SICR = 0x00;			/* SICR: SIM2 = uart mode,dcd does not affect rx */
	psc->SR_CSR = 0xDD00;			/* CSR: RX/TX baud rate from timers */
	psc->CTUR = (uint8) ((divider >> 8) & 0xFF);	/* CTUR: divide counter upper byte */
	psc->CTLR = (uint8) (divider & 0xFF);	/* CTLR: divide counter lower byte */
	psc->CR = 0x20;				/* CR: reset RX and RXFIFO */
	psc->CR = 0x30;				/* CR: reset TX and TXFIFO */
	psc->CR = 0x40;				/* CR: reset error status */
	psc->CR = 0x50;				/* CR: reset break change int */
	psc->CR = 0x10;				/* CR: reset MR pointer */
	psc->IPCR_ACR = 0x00;			/* ACR: disable state change CTS/ DCD interrupts */
#if defined HARDWARE_FLOW_CTRL
	psc->ModeReg = 0xF3;			/* MR1: RX controls RTS, 8bit data, no parity */
	psc->ModeReg = 0x17;			/* MR2: normal mode, 1stop bit, CTS controls TX */
#else
	psc->ModeReg = 0x73;			/* MR1: 8bit data, no parity */
	psc->ModeReg = 0x07;			/* MR2: normal mode,1stop bit */
#endif
	psc->OP1 = 1;				/* OP1: enable RTS to send */
	psc->ISR_IMR = 0x0000;			/* IMR: Mask RxRDY and TxRDY from causing an interrupt */
	psc->RFALARM = 0xF8;			/* RFALARM: */
	psc->TFALARM = 0xF8;			/* TFALARM: */
	psc->CR = 0x05;				/* CR: enable TX and RX. */			
}

/*--------------------------------------------------------------------------
   Function    : int pscuartPollChar (void)
   Description : See if a character is available at the PSC UART and if so
                 retreive it.
   Parameter(s): none
   Returns     : -1, no character available or interface not initialized
                 else, the first character of the FIFO.
  --------------------------------------------------------------------------*/
int pscuartPollChar (void)
{
	char c;
	
	if (psc == NULL) {
		return -1;
	}
	
	if (((psc->SR_CSR) & 0x0100) != 0) {	/* receiver ready? */
		c = *((char *) &psc->RB_TB);	/* get character */

#ifdef RCV_CR_TO_LF
		if (c == '\r') {
			c = '\n';
		}
#endif


#ifdef RCV_ECHO
		pscuartPutChar (c);	
#endif
		return c;
	} else {
		return -1;			/* return error */
	}
}

/*--------------------------------------------------------------------------
   Function    : int pscuartGetChar (void)
   Description : Retrieve a character from the PSC UART. If none is
                 available wait for one.
   Parameter(s): none
   Returns     : -1, interface not initialized
                 else, the first character of the FIFO.
  --------------------------------------------------------------------------*/
int pscuartGetChar (void)
{
	char c;
	
	if (psc == NULL) {
		return -1;
	}
	
	while (((psc->SR_CSR) & 0x0100) == 0) {};	/* wait for receiver ready */
	c =  *((char *) &psc->RB_TB);		/* get character */
	
#ifdef RCV_CR_TO_LF
	if (c == '\r') {
		c = '\n';
	}
#endif

#ifdef RCV_ECHO
	pscuartPutChar (c);
	if (c == 0x08) {			/* Backspace */
		pscuartPutChar (' ');
		pscuartPutChar (c);
	}
#endif
	
	return c;
}

/*--------------------------------------------------------------------------
   Function    : void pscuartPutChar (int c)
   Description : Write a character to the PSC UART. If the FIFO is full,
                 wait until there is space.
   Parameter(s): c - Character to transmit.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void pscuartPutChar (int c)
{
	if (psc == NULL) {
		return;
	}

#ifdef SND_LF_TO_CRLF
	if ((char) c == '\n') {
		while (((psc->SR_CSR) & 0x0400) == 0) {};	/* wait for transmitter ready */
		*((char *) &psc->RB_TB) = '\r';	/* put character */
	}		
#endif
	
	while (((psc->SR_CSR) & 0x0400) == 0) {};	/* wait for transmitter ready */
	*((char *) &psc->RB_TB) = (char) c;		/* put character */
}
