/********************************************************************
 * File:		serial.c
 * Purpose:		Board specific routines for the Pictus EVB
 ********************************************************************/

#include "serial.h"
#include "typedefs.h"
#include "Pictus_Header_v1_09.h"

void ConfigureSciPins(void);
void Linflex_SetSpeed(void);


//*********************************************************
//	Initialise serial port for printf
//	No par / 8 data bits / 1 stop bit
//*********************************************************
void LinFlex_0_Init( void )
{   
   ConfigureSciPins();

   // Enter initialization mode by setting INIT bit
   LINFLEX_0.LINCR1.R = 0x1;

   LINFLEX_0.UARTCR.B.UART = 0x1;  //Linflex working in UART mode

   LINFLEX_0.UARTCR.B.TXEN = 0x1;    // Enable transmission of data now
   LINFLEX_0.UARTCR.B.RXEN = 0x1;    //Receiver enabled
   LINFLEX_0.UARTCR.B.WL   = 0x1;    //8 bit data
   LINFLEX_0.UARTCR.B.PCE  = 0x0;    //No parity

   Linflex_SetSpeed();
  
   // Leave initialization mode by clearing INIT bit
   LINFLEX_0.LINCR1.R = 0x0;	
}


/********************************************************************
 *	SCI_A
 ********************************************************************/
char sci_a_getchar (void)
{
	char cDummy;
   // Wait for the reception to complete   
	while(LINFLEX_0.UARTSR.B.DRF == 0) {
		asm("nop");
	}
   
	cDummy = (uint8_t) LINFLEX_0.BDRM.R;  // Retrieve data received from LinFlex	

   // Clear RMB (Release Message Buffer) and DRF (Data Reception Completed)
   //  flags, which are set when receiving data in UART mode
   LINFLEX_0.UARTSR.B.RMB = 1;
   LINFLEX_0.UARTSR.B.DRF = 1;
   return cDummy;
}

/********************************************************************/
void sci_a_putchar (char ch)
{
	LINFLEX_0.UARTCR.B.TXEN = 1;
	LINFLEX_0.BDRL.R = (uint32_t) ch; // Start transmission by writing the data 
	
	// Wait for the transmission to complete
	while (LINFLEX_0.UARTSR.B.DTF == 0) {
		asm("nop");
	}
	
	LINFLEX_0.UARTSR.B.DTF = 1;		//clear the trasmission completed flag	

	LINFLEX_0.UARTCR.B.TXEN = 0;	// Disable transmission
}
/********************************************************************/

void ConfigureSciPins( void )
{
   // TX_A output
   /* set PA=1, OBE=1, IBE=0, ODE=0, HYS=0, SRC=0, WPE=0, WPS=0 */
   SIU.PCR[PAD_LIN0_TXD].B.PA  =0x1;
   SIU.PCR[PAD_LIN0_TXD].B.OBE =0x1;
	 
   // RX_A input
   /* set PA=1, OBE=0, IBE=1, ODE=0, HYS=0, SRC=0, WPE=0, WPS=0 */
   SIU.PCR[PAD_LIN0_RXD].B.PA  =0x1;
   SIU.PCR[PAD_LIN0_RXD].B.IBE =0x1;	
}

void Linflex_SetSpeed( void )
{
  
#if (FSYS == FSYS_08_MHz)    
#if (UART_BAUDRATE == UART_9600_BAUDRATE)    
   /* baudrate = 9600 with Fsys = 8Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x34;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x1;			
#endif 
#if (UART_BAUDRATE == UART_19200_BAUDRATE)    
   /* baudrate = 19200 with Fsys = 8Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x1A;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x1;			
#endif 
#if (UART_BAUDRATE == UART_38400_BAUDRATE)    
   /* baudrate = 38400 with Fsys = 8Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0xD;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x0;			
#endif 
#if (UART_BAUDRATE == UART_115200_BAUDRATE)    
   /* baudrate = 115200 with Fsys = 8Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x4;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x5;			
#endif   
#endif

   
#if (FSYS == FSYS_16_MHz)    
#if (UART_BAUDRATE == UART_9600_BAUDRATE)    
   /* baudrate = 9600 with Fsys = 16Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x68;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x3;			
#endif 
#if (UART_BAUDRATE == UART_19200_BAUDRATE)    
   /* baudrate = 19200 with Fsys = 16Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x34;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x1;			
#endif 
#if (UART_BAUDRATE == UART_38400_BAUDRATE)    
   /* baudrate = 38400 with Fsys = 16Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x1A;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x1;			
#endif 
#if (UART_BAUDRATE == UART_115200_BAUDRATE)    
   /* baudrate = 115200 with Fsys = 16Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x8;
   LINFLEX_0.LINFBRR.B.DIV_F = 0xB;			
#endif 
#endif

#if (FSYS == FSYS_32_MHz)    
#if (UART_BAUDRATE == UART_9600_BAUDRATE)    
   /* baudrate = 9600 with Fsys = 32Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0xD0;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x5;			
#endif 
#if (UART_BAUDRATE == UART_19200_BAUDRATE)    
   /* baudrate = 19200 with Fsys = 32Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x68;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x3;			
#endif 
#if (UART_BAUDRATE == UART_38400_BAUDRATE)    
   /* baudrate = 38400 with Fsys = 32Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x34;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x1;			
#endif 
#if (UART_BAUDRATE == UART_115200_BAUDRATE)    
   /* baudrate = 115200 with Fsys = 32Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x11;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x6;			
#endif    
#endif

#if (FSYS == FSYS_40_MHz)    
#if (UART_BAUDRATE == UART_9600_BAUDRATE)    
   /* baudrate = 9600 with Fsys = 40Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x104;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x7;			
#endif 
#if (UART_BAUDRATE == UART_19200_BAUDRATE)    
   /* baudrate = 19200 with Fsys = 40Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x82;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x3;
#endif 
#if (UART_BAUDRATE == UART_38400_BAUDRATE)    
   /* baudrate = 38400 with Fsys = 40Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x41;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x4;
#endif 
#if (UART_BAUDRATE == UART_115200_BAUDRATE)    
   /* baudrate = 115200 with Fsys = 40Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x15;
   LINFLEX_0.LINFBRR.B.DIV_F = 0xB;
#endif 
#endif
   
#if (FSYS == FSYS_64_MHz)    
#if (UART_BAUDRATE == UART_9600_BAUDRATE)    
   /* baudrate = 9600 with Fsys = 64Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x1A0;
   LINFLEX_0.LINFBRR.B.DIV_F = 0xB;
#endif 
#if (UART_BAUDRATE == UART_19200_BAUDRATE)    
   /* baudrate = 19200 with Fsys = 64Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0xD0;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x5;
#endif 
#if (UART_BAUDRATE == UART_38400_BAUDRATE)    
   /* baudrate = 38400 with Fsys = 64Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x68;
   LINFLEX_0.LINFBRR.B.DIV_F = 0x3;
#endif 
#if (UART_BAUDRATE == UART_115200_BAUDRATE)    
   /* baudrate = 115200 with Fsys = 64Mz  */
   LINFLEX_0.LINIBRR.B.DIV_M = 0x22;
   LINFLEX_0.LINFBRR.B.DIV_F = 0xC;
#endif 
#endif
    
}
