/*
 * File:		mcf5249.h
 * Purpose:		MCF5249 definitions
 *
 * Notes:		Based on Rev 0.3 of the MCF5249 User's Manual
 *	
 */

#ifndef _CPU_MCF5249_H
#define _CPU_MCF5249_H

/********************************************************************/
/*
 * The basic data types
 */

typedef unsigned char		uint8;  /*  8 bits */
typedef unsigned short int	uint16; /* 16 bits */
typedef unsigned long int	uint32; /* 32 bits */

typedef signed char			int8;   /*  8 bits */
typedef signed short int	int16;  /* 16 bits */
typedef signed long int		int32;  /* 32 bits */

typedef volatile uint8		vuint8;  /*  8 bits */
typedef volatile uint16		vuint16; /* 16 bits */
typedef volatile uint32		vuint32; /* 32 bits */

/********************************************************************/
/*
 * Memory map definitions from linker command files
 */
extern uint8 *__MBAR;
extern uint8 __MBAR2[];
extern uint8 __SRAM0[];
extern uint8 __SRAM1[];

/********************************************************************/
/*
 * Functions provided by mcf5249_lo.s
 */
int 	asm_set_ipl (uint32);
void	mcf5249_wr_vbr (uint32);
void	mcf5249_wr_cacr (uint32);
void	mcf5249_wr_acr0 (uint32);
void	mcf5249_wr_acr1 (uint32);
void	mcf5249_wr_sr (uint32);

/********************************************************************/
/*
 * Misc. Defines
 */

#ifdef	FALSE
#undef	FALSE
#endif
#define FALSE	(0)

#ifdef	TRUE
#undef	TRUE
#endif
#define	TRUE	(1)

#ifdef	NULL
#undef	NULL
#endif
#define NULL	(0)

/********************************************************************/
/*
 * Common M68K & ColdFire definitions
 */

#define ADDRESS			uint32
#define INSTRUCTION		uint16
#define ILLEGAL			0x4AFC
#define CPU_WORD_SIZE	16

/********************************************************************/
/*
 * Routines and macros for accessing Input/Output devices
 */

#define cpu_iord_8(ADDR)        *((vuint8 *)(ADDR))
#define cpu_iord_16(ADDR)       *((vuint16 *)(ADDR))
#define cpu_iord_32(ADDR)       *((vuint32 *)(ADDR))

#define cpu_iowr_8(ADDR,DATA)   *((vuint8 *)(ADDR)) = (DATA)
#define cpu_iowr_16(ADDR,DATA)  *((vuint16 *)(ADDR)) = (DATA)
#define cpu_iowr_32(ADDR,DATA)  *((vuint32 *)(ADDR)) = (DATA)

/********************************************************************/

#define MCF5200_SR_T		(0x8000)
#define MCF5200_SR_S		(0x2000)
#define MCF5200_SR_M		(0x1000)
#define MCF5200_SR_IPL		(0x0700)
#define MCF5200_SR_IPL_0	(0x0000)
#define MCF5200_SR_IPL_1	(0x0100)
#define MCF5200_SR_IPL_2	(0x0200)
#define MCF5200_SR_IPL_3	(0x0300)
#define MCF5200_SR_IPL_4	(0x0400)
#define MCF5200_SR_IPL_5	(0x0500)
#define MCF5200_SR_IPL_6	(0x0600)
#define MCF5200_SR_IPL_7	(0x0700)
#define MCF5200_SR_X		(0x0010)
#define MCF5200_SR_N		(0x0008)
#define MCF5200_SR_Z		(0x0004)
#define MCF5200_SR_V		(0x0002)
#define MCF5200_SR_C		(0x0001)

/********************************************************************/

/*
 * The ColdFire family of processors has a simplified exception stack
 * frame that looks like the following:
 *
 *           8 +----------------+----------------+
 *             |         Program Counter         |
 *           4 +----------------+----------------+
 *             |FS/Fmt/Vector/FS|      SR        |
 *   SP -->  0 +----------------+----------------+
 *
 * The stack self-aligns to a 4-byte boundary at an exception, with
 * the FS/Fmt/Vector/FS field indicating the size of the adjustment
 * (SP += 0,1,2,3 bytes).
 */

#define MCF5200_RD_SF_FORMAT(PTR)	\
	((*((uint16 *)(PTR)) >> 12) & 0x00FF)

#define MCF5200_RD_SF_VECTOR(PTR)	\
	((*((uint16 *)(PTR)) >>  2) & 0x00FF)

#define MCF5200_RD_SF_FS(PTR)		\
	( ((*((uint16 *)(PTR)) & 0x0C00) >> 8) | (*((uint16 *)(PTR)) & 0x0003) )

#define MCF5200_SF_SR(PTR)	*((uint16 *)(PTR)+1)
#define MCF5200_SF_PC(PTR)	*((uint32 *)(PTR)+1)

/********************************************************************
* 
* Local Memories
*
*********************************************************************/
 
/* Bit level definitions and macros */
#define MCF5249_CACR_CENB		(0x80000000)
#define MCF5249_CACR_CPDI		(0x10000000)
#define MCF5249_CACR_CFRZ		(0x08000000)
#define MCF5249_CACR_CINV		(0x01000000)
#define MCF5249_CACR_CEIB		(0x00000400)
#define MCF5249_CACR_DCM		(0x00000200)
#define MCF5249_CACR_DBWE		(0x00000100)
#define MCF5249_CACR_DWP		(0x00000040)
#define MCF5249_CACR_CLNF_00	(0x00000000)
#define MCF5249_CACR_CLNF_01	(0x00000002)
#define MCF5249_CACR_CLNF_10	(0x00000004)
#define MCF5249_CACR_CLNF_11	(0x00000006)

#define MCF5249_ACR_AB(a)		((a)&0xFF000000)
#define MCF5249_ACR_AM(a)		(((a)&0xFF000000) >> 8)
#define MCF5249_ACR_EN			(0x00008000)
#define MCF5249_ACR_SM_USER		(0x00000000)
#define MCF5249_ACR_SM_SUPER	(0x00002000)
#define MCF5249_ACR_SM_IGNORE	(0x00006000)
#define MCF5249_ACR_ENIB		(0x00000080)
#define MCF5249_ACR_CM			(0x00000040)
#define MCF5249_ACR_BWE			(0x00000020)
#define MCF5249_ACR_WP			(0x00000004)

#define MCF5249_RAMBAR_BA(a)	((a)&0xFFFFC000)
#define MCF5249_RAMBAR_PRI_00	(0x00000000)
#define MCF5249_RAMBAR_PRI_01	(0x00004000)
#define MCF5249_RAMBAR_PRI_10	(0x00008000)
#define MCF5249_RAMBAR_PRI_11	(0x0000C000)
#define MCF5249_RAMBAR_WP		(0x00000100)
#define MCF5249_RAMBAR_CI		(0x00000020)
#define MCF5249_RAMBAR_SC		(0x00000010)
#define MCF5249_RAMBAR_SD		(0x00000008)
#define MCF5249_RAMBAR_UC		(0x00000004)
#define MCF5249_RAMBAR_UD		(0x00000002)
#define MCF5249_RAMBAR_V		(0x00000001)

/*********************************************************************
*
* System Configuration Registers (SIM)
*
*********************************************************************/

#define	MCF5249_SIM_RSR				(*(vuint8  *)(void *)(&__MBAR[0x00]))
#define	MCF5249_SIM_SYPCR			(*(vuint8  *)(void *)(&__MBAR[0x01]))
#define	MCF5249_SIM_SWIVR			(*(vuint8  *)(void *)(&__MBAR[0x02]))
#define	MCF5249_SIM_SWSR			(*(vuint8  *)(void *)(&__MBAR[0x03]))
#define	MCF5249_SIM_IRQPAR			(*(vuint8  *)(void *)(&__MBAR[0x06]))
#define	MCF5249_SIM_MPARK			(*(vuint8  *)(void *)(&__MBAR[0x0C]))
#define	MCF5249_SIM_IPR				(*(vuint32 *)(void *)(&__MBAR[0x40]))
#define	MCF5249_SIM_IMR				(*(vuint32 *)(void *)(&__MBAR[0x44]))
									
#define	MCF5249_SIM_AVCR			(*(vuint8  *)(void *)(&__MBAR[0x51]))
#define	MCF5249_SIM_ICR0			(*(vuint8  *)(void *)(&__MBAR[0x4C]))
#define	MCF5249_SIM_ICR1			(*(vuint8  *)(void *)(&__MBAR[0x4D]))
#define	MCF5249_SIM_ICR2			(*(vuint8  *)(void *)(&__MBAR[0x4E]))
#define	MCF5249_SIM_ICR3			(*(vuint8  *)(void *)(&__MBAR[0x4F]))
#define	MCF5249_SIM_ICR4			(*(vuint8  *)(void *)(&__MBAR[0x50]))
#define	MCF5249_SIM_ICR5			(*(vuint8  *)(void *)(&__MBAR[0x51]))
#define	MCF5249_SIM_ICR6			(*(vuint8  *)(void *)(&__MBAR[0x52]))
#define	MCF5249_SIM_ICR7			(*(vuint8  *)(void *)(&__MBAR[0x53]))
#define	MCF5249_SIM_ICR8			(*(vuint8  *)(void *)(&__MBAR[0x54]))
#define	MCF5249_SIM_ICR9			(*(vuint8  *)(void *)(&__MBAR[0x55]))

#define	MCF5249_SIM_GPIO_READ		(*(vuint32 *)(void *)(&__MBAR2[0x00]))
#define	MCF5249_SIM_GPIO_OUT		(*(vuint32 *)(void *)(&__MBAR2[0x04]))
#define	MCF5249_SIM_GPIO_ENABLE		(*(vuint32 *)(void *)(&__MBAR2[0x08]))
#define	MCF5249_SIM_GPIO_FUNC		(*(vuint32 *)(void *)(&__MBAR2[0x0C]))
																			
#define	MCF5249_SIM_DEVID			(*(vuint32 *)(void *)(&__MBAR2[0xAC]))
																			
#define	MCF5249_SIM_GPIO1_READ		(*(vuint32 *)(void *)(&__MBAR2[0xB0]))
#define	MCF5249_SIM_GPIO1_OUT		(*(vuint32 *)(void *)(&__MBAR2[0xB4]))
#define	MCF5249_SIM_GPIO1_ENABLE	(*(vuint32 *)(void *)(&__MBAR2[0xB8]))
#define	MCF5249_SIM_GPIO1_FUNC		(*(vuint32 *)(void *)(&__MBAR2[0xBC]))
#define MCF5249_SIM_GPIO_INT_STAT	(*(vuint32 *)(void *)(&__MBAR2[0xC0]))
#define MCF5249_SIM_GPIO_INT_CLEAR	(*(vuint32 *)(void *)(&__MBAR2[0xC0]))
#define MCF5249_SIM_GPIO_INT_EN		(*(vuint32 *)(void *)(&__MBAR2[0xC4]))

#define MCF5249_SIM_INTERRUPT_STAT3	(*(vuint32 *)(void *)(&__MBAR2[0xE0]))
#define MCF5249_SIM_INTERRUPT_CLR3	(*(vuint32 *)(void *)(&__MBAR2[0xE0]))
#define MCF5249_SIM_INTERRUPT_EN3	(*(vuint32 *)(void *)(&__MBAR2[0xE4]))

#define	MCF5249_SIM_INTLEV1 		(*(vuint32 *)(void *)(&__MBAR2[0x140]))
#define	MCF5249_SIM_INTLEV2 		(*(vuint32 *)(void *)(&__MBAR2[0x144]))
#define	MCF5249_SIM_INTLEV3 		(*(vuint32 *)(void *)(&__MBAR2[0x148]))
#define	MCF5249_SIM_INTLEV4 		(*(vuint32 *)(void *)(&__MBAR2[0x14C]))
#define	MCF5249_SIM_INTLEV5 		(*(vuint32 *)(void *)(&__MBAR2[0x150]))
#define	MCF5249_SIM_INTLEV6 		(*(vuint32 *)(void *)(&__MBAR2[0x154]))
#define	MCF5249_SIM_INTLEV7 		(*(vuint32 *)(void *)(&__MBAR2[0x158]))
#define	MCF5249_SIM_INTLEV8 		(*(vuint32 *)(void *)(&__MBAR2[0x15C]))
#define	MCF5249_SIM_SPURVEC 		(*(vuint32 *)(void *)(&__MBAR2[0x164]))
#define	MCF5249_SIM_INTBASE 		(*(vuint8 *)(void *)(&__MBAR2[0x16B]))

/* Bit level definitions and macros */
#define MCF5249_SIM_RSR_HRST		(0x80)
#define MCF5249_SIM_RSR_SWTR		(0x20)

#define MCF5249_SIM_SYPCR_SWE		(0x80)
#define MCF5249_SIM_SYPCR_SWRI		(0x40)
#define MCF5249_SIM_SYPCR_SWT_2_9	(0x00)
#define MCF5249_SIM_SYPCR_SWT_2_11	(0x08)
#define MCF5249_SIM_SYPCR_SWT_2_13	(0x10)
#define MCF5249_SIM_SYPCR_SWT_2_15	(0x18)
#define MCF5249_SIM_SYPCR_SWT_2_18	(0x20)
#define MCF5249_SIM_SYPCR_SWT_2_20	(0x28)
#define MCF5249_SIM_SYPCR_SWT_2_22	(0x30)
#define MCF5249_SIM_SYPCR_SWT_2_24	(0x38)
#define MCF5249_SIM_SYPCR_SWTA		(0x04)
#define MCF5249_SIM_SYPCR_SWTAVAL	(0x02)

#define MCF5249_SIM_SWSR_55			(0x55)
#define MCF5249_SIM_SWSR_AA			(0xaa)
	
#define MCF5249_MBAR_BA(a)		((a)&0xFFFFF000)
#define MCF5249_MBAR_WP			(0x00000100)
#define MCF5249_MBAR_AM			(0x00000040)
#define MCF5249_MBAR_CI			(0x00000020)
#define MCF5249_MBAR_SC			(0x00000010)
#define MCF5249_MBAR_SD			(0x00000008)
#define MCF5249_MBAR_UC			(0x00000004)
#define MCF5249_MBAR_UD			(0x00000002)
#define MCF5249_MBAR_V			(0x00000001)

#define MCF5249_MBAR2_BA(a)		((a)&0xC0000000)
#define MCF5249_MBAR2_LS7		(0x00000080)
#define MCF5249_MBAR2_LS6		(0x00000040)
#define MCF5249_MBAR2_LS5		(0x00000020)
#define MCF5249_MBAR2_LS4		(0x00000010)
#define MCF5249_MBAR2_LS3		(0x00000008)
#define MCF5249_MBAR2_LS2		(0x00000004)
#define MCF5249_MBAR2_LS1		(0x00000002)
#define MCF5249_MBAR2_V			(0x00000001)

#define MCF5249_SIM_ICR_AVEC		(0x80)    
#define MCF5249_SIM_ICR_IL(a)		(((a)&0x07)<<2)   
#define MCF5249_SIM_ICR_IP_00		(0x00)
#define MCF5249_SIM_ICR_IP_01		(0x01)
#define MCF5249_SIM_ICR_IP_10		(0x02)
#define MCF5249_SIM_ICR_IP_11		(0x03)

#define MCF5249_SIM_IPR_DMA3		(0x00020000)
#define MCF5249_SIM_IPR_DMA2		(0x00010000)
#define MCF5249_SIM_IPR_DMA1		(0x00008000)
#define MCF5249_SIM_IPR_DMA0		(0x00004000)
#define MCF5249_SIM_IPR_UART1		(0x00002000)
#define MCF5249_SIM_IPR_UART0		(0x00001000)
#define MCF5249_SIM_IPR_MBUS		(0x00000800)
#define MCF5249_SIM_IPR_TIMER1		(0x00000400)
#define MCF5249_SIM_IPR_TIMER0		(0x00000200)
#define MCF5249_SIM_IPR_SWT			(0x00000100)
#define MCF5249_SIM_IPR_EINT7		(0x00000080)
#define MCF5249_SIM_IPR_EINT6		(0x00000040)
#define MCF5249_SIM_IPR_EINT5		(0x00000020)
#define MCF5249_SIM_IPR_EINT4		(0x00000010)
#define MCF5249_SIM_IPR_EINT3		(0x00000008)
#define MCF5249_SIM_IPR_EINT2		(0x00000004)
#define MCF5249_SIM_IPR_EINT1		(0x00000002)

#define MCF5249_SIM_IMR_DMA3		(0x00020000)
#define MCF5249_SIM_IMR_DMA2		(0x00010000)
#define MCF5249_SIM_IMR_DMA1		(0x00008000)
#define MCF5249_SIM_IMR_DMA0		(0x00004000)
#define MCF5249_SIM_IMR_UART1		(0x00002000)
#define MCF5249_SIM_IMR_UART0		(0x00001000)
#define MCF5249_SIM_IMR_MBUS		(0x00000800)
#define MCF5249_SIM_IMR_TIMER1		(0x00000400)
#define MCF5249_SIM_IMR_TIMER0		(0x00000200)
#define MCF5249_SIM_IMR_SWT			(0x00000100)
#define MCF5249_SIM_IMR_EINT7		(0x00000080)
#define MCF5249_SIM_IMR_EINT6		(0x00000040)
#define MCF5249_SIM_IMR_EINT5		(0x00000020)
#define MCF5249_SIM_IMR_EINT4		(0x00000010)
#define MCF5249_SIM_IMR_EINT3		(0x00000008)
#define MCF5249_SIM_IMR_EINT2		(0x00000004)
#define MCF5249_SIM_IMR_EINT1		(0x00000002)

#define MCF5249_SIM_AVCR_AVEC7		(0x80)		
#define MCF5249_SIM_AVCR_AVEC6		(0x40)		
#define MCF5249_SIM_AVCR_AVEC5		(0x20)		
#define MCF5249_SIM_AVCR_AVEC4		(0x10)		
#define MCF5249_SIM_AVCR_AVEC3		(0x08)		
#define MCF5249_SIM_AVCR_AVEC2		(0x04)		
#define MCF5249_SIM_AVCR_AVEC1		(0x02)		
#define MCF5249_SIM_AVCR_BLK		(0x01)

#define MCF5249_SIM_DEVID_PART(a)	((a & 0xFFFFFF00) >> 8)
#define MCF5249_SIM_DEVID_MASK(a)	(a & 0x000000FF)


/**********************************************************************
*
* Phase-Locked Loop and Clock Dividers
*
***********************************************************************/

/* Offsets of the registers from the MBAR */
#define MCF5249_PLL_PLLCR			(*(vuint32 *)(void *)(&__MBAR2[0x180]))

/* Bit level definitions and macros */
#define MCF5249_PLL_PLLCR_LOCK		(0x80000000)	

/**********************************************************************
*
* Chip Select Registers
*
***********************************************************************/

/* Offsets of the registers from the MBAR */
#define	MCF5249_CS_CSAR0			(*(vuint16 *)(void *)(&__MBAR[0x080]))
#define	MCF5249_CS_CSMR0			(*(vuint32 *)(void *)(&__MBAR[0x084]))
#define	MCF5249_CS_CSCR0			(*(vuint16 *)(void *)(&__MBAR[0x08A]))
#define	MCF5249_CS_CSAR1			(*(vuint16 *)(void *)(&__MBAR[0x08C]))
#define	MCF5249_CS_CSMR1			(*(vuint32 *)(void *)(&__MBAR[0x090]))
#define	MCF5249_CS_CSCR1			(*(vuint16 *)(void *)(&__MBAR[0x096]))

/* Bit level definitions and macros */
#define MCF5249_CS_CSAR_BA(a)		(((a)&0xFFFF0000)>>16)

#define MCF5249_CS_CSMR_BAM_4G		(0xFFFF0000)
#define MCF5249_CS_CSMR_BAM_2G		(0x7FFF0000)
#define MCF5249_CS_CSMR_BAM_1G		(0x3FFF0000)
#define MCF5249_CS_CSMR_BAM_512M	(0x1FFF0000)
#define MCF5249_CS_CSMR_BAM_256M	(0x0FFF0000)
#define MCF5249_CS_CSMR_BAM_128M	(0x07FF0000)
#define MCF5249_CS_CSMR_BAM_64M		(0x03FF0000)
#define MCF5249_CS_CSMR_BAM_32M		(0x01FF0000)
#define MCF5249_CS_CSMR_BAM_16M		(0x00FF0000)
#define MCF5249_CS_CSMR_BAM_8M		(0x007F0000)
#define MCF5249_CS_CSMR_BAM_4M		(0x003F0000)
#define MCF5249_CS_CSMR_BAM_2M		(0x001F0000)
#define MCF5249_CS_CSMR_BAM_1M		(0x000F0000)
#define MCF5249_CS_CSMR_BAM_512K	(0x00070000)
#define MCF5249_CS_CSMR_BAM_256K	(0x00030000)
#define MCF5249_CS_CSMR_BAM_128K	(0x00010000)
#define MCF5249_CS_CSMR_BAM_64K		(0x00000000)

#define MCF5249_CS_CSMR_WP		(0x00000100)
#define MCF5249_CS_CSMR_AM		(0x00000040)
#define MCF5249_CS_CSMR_CI		(0x00000020)
#define MCF5249_CS_CSMR_SC		(0x00000010)
#define MCF5249_CS_CSMR_SD		(0x00000008)
#define MCF5249_CS_CSMR_UC		(0x00000004)
#define MCF5249_CS_CSMR_UD		(0x00000002)
#define MCF5249_CS_CSMR_V		(0x00000001)

#define MCF5249_CS_CSCR_WS(a)	((a & 0xF)<<10)
#define MCF5249_CS_CSCR_AA		(0x0100)
#define MCF5249_CS_CSCR_PS		(0x00C0)
#define MCF5249_CS_CSCR_BSTR	(0x0010)
#define MCF5249_CS_CSCR_BSTW	(0x0008)

/**********************************************************************
*
* Ports Registers Description
*
***********************************************************************/



/**********************************************************************
*
* QSPI Module Registers Description
*
***********************************************************************/

#define MCF5249_QSPI_QMR		(*(vuint32 *)(void *)(&__MBAR[0x400]))
#define MCF5249_QSPI_QDLYR		(*(vuint32 *)(void *)(&__MBAR[0x404]))
#define MCF5249_QSPI_QWR		(*(vuint32 *)(void *)(&__MBAR[0x408]))
#define MCF5249_QSPI_QIR		(*(vuint32 *)(void *)(&__MBAR[0x40C]))
#define MCF5249_QSPI_QAR		(*(vuint32 *)(void *)(&__MBAR[0x410]))
#define MCF5249_QSPI_QDR		(*(vuint32 *)(void *)(&__MBAR[0x414]))


/**********************************************************************
*
* DMA Module Registers Description
*
***********************************************************************/

#define	MCF5249_DMA_ROUTE		(*(vuint8  *)(void *)(&__MBAR2[0x188]))

#define MCF5249_DMA0_SAR		(*(vuint32 *)(void *)(&__MBAR[0x300]))	
#define MCF5249_DMA0_DAR		(*(vuint32 *)(void *)(&__MBAR[0x304]))	
#define MCF5249_DMA0_DCR		(*(vuint16 *)(void *)(&__MBAR[0x308]))	
#define MCF5249_DMA0_BCR		(*(vuint16 *)(void *)(&__MBAR[0x30C]))	
#define MCF5249_DMA0_DSR		(*(vuint8  *)(void *)(&__MBAR[0x310]))	
#define MCF5249_DMA0_DIVR		(*(vuint8  *)(void *)(&__MBAR[0x314]))	

#define MCF5249_DMA1_SAR		(*(vuint32 *)(void *)(&__MBAR[0x340]))	
#define MCF5249_DMA1_DAR		(*(vuint32 *)(void *)(&__MBAR[0x344]))	
#define MCF5249_DMA1_DCR		(*(vuint16 *)(void *)(&__MBAR[0x348]))	
#define MCF5249_DMA1_BCR		(*(vuint16 *)(void *)(&__MBAR[0x34C]))	
#define MCF5249_DMA1_DSR		(*(vuint8  *)(void *)(&__MBAR[0x350]))	
#define MCF5249_DMA1_DIVR		(*(vuint8  *)(void *)(&__MBAR[0x354]))	

#define MCF5249_DMA2_SAR		(*(vuint32 *)(void *)(&__MBAR[0x380]))	
#define MCF5249_DMA2_DAR		(*(vuint32 *)(void *)(&__MBAR[0x384]))	
#define MCF5249_DMA2_DCR		(*(vuint16 *)(void *)(&__MBAR[0x388]))	
#define MCF5249_DMA2_BCR		(*(vuint16 *)(void *)(&__MBAR[0x38C]))	
#define MCF5249_DMA2_DSR		(*(vuint8  *)(void *)(&__MBAR[0x390]))	
#define MCF5249_DMA2_DIVR		(*(vuint8  *)(void *)(&__MBAR[0x394]))	

#define MCF5249_DMA3_SAR		(*(vuint32 *)(void *)(&__MBAR[0x3C0]))	
#define MCF5249_DMA3_DAR		(*(vuint32 *)(void *)(&__MBAR[0x3C4]))	
#define MCF5249_DMA3_DCR		(*(vuint16 *)(void *)(&__MBAR[0x3C8]))	
#define MCF5249_DMA3_BCR		(*(vuint16 *)(void *)(&__MBAR[0x3CC]))	
#define MCF5249_DMA3_DSR		(*(vuint8  *)(void *)(&__MBAR[0x3D0]))	
#define MCF5249_DMA3_DIVR		(*(vuint8  *)(void *)(&__MBAR[0x3D4]))	

		
/**********************************************************************
*
* Mbus (IIC) Module Registers Description
*
***********************************************************************/

#define MCF5249_MBUS_MADR		(*(vuint8  *)(void *)(&__MBAR[0x280]))	
#define MCF5249_MBUS_MFDR		(*(vuint8  *)(void *)(&__MBAR[0x284]))	
#define MCF5249_MBUS_MBCR		(*(vuint8  *)(void *)(&__MBAR[0x288]))	
#define MCF5249_MBUS_MBSR		(*(vuint8  *)(void *)(&__MBAR[0x28C]))	
#define MCF5249_MBUS_MBDR		(*(vuint8  *)(void *)(&__MBAR[0x290]))	

#define MCF5249_MBUS2_MADR		(*(vuint8  *)(void *)(&__MBAR2[0x440]))	
#define MCF5249_MBUS2_MFDR		(*(vuint8  *)(void *)(&__MBAR2[0x444]))	
#define MCF5249_MBUS2_MBCR		(*(vuint8  *)(void *)(&__MBAR2[0x448]))	
#define MCF5249_MBUS2_MBSR		(*(vuint8  *)(void *)(&__MBAR2[0x44C]))	
#define MCF5249_MBUS2_MBDR		(*(vuint8  *)(void *)(&__MBAR2[0x450]))	

/**********************************************************************
*
* USART Module Registers Description
*
***********************************************************************/

#define MCF5249_UART0_UMR		(*(vuint8  *)(void *)(&__MBAR[0x1C0]))
#define MCF5249_UART0_USR		(*(vuint8  *)(void *)(&__MBAR[0x1C4]))
#define MCF5249_UART0_UCSR		(*(vuint8  *)(void *)(&__MBAR[0x1C4]))
#define MCF5249_UART0_UCR		(*(vuint8  *)(void *)(&__MBAR[0x1C8]))
#define MCF5249_UART0_URB		(*(vuint8  *)(void *)(&__MBAR[0x1CC]))
#define MCF5249_UART0_UTB		(*(vuint8  *)(void *)(&__MBAR[0x1CC]))
#define MCF5249_UART0_UIPCR		(*(vuint8  *)(void *)(&__MBAR[0x1D0]))
#define MCF5249_UART0_UACR		(*(vuint8  *)(void *)(&__MBAR[0x1D0]))
#define MCF5249_UART0_UISR		(*(vuint8  *)(void *)(&__MBAR[0x1D4]))
#define MCF5249_UART0_UIMR		(*(vuint8  *)(void *)(&__MBAR[0x1D4]))
#define MCF5249_UART0_UBG1		(*(vuint8  *)(void *)(&__MBAR[0x1D8]))
#define MCF5249_UART0_UBG2		(*(vuint8  *)(void *)(&__MBAR[0x1DC]))
#define MCF5249_UART0_UIVR		(*(vuint8  *)(void *)(&__MBAR[0x1F0]))
#define MCF5249_UART0_UIP		(*(vuint8  *)(void *)(&__MBAR[0x1F4]))
#define MCF5249_UART0_UOP1		(*(vuint8  *)(void *)(&__MBAR[0x1F8]))
#define MCF5249_UART0_UOP0		(*(vuint8  *)(void *)(&__MBAR[0x1FC]))

#define MCF5249_UART1_UMR		(*(vuint8  *)(void *)(&__MBAR[0x200]))
#define MCF5249_UART1_USR		(*(vuint8  *)(void *)(&__MBAR[0x204]))
#define MCF5249_UART1_UCSR		(*(vuint8  *)(void *)(&__MBAR[0x204]))
#define MCF5249_UART1_UCR		(*(vuint8  *)(void *)(&__MBAR[0x208]))
#define MCF5249_UART1_URB		(*(vuint8  *)(void *)(&__MBAR[0x20C]))
#define MCF5249_UART1_UTB		(*(vuint8  *)(void *)(&__MBAR[0x20C]))
#define MCF5249_UART1_UIPCR		(*(vuint8  *)(void *)(&__MBAR[0x210]))
#define MCF5249_UART1_UACR		(*(vuint8  *)(void *)(&__MBAR[0x210]))
#define MCF5249_UART1_UISR		(*(vuint8  *)(void *)(&__MBAR[0x214]))
#define MCF5249_UART1_UIMR		(*(vuint8  *)(void *)(&__MBAR[0x214]))
#define MCF5249_UART1_UBG1		(*(vuint8  *)(void *)(&__MBAR[0x218]))
#define MCF5249_UART1_UBG2		(*(vuint8  *)(void *)(&__MBAR[0x21C]))
#define MCF5249_UART1_UIVR		(*(vuint8  *)(void *)(&__MBAR[0x230]))
#define MCF5249_UART1_UIP		(*(vuint8  *)(void *)(&__MBAR[0x234]))
#define MCF5249_UART1_UOP1		(*(vuint8  *)(void *)(&__MBAR[0x238]))
#define MCF5249_UART1_UOP0		(*(vuint8  *)(void *)(&__MBAR[0x23C]))


/* Bit level definitions and macros */
#define MCF5249_UART_UMR1_RXRTS			(0x80)
#define MCF5249_UART_UMR1_RXIRQ			(0x40)
#define MCF5249_UART_UMR1_ERR			(0x20)
#define MCF5249_UART_UMR1_PM_MULTI_ADDR	(0x1C)
#define MCF5249_UART_UMR1_PM_MULTI_DATA	(0x18)
#define MCF5249_UART_UMR1_PM_NONE		(0x10)
#define MCF5249_UART_UMR1_PM_FORCE_HI	(0x0C)
#define MCF5249_UART_UMR1_PM_FORCE_LO	(0x08)
#define MCF5249_UART_UMR1_PM_ODD		(0x04)
#define MCF5249_UART_UMR1_PM_EVEN		(0x00)
#define MCF5249_UART_UMR1_BC_5			(0x00)
#define MCF5249_UART_UMR1_BC_6			(0x01)
#define MCF5249_UART_UMR1_BC_7			(0x02)
#define MCF5249_UART_UMR1_BC_8			(0x03)

#define MCF5249_UART_UMR2_CM_NORMAL	  	(0x00)
#define MCF5249_UART_UMR2_CM_ECHO	  	(0x40)
#define MCF5249_UART_UMR2_CM_LOCAL_LOOP	(0x80)
#define MCF5249_UART_UMR2_CM_REMOTE_LOOP	(0xC0)
#define MCF5249_UART_UMR2_TXRTS		 	(0x20)
#define MCF5249_UART_UMR2_TXCTS		 	(0x10)
#define MCF5249_UART_UMR2_STOP_BITS_1 	(0x07)
#define MCF5249_UART_UMR2_STOP_BITS_15	(0x08)
#define MCF5249_UART_UMR2_STOP_BITS_2 	(0x0F)
#define MCF5249_UART_UMR2_STOP_BITS(a)   ((a)&0x0f)

#define MCF5249_UART_USR_RB				(0x80)
#define MCF5249_UART_USR_FE				(0x40)
#define MCF5249_UART_USR_PE				(0x20)
#define MCF5249_UART_USR_OE				(0x10)
#define MCF5249_UART_USR_TXEMP			(0x08)
#define MCF5249_UART_USR_TXRDY			(0x04)
#define MCF5249_UART_USR_FFULL			(0x02)
#define MCF5249_UART_USR_RXRDY			(0x01)

#define MCF5249_UART_UCSR_RCS(a)	(((a)&0x0f)<<4)
#define MCF5249_UART_UCSR_TCS(a)		((a)&0x0f)


#define MCF5249_UART_UCR_NONE			(0x00)
#define MCF5249_UART_UCR_STOP_BREAK		(0x70)
#define MCF5249_UART_UCR_START_BREAK	(0x60)
#define MCF5249_UART_UCR_RESET_BKCHGINT	(0x50)
#define MCF5249_UART_UCR_RESET_ERROR	(0x40)
#define MCF5249_UART_UCR_RESET_TX		(0x30)
#define MCF5249_UART_UCR_RESET_RX		(0x20)
#define MCF5249_UART_UCR_RESET_MR		(0x10)
#define MCF5249_UART_UCR_TX_DISABLED	(0x08)
#define MCF5249_UART_UCR_TX_ENABLED		(0x04)
#define MCF5249_UART_UCR_RX_DISABLED	(0x02)
#define MCF5249_UART_UCR_RX_ENABLED		(0x01)

#define MCF5249_UART_UCCR_COS			(0x10)
#define MCF5249_UART_UCCR_CTS			(0x01)

#define MCF5249_UART_UACR_BRG			(0x80)
#define MCF5249_UART_UACR_CTMS_TIMER	(0x60)
#define MCF5249_UART_UACR_IEC			(0x01)

#define MCF5249_UART_UISR_COS			(0x80)
#define MCF5249_UART_UISR_DB			(0x04)
#define MCF5249_UART_UISR_RXRDY			(0x02)
#define MCF5249_UART_UISR_TXRDY			(0x01)

#define MCF5249_UART_UIMR_COS			(0x80)
#define MCF5249_UART_UIMR_DB			(0x04)
#define MCF5249_UART_UIMR_FFULL			(0x02)
#define MCF5249_UART_UIMR_TXRDY			(0x01)


/**********************************************************************
*
* SDRAM Controller Module Registers Description
*
***********************************************************************/

#define	MCF5249_DRAMC_DCR     	(*(vuint16 *)(void *)(&__MBAR[0x100]))
#define	MCF5249_DRAMC_DACR0		(*(vuint32 *)(void *)(&__MBAR[0x108]))
#define	MCF5249_DRAMC_DMR0   	(*(vuint32 *)(void *)(&__MBAR[0x10C]))
#define	MCF5249_DRAMC_DACR1   	(*(vuint32 *)(void *)(&__MBAR[0x110]))
#define	MCF5249_DRAMC_DMR1   	(*(vuint32 *)(void *)(&__MBAR[0x114]))

/* Controls used by both Synchronous and Asynchronous DRAM */			
#define MCF5249_DRAMC_DCR_SO			(0x8000)		
#define MCF5249_DRAMC_DCR_NAM			(0x2000)		
#define MCF5249_DRAMC_DCR_RC(a)			((a)&0x01FF)	

#define MCF5249_DRAMC_DACR_BASE(a)		((a)&0xFFFC0000)
#define MCF5249_DRAMC_DACR_RE			(0x00008000)	
#define MCF5249_DRAMC_DACR_PS_32		(0x00000000)	
#define MCF5249_DRAMC_DACR_PS_8			(0x00000010)	
#define MCF5249_DRAMC_DACR_PS_16		(0x00000020)	

#define MCF5249_DRAMC_DCMR_MASK_4G		(0xFFFC0000)	
#define MCF5249_DRAMC_DCMR_MASK_2G		(0x7FFC0000)	
#define MCF5249_DRAMC_DCMR_MASK_1G		(0x3FFC0000)	
#define MCF5249_DRAMC_DCMR_MASK_1024M	(0x3FFC0000)	
#define MCF5249_DRAMC_DCMR_MASK_512M	(0x1FFC0000)	
#define MCF5249_DRAMC_DCMR_MASK_256M	(0x0FFC0000)	
#define MCF5249_DRAMC_DCMR_MASK_128M	(0x07FC0000)	
#define MCF5249_DRAMC_DCMR_MASK_64M		(0x03FC0000)	
#define MCF5249_DRAMC_DCMR_MASK_32M		(0x01FC0000)	
#define MCF5249_DRAMC_DCMR_MASK_16M		(0x00FC0000)	
#define MCF5249_DRAMC_DCMR_MASK_8M		(0x007C0000)	
#define MCF5249_DRAMC_DCMR_MASK_4M		(0x003C0000)	
#define MCF5249_DRAMC_DCMR_MASK_2M		(0x001C0000)	
#define MCF5249_DRAMC_DCMR_MASK_1M		(0x000C0000)	
#define MCF5249_DRAMC_DCMR_MASK_1024K	(0x00040000)	
#define MCF5249_DRAMC_DCMR_MASK_512K	(0x00000000)	
#define MCF5249_DRAMC_DCMR_WP			(0x00000100)	
#define MCF5249_DRAMC_DCMR_CPU			(0x00000040)	
#define MCF5249_DRAMC_DCMR_AM			(0x00000020)	
#define MCF5249_DRAMC_DCMR_SC			(0x00000010)	
#define MCF5249_DRAMC_DCMR_SD			(0x00000008)	
#define MCF5249_DRAMC_DCMR_UC			(0x00000004)	
#define MCF5249_DRAMC_DCMR_UD			(0x00000002)	
#define MCF5249_DRAMC_DCMR_V			(0x00000001)	

/* Controls used only by Asynchronous DRAM*/			
#define MCF5249_DRAMC_DCR_RRA_2			(0x0000)		
#define MCF5249_DRAMC_DCR_RRA_3			(0x0800)		
#define MCF5249_DRAMC_DCR_RRA_4			(0x1000)		
#define MCF5249_DRAMC_DCR_RRA_5			(0x1800)		
#define MCF5249_DRAMC_DCR_RRP_1			(0x0000)		
#define MCF5249_DRAMC_DCR_RRP_2			(0x0200)		
#define MCF5249_DRAMC_DCR_RRP_3			(0x0400)		
#define MCF5249_DRAMC_DCR_RRP_4			(0x0600)		

#define MCF5249_DRAMC_DACR_CAS_1		(0x00000000)	
#define MCF5249_DRAMC_DACR_CAS_2		(0x00001000)	
#define MCF5249_DRAMC_DACR_CAS_3		(0x00002000)	
#define MCF5249_DRAMC_DACR_CAS_4		(0x00003000)	
#define MCF5249_DRAMC_DACR_RP_1			(0x00000000)	
#define MCF5249_DRAMC_DACR_RP_2			(0x00000400)	
#define MCF5249_DRAMC_DACR_RP_3			(0x00000800)	
#define MCF5249_DRAMC_DACR_RP_4			(0x00000C00)	
#define MCF5249_DRAMC_DACR_RNCN			(0x00000200)	
#define MCF5249_DRAMC_DACR_RCD_1		(0x00000000)	
#define MCF5249_DRAMC_DACR_RCD_2		(0x00000100)	
#define MCF5249_DRAMC_DACR_EDO			(0x00000040)	
#define MCF5249_DRAMC_DACR_PM_OFF		(0x00000000)	
#define MCF5249_DRAMC_DACR_PM_BURST		(0x00000004)	
#define MCF5249_DRAMC_DACR_PM_ON		(0x0000000C)	

/*	Controls used only by Synchronous DRAM */
#define MCF5249_DRAMC_DCR_COC		(0x1000) /* Command on Clock Enable */
#define MCF5249_DRAMC_DCR_IS		(0x0800) /* Initiate Self Refresh Command */
#define MCF5249_DRAMC_DCR_RTIM_3	(0x0000) /* 3 Clocks Between REF and ACTV Cmds */
#define MCF5249_DRAMC_DCR_RTIM_6	(0x0200) /* 6 Clocks Between REF and ACTV Cmds */
#define MCF5249_DRAMC_DCR_RTIM_9	(0x0400) /* 9 Clocks Between REF and ACTV Cmds */

#define MCF5249_DRAMC_DACR_CASL_1	(0x00000000) /* 1 Clock From CAS to Data */
#define MCF5249_DRAMC_DACR_CASL_2	(0x00001000) /* 2 Clock From CAS to Data */	
#define MCF5249_DRAMC_DACR_CASL_3	(0x00002000) /* 3 Clock From CAS to Data */	
#define MCF5249_DRAMC_DACR_CBM(a)	(((a)&0x00000007)<<8) /* Command and Bank Mux */
#define MCF5249_DRAMC_DACR_IMRS		(0x00000040) /* Initiate Mode Register Set Cmd */
#define MCF5249_DRAMC_DACR_IP		(0x00000008) /* Initiate Precharge All Command */
#define MCF5249_DRAMC_DACR_PM		(0x00000004) /* Continuous Page	Mode */

/**********************************************************************
*
* Timer Module Registers Description
*
***********************************************************************/

#define	MCF5249_TIMER0_TMR     	(*(vuint16 *)(void *)(&__MBAR[0x140]))
#define	MCF5249_TIMER0_TRR   	(*(vuint16 *)(void *)(&__MBAR[0x144]))
#define	MCF5249_TIMER0_TCR   	(*(vuint16 *)(void *)(&__MBAR[0x148]))
#define	MCF5249_TIMER0_TCN   	(*(vuint16 *)(void *)(&__MBAR[0x14C]))
#define	MCF5249_TIMER0_TER   	(*(vuint8  *)(void *)(&__MBAR[0x151]))
#define	MCF5249_TIMER1_TMR     	(*(vuint16 *)(void *)(&__MBAR[0x180]))
#define	MCF5249_TIMER1_TRR   	(*(vuint16 *)(void *)(&__MBAR[0x184]))
#define	MCF5249_TIMER1_TCR   	(*(vuint16 *)(void *)(&__MBAR[0x188]))
#define	MCF5249_TIMER1_TCN   	(*(vuint16 *)(void *)(&__MBAR[0x18C]))
#define	MCF5249_TIMER1_TER   	(*(vuint8  *)(void *)(&__MBAR[0x191]))

/* some special macros */
#define MCF5249_TIMER_TMR(NUM)	(*(vuint16 *)(void *)(&__MBAR[0x140+(NUM * 0x40)]))
#define	MCF5249_TIMER_TRR(NUM)	(*(vuint16 *)(void *)(&__MBAR[0x144+(NUM * 0x40)]))
#define	MCF5249_TIMER_TCR(NUM)	(*(vuint16 *)(void *)(&__MBAR[0x148+(NUM * 0x40)]))
#define	MCF5249_TIMER_TCN(NUM)	(*(vuint16 *)(void *)(&__MBAR[0x14C+(NUM * 0x40)]))
#define	MCF5249_TIMER_TER(NUM)	(*(vuint8  *)(void *)(&__MBAR[0x151+(NUM * 0x40)]))

/* Bit level definitions and macros */
#define MCF5249_TIMER_TMR_PS(a)		(((a)&0x00FF)<<8)
#define MCF5249_TIMER_TMR_CE_ANY	(0x00C0)
#define MCF5249_TIMER_TMR_CE_RISE	(0x0080)
#define MCF5249_TIMER_TMR_CE_FALL	(0x0040)
#define MCF5249_TIMER_TMR_CE_NONE	(0x0000)
#define MCF5249_TIMER_TMR_OM		(0x0020)
#define MCF5249_TIMER_TMR_ORI		(0x0010)
#define MCF5249_TIMER_TMR_FRR		(0x0008)
#define MCF5249_TIMER_TMR_CLK_TIN	(0x0006)
#define MCF5249_TIMER_TMR_CLK_DIV16	(0x0004)
#define MCF5249_TIMER_TMR_CLK_MSCLK	(0x0002)
#define MCF5249_TIMER_TMR_CLK_STOP	(0x0000)
#define MCF5249_TIMER_TMR_RST		(0x0001)

#define MCF5249_TIMER_TER_REF		(0x02)
#define MCF5249_TIMER_TER_CAP		(0x01)

/**********************************************************************
*
* Audio interface Registers Description
*
***********************************************************************/

#define	MCF5249_AUDIO_IIS1_CFG			(*(vuint32 *)(void *)(&__MBAR2[0x010]))
#define	MCF5249_AUDIO_IIS2_CFG			(*(vuint32 *)(void *)(&__MBAR2[0x014]))
#define	MCF5249_AUDIO_IIS3_CFG			(*(vuint32 *)(void *)(&__MBAR2[0x018]))
#define	MCF5249_AUDIO_IIS4_CFG			(*(vuint32 *)(void *)(&__MBAR2[0x01C]))
#define	MCF5249_AUDIO_EBU_CFG			(*(vuint32 *)(void *)(&__MBAR2[0x020]))
#define	MCF5249_AUDIO_EBU_RCV_C_CH_1	(*(vuint32 *)(void *)(&__MBAR2[0x024]))
#define	MCF5249_AUDIO_EBU_TX_C_CH_1		(*(vuint32 *)(void *)(&__MBAR2[0x028]))
#define	MCF5249_AUDIO_EBU_TX_C_CH_2		(*(vuint32 *)(void *)(&__MBAR2[0x02C]))


#define	MCF5249_AUDIO_DATA_IN_CONTROL	(*(vuint16 *)(void *)(&__MBAR2[0x032]))
#define	MCF5249_AUDIO_PDIR1_L			(*(vuint32 *)(void *)(&__MBAR2[0x034]))
#define	MCF5249_AUDIO_PDIR3_L			(*(vuint32 *)(void *)(&__MBAR2[0x044]))
#define	MCF5249_AUDIO_PDIR1_R			(*(vuint32 *)(void *)(&__MBAR2[0x054]))
#define	MCF5249_AUDIO_PDIR3_R			(*(vuint32 *)(void *)(&__MBAR2[0x064]))

#define	MCF5249_AUDIO_PDOR1_L			(*(vuint32 *)(void *)(&__MBAR2[0x034]))
#define	MCF5249_AUDIO_PDOR1_R			(*(vuint32 *)(void *)(&__MBAR2[0x044]))
#define	MCF5249_AUDIO_PDOR2_L			(*(vuint32 *)(void *)(&__MBAR2[0x054]))
#define	MCF5249_AUDIO_PDOR2_R			(*(vuint32 *)(void *)(&__MBAR2[0x064]))
#define	MCF5249_AUDIO_PDOR3				(*(vuint32 *)(void *)(&__MBAR2[0x074]))
#define	MCF5249_AUDIO_PDIR				(*(vuint32 *)(void *)(&__MBAR2[0x074]))

#define	MCF5249_AUDIO_U_CHANNEL_XMIT	(*(vuint32 *)(void *)(&__MBAR2[0x084]))
#define	MCF5249_AUDIO_U_CHANNEL_RCV		(*(vuint32 *)(void *)(&__MBAR2[0x088]))
#define	MCF5249_AUDIO_Q_CHANNEL_RCV		(*(vuint32 *)(void *)(&__MBAR2[0x08C]))

#define	MCF5249_AUDIO_CD_TEXT_CONTROL	(*(vuint8  *)(void *)(&__MBAR2[0x093]))

#define	MCF5249_AUDIO_INTERRUPT_EN		(*(vuint32 *)(void *)(&__MBAR2[0x094]))
#define	MCF5249_AUDIO_INTERRUPT_CLEAR	(*(vuint32 *)(void *)(&__MBAR2[0x098]))
#define	MCF5249_AUDIO_INTERRUPT_STAT	(*(vuint32 *)(void *)(&__MBAR2[0x098]))
#define	MCF5249_AUDIO_DMA_CONFIG		(*(vuint8  *)(void *)(&__MBAR2[0x09F]))
#define	MCF5249_AUDIO_PHASE_CONFIG		(*(vuint8  *)(void *)(&__MBAR2[0x0A3]))
#define	MCF5249_AUDIO_XTRIM				(*(vuint16 *)(void *)(&__MBAR2[0x0A6]))
#define	MCF5249_AUDIO_FREQ_MEAS			(*(vuint32 *)(void *)(&__MBAR2[0x0A8]))

#define	MCF5249_AUDIO_BLOCK_CONTROL		(*(vuint16 *)(void *)(&__MBAR2[0x0CA]))	
#define	MCF5249_AUDIO_GLOB				(*(vuint16 *)(void *)(&__MBAR2[0x0CE]))	
#define	MCF5249_AUDIO_EBU2_CONFIG		(*(vuint32 *)(void *)(&__MBAR2[0x0D0]))	
#define	MCF5249_AUDIO_EBU2_RCV_C_CNL_1	(*(vuint32 *)(void *)(&__MBAR2[0x0D4]))	
#define	MCF5249_AUDIO_U2_CHANNEL_RCV	(*(vuint32 *)(void *)(&__MBAR2[0x0D8]))
#define	MCF5249_AUDIO_Q2_CHANNEL_RCV	(*(vuint32 *)(void *)(&__MBAR2[0x0DC]))

/**********************************************************************
*
* ADC Module Registers Description
*
***********************************************************************/

#define	MCF5249_ADC_ADCONFIG			(*(vuint16 *)(void *)(&__MBAR2[0x402]))	
#define	MCF5249_ADC_ADVALUE				(*(vuint16 *)(void *)(&__MBAR2[0x406]))	

/**********************************************************************
*
* Flash Media Module Registers Description
*
***********************************************************************/

#define	MCF5249_FLASHMEDIA_CONFIG		(*(vuint32 *)(void *)(&__MBAR2[0x460]))	
#define	MCF5249_FLASHMEDIA_CMD1			(*(vuint32 *)(void *)(&__MBAR2[0x464]))
#define	MCF5249_FLASHMEDIA_CMD2			(*(vuint32 *)(void *)(&__MBAR2[0x468]))
#define	MCF5249_FLASHMEDIA_DATA1		(*(vuint32 *)(void *)(&__MBAR2[0x46C]))	
#define	MCF5249_FLASHMEDIA_DATA2		(*(vuint32 *)(void *)(&__MBAR2[0x470]))	
#define	MCF5249_FLASHMEDIA_STATUS		(*(vuint32 *)(void *)(&__MBAR2[0x474]))	
#define	MCF5249_FLASHMEDIA_INT_EN		(*(vuint32 *)(void *)(&__MBAR2[0x478]))	
#define	MCF5249_FLASHMEDIA_INT_STAT		(*(vuint32 *)(void *)(&__MBAR2[0x47C]))
#define	MCF5249_FLASHMEDIA_INT_CLEAR	(*(vuint32 *)(void *)(&__MBAR2[0x47C]))	

/***********************************************************************/

#endif	/* _CPU_MCF5249_H */
