
/*******************************************************************************
 * Defines to control SmartDMA and its tasks.  These defines are used for the 
 * task build process to minimize disconnects at the task/driver interface.
 ******************************************************************************/

#define SDMA_INT_BIT_DBG             31          /* debug interrupt bit        */
#define SDMA_INT_BIT_TEA             28          /* TEA interrupt bit          */
#define SDMA_INT_BIT_TEA_TASK        24          /* lsb for TEA task number    */
#define SDMA_INT_BIT_IMPL            0x9000FFFF

#define SDMA_PTDCTRL_BIT_TEA         14          /* TEA detection enable bit   */

#define SDMA_TCR_BIT_AUTO            15          /* auto start bit             */

#define SDMA_STAT_BIT_ALARM          17
#define SDMA_FIFO_ALARM_MASK         0x0020000

#define SDMA_DRD_BIT_TFD             27   /* mark last buffer of frame         */
#define SDMA_DRD_BIT_INT             26   /* interrupt after buffer processed  */
#define SDMA_DRD_BIT_INIT            21   /* lsb position of initiator         */
#define SDMA_DRD_MASK_FLAGS          0x0C000000
#define SDMA_DRD_MASK_LENGTH         0x03FFFFFF

#define SDMA_PRAGMA_BIT_RL           0  /* read line buffer enable                */
#define SDMA_PRAGMA_BIT_CW           1  /* write line buffer enable               */
#define SDMA_PRAGMA_BIT_SPECREAD     2  /* XLB speculative read enable            */
#define SDMA_PRAGMA_BIT_INTEGER      3  /* data alignment 0=frac(msb), 1=int(lsb) */
#define SDMA_PRAGMA_BIT_PACK         4  /* pack data enable                       */
#define SDMA_PRAGMA_BIT_RST_ERROR_NO 5  /* don't reset errors on task enable      */
#define SDMA_PRAGMA_BIT_PRECISE_INC  6  /* increment 0=when possible, 1=iter end  */
#define SDMA_PRAGMA_BIT_REPARSE      7  /* context save 0=all, 1=necessary data   */

#define SDMA_TASK_ENTRY_BYTES       32  /* Bytes per task in entry table  */
#define SDMA_TASK_GROUP_NUM         16  /* Number of tasks per task group */
#define SDMA_TASK_GROUP_BYTES       (SDMA_TASK_ENTRY_BYTES*SDMA_TASK_GROUP_NUM)


/*******************************************************************************
 * Task group control macros, use when TaskNum > 15
 ******************************************************************************/
#define SDMA_TASKNUM_EXT(OldTaskNum) (OldTaskNum%16)

#define SDMA_TASKBAR_CHANGE(sdma, OldTaskNum) {                 \
   sdma->taskBar += (((int)(OldTaskNum/SDMA_TASK_GROUP_NUM))*SDMA_TASK_GROUP_BYTES); \
}

#define SDMA_TASKBAR_RESTORE(sdma, OldTaskNum) {                \
   sdma->taskBar -= (((int)(OldTaskNum/SDMA_TASK_GROUP_NUM))*SDMA_TASK_GROUP_BYTES); \
}


/*******************************************************************************
 * Task control macros
 ******************************************************************************/

/*
 * SDMA_TASK_ENABLE, SDMA_TASK_DISABLE and SDMA_TASK_STATUS are
 * defined in the BestComm API. SDMA_TASK_AUTO_START replaces
 * TASK_TASK_ENABLE_AUTO.
 */

#ifndef SDMA_TASK_ENABLE
/* Enable SmartDMA task */
#define SDMA_TASK_ENABLE(sdma, TaskNum) {                 \
   *(((uint16 *)&sdma->TCR01)+TaskNum) = (uint16)0x8040;  \
}

#define SDMA_TASK_ENABLE_AUTO(sdma, TaskNum, NextTaskNum) {              \
   *(((uint16 *)&sdma->TCR01)+TaskNum) = (uint16)(0x80C0 | NextTaskNum); \
}
#endif

#ifndef SDMA_TASK_DISABLE
#define SDMA_TASK_DISABLE(sdma, TaskNum) {                 \
   *(((uint16 *)&sdma->TCR01)+TaskNum) &= (uint16)0x7fff;  \
}
#endif

#ifndef SDMA_TASK_STATUS
#define SDMA_TASK_STATUS(sdma, TaskNum) \
   *(((uint16 *)&sdma->TCR01)+TaskNum)
#endif


/*******************************************************************************
 * Interrupt control macros
 ******************************************************************************/

/*
 * SDMA_INT_ENABLE, SDMA_INT_DISABLE, SDMA_INT_SOURCE, SDMA_INT_TEST,
 * SDMA_CLEAR_IEVENT, SDMA_GET_PENDINGBIT and SDMA_GET_MASKBIT are
 * defined in the BestComm API.
 */

#ifndef SDMA_INT_ENABLE
#define SDMA_INT_ENABLE(sdma, Bit) {     \
   sdma->IntMask &= ~((uint32)(1<<Bit)); \
}
#endif

#ifndef SDMA_INT_DISABLE
#define SDMA_INT_DISABLE(sdma, Bit) {    \
   sdma->IntMask |=  ((uint32)(1<<Bit)); \
}
#endif

#ifndef SDMA_INT_SOURCE
#define SDMA_INT_SOURCE(sdma)    \
   (sdma->IntPend & (~sdma->IntMask) & (uint32)SDMA_INT_BIT_IMPL)
#endif

#ifndef SDMA_INT_TEST
#define SDMA_INT_TEST(IntSource, Bit)    \
   (((uint32)IntSource) & ((uint32)(1<<Bit)))
#endif

// define SDMA_INT_FIND to get int bit rather than scan all bits
// use cntlzw

#ifndef SDMA_CLEAR_IEVENT
/* Clear the IntPend bit */
#define SDMA_CLEAR_IEVENT(sdma, Bit) {   \
   sdma->IntPend  =  ((uint32)(1<<Bit)); \
}
#endif

#ifndef SDMA_GET_PENDINGBIT
#define SDMA_GET_PENDINGBIT(sdma, Bit)   \
   (sdma->IntPend & (uint32)(1<<Bit))
#endif

#ifndef SDMA_GET_MASKBIT
// do we really need this?
#define SDMA_GET_MASKBIT(sdma, Bit)      \
   (sdma->IntMask & (uint32)(1<<Bit))
#endif


/*******************************************************************************
 * SmartDMA FIFO control macros
 ******************************************************************************/

/*******************************************************************************
 * SmartDMA TEA detection control macros
 ******************************************************************************/
/* Enable SmartDMA TEA detection and TEA interrupt */
#define SDMA_TEA_ENABLE(sdma) {                            \
   SDMA_INT_ENABLE(sdma, SDMA_INT_BIT_TEA);                \
   sdma->PtdCntrl &= ~((uint32)(1<<SDMA_PTDCTRL_BIT_TEA)); \
}

/* Disable SmartDMA TEA detection and TEA interrupt */
#define SDMA_TEA_DISABLE(sdma) {                           \
   SDMA_INT_DISABLE(sdma, SDMA_INT_BIT_TEA);               \
   sdma->PtdCntrl |=  ((uint32)(1<<SDMA_PTDCTRL_BIT_TEA)); \
}

/* Clear the TEA interrupt */
#define SDMA_TEA_CLEAR(sdma) {                                 \
   sdma->IntPend   =  ((uint32)(0x1F<<SDMA_INT_BIT_TEA_TASK)); \
}

#ifndef SDMA_TEA_SOURCE
/* Determine which task caused a TEA on the XLB */
#define SDMA_TEA_SOURCE(sdma)                                 \
   (uint32)((sdma->IntPend>>SDMA_INT_BIT_TEA_TASK) & 0xF)
#endif


/*******************************************************************************
 * SmartDMA debug control macros
 ******************************************************************************/
/* Enable the SmartDMA debug unit and DBG interrupt */
/* add sdma->dbg_regs setup? */
#define SDMA_DBG_ENABLE(sdma) {               \
   SDMA_INT_ENABLE(sdma, SDMA_INT_BIT_DBG);   \
}

#define SDMA_DBG_DISABLE(sdma) {              \
   SDMA_INT_DISABLE(sdma, SDMA_INT_BIT_DBG);  \
}

/* Clear the debug interrupt */
#define SDMA_DBG_CLEAR(sdma) {                \
   SDMA_CLEAR_IEVENT(sdma, SDMA_INT_BIT_DBG); \
}

#define SDMA_DBG_MDE(dst, sdma, addr) {   \
   sdma->MDEDebug = addr;                 \
   dst = sdma->MDEDebug;                  \
}

#define SDMA_DBG_ADS(dst, sdma, addr) {   \
   sdma->ADSDebug = addr;                 \
   dst = sdma->ADSDebug;                  \
}

#define SDMA_DBG_PTD(dst, sdma, addr) {   \
   sdma->PTDDebug = addr;                 \
   dst = sdma->PTDDebug;                  \
}


/*******************************************************************************
 * Initiator control macros
 ******************************************************************************/

/* Set the Initiator Priority */
#define SDMA_SET_INITIATOR_PRIORITY(sdma, initiator, priority)	\
	*(((uint8 *)&sdma->IPR0)+initiator) = priority;


/* Read DRD initiator number */
#define SDMA_INIT_READ(PtrDRD)              \
   (((*(uint32 *)PtrDRD)>>SDMA_DRD_BIT_INIT) & (uint32)0x1F)

/* Write DRD initiator number */
#define SDMA_INIT_WRITE(PtrDRD, Initiator) {                \
   *(uint32 *)PtrDRD = ((*(uint32 *)PtrDRD) & 0xFC1FFFFF) | \
                            (Initiator<<SDMA_DRD_BIT_INIT); \
}

/* Change DRD initiator number */
#define SDMA_INIT_CHANGE(task, oldInitiator, newInitiator) {      \
   int i;                                                         \
   for (i=0; i<task->NumDRD; i++) {                               \
      if (SDMA_INIT_READ(task->DRD[i]) == (uint32)oldInitiator) { \
         SDMA_INIT_WRITE(task->DRD[i],newInitiator);              \
      }                                                           \
   }                                                              \
}


