/******************************************************************************
*
*       COPYRIGHT (c) 2001-2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
*       The code is the property of Motorola.
*
*       The copyright notice above does not evidence any
*       actual or intended publication of such source code.
*
* Filename:     $Source: /proj/cvsroot/mgt/MGT5200/apps/AC97Sample/modules/Core/core5200.c,v $
* Author:       $Author: ra6707 $
* Locker:       $Locker:  $
* State:        $State: Exp $
* Revision:     $Revision: 1.1 $
*
* Functions:    
*
* History:      Use the RCS command rlog to display revision history
*               information.
*
* Description:  
*
* Notes:                
*
******************************************************************************/

#include <stdio.h>

#include "ppctypes.h"

#include "configure.h"

#if !(defined(MGT5200) ^ defined(MPC5200B))
	#error Either MGT5200 or MPC5200B has to be defined.
#endif

#if defined(MPC5200B)
	#include "mpc5200b/mpc5200b.h"
	#include "mpc5200b/ipbi.h"
#elif defined(MGT5200)
	#include "mgt5200/mgt5200.h"
	#include "mgt5200/ipbi.h"
#endif

#include "core5200.h"
#include "spr5200.h"

/*--------------------------------------------------------------------------
   Function    : void initDCache (int copyback)
   Description : Initialize DBAT registers to allow caching of SDRAM
                 disallow caching of Flash and internal peripheral address
                 space.
                 ATTENTION: This code assume that the Flash is connected to
                            CSBoot or CS0 only. Access to other Flash will
                            crash the system. In this case write your own
                            version of this function.
   Parameter(s): If copyback is true copy-back caching will be enabled, else
                 write-trough caching will be enabled.
   Returns     : nothing
   Note        : Keep in mind coherency when using copy-back caching and
                 there are multiple writers (core, SmartComm, PCI) to SDRAM.
  --------------------------------------------------------------------------*/
void initDCache (int copyback)
{
	ipbi_regs *ipbi;
	int i;
	uint32 flashaddr, flashsize;
	uint32 regspaceaddr;
	uint32 sdramaddr0, sdramaddr1, sdramsize0, sdramsize1;
	const uint32 sizemmu[12] = {
		BATU_BL_128K, BATU_BL_256K, BATU_BL_512K,
		BATU_BL_1M, BATU_BL_2M, BATU_BL_4M, BATU_BL_8M, BATU_BL_16M,
		BATU_BL_32M, BATU_BL_64M, BATU_BL_128M, BATU_BL_256M	
	};
	
	ipbi = (ipbi_regs *) (readMBAR () + MBAR_CS);
	
	if (ipbi->control_reg & CSCTRL_CS0_EN) {
		flashaddr = ipbi->cs0_start_addr << 16;
		flashsize = ((ipbi->cs0_stop_addr << 16) - flashaddr) + 0x00010000UL;    	
	} else {
		flashaddr = ipbi->boot_start_addr << 16;
		flashsize = ((ipbi->boot_stop_addr << 16) - flashaddr) + 0x00010000UL;    	    	
	}
	for (i=0x11; i<=0x1B; i++) {
		if (1<<i >= flashsize) {
			break;
		}
	}
	if (i<=0x1B) {
		flashsize = sizemmu[i - 0x11];
	} else {
		flashsize = BATU_BL_256M;
	}

	regspaceaddr = readMBAR ();

	sdramaddr0 = ipbi->sdram_cs0_cfg & 0xFFF00000UL;
	sdramsize0 = ipbi->sdram_cs0_cfg & 0x0000001FUL;
	if ((sdramsize0 >= 0x13) && (sdramsize0 <= 0x1B)) {
		sdramsize0 = sizemmu[(sdramsize0 - 0x10)];
	} else {
		sdramsize0 = 0;
	}
	
	sdramaddr1 = ipbi->sdram_cs1_cfg & 0xFFF00000UL;
	sdramsize1 = ipbi->sdram_cs1_cfg & 0x0000001FUL;
	if ((sdramsize1 >= 0x13) && (sdramsize1 <= 0x1B)) {
		sdramsize1 = sizemmu[(sdramsize1 - 0x10)];
	} else {
		sdramsize1 = 0;
	}

	writeDBAT0U ((flashaddr & BATU_BEPI) | BATU_SUPERVISOR | BATU_USER | BATU_BL_32M);
	writeDBAT0L ((flashaddr & BATL_BRPN) | BATL_CACHE_INH | BATL_READ_WRITE);

	if (sdramsize0 != 0) {
		writeDBAT1U ((sdramaddr0 & BATU_BEPI) | BATU_SUPERVISOR | BATU_USER | sdramsize0);
		if (copyback) {
			writeDBAT1L ((sdramaddr0 & BATL_BRPN) | BATL_READ_WRITE);
		} else {	
			writeDBAT1L ((sdramaddr0 & BATL_BRPN) | BATL_WRITE_THR | BATL_READ_WRITE);
		}
	} else {
		writeDBAT1U (BATU_INVALID);
		writeDBAT1L (BATL_NO_ACCESS);
	}

	if (sdramsize1 != 0) {
		writeDBAT2U ((sdramaddr1 & BATU_BEPI) | BATU_SUPERVISOR | BATU_USER | sdramsize1);
		if (copyback) {
			writeDBAT2L ((sdramaddr1 & BATL_BRPN) | BATL_READ_WRITE);
		} else {	
			writeDBAT2L ((sdramaddr1 & BATL_BRPN) | BATL_WRITE_THR | BATL_READ_WRITE);
		}
	} else {
		writeDBAT2U (BATU_INVALID);
		writeDBAT2L (BATL_NO_ACCESS);
	}

	writeDBAT3U ((regspaceaddr & BATU_BEPI) | BATU_SUPERVISOR | BATU_USER | BATU_BL_128K);
	writeDBAT3L ((regspaceaddr & BATL_BRPN) | BATL_CACHE_INH | BATL_READ_WRITE);

	writeDBAT4U (BATU_INVALID);
	writeDBAT4L (BATL_NO_ACCESS);

	writeDBAT5U (BATU_INVALID);
	writeDBAT5L (BATL_NO_ACCESS);

	writeDBAT6U (BATU_INVALID);
	writeDBAT6L (BATL_NO_ACCESS);
	
	writeDBAT7U (BATU_INVALID);
	writeDBAT7L (BATL_NO_ACCESS);
}

/*--------------------------------------------------------------------------
   Function    : void enableDCache (void)
   Description : Enable data cache and memory address translation. Before
                 calling this function initDCache must be called.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void enableDCache (void)
{
	nofralloc

	/*
	 * Enable memory address translation
	 */
	mfmsr r3
	ori r3,r3,MSR_DR
	mtmsr r3
	isync

	/*
	 * Invalidate D-Cache
	 */
	mfspr r3,SPR_HID0
	ori r3,r3,(HID0_DCE | HID0_DCFI)
	mtspr SPR_HID0,r3
	sync
	
	/*
	 * Enable D-Cache (by leaving it enabled)
	 */
	lis r4,(~HID0_DCFI)@h
	ori r4,r4,(~HID0_DCFI)@l
	and r3,r3,r4
	mtspr SPR_HID0,r3
	sync
	
	blr
}

/*--------------------------------------------------------------------------
   Function    : void disableDCache (void)
   Description : Disable data cache and memory address translation.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void disableDCache (void)
{
	nofralloc
	
	/*
	 * Disable D-Cache
	 */
	mfspr r3,SPR_HID0
	lis r4,(~HID0_DCE)@h
	ori r4,r4,(~HID0_DCE)@l
	and r3,r3,r4
	mtspr SPR_HID0,r3
	sync

	/*
	 * Disable memory address translation
	 */
	mfmsr r3
	lis r4,(~MSR_DR)@h
	ori r4,r4,(~MSR_DR)@l
	and r3,r3,r4
	mtmsr r3
	isync
	
	blr
}	

/*--------------------------------------------------------------------------
   Function    : void flushDCache (void)
   Description : Flush data cache.
   Parameter(s): Effective starting address of the data and length of the data
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void flushDCache (uint32 address, uint32 length)
{
#pragma unused (address, length)
	nofralloc
	
	clrrwi  r6,r3,5			/* address_aligned = address & 0xFFFFFFE0 */
	sub     r0,r3,r6		/* diff = address - address_aligned */
	add     r3,r4,r0		/* length += diff */
	addi    r0,r3,31		/* blocks = (length + 31) / 32 */
	srwi    r5,r0,5
	mtctr   r5				/* ctr = blocks */
	loop:					/* do { */
	dcbf    0,r6			/*   data cache block flush */
	addi    r6,r6,32		/*   address_aligned += 32 */
	bdnz    loop			/* } while (--blocks) */

	sync    

	blr
}
	
#if 0	
/*--------------------------------------------------------------------------
   Function    : void flushDCache (void)
   Description : Flush data cache.
   Parameter(s): Effective starting address of the data and length of the data
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void flushDCache (uint32 address, uint32 length)
{
#pragma unused (address, length)
	nofralloc
	
	li  	r5,5
	/* divide length by 32 */
	srw		r5,r4,r5
	
	/* 
	 * check the remainder of the calculation, 
	 * if it is not zero, than one more cache block must be flushed 
	 */
	andi.	r4,r4,0x1F
	beq		jump_over
	addi	r5,r5,1
jump_over:	
	/* create the remainder of the calculation r3 / 32 */
	andi.	r6,r3,0x1F
	li		r7,32
	/* 32 - remainder of (address / 32) */ 
	subf	r6,r6,r7
	subf. 	r4,r6,r4
	/* 
	 * if the remainder of the calculation (length / 32) is 
	 * greater than the result of (32 - remainder of (address / 32)), 
	 * than one more cache block must be flushed 
	 */ 
	bgt		add_one_more
	
set_counter:		
	mtctr	r5
	
	/*
	 * flush the defined d-cache area
	 */
loop:
	dcbf	r0,r3
	addi	r3,r3,32
	bdnz 	loop
	
	isync
	
	blr
	
add_one_more:
	addi	r5,r5,1
	b		set_counter	
	
}	
#endif

/*--------------------------------------------------------------------------
   Function    : void enableICache (void)
   Description : Enable instruction cache.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void enableICache (void)
{
	nofralloc

	/*
	 * Invalidate I-Cache
	 */
	mfspr r3,SPR_HID0
	ori r3,r3,(HID0_ICE | HID0_ICFI)
	mtspr SPR_HID0,r3
	sync
	
	/*
	 * Enable I-Cache (by leaving it enabled)
	 */
	lis r4,(~HID0_ICFI)@h
	ori r4,r4,(~HID0_ICFI)@l
	and r3,r3,r4
	mtspr SPR_HID0,r3
	sync
	
	blr
}

/*--------------------------------------------------------------------------
   Function    : void disableICache (void)
   Description : Disable instruction cache.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void disableICache (void)
{
	nofralloc
	
	/*
	 * Disable I-Cache
	 */
	mfspr r3,SPR_HID0
	lis r4,(~HID0_ICE)@h
	ori r4,r4,(~HID0_ICE)@l
	and r3,r3,r4
	mtspr SPR_HID0,r3
	isync
	
	blr
}	

/*--------------------------------------------------------------------------
   Function    : void lockICache (void)
   Description : Lock instruction cache.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void lockICache (void)
{
	nofralloc

	mfspr r3,SPR_HID0
	ori r3,r3,HID0_ILOCK
	mtspr SPR_HID0,r3
	sync
		
	blr
}

/*--------------------------------------------------------------------------
   Function    : void unlockICache (void)
   Description : Unlock instruction cache.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void unlockICache (void)
{
	nofralloc

	mfspr r3,SPR_HID0
	lis r4,(~HID0_ILOCK)@h
	ori r4,r4,(~HID0_ILOCK)@l
	and r3,r3,r4
	mtspr SPR_HID0,r3
	sync
	
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readMBAR (void)
   Description : Return the contents of the MBAR register. The core SPR MBAR
                 mirrors the peripheral MBAR register.
   Parameter(s): none
   Returns     : MBAR address
  --------------------------------------------------------------------------*/
asm uint32 readMBAR (void)
{
	nofralloc	
	mfspr r3, SPR_MBAR	
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 writeMBAR (register uint32 mbar)
   Description : Set a new MBAR address and write it to peripheral MBAR and
                 to core SPR MBAR. The core SPR MBAR mirrors the peripheral
                 MBAR register. The function returns the MBAR address which
                 has been written.
   Parameter(s): new MBAR address
   Returns     : set MBAR address
  --------------------------------------------------------------------------*/
uint32 writeMBAR (register uint32 mbar)
{
	uint32 *mbar_p;
	uint32 msr;
	
	/*
	 * Writing a new value to MBAR and its shadow SPR MBAR must be intrinsic.
	 * Therefore, external and critical interrupts and machine check, floating
	 * point, single step, branch trace and recoverable exceptions will be
	 * forbidden while this happens.
	 */
	msr = readMSR ();				/* save MSR */
	writeMSR (msr & 0xFFFF607D);	/* disable interrupt and exceptions */
	
	mbar &= 0xFFFF0000UL;			/* only the upper 16 bits are valid */
	
	/*
	 * Write new MBAR address to MBAR
	 */
	mbar_p = (uint32 *) readMBAR ();	/* get current MBAR address */
	*mbar_p = mbar >> 16;			/* write new MBAR address */
	EIEIO;
	mbar_p = (uint32 *) mbar;		/* update MBAR pointer */
	
	/*
	 * Write new MBAR address to core MBAR
	 */
	asm {
		mtspr SPR_MBAR, mbar
	}

	writeMSR (msr);					/* resore MSR */
	
	return mbar;
}

/*--------------------------------------------------------------------------
   Function    : uint32 readRSP (void)
   Description : Read the EABI stack pointer.
   Parameter(s): none
   Returns     : stack pointer
  --------------------------------------------------------------------------*/
asm uint32 readRSP (void)
{
	nofralloc
	mr r3,rsp
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readMSR (void)
   Description : Read MSR register. 
   Parameter(s): none
   Returns     : MSR register contents
  --------------------------------------------------------------------------*/
asm uint32 readMSR (void)
{
	nofralloc
	mfmsr r3
	blr
}

/*--------------------------------------------------------------------------
   Function    : void writeMSR (uint32 msr)
   Description : Write MSR register.
   Parameter(s): Value to write in MSR register.
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void writeMSR (uint32 msr)
{
#pragma unused (msr)
	nofralloc
	mtmsr r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint64 readTBx (void)
   Description : Read TBL and TBU register.
   Parameter(s): none
   Returns     : Time base register contents
  --------------------------------------------------------------------------*/
asm uint64 readTBx (void)
{
	nofralloc
@loop:
	mftbu r3		/* get TBU */
	mftb r4			/* get TBL */
	mftbu r5		/* get TBU once again */
	cmpw r5,r3		/* see if 'old' = 'new' */
	bne @loop		/* loop if carry occured */
	blr
}

/*--------------------------------------------------------------------------
   Function    : void writeTBx (uint32 tbu, uint32 tbl)
   Description : Write TBL and TBU register.
   Parameter(s): Values to write to TBU and TBL.
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void writeTBx (uint32 tbu, uint32 tbl)
{
#pragma unused (tbu, tbl)	
	nofralloc
	li r5,0
	mttbl r5		/* force TBL to zero */
	mttbu r3		/* set TBU */
	mttbl r4		/* set TBL */
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readDEC (void)
   Description : Read Decrementer.
   Parameter(s): none
   Returns     : Decrementer contents.
  --------------------------------------------------------------------------*/
asm uint32 readDEC (void)
{
	nofralloc
	mfdec r3	
	blr
}

/*--------------------------------------------------------------------------
   Function    : void writeDEC (uint32 dec)
   Description : Write Decrementer.
   Parameter(s): Value to write to Decrementer
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void writeDEC (uint32 dec)
{
#pragma unused (dec)
	nofralloc
	mtdec r3
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readPVR (void)
   Description : Read PVR register.
   Parameter(s): none
   Returns     : PVR register contents.
  --------------------------------------------------------------------------*/
asm uint32 readPVR (void)
{
	nofralloc
	mfspr r3,SPR_PVR
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readSVR (void)
   Description : Read SVR register.
   Parameter(s): none
   Returns     : SVR register contents.
  --------------------------------------------------------------------------*/
asm uint32 readSVR (void)
{
	nofralloc
	mfspr r3,SPR_SVR
	blr
}

/*! \brief Read HID0 register.
 *
 * \return HID0 register contents.
 */
asm uint32 readHID0 (void)
{
	nofralloc
	mfspr r3,SPR_HID0
	blr
}

/*! \brief Write HID0 register.
 *
 * \param	hid0	Value to write to HID0.
 */
asm void writeHID0 (uint32 hid0)
{
#pragma unused (hid0)
	nofralloc
	mtspr SPR_HID0,r3
	sync
	blr
}

/*! \brief Read HID1 register.
 *
 * \return HID1 register contents.
 */
asm uint32 readHID1 (void)
{
	nofralloc
	mfspr r3,SPR_HID1
	blr
}

/*! \brief Write HID1 register.
 *
 * \return Value to write to HID1.
 */
asm void writeHID1 (uint32 hid1)
{
#pragma unused (hid1)
	nofralloc
	mtspr SPR_HID1,r3
	sync
	blr
}

/*! \brief Read HID2 register.
 *
 * \return HID2 register contents.
 */
asm uint32 readHID2 (void)
{
	nofralloc
	mfspr r3,SPR_HID2
	blr
}

/*! \brief Write HID2 register.
 *
 * \param	hid2	Value to write to HID2.
 */
asm void writeHID2 (uint32 hid2)
{
#pragma unused (hid2)
	nofralloc
	mtspr SPR_HID2,r3
	sync
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readDBAT0U (void)
   Description : Read DBAT0U register.
   Parameter(s): none
   Returns     : DBAT0U register contents.
   Note        : This function also exists for DBAT0L, DBAT1U, ..., DBAT7L                 
  --------------------------------------------------------------------------*/
asm uint32 readDBAT0U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT0U
	blr
}

/*--------------------------------------------------------------------------
   Function    : void writeDBAT0U (uint32 dbat0u)
   Description : Write DBAT0U register.
   Parameter(s): Value to write to DBAT0U register.
   Returns     : nothing
   Note        : This function also exists for DBAT0L, DBAT1U, ..., DBAT7L                 
  --------------------------------------------------------------------------*/
asm void writeDBAT0U (uint32 dbat0u)
{
#pragma unused (dbat0u)
	nofralloc
	mtspr SPR_DBAT0U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT0L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT0L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT0L (uint32 dbat0l)
{
#pragma unused (dbat0l)
	nofralloc
	mtspr SPR_DBAT0L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT1U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT1U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT1U (uint32 dbat1u)
{
#pragma unused (dbat1u)
	nofralloc
	mtspr SPR_DBAT1U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT1L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT1L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT1L (uint32 dbat1l)
{
#pragma unused (dbat1l)
	nofralloc
	mtspr SPR_DBAT1L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT2U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT2U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT2U (uint32 dbat2u)
{
#pragma unused (dbat2u)
	nofralloc
	mtspr SPR_DBAT2U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT2L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT2L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT2L (uint32 dbat2l)
{
#pragma unused (dbat2l)
	nofralloc
	mtspr SPR_DBAT2L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT3U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT3U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT3U (uint32 dbat3u)
{
#pragma unused (dbat3u)
	nofralloc
	mtspr SPR_DBAT3U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT3L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT3L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT3L (uint32 dbat3l)
{
#pragma unused (dbat3l)
	nofralloc
	mtspr SPR_DBAT3L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT4U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT4U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT4U (uint32 dbat4u)
{
#pragma unused (dbat4u)
	nofralloc
	mtspr SPR_DBAT4U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT4L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT4L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT4L (uint32 dbat4l)
{
#pragma unused (dbat4l)
	nofralloc
	mtspr SPR_DBAT4L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT5U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT5U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT5U (uint32 dbat5u)
{
#pragma unused (dbat5u)
	nofralloc
	mtspr SPR_DBAT5U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT5L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT5L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT5L (uint32 dbat5l)
{
#pragma unused (dbat5l)
	nofralloc
	mtspr SPR_DBAT5L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT6U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT6U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT6U (uint32 dbat6u)
{
#pragma unused (dbat6u)
	nofralloc
	mtspr SPR_DBAT6U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT6L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT6L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT6L (uint32 dbat6l)
{
#pragma unused (dbat6l)
	nofralloc
	mtspr SPR_DBAT6L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT7U (void)
{
	nofralloc
	mfspr r3,SPR_DBAT7U
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT7U (uint32 dbat7u)
{
#pragma unused (dbat7u)
	nofralloc
	mtspr SPR_DBAT7U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readDBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readDBAT7L (void)
{
	nofralloc
	mfspr r3,SPR_DBAT7L
	blr
}

/*--------------------------------------------------------------------------
   see void writeDBAT0U (uint32 dbat0u)
  --------------------------------------------------------------------------*/
asm void writeDBAT7L (uint32 dbat7l)
{
#pragma unused (dbat7l)
	nofralloc
	mtspr SPR_DBAT7L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   Function    : uint32 readIBAT0U (void)
   Description : Read IBAT0U register.
   Parameter(s): none
   Returns     : IBAT0U register contents.
   Note        : This function also exists for IBAT0L, IBAT1U, ..., IBAT7L                 
  --------------------------------------------------------------------------*/
asm uint32 readIBAT0U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT0U
	blr
}

/*--------------------------------------------------------------------------
   Function    : void writeIBAT0U (uint32 ibat0u)
   Description : Write IBAT0U register.
   Parameter(s): Value to write to IBAT0U register.
   Returns     : nothing
   Note        : This function also exists for IBAT0L, IBAT1U, ..., IBAT7L                 
  --------------------------------------------------------------------------*/
asm void writeIBAT0U (uint32 ibat0u)
{
#pragma unused (ibat0u)
	nofralloc
	mtspr SPR_IBAT0U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT0L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT0L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT0L (uint32 ibat0l)
{
#pragma unused (ibat0l)
	nofralloc
	mtspr SPR_IBAT0L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT1U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT1U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT1U (uint32 ibat1u)
{
#pragma unused (ibat1u)
	nofralloc
	mtspr SPR_IBAT1U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT1L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT1L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT1L (uint32 ibat1l)
{
#pragma unused (ibat1l)
	nofralloc
	mtspr SPR_IBAT1L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT2U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT2U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT2U (uint32 ibat2u)
{
#pragma unused (ibat2u)
	nofralloc
	mtspr SPR_IBAT2U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT2L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT2L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT2L (uint32 ibat2l)
{
#pragma unused (ibat2l)
	nofralloc
	mtspr SPR_IBAT2L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT3U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT3U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT3U (uint32 ibat3u)
{
#pragma unused (ibat3u)
	nofralloc
	mtspr SPR_IBAT3U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT3L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT3L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT3L (uint32 ibat3l)
{
#pragma unused (ibat3l)
	nofralloc
	mtspr SPR_IBAT3L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT4U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT4U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT4U (uint32 ibat4u)
{
#pragma unused (ibat4u)
	nofralloc
	mtspr SPR_IBAT4U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT4L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT4L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT4L (uint32 ibat4l)
{
#pragma unused (ibat4l)
	nofralloc
	mtspr SPR_IBAT4L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT5U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT5U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT5U (uint32 ibat5u)
{
#pragma unused (ibat5u)
	nofralloc
	mtspr SPR_IBAT5U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT5L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT5L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT5L (uint32 ibat5l)
{
#pragma unused (ibat5l)
	nofralloc
	mtspr SPR_IBAT5L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT6U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT6U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT6U (uint32 ibat6u)
{
#pragma unused (ibat6u)
	nofralloc
	mtspr SPR_IBAT6U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT6L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT6L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT6L (uint32 ibat6l)
{
#pragma unused (ibat6l)
	nofralloc
	mtspr SPR_IBAT6L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT7U (void)
{
	nofralloc
	mfspr r3,SPR_IBAT7U
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT7U (uint32 ibat7u)
{
#pragma unused (ibat7u)
	nofralloc
	mtspr SPR_IBAT7U,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   see uint32 readIBAT0U (void)
  --------------------------------------------------------------------------*/
asm uint32 readIBAT7L (void)
{
	nofralloc
	mfspr r3,SPR_IBAT7L
	blr
}

/*--------------------------------------------------------------------------
   see void writeIBAT0U (uint32 ibat0u)
  --------------------------------------------------------------------------*/
asm void writeIBAT7L (uint32 ibat7l)
{
#pragma unused (ibat7l)
	nofralloc
	mtspr SPR_IBAT7L,r3
	isync
	blr
}

/*--------------------------------------------------------------------------
   Function    : void writeSRR0 (uint32 srr0)
   Description : Write SRR0 register.
   Parameter(s): Value to write to SRR0 register.
   Returns     : nothing
  --------------------------------------------------------------------------*/
asm void writeSRR0 (uint32 srr0)
{
#pragma unused (srr0)
	nofralloc
	mtspr SPR_SRR0,r3
	blr
}

/*--------------------------------------------------------------------------
   see void writeSRR0 (uint32 srr0)
  --------------------------------------------------------------------------*/
asm void writeSRR1 (uint32 srr1)
{
#pragma unused (srr1)
	nofralloc
	mtspr SPR_SRR1,r3
	blr
}
