/******************************************************************************
*
*       COPYRIGHT (c) 2001-2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
*       The code is the property of Motorola.
*
*       The copyright notice above does not evidence any
*       actual or intended publication of such source code.
*
* Filename:     $Source: /proj/cvsroot/mgt/MGT5200/apps/AC97Sample/modules/Exceptions/exc5xxx.c,v $
* Author:       $Author: ra6707 $
* Locker:       $Locker:  $
* State:        $State: Exp $
* Revision:     $Revision: 1.1 $
*
* Functions:    
*
* History:      Use the RCS command rlog to display revision history
*               information.
*
* Description:  
*
* Notes:                
*
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ppctypes.h"

#include "configure.h"

#if !((defined(MGT5100) ^ defined(MGT5200) ^ defined(MPC5200B)) & \
     !(defined(MGT5100) & defined(MGT5200) & defined(MPC5200B)))
	#error Either MGT5100, MGT5200 or MPC5200B has to be defined.
#endif

#if defined(MGT5200) || defined(MPC5200B)
	#include "core5200.h"
	#include "spr5200.h"
#elif defined(MGT5100)
	#include "core5100.h"
	#include "spr5100.h"
#endif
#include "exc5xxx.h"

/*
 * non-public functions
 */
void exceptionHandlerMaster (uint32 excvec);
void exceptionInit (void);
void exceptionRestore (void);

/*
 * Exception table constants
 */
#define EXC_TABLE_HIGH		0xFFF00000UL
#define EXC_TABLE_LOW		0x00000000UL
#define EXC_TABLE_SIZE		0x1500

/*
 * Former contents of RAM were our exception table goes and location of
 * original exception table.
 */ 
uint32 orig_exceptiontable[EXC_TABLE_SIZE];
uint32 orig_exceptiontable_addr;

/*! \brief List member of registered exception handlers list.
 *
 * The list is sorted (ascending) by
 * exception vector and within the exception vector descending by priority.
 * I.e. priority 9 is highest priority, priority 0 is lowest priority. If there
 * are two exception handlers with the same priority the one which was
 * registered first will be called first.
 */
typedef struct {
	int vector;							/*!< exception vector handler is responsible for */
	int priority;						/*!< priority of handler from [0..9] where 0 is lowest and 9 is highest priority */
	int (*handler) (void *, void *);	/*!< pointer to the handler function */
	void *arg0;							/*!< argument 0 of the handler function; user definied */
	void *arg1;							/*!< argument 1 of the handler function; user definied */
} exceptionhandler_T;

/*! Maximum number of exception handlers */
#define MAX_EXC_HANDLERS	20

exceptionhandler_T exchandlerlist[MAX_EXC_HANDLERS];
int exchandlernum;

/*
 * Pointer to our exception table
 */
extern void exceptionTable (void);

/*! \brief Register a function to become the handler for an exception.
 *
 * A function must not be registered multiple times.
 * \param	vector		Exception vector to register handler for.
 * \param	priority	Priority of the handler (0 lowest, 9 highest).
 * \param	*handler	Pointer to the function that will be the exception
 *                      handler.
 * \param	*arg0		Arbitrary pointer which will be passed as an argument
 *                      to the exception handler when it's called.
 * \param	*arg1		Pointer like arg0.
 * \return 0, if registering the exception handler passed. Not 0 if registering
 *         the exception handler failed.
 */
int exceptionRegister (int vector, int priority, int (*handler) (void *, void *), void *arg0, void *arg1)
{
	int i;
	int ins_pos;

	/*
	 * Make priority valid
	 */
	if (priority < 0) {
		priority = 0;
	}
	if (priority > 9) {
		priority = 9;
	}
	
	/*
	 * If the list is already full, no more exception handlers can be
	 * registerd.
	 */
	if (exchandlernum == MAX_EXC_HANDLERS) {
		return -1;
	}
	
	/*
	 * Find the place in exception handler list to insert handler
	 */
	ins_pos = 0;
	while ((ins_pos < exchandlernum) &&
	       (exchandlerlist[ins_pos].vector <= vector)) {
		ins_pos++;
	}
	while ((ins_pos < exchandlernum) &&
	       (exchandlerlist[ins_pos].vector == vector) &&
	       (exchandlerlist[ins_pos].priority >= priority)) {
		ins_pos++;
	}

	/*
	 * Move down the rest of the list by one position to make space
	 */
	i = exchandlernum - 1;
	while (i >= ins_pos) {
		memcpy (&exchandlerlist[i+1], &exchandlerlist[i], sizeof (exceptionhandler_T));
		i--;
	}
	
	/*
	 * Insert new exception handler list entry
	 */
	exchandlerlist[ins_pos].vector   = vector;
	exchandlerlist[ins_pos].priority = priority;
	exchandlerlist[ins_pos].handler  = handler;
	exchandlerlist[ins_pos].arg0     = arg0;
	exchandlerlist[ins_pos].arg1     = arg1;
	exchandlernum++;

	return 0;
}

/*! \brief Remove a function as the handler for an exception.
 * \param	*handler	Pointer to the function to remove.
 * \return	0, if the function was removed successfully. Not 0, if removing the
 *          function failed.
 */
int exceptionRemove (int (*handler) (void *, void *))
{
	int i;
	int rem_pos;

	/*
	 * Find exception handler entry in the list.
	 */
	rem_pos = 0;
	while (rem_pos < exchandlernum) {
		if (exchandlerlist[rem_pos].handler == handler) {
			break;
		}
		rem_pos++;
	}

	if (rem_pos == exchandlernum) {
		return -1;
	}

	/*
	 * Move up the rest of the list to remove that entry.
	 */
	i = rem_pos + 1;
	while (i < exchandlernum) {
		memcpy (&exchandlerlist[i-1], &exchandlerlist[i], sizeof (exceptionhandler_T));
		i++;
	}
	exchandlernum--;

	return 0;
}

void exceptionHandlerMaster (uint32 excvec)
{
	int pos;
	int handled, regged;
	uint32 *excstack;
	uint32 *rsp;

	/*
	 * Go through list of exception handlers and look for one handling this
	 * exception.
	 */
	regged = 0;
	handled = 0;
	for (pos=0; (pos < exchandlernum) && !handled; pos++) {
		if (exchandlerlist[pos].vector == excvec) {
			regged = -1;
			handled = exchandlerlist[pos].handler (exchandlerlist[pos].arg0, exchandlerlist[pos].arg1);
		}
	}
	
	/*
	 * If the exception was handled by a registered exception handler there is
	 * nothing more to do.
	 */
	if (handled) {
		return;
	}

	/*
	 * If there is no registered Decrementer exception handler don't print the
	 * warning about an unhandled Decrementer exception.
	 */
	if (!regged && (excvec == EXC_DECREMENTER)) {
		return;
	}
	
	/*
	 * If no exception handler was found, print information about unhandled
	 * exception
	 */
	printf ("\n############################################################\n");
	printf ("Unhandled ");
	switch (excvec) {
		case EXC_SYSTEM_RESET:
			printf ("System Reset Exception");
			break;
			
		case EXC_MACHINE_CHECK:
			printf ("Machine Check Exception");
			break;
			
		case EXC_DSI:
			printf ("Data Storage Interrupt");
			break;
			
		case EXC_ISI:
			printf ("Instruction Storage Interrupt");
			break;
			
		case EXC_EXTERNAL_INT:
			printf ("External Interrupt");
			break;
			
		case EXC_ALIGNMENT:
			printf ("Aligment Exception");
			break;
			
		case EXC_PROGRAM:
			printf ("Program Exception");
			break;
			
		case EXC_FP_UNAVAILABLE:
			printf ("Floating-Point Unavailable Exception");
			break;
			
		case EXC_DECREMENTER:
			printf ("Decrementer Exception");
			break;
			
		case EXC_CRITICAL_INT:
			printf ("Critical Interrupt");
			break;
			
		case EXC_SYSTEM_CALL:
			printf ("System Call Exception");
			break;
		
		case EXC_TRACE:
			printf ("Trace Exception");
			break;
			
		case EXC_FP_ASSIST:
			printf ("Floating-Point Assist Exception");
			break;

		case EXC_ITLB_MISS:
			printf ("Instruction TLB Miss Exception");
			break;
			
		case EXC_DTLB_MISS_LOAD:
			printf ("Data TLB Miss on Load Exception");
			break;
			
		case EXC_DTLB_MISS_STORE:
			printf ("Data TLB Miss on Store Exception");
			break;
			
		case EXC_IABREAK:
			printf ("Instruction Address Breakpoint Exception");
			break;
			
		case EXC_SMI:
			printf ("System Management Interrupt");
			break;
			
		default:
			printf ("Unknown Exception");
			break;
	}
	printf (" (%04X)\n", excvec);
	rsp = (uint32 *) readRSP ();
	excstack = (uint32 *) rsp[0];
	printf ("Address where exception occured: %08X\n", excstack[6]);
	printf ("############################################################\n");

	/*
	 * Exit program
	 */
	exit (0);
}

void exceptionInit (void)
{
	uint32 msr;

	msr = readMSR ();

	/*
	 * Initialize exception handler table.
	 */
	exchandlernum = 0;
	
	/*
	 * Store location of original exception table
	 */
	if (msr & MSR_IP) {
		orig_exceptiontable_addr = EXC_TABLE_HIGH;
	} else {
		orig_exceptiontable_addr = EXC_TABLE_LOW;
	}

	/*
	 * Disable exception sources that can be disabled
	 */
	writeMSR (msr & 0xFFFF607D);
	
	/*
	 * Store original exception table
	 * (or whatever is located at the place the new exception table will be)
	 */
	memcpy ((void *) orig_exceptiontable, (void *) EXC_TABLE_LOW, EXC_TABLE_SIZE);
		
	/*
	 * Write new exception table
	 */
	memcpy ((void *) EXC_TABLE_LOW, exceptionTable, EXC_TABLE_SIZE);
	
	/*
	 * Enable new exception table located at 0x00000000 and reenable
	 * already enabled exceptions.
	 */
	clrbit (msr, MSR_IP);
	writeMSR (msr);
}

void exceptionRestore (void)
{
	uint32 msr;

	msr = readMSR ();

	/*
	 * Disable exception sources possible to disabled
	 */
	writeMSR (msr & 0xFFFF607D);

	/*
	 * Restore original exception table
	 * (or whatever has been at the place the new exception table has been)
	 */
	memcpy ((void *) EXC_TABLE_LOW, orig_exceptiontable, EXC_TABLE_SIZE);
	
	/*
	 * Re-enable original exception table
	 */
	if (orig_exceptiontable_addr == EXC_TABLE_LOW) {
		clrbit (msr, MSR_IP);
	} else {
		setbit (msr, MSR_IP);
	}
	writeMSR (msr);
}
