/*******************************************************************************/
/**
Copyright (c) 2007 Freescale Semiconductor
Freescale Confidential Proprietary
\file       FlexRay_handler.c
\brief      FlexRay module handling routines.
            Code based on the Transmit/Receive application example 
            for the FlexRay UNIFIED driver in interrupt driven mode
            (transmit_receive_node1.c) made by R62779.            
\author     Freescale Semiconductor
\author     Guadalajara Applications Laboratory RTAC Americas
\author     Jaime Orozco
\version    1.1
\date       March/12/2007
*/
/*******************************************************************************/
/*                                                                             */
/* All software, source code, included documentation, and any implied know-how */
/* are property of Freescale Semiconductor and therefore considered            */ 
/* CONFIDENTIAL INFORMATION.                                                   */
/*                                                                             */
/* This confidential information is disclosed FOR DEMONSTRATION PURPOSES ONLY. */
/*                                                                             */
/* All Confidential Information remains the property of Freescale Semiconductor*/
/* and will not be copied or reproduced without the express written permission */
/* of the Discloser, except for copies that are absolutely necessary in order  */
/* to fulfill the Purpose.                                                     */
/*                                                                             */
/* Services performed by FREESCALE in this matter are performed AS IS and      */
/* without any warranty. CUSTOMER retains the final decision relative to the   */
/* total design and functionality of the end product.                          */
/*                                                                             */
/* FREESCALE neither guarantees nor will be held liable by CUSTOMER for the    */
/* success of this project.                                                    */
/*                                                                             */
/* FREESCALE disclaims all warranties, express, implied or statutory including,*/
/* but not limited to, implied warranty of merchantability or fitness for a    */
/* particular purpose on any hardware, software or advise supplied to the      */
/* project by FREESCALE, and or any product resulting from FREESCALE services. */
/*                                                                             */
/* In no event shall FREESCALE be liable for incidental or consequential       */
/* damages arising out of this agreement. CUSTOMER agrees to hold FREESCALE    */
/* harmless against any and all claims demands or actions by anyone on account */
/* of any damage,or injury, whether commercial, contractual, or tortuous,      */
/* rising directly or indirectly as a result of the advise or assistance       */
/* supplied CUSTOMER in connection with product, services or goods supplied    */
/* under this Agreement.                                                       */
/*                                                                             */
/*******************************************************************************/


/** S12X derivative information */ 
#include "M9S12XF512.h"       

/** Macro definitions for general purpose I/O handling  */
#include "GPIO_macros.h"

/** Standard and driver types */
#include "Fr_UNIFIED_types.h"  

/** UNIFIED driver implementation */         
#include "Fr_UNIFIED.h"        
         
/** Configuration data for the FlexRay node */
#include "Fr_UNIFIED_cfg.h"

				
/** Double transmit MB 0 - commit side, slot 1 */
#define TX_SLOT_1                   0   
/** Double transmit MB 1 - transmit side, slot 1 */
#define TX_SLOT_1_TRANSMIT_SIDE     1   
/** Receive MB 3, slot 4 */
#define RX_SLOT_4                   3   

/** Return values */
Fr_return_type return_value;      
/** Current protocol state */      
Fr_POC_state_type protocol_state;       
/** Current wakeup status */
Fr_wakeup_state_type wakeup_status;     
/** Current cycle value */
uint8 current_cycle;                    
/** Current macrotick value */
uint16 current_macrotick;               


/** Definition of the variables used for updating of the transmit MBs */
/** Data array - static segment, slot 1 */
uint16 tx_data_1[16] = {0};             
/** Transmission return values */
Fr_tx_MB_status_type tx_return_value;   

/** Variables used for storing data and status from the receive MBs */
/** Data array - static segment, slot 4 */
uint16 rx_data_4[16] = {0};             
/** Received data length */
uint8 rx_data_length = 0;               
/** Received frame status */
uint16 rx_status_slot = 0;              
/** Reception return values */
Fr_rx_MB_status_type rx_return_value;    

/** Variables used for storing data and status of the FIFO A */
/** Data array */
uint16 fifo_data[8] = {0};              
/** Received data length */
uint8 fifo_data_length = 0;             
/** Received frame status */
uint16 fifo_status_slot = 0;            
/** Received frame index */
uint16 fifo_slot_idx = 0;               
/** Return values */
Fr_FIFO_status_type fifo_return_value;  


/******************************************************************************/
/**
* \brief    Error function for debugging 
* \author   Jaime Orozco
* \param    u8number: error code
* \return   void
*/
void Failed(uint8 u8number)               
{    
    while(u8number);       /* Function only for debugging, CC should be restarted  */
}

/*******************************************************************************/
/**
* \brief    Function for transmission on Slot 1
* \author   R62779
* \param    buffer_idx: Message Buffer identifier
* \return   void
*/
void CC_interrupt_slot_1(uint8 buffer_idx)
{
    /* Update double transmit MB with new data (commit side) */
    tx_return_value = Fr_transmit_data(TX_SLOT_1, &tx_data_1[0], 16);
    
    /* It is necessary to clear the flag at tranmsit side */
    Fr_clear_MB_interrupt_flag(TX_SLOT_1_TRANSMIT_SIDE);
    
    if(tx_return_value == FR_TXMB_UPDATED)
    {
        tx_data_1[0]++;     /* Increment variable */
    }    
    
    LED_OFF(D23);  /* Clear LED indicator */ 
}

/*******************************************************************************/
/**
* \brief    Function for reception on Slot 4  
* \author   R62779
* \param    buffer_idx: Message Buffer identifier
* \return   void
*/
void CC_interrupt_slot_4(uint8 buffer_idx)
{
    /* Copy received data into given array */
    rx_return_value = Fr_receive_data(buffer_idx, &rx_data_4[0],
                                      &rx_data_length, &rx_status_slot);
                                      
    tx_data_1[1] = rx_status_slot;  /* Store status slot */
    
    /* Increment data array which should be transmitted in slot 1 */
    tx_data_1[2]++;     
    
    LED_OFF(D25);   /* Clear LED indicator */
}

/*******************************************************************************/
/**
* \brief    Function for FlexRay Timer 1 
* \author   R62779
* \param    void
* \return   void
*/
void CC_interrupt_timer_1(void)
{
    /* Get the global time */
    Fr_get_global_time(&current_cycle, &current_macrotick);     
    tx_data_1[14] = current_macrotick;      /* Store current macrotick value */
    tx_data_1[15] = (uint16) (current_cycle);   /* Store current cycle value */
}

/*******************************************************************************/
/**
* \brief    Function for FlexRay Timer 2 
* \author   R62779
* \param    void
* \return   void
*/
void CC_interrupt_timer_2(void)
{
    /* Get the global time */
    Fr_get_global_time(&current_cycle, &current_macrotick);
    tx_data_1[12] = current_macrotick;      /* Store current macrotick value */
    tx_data_1[13] = (uint16) (current_cycle);   /* Store current cycle value */
}

/*******************************************************************************/
/**
* \brief    Function for FlexRay cycle start interrupt
* \author   R62779
* \param    void
* \return   void
*/
void CC_interrupt_cycle_start(void)
{
    /* Get the global time */
    Fr_get_global_time(&current_cycle, &current_macrotick); 
    tx_data_1[11] = current_cycle;  /* Store current cycle value */
}

/*******************************************************************************/
/**
* \brief    Function for FlexRay FIFO A interrupt
* \author   R62779
* \param    header_idx: Header index
* \return   void
*/
void CC_interrupt_FIFO_A(uint16 header_idx)
{
    /* Copy received data into given array  */
    fifo_return_value = Fr_receive_fifo_data(header_idx, &fifo_data[0],
                        &fifo_data_length, &fifo_slot_idx, &fifo_status_slot);
    
    if(fifo_return_value == FR_FIFO_RECEIVED)   /* Data has been received? */
    {                                  
        /* Store header index into data array - position 8 */
        tx_data_1[3] = header_idx;     
        /* Store value of the first field into data array - position 9 */     
        tx_data_1[4] = fifo_data[0];        
        
        /* Update appropriate variables in case that a frame has been received */
        switch(fifo_slot_idx)
        {
        case 4:          
            /* Slot 4,fault:
            Store 0xF0F3 value into data array - position 5 */
            tx_data_1[5] = 0xF0F3;   
            break;  
            
        case 5:                             
            /* Slot 5, fault:
            Store 0xF0F2 value into data array - position 6 */
            tx_data_1[6] = 0xF0F2;          
            break;
            
        case 62:                            
            /* Slot 62: Increment variable in case that a frame
            has been received - position 7 */
            tx_data_1[7]++;  
            LED_OFF(D28);   /* Clear LED indicator */               
            break;
            
        case 63:   
            /* Slot 63: Increment variable in case that a frame
            has been received - position 8 */     
            tx_data_1[8]++;                            
            break;
            
        default:                            
            /* Another frame - fault */
            /* Store 0xF0F0 value into data array - position 9 */
            tx_data_1[9] = 0xF0F0;          
            break;
        }
    }
    
    /* If data has not been received */
    else if(fifo_return_value == FR_FIFO_NOT_RECEIVED)  
    {
        /* Store 0xFFFF value to the transmit data array, position 10 */
        tx_data_1[10] = 0xFFFF;              
    }

}

/*******************************************************************************/
/**
* \brief    FlexRay module configuration
* \author   Jaime Orozco
* \param    void
* \return   void
*/
void vfnFlexRay_Init(void) {     
 
    /* Enable the FlexRay CC and force it into FR_POCSTATE_CONFIG */
    return_value = Fr_init(&Fr_HW_cfg_00, &Fr_low_level_cfg_set_00);
    
        if(return_value == FR_NOT_SUCCESS)
            Failed(1);   /* Call debug function in case of any error */

    /* Initialization of the FlexRay CC with protocol configuration parameter */
    Fr_set_configuration(&Fr_HW_cfg_00, &Fr_low_level_cfg_set_00);
    
    /* Initialization of message buffers, receive shadow buffers and FIFO storages */
    Fr_buffers_init(&Fr_buffer_cfg_00[0], &Fr_buffer_cfg_set_00[0]);
    
    /* Set callback function in case that an interrupt from MB 0 occurs */
    Fr_set_MB_callback(&CC_interrupt_slot_1, TX_SLOT_1_TRANSMIT_SIDE);  

    /* Set callback function in case that an interrupt from MB 3 occurs */
    Fr_set_MB_callback(&CC_interrupt_slot_4, RX_SLOT_4);

    /* Initialization of the timers */
    Fr_timers_init(&Fr_timers_cfg_00_ptr[0]);

    /* Set callback function in case that an interrupt from timer 1 occurs */
    Fr_set_protocol_0_IRQ_callback(&CC_interrupt_timer_1, FR_TIMER_1_EXPIRED_IRQ);

    /* Set callback function in case that an interrupt from timer 2 occurs */
    Fr_set_protocol_0_IRQ_callback(&CC_interrupt_timer_2, FR_TIMER_2_EXPIRED_IRQ);

    /* Set callback function in case that cycle start interrupt occurs */
    Fr_set_protocol_0_IRQ_callback(&CC_interrupt_cycle_start, FR_CYCLE_START_IRQ);

    /* Set callback function in case that an Interrupt occurs */
    Fr_set_fifo_IRQ_callback(&CC_interrupt_FIFO_A, FR_FIFO_A_IRQ);    
        
    
    /* Leave FR_POCSTATE_CONFIG state */
    return_value = Fr_leave_configuration_mode();
    
        if(return_value == FR_NOT_SUCCESS)
            Failed(2);   /* Call debug function in case of any error  */
    
    wakeup_status = Fr_get_wakeup_state();  /* Retrieve the wakeup state */
    
        /* Check whether a wakeup pattern has been received */
        if(wakeup_status == FR_WAKEUPSTATE_UNDEFINED)
        {   /* No wakeup pattern has been received, initiate wakeup procedure */
            return_value = Fr_send_wakeup();
                if(return_value == FR_NOT_SUCCESS)
                    Failed(3);   /* Call debug function in case of any error */
        }
   
    protocol_state = Fr_get_POC_state();    /* Load the current POC state */
    
    
        /* Wait till the FR CC is not in the FR_POCSTATE_READY */
        while(Fr_get_POC_state() != FR_POCSTATE_READY)  
            protocol_state = Fr_get_POC_state(); /* Current POC state */
        
    return_value = Fr_start_communication();    /* Initialize startup  */
    
        if(return_value == FR_NOT_SUCCESS)
            Failed(4);   /* Call debug function in case of any error */
    
    
    protocol_state = Fr_get_POC_state();    /* Load the current POC state */
    
        /* Wait till the FR CC is not in the FR_POCSTATE_NORMAL_ACTIVE */
        while(Fr_get_POC_state() != FR_POCSTATE_NORMAL_ACTIVE)
            protocol_state = Fr_get_POC_state(); /* Current POC state */       
  
    protocol_state = Fr_get_POC_state();    /* Load the current POC state */
    
    /* First initialization of the message buffer 0 */
    tx_return_value = Fr_transmit_data(TX_SLOT_1, &tx_data_1[0], 16);
        if(tx_return_value == FR_TXMB_NO_ACCESS)
            Failed(5);   /* Call debug function in case of any error */

    wakeup_status = Fr_get_wakeup_state();  /* Load current wakeup status */      

    /* Enable appropriate interrupts */
    Fr_enable_interrupts((FR_PROTOCOL_IRQ | FR_FIFO_A_IRQ | FR_RECEIVE_IRQ | FR_TRANSMIT_IRQ), 
                    (FR_TIMER_1_EXPIRED_IRQ | FR_TIMER_2_EXPIRED_IRQ | FR_CYCLE_START_IRQ), 0);
                        
    Fr_start_timer(FR_TIMER_T1);    /* Start Timer T1 */
    Fr_start_timer(FR_TIMER_T2);    /* Start Timer T2 */
                        
} 

/************************************************************************************/

