/*------------------------------------------------------------------------------

*

*   Filename:  705B16.h

*   

*   Current Revision: 1.0

*   Current Revision Release Date: 08/28/98

*

*   Description:    This file contains the memory map definition and the register

*                   bit definition for the MC68HC705B16N microcontroller, as defined

*                   in the databook MC68HC05B6/D Rev 4.

*

*   This header file has been validated with the following compilers:

*       Archemedies IDE HC05 compiler, Rev 2.7.10

*       Cosmic MC68HC05 compiler, Rev 4.1c - must use #pragma space [] @near setting

*

*   This file is provided "as is" without warranty of any kind either expressed or

*   implied, including, but not limited to any warranties of merchantability and

*   fitness for a particular purpose.  All risks of using this product including

*   the entire costs of any necessary remedies are those of the user and MOTOROLA

*   assumes no liability of any kind.

*   

*   

*   History:

*

*   08/28/98    rsf     Created

*

------------------------------------------------------------------------------*/



/*--------------------------------------

*   General types that are needed

--------------------------------------*/



typedef char                    BYTE;

typedef unsigned char           UBYTE;

typedef volatile unsigned char  VUBYTE;



/*------------------------------------------------------------------------------

*   Memory Map definition of I/O registers, as a structure.

*   

*   

*   This structure is defined as a type IOREGS.  There is also a label defined, LMREGS, that

*   you can make use of in your code to reference the structure.  See the example code

*   below the structure definition.

*

*   Please note that the Option Register (OPTION) and the Mask Option Register (MOR) are

*   not defined in this struct since they are at the end of the memory map.

------------------------------------------------------------------------------*/



typedef struct ioregs {

    VUBYTE  PORTA;          /* Port A Data Register */

    VUBYTE  PORTB;          /* Port B Data Register */

    VUBYTE  PORTC;          /* Port C Data Register */

    VUBYTE  PORTD;          /* Port D Data Register */

    VUBYTE  DDRA;           /* Data Direction Register A */

    VUBYTE  DDRB;           /* Data Direction Register B */

    VUBYTE  DDRC;           /* Data Direction Register C */

    VUBYTE  EEPROMCTL;      /* E/EEPROM/ECLK Control */

    VUBYTE  ADDATA;         /* A/D Data Register */

    VUBYTE  ADSTAT;         /* A/D Status/Control Register */

    VUBYTE  PLMA;           /* Pulse Length Modulation A */

    VUBYTE  PLMB;           /* Pulse Length Modulation B */

    VUBYTE  MISC;           /* Miscelaneous */

    VUBYTE  BAUD;           /* SCI Baud Rate Register */

    VUBYTE  SCCR1;          /* SCI Control Register 1 */

    VUBYTE  SCCR2;          /* SCI Control Register 2 */

    VUBYTE  SCSR;           /* SCI Status Register */

    VUBYTE  SCDR;           /* SCI Data Register */

    VUBYTE  TCR;            /* Timer Control Register */

    VUBYTE  TSR;            /* Timer Status Register */

    VUBYTE  ICRH1;          /* Input Capture Register 1 MSB */

    VUBYTE  ICRL1;          /* Input Capture Register 1 LSB */

    VUBYTE  OCRH1;          /* Output Compare Register 1 MSB */

    VUBYTE  OCRL1;          /* Output Compare Register 1 LSB */

    VUBYTE  TRH;            /* Timer Register MSB */

    VUBYTE  TRL;            /* Timer Register LSB */

    VUBYTE  ATRH;           /* Alternate Timer Register MSB */

    VUBYTE  ATRL;           /* Alternate Timer Register LSB */

    VUBYTE  ICRH2;          /* Input Capture Register 2 MSB */

    VUBYTE  ICRL2;          /* Input Capture Register 2 LSB */

    VUBYTE  OCRH2;          /* Output Compare Register 2 MSB */

    VUBYTE  OCRL2;          /* Output Compare Register 2 LSB */

} IOREGS;



#define LMREGS  (*(IOREGS *)(0x0000))   /* LMREGS (i.e. low memory registers) defines the */

                                        /* IOREGS structure at its natural base in memory */

                                        

/*------------------------------------------------------------------------------

*   Here is an example of how to use this structure definition in your code:

------------------------------------------------------------------------------*/

#ifdef  not_def

   

   UBYTE   tempa, tempb;         /* allocate some variables */

   

   LMREGS.PORTA = PA4 | PA3;     /* set bits 4 and 3 in port a so we can use them as outputs */

   LMREGS.DDRA = DDRA4 | DDRA3;  /* make bits 4 and 3 in port a output bits */

   LMREGS.PORTB = PB6 | PB5;     /* set bits 6 and 5 in port b so we can use them as outputs */

   LMREGS.DDRB = DDRB6 | DDRB5;  /* make bits 6 and 5 in port b output bits */

   

   tempa = LMREGS.PORTA          /* read port a */

   tempb = LMREGS.PORTB          /* read port b */

   

   /* set PB6:PB5 to the value of PA1:PA0 */

   LMREGS.PORTB = (LMREGS.PORTA & (PA1 | PA0)) << B_PB5;



#endif       



/*--------------------------------------

*   Memory Map definition, as constants.

*

*   

*   This time each register is defined as a constant pointer value, you can make

*   use of this in your code by simply referencing the value at the pointer.

*   See the example code below the memory definitions.

*   

--------------------------------------*/



/*--------------------------------------

*   I/O Registers

--------------------------------------*/



#define P_PORTA     ((VUBYTE *)(0x0000))    /* Port A Data Register */

#define P_PORTB     ((VUBYTE *)(0x0001))    /* Port B Data Register */

#define P_PORTC     ((VUBYTE *)(0x0002))    /* Port C Data Register */

#define P_PORTD     ((VUBYTE *)(0x0003))    /* Port D Data Register */

#define P_DDRA      ((VUBYTE *)(0x0004))    /* Data Direction Register A */

#define P_DDRB      ((VUBYTE *)(0x0005))    /* Data Direction Register B */

#define P_DDRC      ((VUBYTE *)(0x0006))    /* Data Direction Register C */

#define P_EEPROMCTL ((VUBYTE *)(0x0007))    /* E/EEPROM/ECLK Control */

#define P_ADDATA    ((VUBYTE *)(0x0008))    /* A/D Data Register */

#define P_ADSTAT    ((VUBYTE *)(0x0009))    /* A/D Status/Control Register */

#define P_PLMA      ((VUBYTE *)(0x000a))    /* Pulse Length Modulation A */

#define P_PLMB      ((VUBYTE *)(0x000b))    /* Pulse Length Modulation B */

#define P_MISC      ((VUBYTE *)(0x000c))    /* Miscelaneous */

#define P_BAUD      ((VUBYTE *)(0x000d))    /* SCI Baud Rate Register */

#define P_SCCR1     ((VUBYTE *)(0x000e))    /* SCI Control Register 1 */

#define P_SCCR2     ((VUBYTE *)(0x000f))    /* SCI Control Register 2 */

#define P_SCSR      ((VUBYTE *)(0x0010))    /* SCI Status Register */

#define P_SCDR      ((VUBYTE *)(0x0011))    /* SCI Data Register */

#define P_TCR       ((VUBYTE *)(0x0012))    /* Timer Control Register */

#define P_TSR       ((VUBYTE *)(0x0013))    /* Timer Status Register */

#define P_ICRH1     ((VUBYTE *)(0x0014))    /* Input Capture Register 1 MSB */

#define P_ICRL1     ((VUBYTE *)(0x0015))    /* Input Capture Register 1 LSB */

#define P_OCRH1     ((VUBYTE *)(0x0016))    /* Output Compare Register 1 MSB */

#define P_OCRL1     ((VUBYTE *)(0x0017))    /* Output Compare Register 1 LSB */

#define P_TRH       ((VUBYTE *)(0x0018))    /* Timer Register MSB */

#define P_TRL       ((VUBYTE *)(0x0019))    /* Timer Register LSB */

#define P_ATRH      ((VUBYTE *)(0x001a))    /* Alternate Timer Register MSB */

#define P_ATRL      ((VUBYTE *)(0x001b))    /* Alternate Timer Register LSB */

#define P_ICRH2     ((VUBYTE *)(0x001c))    /* Input Capture Register 2 MSB */

#define P_ICRL2     ((VUBYTE *)(0x001d))    /* Input Capture Register 2 LSB */

#define P_OCRH2     ((VUBYTE *)(0x001e))    /* Output Compare Register 2 MSB */

#define P_OCRL2     ((VUBYTE *)(0x001f))    /* Output Compare Register 2 LSB */

#define P_OPTION    ((VUBYTE *)(0x0100))    /* Option Register */

#define P_MOR       ((VUBYTE *)(0x3dfe))    /* Mask Option Register */



/*--------------------------------------

*   Exception Vectors

--------------------------------------*/



#define P_SCIV_H    ((UBYTE *)(0x3ff2))     /* SCI Interrupt Vector High */

#define P_SCIV_L    ((UBYTE *)(0x3ff3))     /* SCI Interrupt Vector Low */

#define P_TOVFL_H   ((UBYTE *)(0x3ff4))     /* Timer Overflow Interrupt Vector High */

#define P_TOVFL_L   ((UBYTE *)(0x3ff5))     /* Timer Overflow Interrupt Vector Low */

#define P_TOCV_H    ((UBYTE *)(0x3ff6))     /* Timer Output Capture Interrupt Vector High */

#define P_TOCV_L    ((UBYTE *)(0x3ff7))     /* Timer Output Capture Interrupt Vector Low */

#define P_TICV_H    ((UBYTE *)(0x3ff8))     /* Timer Input Capture Interrupt Vector High */

#define P_TICV_L    ((UBYTE *)(0x3ff9))     /* Timer Input Capture Interrupt Vector Low */

#define P_EIV_H     ((UBYTE *)(0x3ffa))     /* External Interrupt Vector High */

#define P_EIV_L     ((UBYTE *)(0x3ffb))     /* External Interrupt Vector Low */

#define P_SIV_H     ((UBYTE *)(0x3ffc))     /* Software Interrupt Vector High */

#define P_SIV_L     ((UBYTE *)(0x3ffd))     /* Software Interrupt Vector Low */

#define P_RESET_H   ((UBYTE *)(0x3ffe))     /* Reset Vector High */

#define P_RESET_L   ((UBYTE *)(0x3fff))     /* Reset Vector Low */



/*--------------------------------------

*   Memory Regions

--------------------------------------*/



#define P_EPROM0_BASE   ((UBYTE *)(0x0020))     /* first address of Page 0 EPROM */

#define P_EPROM0_TOP    ((UBYTE *)(0x004f))     /* last address of Page 0 EPROM */

#define P_RAM1_BASE     ((UBYTE *)(0x0050))     /* first address of RAM1 */

#define P_RAM1_TOP      ((UBYTE *)(0x00bf))     /* last address of RAM1 */

#define P_STACK_BASE    ((UBYTE *)(0x00c0))     /* first address of STACK */

#define P_STACK_TOP     ((UBYTE *)(0x00ff))     /* last address of STACK */   

#define P_EEPROM_BASE   ((UBYTE *)(0x0101))     /* first address of EEPROM */

#define P_EEPROM_TOP    ((UBYTE *)(0x01ff))     /* last address of EEPROM */

#define P_RAM11_BASE    ((UBYTE *)(0x0250))     /* first address of RAM11 */

#define P_RAM11_TOP     ((UBYTE *)(0x02ff))     /* last address of RAM11 */

#define P_EPROM_BASE    ((UBYTE *)(0x0300))     /* first address of EPROM */

#define P_EPROM_TOP     ((UBYTE *)(0x3dfd))     /* last address of EPROM */



/*------------------------------------------------------------------------------

*   Here is an example of how to use these pointer definitions in your code:

------------------------------------------------------------------------------*/

#ifdef  not_def

   

   UBYTE   tempa, tempb;       /* allocate some variables */

   

   *P_PORTA = PA4 | PA3;       /* set bits 4 and 3 in port a so we can use them as outputs */

   *P_DDRA = DDRA4 | DDRA3;    /* make bits 4 and 3 in port a output bits */

   *P_PORTB = PB6 | PB5;       /* set bits 6 and 5 in port b so we can use them as outputs */

   *P_DDRB = DDRB6 | DDRB5;    /* make bits 6 and 5 in port b output bits */

   

   tempa = *P_PORTA                                /* read port a */

   tempb = *P_PORTB                                /* read port b */

   *P_PORTB = (*P_PORTA & (PA1 | PA0)) << B_PB5;   /* set PB6:PB5 to the value of PA1:PA0 */

   

#endif



/*------------------------------------------------------------------------------

*   Register Definitions

*   

*   The registers are defined in the same order as they seqeuntially appear in

*   the memory map.

------------------------------------------------------------------------------*/



/*--------------------------------------

*   PORTA

*   Port A Data Register

*   

*   Address: 0x0000

*   Manual Page: 4-4

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PA0 - Port A Data Bit 0

*     |   |   |   |   |   |   |-------- PA1 - Port A Data Bit 1

*     |   |   |   |   |   |------------ PA2 - Port A Data Bit 2

*     |   |   |   |   |---------------- PA3 - Port A Data Bit 3

*     |   |   |   |-------------------- PA4 - Port A Data Bit 4

*     |   |   |------------------------ PA5 - Port A Data Bit 5

*     |   |---------------------------- PA6 - Port A Data Bit 6

*     |-------------------------------- PA7 - Port A Data Bit 7

*     

*   These read/write bits are software programmable. Data direction of each port

*   A pin is under the control of the corresponding bit in data direction register A.

*   Reset has no effect on port A data.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PA0 0x01    /* Port A Data Bit 0 */

#define PA1 0x02    /* Port A Data Bit 1 */

#define PA2 0x04    /* Port A Data Bit 2 */

#define PA3 0x08    /* Port A Data Bit 3 */

#define PA4 0x10    /* Port A Data Bit 4 */

#define PA5 0x20    /* Port A Data Bit 5 */

#define PA6 0x40    /* Port A Data Bit 6 */

#define PA7 0x80    /* Port A Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PA0   0   /* Port A Data Bit 0 */

#define B_PA1   1   /* Port A Data Bit 1 */

#define B_PA2   2   /* Port A Data Bit 2 */

#define B_PA3   3   /* Port A Data Bit 3 */

#define B_PA4   4   /* Port A Data Bit 4 */

#define B_PA5   5   /* Port A Data Bit 5 */

#define B_PA6   6   /* Port A Data Bit 6 */

#define B_PA7   7   /* Port A Data Bit 7 */



/*--------------------------------------

*   PORTB

*   Port B Data Register

*   

*   Address: 0x0001

*   Manual Page: 4-4

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PB0 - Port B Data Bit 0

*     |   |   |   |   |   |   |-------- PB1 - Port B Data Bit 1

*     |   |   |   |   |   |------------ PB2 - Port B Data Bit 2

*     |   |   |   |   |---------------- PB3 - Port B Data Bit 3

*     |   |   |   |-------------------- PB4 - Port B Data Bit 4

*     |   |   |------------------------ PB5 - Port B Data Bit 5

*     |   |---------------------------- PB6 - Port B Data Bit 6

*     |-------------------------------- PB7 - Port B Data Bit 7

*   

*   These read/write bits are software programmable. Data direction of each port

*   B pin is under the control of the corresponding bit in data direction register B.

*   Reset has no effect on port B data.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PB0 0x01    /* Port B Data Bit 0 */

#define PB1 0x02    /* Port B Data Bit 1 */

#define PB2 0x04    /* Port B Data Bit 2 */

#define PB3 0x08    /* Port B Data Bit 3 */

#define PB4 0x10    /* Port B Data Bit 4 */

#define PB5 0x20    /* Port B Data Bit 5 */

#define PB6 0x40    /* Port B Data Bit 6 */

#define PB7 0x80    /* Port B Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PB0   0   /* Port B Data Bit 0 */

#define B_PB1   1   /* Port B Data Bit 1 */

#define B_PB2   2   /* Port B Data Bit 2 */

#define B_PB3   3   /* Port B Data Bit 3 */

#define B_PB4   4   /* Port B Data Bit 4 */

#define B_PB5   5   /* Port B Data Bit 5 */

#define B_PB6   6   /* Port B Data Bit 6 */

#define B_PB7   7   /* Port B Data Bit 7 */



/*--------------------------------------

*   PORTC

*   Port C Data Register

*   

*   Address: 0x0002

*   Manual Page: 4-4

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PC0 - Port C Data Bit 0

*     |   |   |   |   |   |   |-------- PC1 - Port C Data Bit 1

*     |   |   |   |   |   |------------ PC2/ECLK - Port C Data Bit 2 / ECLK

*     |   |   |   |   |---------------- PC3 - Port C Data Bit 3

*     |   |   |   |-------------------- PC4 - Port C Data Bit 4

*     |   |   |------------------------ PC5 - Port C Data Bit 5

*     |   |---------------------------- PC6 - Port C Data Bit 6

*     |-------------------------------- PC7 - Port C Data Bit 7

*     

*   These read/write bits are software programmable. Data direction of each port

*   C pin is under the control of the corresponding bit in data direction register C.

*   Reset has no effect on port C data.

*   

*   PC2 can be used to output the processor clock if the ECLK bit is set in the EEPROM/ECLK

*   control register.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PC0   0x01    /* Port C Data Bit 0 */

#define PC1   0x02    /* Port C Data Bit 1 */

#define PC2   0x04    /* Port C Data Bit 2 */

#define ECLKO 0x04    /* ECLK out */

#define PC3   0x08    /* Port C Data Bit 3 */

#define PC4   0x10    /* Port C Data Bit 4 */

#define PC5   0x20    /* Port C Data Bit 5 */

#define PC6   0x40    /* Port C Data Bit 6 */

#define PC7   0x80    /* Port C Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PC0   0   /* Port C Data Bit 0 */

#define B_PC1   1   /* Port C Data Bit 1 */

#define B_PC2   2   /* Port C Data Bit 2 */

#define B_ECLKO 2   /* ECLK out */

#define B_PC3   3   /* Port C Data Bit 3 */

#define B_PC4   4   /* Port C Data Bit 4 */

#define B_PC5   5   /* Port C Data Bit 5 */

#define B_PC6   6   /* Port C Data Bit 6 */

#define B_PC7   7   /* Port C Data Bit 7 */



/*--------------------------------------

*   PORTD

*   Port D Data Register

*   

*   Address: 0x0003

*   Manual Page: 4-5

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PD0 - Port D Data Bit 0

*     |   |   |   |   |   |   |-------- PD1 - Port D Data Bit 1

*     |   |   |   |   |   |------------ PD2 - Port D Data Bit 2

*     |   |   |   |   |---------------- PD3 - Port D Data Bit 3

*     |   |   |   |-------------------- PD4 - Port D Data Bit 4

*     |   |   |------------------------ PD5 - Port D Data Bit 5

*     |   |---------------------------- PD6 - Port D Data Bit 6

*     |-------------------------------- PD7 - Port D Data Bit 7

*   

*   This port is shared with the A/D pins, and their function depends on the

*   ADON bit in the ADSTAT register.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PD0     0x01    /* Port D Data Bit 0 */

#define PD1     0x02    /* Port D Data Bit 1 */

#define PD2     0x04    /* Port D Data Bit 2 */

#define PD3     0x08    /* Port D Data Bit 3 */

#define PD4     0x10    /* Port D Data Bit 4 */

#define PD5     0x20    /* Port D Data Bit 5 */

#define PD6     0x40    /* Port D Data Bit 6 */

#define PD7     0x80    /* Port D Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PD0   0   /* Port D Data Bit 0 */

#define B_PD1   1   /* Port D Data Bit 1 */

#define B_PD2   2   /* Port D Data Bit 2 */

#define B_PD3   3   /* Port D Data Bit 3 */

#define B_PD4   4   /* Port D Data Bit 4 */

#define B_PD5   5   /* Port D Data Bit 5 */

#define B_PD6   6   /* Port D Data Bit 6 */

#define B_PD7   7   /* Port D Data Bit 7 */



/*--------------------------------------

*   DDRA

*   Data Direction Register A

*   

*   Address: 0x0004

*   Manual Page: 4-5

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- DDRA0 - Data Direction for Port A Data Bit 0

*     |   |   |   |   |   |   |-------- DDRA1 - Data Direction for Port A Data Bit 1

*     |   |   |   |   |   |------------ DDRA2 - Data Direction for Port A Data Bit 2

*     |   |   |   |   |---------------- DDRA3 - Data Direction for Port A Data Bit 3

*     |   |   |   |-------------------- DDRA4 - Data Direction for Port A Data Bit 4

*     |   |   |------------------------ DDRA5 - Data Direction for Port A Data Bit 5

*     |   |---------------------------- DDRA6 - Data Direction for Port A Data Bit 6

*     |-------------------------------- DDRA7 - Data Direction for Port A Data Bit 7

*     

*   These read/write bits control port A data direction.  Reset clears DDRA[7:0],

*   configuring all port A pins as inputs.

*       1 = corresponding port A pin configured as output

*       0 = corresponding port A pin configured as input

*       

*   Note:   Avoid glitches on port A pin by writing to the port A data register

*           before changing data direction bits from 0 to 1.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRA0   0x01    /* Data Direction for Port A Data Bit 0 */

#define DDRA1   0x02    /* Data Direction for Port A Data Bit 1 */

#define DDRA2   0x04    /* Data Direction for Port A Data Bit 2 */

#define DDRA3   0x08    /* Data Direction for Port A Data Bit 3 */

#define DDRA4   0x10    /* Data Direction for Port A Data Bit 4 */

#define DDRA5   0x20    /* Data Direction for Port A Data Bit 5 */

#define DDRA6   0x40    /* Data Direction for Port A Data Bit 6 */

#define DDRA7   0x80    /* Data Direction for Port A Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRA0 0   /* Data Direction for Port A Data Bit 0 */

#define B_DDRA1 1   /* Data Direction for Port A Data Bit 1 */

#define B_DDRA2 2   /* Data Direction for Port A Data Bit 2 */

#define B_DDRA3 3   /* Data Direction for Port A Data Bit 3 */

#define B_DDRA4 4   /* Data Direction for Port A Data Bit 4 */

#define B_DDRA5 5   /* Data Direction for Port A Data Bit 5 */

#define B_DDRA6 6   /* Data Direction for Port A Data Bit 6 */

#define B_DDRA7 7   /* Data Direction for Port A Data Bit 7 */



/*--------------------------------------

*   DDRB

*   Data Direction Register B

*   

*   Address: 0x0005

*   Manual Page: 4-5

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- DDRB0 - Data Direction for Port B Data Bit 0

*     |   |   |   |   |   |   |-------- DDRB1 - Data Direction for Port B Data Bit 1

*     |   |   |   |   |   |------------ DDRB2 - Data Direction for Port B Data Bit 2

*     |   |   |   |   |---------------- DDRB3 - Data Direction for Port B Data Bit 3

*     |   |   |   |-------------------- DDRB4 - Data Direction for Port B Data Bit 4

*     |   |   |------------------------ DDRB5 - Data Direction for Port B Data Bit 5

*     |   |---------------------------- DDRB6 - Data Direction for Port B Data Bit 6

*     |-------------------------------- DDRB7 - Data Direction for Port B Data Bit 7

*     

*   These read/write bits control port B data direction.  Reset clears DDRB[7:0],

*   configuring all port B pins as inputs.

*       1 = corresponding port B pin configured as output

*       0 = corresponding port B pin configured as input

*       

*   Note:   Avoid glitches on port B pin by writing to the port B data register

*           before changing data direction bits from 0 to 1.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRB0   0x01    /* Data Direction for Port B Data Bit 0 */

#define DDRB1   0x02    /* Data Direction for Port B Data Bit 1 */

#define DDRB2   0x04    /* Data Direction for Port B Data Bit 2 */

#define DDRB3   0x08    /* Data Direction for Port B Data Bit 3 */

#define DDRB4   0x10    /* Data Direction for Port B Data Bit 4 */

#define DDRB5   0x20    /* Data Direction for Port B Data Bit 5 */

#define DDRB6   0x40    /* Data Direction for Port B Data Bit 6 */

#define DDRB7   0x80    /* Data Direction for Port B Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRB0 0   /* Data Direction for Port B Data Bit 0 */

#define B_DDRB1 1   /* Data Direction for Port B Data Bit 1 */

#define B_DDRB2 2   /* Data Direction for Port B Data Bit 2 */

#define B_DDRB3 3   /* Data Direction for Port B Data Bit 3 */

#define B_DDRB4 4   /* Data Direction for Port B Data Bit 4 */

#define B_DDRB5 5   /* Data Direction for Port B Data Bit 5 */

#define B_DDRB6 6   /* Data Direction for Port B Data Bit 6 */

#define B_DDRB7 7   /* Data Direction for Port B Data Bit 7 */



/*--------------------------------------

*   DDRC

*   Data Direction Register C

*   

*   Address: 0x0006

*   Manual Page: 4-5

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- DDRC0 - Data Direction for Port C Data Bit 0

*     |   |   |   |   |   |   |-------- DDRC1 - Data Direction for Port C Data Bit 1

*     |   |   |   |   |   |------------ DDRC2 - Data Direction for Port C Data Bit 2

*     |   |   |   |   |---------------- DDRC3 - Data Direction for Port C Data Bit 3

*     |   |   |   |-------------------- DDRC4 - Data Direction for Port C Data Bit 4

*     |   |   |------------------------ DDRC5 - Data Direction for Port C Data Bit 5

*     |   |---------------------------- DDRC6 - Data Direction for Port C Data Bit 6

*     |-------------------------------- DDRC7 - Data Direction for Port C Data Bit 7

*     

*   These read/write bits control port C data direction.  Reset clears DDRC[7:0],

*   configuring all port C pins as inputs.

*       1 = corresponding port C pin configured as output

*       0 = corresponding port C pin configured as input

*       

*   Note:   Avoid glitches on port C pin by writing to the port C data register

*           before changing data direction bits from 0 to 1.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRC0   0x01    /* Data Direction for Port C Data Bit 0 */

#define DDRC1   0x02    /* Data Direction for Port C Data Bit 1 */

#define DDRC2   0x04    /* Data Direction for Port C Data Bit 2 */

#define DDRC3   0x08    /* Data Direction for Port C Data Bit 3 */

#define DDRC4   0x10    /* Data Direction for Port C Data Bit 4 */

#define DDRC5   0x20    /* Data Direction for Port C Data Bit 5 */

#define DDRC6   0x40    /* Data Direction for Port C Data Bit 6 */

#define DDRC7   0x80    /* Data Direction for Port C Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRC0 0   /* Data Direction for Port C Data Bit 0 */

#define B_DDRC1 1   /* Data Direction for Port C Data Bit 1 */

#define B_DDRC2 2   /* Data Direction for Port C Data Bit 2 */

#define B_DDRC3 3   /* Data Direction for Port C Data Bit 3 */

#define B_DDRC4 4   /* Data Direction for Port C Data Bit 4 */

#define B_DDRC5 5   /* Data Direction for Port C Data Bit 5 */

#define B_DDRC6 6   /* Data Direction for Port C Data Bit 6 */

#define B_DDRC7 7   /* Data Direction for Port C Data Bit 7 */



/*--------------------------------------

*   EEPROMCTL

*   EPROM/EEPROM/ECLK Control

*   

*   Address: 0x0007

*   Manual Page: F-6

*

*    --  --  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- E1PGM - EEPROM Charge Pump Enable Bit

*     |   |   |   |   |   |   |-------- E1LAT - EEPROM Programming Latch Enable Bit

*     |   |   |   |   |   |------------ E1ERA - EEPROM Erase/Programming Bit

*     |   |   |   |   |---------------- ECLK - External Clock Output Bit

*     |   |   |   |-------------------- E6PGM - EPROM Program Enable Bit

*     |   |   |------------------------ E6LAT - EPROM Programming Latch Enable Bit

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define E1PGM   0x01    /* EEPROM Charge Pump Enable Bit */

#define E1LAT   0x02    /* EEPROM Programming Latch Enable Bit */

#define E1ERA   0x04    /* EEPROM Erase/Programming Bit */

#define ECLK    0x08    /* External Clock Output Bit */

#define E6PGM   0x10    /* EPROM Program Enable Bit */

#define E6LAT   0x20    /* EPROM Programming Latch Enable Bit */



#define EEPROMCTL_UNUSED 0xc0    /* bit mask of unused bits in EEPROMCTL register */

/*------------------

*   Bit Positions

------------------*/

#define B_E1PGM   0    /* EEPROM Charge Pump Enable Bit */

#define B_E1LAT   1    /* EEPROM Programming Latch Enable Bit */

#define B_E1ERA   2    /* EEPROM Erase/Programming Bit */

#define B_ECLK    3    /* External Clock Output Bit */

#define B_E6PGM   4    /* EPROM Program Enable Bit */

#define B_E6LAT   5    /* EPROM Programming Latch Enable Bit */



/*--------------------------------------

*   ADDATA

*   A/D Result Data Register

*   

*   Address: 0x0008

*   Manual Page: 8-3

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- AD0 - Analog Data Bit 0

*     |   |   |   |   |   |   |-------- AD1 - Analog Data Bit 1

*     |   |   |   |   |   |------------ AD2 - Analog Data Bit 2

*     |   |   |   |   |---------------- AD3 - Analog Data Bit 3

*     |   |   |   |-------------------- AD4 - Analog Data Bit 4

*     |   |   |------------------------ AD5 - Analog Data Bit 5

*     |   |---------------------------- AD6 - Analog Data Bit 6

*     |-------------------------------- AD7 - Analog Data Bit 7

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define AD0 0x01   /* Analog Data Bit 0 */

#define AD1 0x02   /* Analog Data Bit 1 */

#define AD2 0x04   /* Analog Data Bit 2 */

#define AD3 0x08   /* Analog Data Bit 3 */

#define AD4 0x10   /* Analog Data Bit 4 */

#define AD5 0x20   /* Analog Data Bit 5 */

#define AD6 0x40   /* Analog Data Bit 6 */

#define AD7 0x80   /* Analog Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_AD0 0   /* Analog Data Bit 0 */

#define B_AD1 1   /* Analog Data Bit 1 */

#define B_AD2 2   /* Analog Data Bit 2 */

#define B_AD3 3   /* Analog Data Bit 3 */

#define B_AD4 4   /* Analog Data Bit 4 */

#define B_AD5 5   /* Analog Data Bit 5 */

#define B_AD6 6   /* Analog Data Bit 6 */

#define B_AD7 7   /* Analog Data Bit 7 */



/*--------------------------------------

*   ADSTAT

*   A/D Status/Control Register

*   

*   Address: 0x0009

*   Manual Page: 8-4

*

*    RO  RW  RW  --  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- CH0 - A/D Channel 0

*     |   |   |   |   |   |   |-------- CH1 - A/D Channel 1

*     |   |   |   |   |   |------------ CH2 - A/D Channel 2

*     |   |   |   |   |---------------- CH3 - A/D Channel 3

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ ADON - A/D Converter On

*     |   |---------------------------- ADRC - A/D RC Oscillator Control

*     |-------------------------------- COCO - Conversion Complete Flag

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define CH0  0x01   /* A/D Channel 0 */

#define CH1  0x02   /* A/D Channel 1 */

#define CH2  0x04   /* A/D Channel 2 */

#define CH3  0x08   /* A/D Channel 3 */

#define ADON 0x20   /* A/D Converter On */

#define ADRC 0x40   /* A/D RC Oscillator Control */

#define COCO 0x80   /* Conversion Complete Flag */



#define ADSTAT_UNUSED  0x10    /* bitmask of unused bits in ADSTAT */



/*------------------

*   Bit Positions

------------------*/

#define B_CH0  0   /* A/D Channel 0 */

#define B_CH1  1   /* A/D Channel 1 */

#define B_CH2  2   /* A/D Channel 2 */

#define B_CH3  3   /* A/D Channel 3 */

#define B_ADON 5   /* A/D Converter On */

#define B_ADRC 6   /* A/D RC Oscillator Control */

#define B_COCO 7   /* Conversion Complete Flag */



/*--------------------------------------

*   PLMA

*   Pulse Length Modulation A Register

*   

*   Address: 0x000a

*   Manual Page: 7-2

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PLMA0 - PLMA Data Bit 0

*     |   |   |   |   |   |   |-------- PLMA1 - PLMA Data Bit 1

*     |   |   |   |   |   |------------ PLMA2 - PLMA Data Bit 2

*     |   |   |   |   |---------------- PLMA3 - PLMA Data Bit 3

*     |   |   |   |-------------------- PLMA4 - PLMA Data Bit 4

*     |   |   |------------------------ PLMA5 - PLMA Data Bit 5

*     |   |---------------------------- PLMA6 - PLMA Data Bit 6

*     |-------------------------------- PLMA7 - PLMA Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PLMA0 0x01    /* PLMA Data Bit 0 */

#define PLMA1 0x02    /* PLMA Data Bit 1 */

#define PLMA2 0x04    /* PLMA Data Bit 2 */

#define PLMA3 0x08    /* PLMA Data Bit 3 */

#define PLMA4 0x10    /* PLMA Data Bit 4 */

#define PLMA5 0x20    /* PLMA Data Bit 5 */

#define PLMA6 0x40    /* PLMA Data Bit 6 */

#define PLMA7 0x80    /* PLMA Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PLMA0   0   /* PLMA Data Bit 0 */

#define B_PLMA1   1   /* PLMA Data Bit 1 */

#define B_PLMA2   2   /* PLMA Data Bit 2 */

#define B_PLMA3   3   /* PLMA Data Bit 3 */

#define B_PLMA4   4   /* PLMA Data Bit 4 */

#define B_PLMA5   5   /* PLMA Data Bit 5 */

#define B_PLMA6   6   /* PLMA Data Bit 6 */

#define B_PLMA7   7   /* PLMA Data Bit 7 */



/*--------------------------------------

*   PLMB

*   Pulse Length Modulation B Register

*   

*   Address: 0x000b

*   Manual Page: 7-2

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PLMB0 - PLMB Data Bit 0

*     |   |   |   |   |   |   |-------- PLMB1 - PLMB Data Bit 1

*     |   |   |   |   |   |------------ PLMB2 - PLMB Data Bit 2

*     |   |   |   |   |---------------- PLMB3 - PLMB Data Bit 3

*     |   |   |   |-------------------- PLMB4 - PLMB Data Bit 4

*     |   |   |------------------------ PLMB5 - PLMB Data Bit 5

*     |   |---------------------------- PLMB6 - PLMB Data Bit 6

*     |-------------------------------- PLMB7 - PLMB Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PLMB0 0x01    /* PLMB Data Bit 0 */

#define PLMB1 0x02    /* PLMB Data Bit 1 */

#define PLMB2 0x04    /* PLMB Data Bit 2 */

#define PLMB3 0x08    /* PLMB Data Bit 3 */

#define PLMB4 0x10    /* PLMB Data Bit 4 */

#define PLMB5 0x20    /* PLMB Data Bit 5 */

#define PLMB6 0x40    /* PLMB Data Bit 6 */

#define PLMB7 0x80    /* PLMB Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PLMB0   0   /* PLMB Data Bit 0 */

#define B_PLMB1   1   /* PLMB Data Bit 1 */

#define B_PLMB2   2   /* PLMB Data Bit 2 */

#define B_PLMB3   3   /* PLMB Data Bit 3 */

#define B_PLMB4   4   /* PLMB Data Bit 4 */

#define B_PLMB5   5   /* PLMB Data Bit 5 */

#define B_PLMB6   6   /* PLMB Data Bit 6 */

#define B_PLMB7   7   /* PLMB Data Bit 7 */



/*--------------------------------------

*   MISC

*   Miscellaneous Register

*   

*   Address: 0x000c

*   Manual Page: 7-3, 9-2, 9-9

*

*    RO  RO  RO  RO  RW  RW  RW  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- WDOG - Watchdog Interrupt Enable Bit

*     |   |   |   |   |   |   |-------- SM - Slow Mode Bit

*     |   |   |   |   |   |------------ SFB - Slow/Fast Mode Bit PLMB

*     |   |   |   |   |---------------- SFA - Slow/Fast Mode Bit PLMA

*     |   |   |   |-------------------- INTE - External Interrupt Enable Bit

*     |   |   |------------------------ INTN - External Interrupt Negative Polarity Bit

*     |   |---------------------------- INTP - External Interrupt Positive Polarity Bit

*     |-------------------------------- POR - Power On Reset Bit

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define WDOG  0x01    /* Watchdog Interrupt Enable Bit */

#define SM    0x02    /* Slow Mode Bit */

#define SFB   0x04    /* Slow/Fast Mode Bit PLMB */

#define SFA   0x08    /* Slow/Fast Mode Bit PLMA */

#define INTE  0x10    /* External Interrupt Enable Bit */

#define INTN  0x20    /* External Interrupt Negative Polarity Bit */

#define INTP  0x40    /* External Interrupt Positive Polarity Bit */

#define POR   0x80    /* Power On Reset Bit */



/*------------------

*   Bit Positions

------------------*/

#define B_WDOG  0    /* Watchdog Interrupt Enable Bit */

#define B_SM    1    /* Slow Mode Bit */

#define B_SFB   2    /* Slow/Fast Mode Bit PLMB */

#define B_SFA   3    /* Slow/Fast Mode Bit PLMA */

#define B_INTE  4    /* External Interrupt Enable Bit */

#define B_INTN  5    /* External Interrupt Negative Polarity Bit */

#define B_INTP  6    /* External Interrupt Positive Polarity Bit */

#define B_POR   7    /* Power On Reset Bit */



/*--------------------------------------

*   BAUD

*   BAUD Rate Register

*   

*   Address: 0x000d

*   Manual Page: 6-18

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- SCR0 - SCI Receive Baud Rate Select Bit 0

*     |   |   |   |   |   |   |-------- SCR1 - SCI Receive Baud Rate Select Bit 1

*     |   |   |   |   |   |------------ SCR2 - SCI Receive Baud Rate Select Bit 2

*     |   |   |   |   |---------------- SCT0 - SCI Transmit Baud Rate Select Bit 0

*     |   |   |   |-------------------- SCT1 - SCI Transmit Baud Rate Select Bit 1

*     |   |   |------------------------ SCT2 - SCI Transmit Baud Rate Select Bit 2

*     |   |---------------------------- SCP0 - SCI Prescaler Select Bit 0

*     |-------------------------------- SCP1 - SCI Prescaler Select Bit 1

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define SCR0    0x01    /* SCI Receive Baud Rate Select Bit 0 */

#define SCR1    0x02    /* SCI Receive Baud Rate Select Bit 1 */

#define SCR2    0x04    /* SCI Receive Baud Rate Select Bit 2 */

#define SCT0    0x08    /* SCI Transmit Baud Rate Select Bit 0 */

#define SCT1    0x10    /* SCI Transmit Baud Rate Select Bit 1 */

#define SCT2    0x20    /* SCI Transmit Baud Rate Select Bit 2 */

#define SCP0    0x40    /* SCI Prescaler Select Bit 0 */

#define SCP1    0x80    /* SCI Prescaler Select Bit 1 */



/*------------------

*   Bit Positions

------------------*/

#define B_SCR0    0    /* SCI Receive Baud Rate Select Bit 0 */

#define B_SCR1    1    /* SCI Receive Baud Rate Select Bit 1 */

#define B_SCR2    2    /* SCI Receive Baud Rate Select Bit 2 */

#define B_SCT0    3    /* SCI Transmit Baud Rate Select Bit 0 */

#define B_SCT1    4    /* SCI Transmit Baud Rate Select Bit 1 */

#define B_SCT2    5    /* SCI Transmit Baud Rate Select Bit 2 */

#define B_SCP0    6    /* SCI Prescaler Select Bit 0 */

#define B_SCP1    7    /* SCI Prescaler Select Bit 1 */



/*--------------------------------------

*   SCCR1

*   SCI Control Register 1

*   

*   Address: 0x000e

*   Manual Page: 6-10

*

*    RW  RW  --  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- LBCL - Last Bit Clock

*     |   |   |   |   |   |   |-------- CPHA - Clock Phase Bit

*     |   |   |   |   |   |------------ CPOL - Clock Polarity Bit

*     |   |   |   |   |---------------- WAKE - Wakeup Bit

*     |   |   |   |-------------------- M - Character Length

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- T8 - Bit 8 ( Transmitted )

*     |-------------------------------- R8 - Bit 8 ( Received )

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define LBCL    0x01    /* Last Bit Clock */

#define CPHA    0x02    /* Clock Phase Bit */

#define CPOL    0x04    /* Clock Polarity Bit */

#define WAKE    0x08    /* Wakeup Bit */

#define M       0x10    /* Character Length */

#define T8      0x40    /* Bit 8 ( Transmitted ) */

#define R8      0x80    /* Bit 8 ( Received ) */



#define SCCR1_UNUSED    0x20    /* bit mask of unused bits in SCCR1 */



/*------------------

*   Bit Positions

------------------*/

#define B_LBCL  0   /* Last Bit Clock */

#define B_CPHA  1   /* Clock Phase Bit */

#define B_CPOL  2   /* Clock Polarity Bit */

#define B_WAKE  3   /* Wakeup Bit */

#define B_M     4   /* Character Length */

#define B_T8    6   /* Bit 8 ( Transmitted ) */

#define B_R8    7   /* Bit 8 ( Received ) */



/*--------------------------------------

*   SCCR2

*   SCI Control Register 2

*   

*   Address: 0x000f

*   Manual Page: 6-14

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- SBK - Send Break

*     |   |   |   |   |   |   |-------- RWU - Receiver Wakeup Enable

*     |   |   |   |   |   |------------ RE - Receive Enable

*     |   |   |   |   |---------------- TE - Transmit Enable

*     |   |   |   |-------------------- ILIE - Idle Line Interrupt Enable

*     |   |   |------------------------ RIE - Receive Interrupt Enable

*     |   |---------------------------- TCIE - Transmission Complete Interrupt Enable

*     |-------------------------------- TIE - Transmit Interrupt Enable

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define SBK     0x01    /* Send Break */

#define RWU     0x02    /* Receiver Wakeup Enable */

#define RE      0x04    /* Receive Enable */

#define TE      0x08    /* Transmit Enable */

#define ILIE    0x10    /* Idle Line Interrupt Enable */

#define RIE     0x20    /* Receive Interrupt Enable */

#define TCIE    0x40    /* Transmission Complete Interrupt Enable */

#define TIE     0x80    /* Transmit Interrupt Enable */



/*------------------

*   Bit Positions

------------------*/

#define B_SBK   0   /* Send Break */

#define B_RWU   1   /* Receiver Wakeup Enable */

#define B_RE    2   /* Receive Enable */

#define B_TE    3   /* Transmit Enable */

#define B_ILIE  4   /* Idle Line Interrupt Enable */

#define B_RIE   5   /* Receive Interrupt Enable */

#define B_TCIE  6   /* Transmission Complete Interrupt Enable */

#define B_TIE   7   /* Transmit Interrupt Enable */



/*--------------------------------------

*   SCSR

*   SCI Status Register

*   

*   Address: 0x0010

*   Manual Page: 6-16

*

*    RO  RO  RO  RO  RO  RO  RO  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- FE - Receiver Framing Error

*     |   |   |   |   |   |------------ NF - Receiver Noise Flag

*     |   |   |   |   |---------------- OR - Receiver Overrun

*     |   |   |   |-------------------- IDLE - Receiver Idle

*     |   |   |------------------------ RDRF - Receiver Data Register Full

*     |   |---------------------------- TC - Transmission Complete

*     |-------------------------------- TDRE - Transmit Data Register Empty

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define FE      0x02    /* Receiver Framing Error */

#define NF      0x04    /* Receiver Noise Flag */

#define OR      0x08    /* Receiver Overrun */

#define IDLE    0x10    /* Receiver Idle */

#define RDRF    0x20    /* Receiver Data Register Full */

#define TC      0x40    /* Transmission Complete */

#define TDRE    0x80    /* Transmit Data Register Empty */



#define SCSR_UNUSED 0x01    /* bit mask of unused bits in SCSR */



/*------------------

*   Bit Positions

------------------*/

#define B_FE    1   /* Receiver Framing Error */

#define B_NF    2   /* Receiver Noise Flag */

#define B_OR    3   /* Receiver Overrun */

#define B_IDLE  4   /* Receiver Idle */

#define B_RDRF  5   /* Receiver Data Register Full */

#define B_TC    6   /* Transmission Complete */

#define B_TDRE  7   /* Transmit Data Register Empty */



/*--------------------------------------

*   SCDR

*   SCI Data Register

*   

*   Address: 0x0011

*   Manual Page: 6-10

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- SCD0 - SCI Data Bit 0

*     |   |   |   |   |   |   |-------- SCD1 - SCI Data Bit 1

*     |   |   |   |   |   |------------ SCD2 - SCI Data Bit 2

*     |   |   |   |   |---------------- SCD3 - SCI Data Bit 3

*     |   |   |   |-------------------- SCD4 - SCI Data Bit 4

*     |   |   |------------------------ SCD5 - SCI Data Bit 5

*     |   |---------------------------- SCD6 - SCI Data Bit 6

*     |-------------------------------- SCD7 - SCI Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define SCD0 0x01    /* SCI Data Bit 0 */

#define SCD1 0x02    /* SCI Data Bit 1 */

#define SCD2 0x04    /* SCI Data Bit 2 */

#define SCD3 0x08    /* SCI Data Bit 3 */

#define SCD4 0x10    /* SCI Data Bit 4 */

#define SCD5 0x20    /* SCI Data Bit 5 */

#define SCD6 0x40    /* SCI Data Bit 6 */

#define SCD7 0x80    /* SCI Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_SCD0   0   /* SCI Data Bit 0 */

#define B_SCD1   1   /* SCI Data Bit 1 */

#define B_SCD2   2   /* SCI Data Bit 2 */

#define B_SCD3   3   /* SCI Data Bit 3 */

#define B_SCD4   4   /* SCI Data Bit 4 */

#define B_SCD5   5   /* SCI Data Bit 5 */

#define B_SCD6   6   /* SCI Data Bit 6 */

#define B_SCD7   7   /* SCI Data Bit 7 */



/*--------------------------------------

*   TCR

*   Timer Control Register

*   

*   Address: 0x0012

*   Manual Page: 5-4

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OLVL1 - Output Level 1

*     |   |   |   |   |   |   |-------- IEDG1 - Input Edge 1

*     |   |   |   |   |   |------------ OLV2 - Output Level 2

*     |   |   |   |   |---------------- FOLV1 - Force Output Compare 1

*     |   |   |   |-------------------- FOLV2 - Force Output Compare 2

*     |   |   |------------------------ TOIE - Timer Overflow Interrupt Enable

*     |   |---------------------------- OCIE - Output Compare Interrupt Enable

*     |-------------------------------- ICIE - Input Capture Interrupt Enable

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OLVL1 0x01    /* Output Level 1 */

#define IEDG1 0x02    /* Input Edge 1 */

#define OLV2  0x04    /* Output Level 2 */

#define FOLV1 0x08    /* Force Output Compare 1 */

#define FOLV2 0x10    /* Force Output Compare 2 */

#define TOIE  0x20    /* Timer Overflow Interrupt Enable */

#define OCIE  0x40    /* Output Compare Interrupt Enable */

#define ICIE  0x80    /* Input Capture Interrupt Enable */



/*------------------

*   Bit Positions

------------------*/

#define B_OLVL1  0   /* Output Level 1 */

#define B_IEDG1  1   /* Input Edge 1 */

#define B_OLV2   2   /* Output Level 2 */

#define B_FOLV1  3   /* Force Output Compare 1 */

#define B_FOLV2  4   /* Force Output Compare 2 */

#define B_TOIE   5   /* Timer Overflow Interrupt Enable */

#define B_OCIE   6   /* Output Compare Interrupt Enable */

#define B_ICIE   7   /* Input Capture Interrupt Enable */



/*--------------------------------------

*   TSR

*   Timer Status Register

*   

*   Address: 0x0013

*   Manual Page: 5-6

*

*    RO  RO  RO  RO  RO  --  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- OCF2 - Output Compare Flag 2

*     |   |   |   |-------------------- ICF2 - Input Capture Flag 2

*     |   |   |------------------------ TOF - Timer Overflow Flag

*     |   |---------------------------- OCF1 - Output Compare Flag 1

*     |-------------------------------- ICF1 - Input Capture Flag 1

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OCF2 0x08    /* Output Compare Flag 2 */

#define ICF2 0x10    /* Input Capture Flag 2 */

#define TOF  0x20    /* Timer Overflow Flag */

#define OCF1 0x40    /* Output Compare Flag 1 */

#define ICF1 0x80    /* Input Capture Flag 1 */



#define TSR_UNUSED  0x07    /* bit mask of unused bits in TSR */



/*------------------

*   Bit Positions

------------------*/

#define B_OCF2   3   /* Output Compare Flag 2 */

#define B_ICF2   4   /* Input Capture Flag 2 */

#define B_TOF    5   /* Timer Overflow Flag */

#define B_OCF1   6   /* Output Compare Flag 1 */

#define B_ICF1   7   /* Input Capture Flag 1 */



/*--------------------------------------

*   ICRH1

*   Input Capture Register High 1

*   

*   Address: 0x0014

*   Manual Page: 5-7

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ICRH1_8  - Input Capture High 1 Data Bit 8 

*     |   |   |   |   |   |   |-------- ICRH1_9  - Input Capture High 1 Data Bit 9 

*     |   |   |   |   |   |------------ ICRH1_10 - Input Capture High 1 Data Bit 10

*     |   |   |   |   |---------------- ICRH1_11 - Input Capture High 1 Data Bit 11

*     |   |   |   |-------------------- ICRH1_12 - Input Capture High 1 Data Bit 12

*     |   |   |------------------------ ICRH1_13 - Input Capture High 1 Data Bit 13

*     |   |---------------------------- ICRH1_14 - Input Capture High 1 Data Bit 14

*     |-------------------------------- ICRH1_15 - Input Capture High 1 Data Bit 15

*     

*   NOTE:   To prevent interrupts from occurring between readings of ICRH1 and ICRL1,

*           set the interrupt flag in the condition code register before reading

*           ICRH1 and clear the flag after reading ICRL1.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ICRH1_8  0x01    /* Input Capture High 1 Data Bit 8  */

#define ICRH1_9  0x02    /* Input Capture High 1 Data Bit 9  */

#define ICRH1_10 0x04    /* Input Capture High 1 Data Bit 10 */

#define ICRH1_11 0x08    /* Input Capture High 1 Data Bit 11 */

#define ICRH1_12 0x10    /* Input Capture High 1 Data Bit 12 */

#define ICRH1_13 0x20    /* Input Capture High 1 Data Bit 13 */

#define ICRH1_14 0x40    /* Input Capture High 1 Data Bit 14 */

#define ICRH1_15 0x80    /* Input Capture High 1 Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_ICRH1_8   0  /* Input Capture High 1 Data Bit 8  */

#define B_ICRH1_9   1  /* Input Capture High 1 Data Bit 9  */

#define B_ICRH1_10  2  /* Input Capture High 1 Data Bit 10 */

#define B_ICRH1_11  3  /* Input Capture High 1 Data Bit 11 */

#define B_ICRH1_12  4  /* Input Capture High 1 Data Bit 12 */

#define B_ICRH1_13  5  /* Input Capture High 1 Data Bit 13 */

#define B_ICRH1_14  6  /* Input Capture High 1 Data Bit 14 */

#define B_ICRH1_15  7  /* Input Capture High 1 Data Bit 15 */



/*--------------------------------------

*   ICRL1

*   Input Capture Register Low 1

*   

*   Address: 0x0015

*   Manual Page: 5-7

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ICRL1_0 - Input Capture Low 1 Data Bit 0 

*     |   |   |   |   |   |   |-------- ICRL1_1 - Input Capture Low 1 Data Bit 1 

*     |   |   |   |   |   |------------ ICRL1_2 - Input Capture Low 1 Data Bit 2

*     |   |   |   |   |---------------- ICRL1_3 - Input Capture Low 1 Data Bit 3

*     |   |   |   |-------------------- ICRL1_4 - Input Capture Low 1 Data Bit 4

*     |   |   |------------------------ ICRL1_5 - Input Capture Low 1 Data Bit 5

*     |   |---------------------------- ICRL1_6 - Input Capture Low 1 Data Bit 6

*     |-------------------------------- ICRL1_7 - Input Capture Low 1 Data Bit 7

*     

*   NOTE:   To prevent interrupts from occurring between readings of ICRH1 and ICRL1,

*           set the interrupt flag in the condition code register before reading

*           ICRH1 and clear the flag after reading ICRL1.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ICRL1_0  0x01    /* Input Capture Low 1 Data Bit 0 */

#define ICRL1_1  0x02    /* Input Capture Low 1 Data Bit 1 */

#define ICRL1_2  0x04    /* Input Capture Low 1 Data Bit 2 */

#define ICRL1_3  0x08    /* Input Capture Low 1 Data Bit 3 */

#define ICRL1_4  0x10    /* Input Capture Low 1 Data Bit 4 */

#define ICRL1_5  0x20    /* Input Capture Low 1 Data Bit 5 */

#define ICRL1_6  0x40    /* Input Capture Low 1 Data Bit 6 */

#define ICRL1_7  0x80    /* Input Capture Low 1 Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_ICRL1_0  0  /* Input Capture Low 1 Data Bit 0 */

#define B_ICRL1_1  1  /* Input Capture Low 1 Data Bit 1 */

#define B_ICRL1_2  2  /* Input Capture Low 1 Data Bit 2 */

#define B_ICRL1_3  3  /* Input Capture Low 1 Data Bit 3 */

#define B_ICRL1_4  4  /* Input Capture Low 1 Data Bit 4 */

#define B_ICRL1_5  5  /* Input Capture Low 1 Data Bit 5 */

#define B_ICRL1_6  6  /* Input Capture Low 1 Data Bit 6 */

#define B_ICRL1_7  7  /* Input Capture Low 1 Data Bit 7 */



/*--------------------------------------

*   OCRH1

*   Output Compare Register High 1

*   

*   Address: 0x0016

*   Manual Page: 5-9

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OCRH1_8  - Output Compare High 1 Data Bit 8 

*     |   |   |   |   |   |   |-------- OCRH1_9  - Output Compare High 1 Data Bit 9 

*     |   |   |   |   |   |------------ OCRH1_10 - Output Compare High 1 Data Bit 10

*     |   |   |   |   |---------------- OCRH1_11 - Output Compare High 1 Data Bit 11

*     |   |   |   |-------------------- OCRH1_12 - Output Compare High 1 Data Bit 12

*     |   |   |------------------------ OCRH1_13 - Output Compare High 1 Data Bit 13

*     |   |---------------------------- OCRH1_14 - Output Compare High 1 Data Bit 14

*     |-------------------------------- OCRH1_15 - Output Compare High 1 Data Bit 15

*     

*   NOTE:   To prevent OCF1 from being set between the time it is read and the time the

*           output compare registers are updated, you should disable interrupts in the

*           condition code register, write to OCFH1, clear the OCF1 bit by reading the TSR,

*           write OCRL1, and then re-enable interrupts in the condition code register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OCRH1_8  0x01    /* Output Compare High 1 Data Bit 8  */

#define OCRH1_9  0x02    /* Output Compare High 1 Data Bit 9  */

#define OCRH1_10 0x04    /* Output Compare High 1 Data Bit 10 */

#define OCRH1_11 0x08    /* Output Compare High 1 Data Bit 11 */

#define OCRH1_12 0x10    /* Output Compare High 1 Data Bit 12 */

#define OCRH1_13 0x20    /* Output Compare High 1 Data Bit 13 */

#define OCRH1_14 0x40    /* Output Compare High 1 Data Bit 14 */

#define OCRH1_15 0x80    /* Output Compare High 1 Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_OCRH1_8   0  /* Output Compare High 1 Data Bit 8  */

#define B_OCRH1_9   1  /* Output Compare High 1 Data Bit 9  */

#define B_OCRH1_10  2  /* Output Compare High 1 Data Bit 10 */

#define B_OCRH1_11  3  /* Output Compare High 1 Data Bit 11 */

#define B_OCRH1_12  4  /* Output Compare High 1 Data Bit 12 */

#define B_OCRH1_13  5  /* Output Compare High 1 Data Bit 13 */

#define B_OCRH1_14  6  /* Output Compare High 1 Data Bit 14 */

#define B_OCRH1_15  7  /* Output Compare High 1 Data Bit 15 */



/*--------------------------------------

*   OCRL1

*   Output Compare Register Low 1

*   

*   Address: 0x0017

*   Manual Page: 5-9

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OCRL1_0 - Output Compare Low 1 Data Bit 0 

*     |   |   |   |   |   |   |-------- OCRL1_1 - Output Compare Low 1 Data Bit 1 

*     |   |   |   |   |   |------------ OCRL1_2 - Output Compare Low 1 Data Bit 2

*     |   |   |   |   |---------------- OCRL1_3 - Output Compare Low 1 Data Bit 3

*     |   |   |   |-------------------- OCRL1_4 - Output Compare Low 1 Data Bit 4

*     |   |   |------------------------ OCRL1_5 - Output Compare Low 1 Data Bit 5

*     |   |---------------------------- OCRL1_6 - Output Compare Low 1 Data Bit 6

*     |-------------------------------- OCRL1_7 - Output Compare Low 1 Data Bit 7

*     

*   NOTE:   To prevent OCF1 from being set between the time it is read and the time the

*           output compare registers are updated, you should disable interrupts in the

*           condition code register, write to OCFH1, clear the OCF1 bit by reading the TSR,

*           write OCRL1, and then re-enable interrupts in the condition code register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OCRL1_0  0x01    /* Output Compare Low 1 Data Bit 0 */

#define OCRL1_1  0x02    /* Output Compare Low 1 Data Bit 1 */

#define OCRL1_2  0x04    /* Output Compare Low 1 Data Bit 2 */

#define OCRL1_3  0x08    /* Output Compare Low 1 Data Bit 3 */

#define OCRL1_4  0x10    /* Output Compare Low 1 Data Bit 4 */

#define OCRL1_5  0x20    /* Output Compare Low 1 Data Bit 5 */

#define OCRL1_6  0x40    /* Output Compare Low 1 Data Bit 6 */

#define OCRL1_7  0x80    /* Output Compare Low 1 Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_OCRL1_0  0  /* Output Compare Low 1 Data Bit 0 */

#define B_OCRL1_1  1  /* Output Compare Low 1 Data Bit 1 */

#define B_OCRL1_2  2  /* Output Compare Low 1 Data Bit 2 */

#define B_OCRL1_3  3  /* Output Compare Low 1 Data Bit 3 */

#define B_OCRL1_4  4  /* Output Compare Low 1 Data Bit 4 */

#define B_OCRL1_5  5  /* Output Compare Low 1 Data Bit 5 */

#define B_OCRL1_6  6  /* Output Compare Low 1 Data Bit 6 */

#define B_OCRL1_7  7  /* Output Compare Low 1 Data Bit 7 */



/*--------------------------------------

*   TRH

*   Timer Register High

*   

*   Address: 0x0018

*   Manual Page: 5-3

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- TRH8  - Timer High Data Bit 8 

*     |   |   |   |   |   |   |-------- TRH9  - Timer High Data Bit 9 

*     |   |   |   |   |   |------------ TRH10 - Timer High Data Bit 10

*     |   |   |   |   |---------------- TRH11 - Timer High Data Bit 11

*     |   |   |   |-------------------- TRH12 - Timer High Data Bit 12

*     |   |   |------------------------ TRH13 - Timer High Data Bit 13

*     |   |---------------------------- TRH14 - Timer High Data Bit 14

*     |-------------------------------- TRH15 - Timer High Data Bit 15

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TRH8  0x01    /* Timer High Data Bit 8  */

#define TRH9  0x02    /* Timer High Data Bit 9  */

#define TRH10 0x04    /* Timer High Data Bit 10 */

#define TRH11 0x08    /* Timer High Data Bit 11 */

#define TRH12 0x10    /* Timer High Data Bit 12 */

#define TRH13 0x20    /* Timer High Data Bit 13 */

#define TRH14 0x40    /* Timer High Data Bit 14 */

#define TRH15 0x80    /* Timer High Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_TRH8   0  /* Timer High Data Bit 8  */

#define B_TRH9   1  /* Timer High Data Bit 9  */

#define B_TRH10  2  /* Timer High Data Bit 10 */

#define B_TRH11  3  /* Timer High Data Bit 11 */

#define B_TRH12  4  /* Timer High Data Bit 12 */

#define B_TRH13  5  /* Timer High Data Bit 13 */

#define B_TRH14  6  /* Timer High Data Bit 14 */

#define B_TRH15  7  /* Timer High Data Bit 15 */



/*--------------------------------------

*   TRL

*   Timer Register Low

*   

*   Address: 0x0019

*   Manual Page: 5-3

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- TRL0 - Timer Low Data Bit 0 

*     |   |   |   |   |   |   |-------- TRL1 - Timer Low Data Bit 1 

*     |   |   |   |   |   |------------ TRL2 - Timer Low Data Bit 2

*     |   |   |   |   |---------------- TRL3 - Timer Low Data Bit 3

*     |   |   |   |-------------------- TRL4 - Timer Low Data Bit 4

*     |   |   |------------------------ TRL5 - Timer Low Data Bit 5

*     |   |---------------------------- TRL6 - Timer Low Data Bit 6

*     |-------------------------------- TRL7 - Timer Low Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TRL0  0x01    /* Timer Low Data Bit 0 */

#define TRL1  0x02    /* Timer Low Data Bit 1 */

#define TRL2  0x04    /* Timer Low Data Bit 2 */

#define TRL3  0x08    /* Timer Low Data Bit 3 */

#define TRL4  0x10    /* Timer Low Data Bit 4 */

#define TRL5  0x20    /* Timer Low Data Bit 5 */

#define TRL6  0x40    /* Timer Low Data Bit 6 */

#define TRL7  0x80    /* Timer Low Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_TRL0  0   /* Timer Low Data Bit 0 */

#define B_TRL1  1   /* Timer Low Data Bit 1 */

#define B_TRL2  2   /* Timer Low Data Bit 2 */

#define B_TRL3  3   /* Timer Low Data Bit 3 */

#define B_TRL4  4   /* Timer Low Data Bit 4 */

#define B_TRL5  5   /* Timer Low Data Bit 5 */

#define B_TRL6  6   /* Timer Low Data Bit 6 */

#define B_TRL7  7   /* Timer Low Data Bit 7 */



/*--------------------------------------

*   ATRH

*   Alternate Timer Register High

*   

*   Address: 0x001a

*   Manual Page: 5-3

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ATRH8  - Alternate Timer High Data Bit 8 

*     |   |   |   |   |   |   |-------- ATRH9  - Alternate Timer High Data Bit 9 

*     |   |   |   |   |   |------------ ATRH10 - Alternate Timer High Data Bit 10

*     |   |   |   |   |---------------- ATRH11 - Alternate Timer High Data Bit 11

*     |   |   |   |-------------------- ATRH12 - Alternate Timer High Data Bit 12

*     |   |   |------------------------ ATRH13 - Alternate Timer High Data Bit 13

*     |   |---------------------------- ATRH14 - Alternate Timer High Data Bit 14

*     |-------------------------------- ATRH15 - Alternate Timer High Data Bit 15

*     

*   NOTE:   To prevent interrupts from occurring between readings of ATRH and ATRL,

*           set the interrupt flag in the condition code register before reading

*           ATRH and clear the flag after reading ATRL.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ATRH8  0x01    /* Alternate Timer High Data Bit 8  */

#define ATRH9  0x02    /* Alternate Timer High Data Bit 9  */

#define ATRH10 0x04    /* Alternate Timer High Data Bit 10 */

#define ATRH11 0x08    /* Alternate Timer High Data Bit 11 */

#define ATRH12 0x10    /* Alternate Timer High Data Bit 12 */

#define ATRH13 0x20    /* Alternate Timer High Data Bit 13 */

#define ATRH14 0x40    /* Alternate Timer High Data Bit 14 */

#define ATRH15 0x80    /* Alternate Timer High Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_ATRH8   0  /* Alternate Timer High Data Bit 8  */

#define B_ATRH9   1  /* Alternate Timer High Data Bit 9  */

#define B_ATRH10  2  /* Alternate Timer High Data Bit 10 */

#define B_ATRH11  3  /* Alternate Timer High Data Bit 11 */

#define B_ATRH12  4  /* Alternate Timer High Data Bit 12 */

#define B_ATRH13  5  /* Alternate Timer High Data Bit 13 */

#define B_ATRH14  6  /* Alternate Timer High Data Bit 14 */

#define B_ATRH15  7  /* Alternate Timer High Data Bit 15 */



/*--------------------------------------

*   ATRL

*   Alternate Timer Register Low

*   

*   Address: 0x001b

*   Manual Page: 5-3

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ATRL0 - Alternate Timer Low Data Bit 0 

*     |   |   |   |   |   |   |-------- ATRL1 - Alternate Timer Low Data Bit 1 

*     |   |   |   |   |   |------------ ATRL2 - Alternate Timer Low Data Bit 2

*     |   |   |   |   |---------------- ATRL3 - Alternate Timer Low Data Bit 3

*     |   |   |   |-------------------- ATRL4 - Alternate Timer Low Data Bit 4

*     |   |   |------------------------ ATRL5 - Alternate Timer Low Data Bit 5

*     |   |---------------------------- ATRL6 - Alternate Timer Low Data Bit 6

*     |-------------------------------- ATRL7 - Alternate Timer Low Data Bit 7

*     

*   NOTE:   To prevent interrupts from occurring between readings of ATRH and ATRL,

*           set the interrupt flag in the condition code register before reading

*           ATRH and clear the flag after reading ATRL.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ATRL0  0x01    /* Alternate Timer Low Data Bit 0 */

#define ATRL1  0x02    /* Alternate Timer Low Data Bit 1 */

#define ATRL2  0x04    /* Alternate Timer Low Data Bit 2 */

#define ATRL3  0x08    /* Alternate Timer Low Data Bit 3 */

#define ATRL4  0x10    /* Alternate Timer Low Data Bit 4 */

#define ATRL5  0x20    /* Alternate Timer Low Data Bit 5 */

#define ATRL6  0x40    /* Alternate Timer Low Data Bit 6 */

#define ATRL7  0x80    /* Alternate Timer Low Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_ATRL0  0  /* Alternate Timer Low Data Bit 0 */

#define B_ATRL1  1  /* Alternate Timer Low Data Bit 1 */

#define B_ATRL2  2  /* Alternate Timer Low Data Bit 2 */

#define B_ATRL3  3  /* Alternate Timer Low Data Bit 3 */

#define B_ATRL4  4  /* Alternate Timer Low Data Bit 4 */

#define B_ATRL5  5  /* Alternate Timer Low Data Bit 5 */

#define B_ATRL6  6  /* Alternate Timer Low Data Bit 6 */

#define B_ATRL7  7  /* Alternate Timer Low Data Bit 7 */



/*--------------------------------------

*   ICRH2

*   Input Capture Register High 2

*   

*   Address: 0x001c

*   Manual Page: 5-8

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ICRH2_8  - Input Capture High 2 Data Bit 8 

*     |   |   |   |   |   |   |-------- ICRH2_9  - Input Capture High 2 Data Bit 9 

*     |   |   |   |   |   |------------ ICRH2_10 - Input Capture High 2 Data Bit 10

*     |   |   |   |   |---------------- ICRH2_11 - Input Capture High 2 Data Bit 11

*     |   |   |   |-------------------- ICRH2_12 - Input Capture High 2 Data Bit 12

*     |   |   |------------------------ ICRH2_13 - Input Capture High 2 Data Bit 13

*     |   |---------------------------- ICRH2_14 - Input Capture High 2 Data Bit 14

*     |-------------------------------- ICRH2_15 - Input Capture High 2 Data Bit 15

*     

*   NOTE:   To prevent interrupts from occurring between readings of ICRH2 and ICRL2,

*           set the interrupt flag in the condition code register before reading

*           ICRH2 and clear the flag after reading ICRL2.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ICRH2_8  0x01    /* Input Capture High 2 Data Bit 8  */

#define ICRH2_9  0x02    /* Input Capture High 2 Data Bit 9  */

#define ICRH2_10 0x04    /* Input Capture High 2 Data Bit 10 */

#define ICRH2_11 0x08    /* Input Capture High 2 Data Bit 11 */

#define ICRH2_12 0x10    /* Input Capture High 2 Data Bit 12 */

#define ICRH2_13 0x20    /* Input Capture High 2 Data Bit 13 */

#define ICRH2_14 0x40    /* Input Capture High 2 Data Bit 14 */

#define ICRH2_15 0x80    /* Input Capture High 2 Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_ICRH2_8   0  /* Input Capture High 2 Data Bit 8  */

#define B_ICRH2_9   1  /* Input Capture High 2 Data Bit 9  */

#define B_ICRH2_10  2  /* Input Capture High 2 Data Bit 10 */

#define B_ICRH2_11  3  /* Input Capture High 2 Data Bit 11 */

#define B_ICRH2_12  4  /* Input Capture High 2 Data Bit 12 */

#define B_ICRH2_13  5  /* Input Capture High 2 Data Bit 13 */

#define B_ICRH2_14  6  /* Input Capture High 2 Data Bit 14 */

#define B_ICRH2_15  7  /* Input Capture High 2 Data Bit 15 */



/*--------------------------------------

*   ICRL2

*   Input Capture Register Low 2

*   

*   Address: 0x001d

*   Manual Page: 5-8

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ICRL2_0 - Input Capture Low 2 Data Bit 0 

*     |   |   |   |   |   |   |-------- ICRL2_1 - Input Capture Low 2 Data Bit 1

*     |   |   |   |   |   |------------ ICRL2_2 - Input Capture Low 2 Data Bit 2

*     |   |   |   |   |---------------- ICRL2_3 - Input Capture Low 2 Data Bit 3

*     |   |   |   |-------------------- ICRL2_4 - Input Capture Low 2 Data Bit 4

*     |   |   |------------------------ ICRL2_5 - Input Capture Low 2 Data Bit 5

*     |   |---------------------------- ICRL2_6 - Input Capture Low 2 Data Bit 6

*     |-------------------------------- ICRL2_7 - Input Capture Low 2 Data Bit 7

*     

*   NOTE:   To prevent interrupts from occurring between readings of ICRH2 and ICRL2,

*           set the interrupt flag in the condition code register before reading

*           ICRH2 and clear the flag after reading ICRL2.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ICRL2_0  0x01    /* Input Capture Low 2 Data Bit 0 */

#define ICRL2_1  0x02    /* Input Capture Low 2 Data Bit 1 */

#define ICRL2_2  0x04    /* Input Capture Low 2 Data Bit 2 */

#define ICRL2_3  0x08    /* Input Capture Low 2 Data Bit 3 */

#define ICRL2_4  0x10    /* Input Capture Low 2 Data Bit 4 */

#define ICRL2_5  0x20    /* Input Capture Low 2 Data Bit 5 */

#define ICRL2_6  0x40    /* Input Capture Low 2 Data Bit 6 */

#define ICRL2_7  0x80    /* Input Capture Low 2 Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_ICRL2_0  0  /* Input Capture Low 2 Data Bit 0 */

#define B_ICRL2_1  1  /* Input Capture Low 2 Data Bit 1 */

#define B_ICRL2_2  2  /* Input Capture Low 2 Data Bit 2 */

#define B_ICRL2_3  3  /* Input Capture Low 2 Data Bit 3 */

#define B_ICRL2_4  4  /* Input Capture Low 2 Data Bit 4 */

#define B_ICRL2_5  5  /* Input Capture Low 2 Data Bit 5 */

#define B_ICRL2_6  6  /* Input Capture Low 2 Data Bit 6 */

#define B_ICRL2_7  7  /* Input Capture Low 2 Data Bit 7 */



/*--------------------------------------

*   OCRH2

*   Output Compare Register High 2

*   

*   Address: 0x001e

*   Manual Page: 5-10

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OCRH2_8  - Output Compare High 2 Data Bit 8 

*     |   |   |   |   |   |   |-------- OCRH2_9  - Output Compare High 2 Data Bit 9 

*     |   |   |   |   |   |------------ OCRH2_10 - Output Compare High 2 Data Bit 10

*     |   |   |   |   |---------------- OCRH2_11 - Output Compare High 2 Data Bit 11

*     |   |   |   |-------------------- OCRH2_12 - Output Compare High 2 Data Bit 12

*     |   |   |------------------------ OCRH2_13 - Output Compare High 2 Data Bit 13

*     |   |---------------------------- OCRH2_14 - Output Compare High 2 Data Bit 14

*     |-------------------------------- OCRH2_15 - Output Compare High 2 Data Bit 15

*     

*   NOTE:   To prevent OCF2 from being set between the time it is read and the time the

*           output compare registers are updated, you should disable interrupts in the

*           condition code register, write to OCFH2, clear the OCF2 bit by reading the TSR,

*           write OCRL2, and then re-enable interrupts in the condition code register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OCRH2_8  0x01    /* Output Compare High 2 Data Bit 8  */

#define OCRH2_9  0x02    /* Output Compare High 2 Data Bit 9  */

#define OCRH2_10 0x04    /* Output Compare High 2 Data Bit 10 */

#define OCRH2_11 0x08    /* Output Compare High 2 Data Bit 11 */

#define OCRH2_12 0x10    /* Output Compare High 2 Data Bit 12 */

#define OCRH2_13 0x20    /* Output Compare High 2 Data Bit 13 */

#define OCRH2_14 0x40    /* Output Compare High 2 Data Bit 14 */

#define OCRH2_15 0x80    /* Output Compare High 2 Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_OCRH2_8   0  /* Output Compare High 2 Data Bit 8  */

#define B_OCRH2_9   1  /* Output Compare High 2 Data Bit 9  */

#define B_OCRH2_10  2  /* Output Compare High 2 Data Bit 10 */

#define B_OCRH2_11  3  /* Output Compare High 2 Data Bit 11 */

#define B_OCRH2_12  4  /* Output Compare High 2 Data Bit 12 */

#define B_OCRH2_13  5  /* Output Compare High 2 Data Bit 13 */

#define B_OCRH2_14  6  /* Output Compare High 2 Data Bit 14 */

#define B_OCRH2_15  7  /* Output Compare High 2 Data Bit 15 */



/*--------------------------------------

*   OCRL2

*   Output Compare Register Low 2

*   

*   Address: 0x001f

*   Manual Page: 5-10

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OCRL2_0 - Output Compare Low 2 Data Bit 0 

*     |   |   |   |   |   |   |-------- OCRL2_1 - Output Compare Low 2 Data Bit 1 

*     |   |   |   |   |   |------------ OCRL2_2 - Output Compare Low 2 Data Bit 2

*     |   |   |   |   |---------------- OCRL2_3 - Output Compare Low 2 Data Bit 3

*     |   |   |   |-------------------- OCRL2_4 - Output Compare Low 2 Data Bit 4

*     |   |   |------------------------ OCRL2_5 - Output Compare Low 2 Data Bit 5

*     |   |---------------------------- OCRL2_6 - Output Compare Low 2 Data Bit 6

*     |-------------------------------- OCRL2_7 - Output Compare Low 2 Data Bit 7

*     

*   NOTE:   To prevent OCF2 from being set between the time it is read and the time the

*           output compare registers are updated, you should disable interrupts in the

*           condition code register, write to OCFH2, clear the OCF2 bit by reading the TSR,

*           write OCRL2, and then re-enable interrupts in the condition code register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OCRL2_0  0x01    /* Output Compare Low 2 Data Bit 0 */

#define OCRL2_1  0x02    /* Output Compare Low 2 Data Bit 1 */

#define OCRL2_2  0x04    /* Output Compare Low 2 Data Bit 2 */

#define OCRL2_3  0x08    /* Output Compare Low 2 Data Bit 3 */

#define OCRL2_4  0x10    /* Output Compare Low 2 Data Bit 4 */

#define OCRL2_5  0x20    /* Output Compare Low 2 Data Bit 5 */

#define OCRL2_6  0x40    /* Output Compare Low 2 Data Bit 6 */

#define OCRL2_7  0x80    /* Output Compare Low 2 Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_OCRL2_0  0  /* Output Compare Low 2 Data Bit 0 */

#define B_OCRL2_1  1  /* Output Compare Low 2 Data Bit 1 */

#define B_OCRL2_2  2  /* Output Compare Low 2 Data Bit 2 */

#define B_OCRL2_3  3  /* Output Compare Low 2 Data Bit 3 */

#define B_OCRL2_4  4  /* Output Compare Low 2 Data Bit 4 */

#define B_OCRL2_5  5  /* Output Compare Low 2 Data Bit 5 */

#define B_OCRL2_6  6  /* Output Compare Low 2 Data Bit 6 */

#define B_OCRL2_7  7  /* Output Compare Low 2 Data Bit 7 */



/*--------------------------------------

*   OPTION

*   Option Register

*   

*   Address: 0x0100

*   Manual Page: F-9

*

*    --  --  --  --  --  --  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- SEC - Secure Bit

*     |   |   |   |   |   |   |-------- EE1P - EEPROM protect bit

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

*     Note: This register is implemented in EEPROM, so reset has no affect on it.

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define SEC  0x01   /* Secure Bit */

#define EE1P 0x02   /* EEPROM protect bit */



/*------------------

*   Bit Positions

------------------*/

#define B_SEC  0   /* Secure Bit */

#define B_EE1P 1   /* EEPROM protect bit */



/*--------------------------------------

*   MOR

*   Mask Option Register

*   

*   Address: 0x3dfe

*   Manual Page: F-8

*

*    --  --  --  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PCPD - Port C pull-down

*     |   |   |   |   |   |   |-------- PBPD - Port B pull-down

*     |   |   |   |   |   |------------ WWAT - Watchdog during wait mode

*     |   |   |   |   |---------------- RWAT - Watchdog after reset

*     |   |   |   |-------------------- RTIM - Reset Time

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

*   Note:  This register is implemented as EPROM cells so in order to write a new

*          value to this register, you must follow the procedure for programming

*          the EPROM.  Typically this register is only written at EPROM program time.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PCPD 0x01    /* Port C pull-down */

#define PBPD 0x02    /* Port B pull-down */

#define WWAT 0x04    /* Watchdog during wait mode */

#define RWAT 0x08    /* Watchdog after reset */

#define RTIM 0x10    /* Reset Time */

*



/*------------------

*   Bit Positions

------------------*/

#define B_PCPD 0    /* Port C pull-down */

#define B_PBPD 1    /* Port B pull-down */

#define B_WWAT 2    /* Watchdog during wait mode */

#define B_RWAT 3    /* Watchdog after reset */

#define B_RTIM 4    /* Reset Time */



/*--------------------------------------

*   Register Manipulation Macros

--------------------------------------*/



/* PUT_REG writes an immediate value into the register */



#define PUT_REG(reg, value) reg = value



/* GET_REG reads the current value of a register and stores it in a variable */



#define GET_REG(reg, var)   var = reg



/* SET_REG_BITS or's the mask value into the register*/



#define SET_REG_BITS(reg, mask) reg |= mask



/* CLR_REG_BITS and's the inverse of the mask value into the register*/



#define CLR_REG_BITS(reg, mask) reg &= ~(mask)

