/*------------------------------------------------------------------------------

*

*   Filename:  705KJ1.h

*   

*   Current Revision: 1.1

*   Current Revision Release Date: 04/13/98

*

*   Description:    This file contains the memory map definition and the register

*                   bit definition for the MC68HC705KJ1 microcontroller, as defined

*                   in the databook MC68HC705KJ1/D Rev 1.0.

*

*   This header file has been validated with the following compilers:

*       Archemedies IDE HC05 compiler, Rev 2.7.10

*       Cosmic MC68HC05 compiler, Rev 4.1c - must use #pragma space [] @near setting

*

*   This file is provided "as is" without warranty of any kind either expressed or

*   implied, including, but not limited to any warranties of merchantability and

*   fitness for a particular purpose.  All risks of using this product including

*   the entire costs of any necessary remedies are those of the user and MOTOROLA

*   assumes no liability of any kind.

*   

*   

*   History:

*

*   04/09/98    rsf     Created

*

*   04/13/98    rsf     corrected comment errors in example code

*

------------------------------------------------------------------------------*/



/*--------------------------------------

*   General types that are needed

--------------------------------------*/



typedef char                    BYTE;

typedef unsigned char           UBYTE;

typedef volatile unsigned char  VUBYTE;



/*------------------------------------------------------------------------------

*   Memory Map definition of I/O registers, as a structure.

*   

*   

*   This structure is defined as a type IOREGS.  There is also a label defined, LMREGS, that

*   you can make use of in your code to reference the structure.  See the example code

*   below the structure definition.

*

*   Please note that the COP Register (COPR) and the Mask Option Register (MOR) are

*   not defined in this struct since they are at the end of the memory map.

------------------------------------------------------------------------------*/



typedef struct ioregs {

    VUBYTE  PORTA;          /* Port A Data Register */

    VUBYTE  PORTB;          /* Port B Data Register */

    VUBYTE  RESERVED1[2];

    VUBYTE  DDRA;           /* Data Direction Register A */

    VUBYTE  DDRB;           /* Data Direction Register B */

    VUBYTE  RESERVED2[2];

    VUBYTE  TSCR;           /* Timer Status and Control Register */

    VUBYTE  TCR;            /* Timer Counter Register */

    VUBYTE  ISCR;           /* IRQ Status and Control Register */

    VUBYTE  RESERVED3[5];

    VUBYTE  PDRA;           /* Pulldown Register Port A */

    VUBYTE  PDRB;           /* Pulldown Register Port B */

    VUBYTE  RESERVED4[6];

    VUBYTE  EPROG;          /* EPROM Programming Register */

    VUBYTE  RESERVED5[7];

} IOREGS;



#define LMREGS  (*(IOREGS *)(0x0000))   /* LMREGS (i.e. low memory registers) defines the */

                                        /* IOREGS structure at its natural base in memory */

                                        

/*------------------------------------------------------------------------------

*   Here is an example of how to use this structure definition in your code:

------------------------------------------------------------------------------*/

#ifdef  not_def

   

   UBYTE   tempa, tempb;         /* allocate some variables */

   

   LMREGS.PORTA = PA4 | PA3;     /* set bits 4 and 3 in port a so we can use them as outputs */

   LMREGS.DDRA = DDRA4 | DDRA3;  /* make bits 4 and 3 in port a output bits */

   LMREGS.PORTB = PB3 | PB2;     /* set bits 3 and 2 in port b so we can use them as outputs */

   LMREGS.DDRB = DDRB3 | DDRB2;  /* make bits 3 and 2 in port b output bits */

   

   tempa = LMREGS.PORTA          /* read port a */

   tempb = LMREGS.PORTB          /* read port b */

   

   /* set PB3:PB2 to the value of PA1:PA0 */

   LMREGS.PORTB = (LMREGS.PORTA & (PA1 | PA0)) << B_PB2;



#endif       



/*--------------------------------------

*   Memory Map definition, as constants.

*

*   

*   This time each register is defined as a constant pointer value, you can make

*   use of this in your code by simply referencing the value at the pointer.

*   See the example code below the memory definitions.

*   

--------------------------------------*/



/*--------------------------------------

*   I/O Registers

--------------------------------------*/



#define P_PORTA ((VUBYTE *)(0x0000))        /* Port A Data Register */

#define P_PORTB ((VUBYTE *)(0x0001))        /* Port B Data Register */

#define P_DDRA  ((VUBYTE *)(0x0004))        /* Data Direction Register A */

#define P_DDRB  ((VUBYTE *)(0x0005))        /* Data Direction Register B */

#define P_TSCR  ((VUBYTE *)(0x0008))        /* Timer Status and Control Register */

#define P_TCR   ((VUBYTE *)(0x0009))        /* Timer Counter Register */

#define P_ISCR  ((VUBYTE *)(0x000a))        /* IRQ Status and Control Register */

#define P_PDRA  ((VUBYTE *)(0x0010))        /* Pulldown Register Port A */

#define P_PDRB  ((VUBYTE *)(0x0011))        /* Pulldown Register Port B */

#define P_EPROG ((VUBYTE *)(0x0018))        /* EPROM Programming Register */

#define P_COPR  ((VUBYTE *)(0x07f0))        /* COP Register */

#define P_MOR   ((VUBYTE *)(0x07f1))        /* Mask Option Register */



/*--------------------------------------

*   Exception Vectors

--------------------------------------*/



#define P_TIV_H     ((UBYTE *)(0x07f8))     /* Timer Interrupt Vector High */

#define P_TIV_L     ((UBYTE *)(0x07f9))     /* Timer Interrupt Vector Low */

#define P_EIV_H     ((UBYTE *)(0x07fa))     /* External Interrupt Vector High */

#define P_EIV_L     ((UBYTE *)(0x07fb))     /* External Interrupt Vector Low */

#define P_SIV_H     ((UBYTE *)(0x07fc))     /* Software Interrupt Vector High */

#define P_SIV_L     ((UBYTE *)(0x07fd))     /* Software Interrupt Vector Low */

#define P_RESET_H   ((UBYTE *)(0x07fe))     /* Reset Vector High */

#define P_RESET_L   ((UBYTE *)(0x07ff))     /* Reset Vector Low */



/*--------------------------------------

*   Memory Regions

--------------------------------------*/



#define P_RAM_BASE      ((UBYTE *)(0x00c0))     /* first address of RAM */

#define P_RAM_TOP       ((UBYTE *)(0x00ff))     /* last address of RAM */

#define P_EPROM_BASE    ((UBYTE *)(0x0300))     /* first address of EPROM */

#define P_EPROM_TOP     ((UBYTE *)(0x07cf))     /* last address of EPROM */



/*------------------------------------------------------------------------------

*   Here is an example of how to use these pointer definitions in your code:

------------------------------------------------------------------------------*/

#ifdef  not_def

   

   UBYTE   tempa, tempb;       /* allocate some variables */

   

   *P_PORTA = PA4 | PA3;       /* set bits 4 and 3 in port a so we can use them as outputs */

   *P_DDRA = DDRA4 | DDRA3;    /* make bits 4 and 3 in port a output bits */

   *P_PORTB = PB3 | PB2;       /* set bits 3 and 2 in port b so we can use them as outputs */

   *P_DDRB = DDRB3 | DDRB2;    /* make bits 3 and 2 in port b output bits */

   

   tempa = *P_PORTA                                /* read port a */

   tempb = *P_PORTB                                /* read port b */

   *P_PORTB = (*P_PORTA & (PA1 | PA0)) << B_PB2;   /* set PB3:PB2 to the value of PA1:PA0 */

   

#endif



/*------------------------------------------------------------------------------

*   Register Definitions

*   

*   The registers are defined in the same order as they seqeuntially appear in

*   the memory map.

------------------------------------------------------------------------------*/



/*--------------------------------------

*   PORTA

*   Port A Data Register

*   

*   Address: 0x0000

*   Manual Page: 73

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PA0 - Port A Data Bit 0

*     |   |   |   |   |   |   |-------- PA1 - Port A Data Bit 1

*     |   |   |   |   |   |------------ PA2 - Port A Data Bit 2

*     |   |   |   |   |---------------- PA3 - Port A Data Bit 3

*     |   |   |   |-------------------- PA4 - Port A Data Bit 4

*     |   |   |------------------------ PA5 - Port A Data Bit 5

*     |   |---------------------------- PA6 - Port A Data Bit 6

*     |-------------------------------- PA7 - Port A Data Bit 7

*     

*   These read/write bits are software programmable. Data direction of each port

*   A pin is under the control of the corresponding bit in data direction register A.

*   Reset has no effect on port A data.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PA0 0x01    /* Port A Data Bit 0 */

#define PA1 0x02    /* Port A Data Bit 1 */

#define PA2 0x04    /* Port A Data Bit 2 */

#define PA3 0x08    /* Port A Data Bit 3 */

#define PA4 0x10    /* Port A Data Bit 4 */

#define PA5 0x20    /* Port A Data Bit 5 */

#define PA6 0x40    /* Port A Data Bit 6 */

#define PA7 0x80    /* Port A Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PA0   0   /* Port A Data Bit 0 */

#define B_PA1   1   /* Port A Data Bit 1 */

#define B_PA2   2   /* Port A Data Bit 2 */

#define B_PA3   3   /* Port A Data Bit 3 */

#define B_PA4   4   /* Port A Data Bit 4 */

#define B_PA5   5   /* Port A Data Bit 5 */

#define B_PA6   6   /* Port A Data Bit 6 */

#define B_PA7   7   /* Port A Data Bit 7 */



/*--------------------------------------

*   PORTB

*   Port B Data Register

*   

*   Address: 0x0001

*   Manual Page: 76

*

*    RO  RO  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- See Note

*     |   |   |   |   |   |   |-------- See Note

*     |   |   |   |   |   |------------ PB2 - Port B Data Bit 2

*     |   |   |   |   |---------------- PB3 - Port B Data Bit 3

*     |   |   |   |-------------------- See Note

*     |   |   |------------------------ See Note

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

*   Note:   Bits 0, 1, 4 and 5 represent unimplemented bits PB0, PB1, PB4 and PB5.

*           To ensure proper termination of these unused bits, you should set

*           them as input bits in the DDRB register.

*

*   These read/write bits are software programmable. Data direction of each port

*   B pin is under the control of the corresponding bit in data direction register B.

*   Reset has no effect on port B data.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PB2 0x04    /* Port B Data Bit 2 */

#define PB3 0x08    /* Port B Data Bit 3 */



#define PORTB_UNUSED    0xf3    /* bit mask of unused bits in PORTB */



/*------------------

*   Bit Positions

------------------*/

#define B_PB2   2   /* Port B Data Bit 2 */

#define B_PB3   3   /* Port B Data Bit 3 */



/*--------------------------------------

*   DDRA

*   Data Direction Register A

*   

*   Address: 0x0004

*   Manual Page: 73

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- DDRA0 - Data Direction for Port A Data Bit 0

*     |   |   |   |   |   |   |-------- DDRA1 - Data Direction for Port A Data Bit 1

*     |   |   |   |   |   |------------ DDRA2 - Data Direction for Port A Data Bit 2

*     |   |   |   |   |---------------- DDRA3 - Data Direction for Port A Data Bit 3

*     |   |   |   |-------------------- DDRA4 - Data Direction for Port A Data Bit 4

*     |   |   |------------------------ DDRA5 - Data Direction for Port A Data Bit 5

*     |   |---------------------------- DDRA6 - Data Direction for Port A Data Bit 6

*     |-------------------------------- DDRA7 - Data Direction for Port A Data Bit 7

*     

*   These read/write bits control port A data direction.  Reset clears DDRA[7:0],

*   configuring all port A pins as inputs.

*       1 = corresponding port A pin configured as output

*       0 = corresponding port A pin configured as input

*       

*   Note:   Avoid glitches on port A pin by writing to the port A data register

*           before changing data direction bits from 0 to 1.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRA0   0x01    /* Data Direction for Port A Data Bit 0 */

#define DDRA1   0x02    /* Data Direction for Port A Data Bit 1 */

#define DDRA2   0x04    /* Data Direction for Port A Data Bit 2 */

#define DDRA3   0x08    /* Data Direction for Port A Data Bit 3 */

#define DDRA4   0x10    /* Data Direction for Port A Data Bit 4 */

#define DDRA5   0x20    /* Data Direction for Port A Data Bit 5 */

#define DDRA6   0x40    /* Data Direction for Port A Data Bit 6 */

#define DDRA7   0x80    /* Data Direction for Port A Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRA0 0   /* Data Direction for Port A Data Bit 0 */

#define B_DDRA1 1   /* Data Direction for Port A Data Bit 1 */

#define B_DDRA2 2   /* Data Direction for Port A Data Bit 2 */

#define B_DDRA3 3   /* Data Direction for Port A Data Bit 3 */

#define B_DDRA4 4   /* Data Direction for Port A Data Bit 4 */

#define B_DDRA5 5   /* Data Direction for Port A Data Bit 5 */

#define B_DDRA6 6   /* Data Direction for Port A Data Bit 6 */

#define B_DDRA7 7   /* Data Direction for Port A Data Bit 7 */



/*--------------------------------------

*   DDRB

*   Data Direction Register B

*   

*   Address: 0x0005

*   Manual Page: 77

*

*    RO  RO  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- See Note1

*     |   |   |   |   |   |   |-------- See Note1

*     |   |   |   |   |   |------------ DDRB2 - Data Direction for Port B Data Bit 2

*     |   |   |   |   |---------------- DDRB3 - Data Direction for Port B Data Bit 3

*     |   |   |   |-------------------- See Note1

*     |   |   |------------------------ See Note1

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

*   Note1:  Bits 0, 1, 4 and 5 represent unimplemented bits DDRB0, DDRB1, DDRB4 and DDRB5.

*           To ensure proper termination of these unused bits, you should set

*           them to 1.

*   

*   These read/write bits control port B data direction.  Reset clears DDRB[3:2],

*   configuring all port B pins as inputs.

*       1 = corresponding port B pin configured as output

*       0 = corresponding port B pin configured as input

*       

*   Note2:  Avoid glitches on port B pin by writing to the port B data register

*           before changing data direction bits from 0 to 1.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRB2   0x04    /* Data Direction for Port B Data Bit 2 */

#define DDRB3   0x08    /* Data Direction for Port B Data Bit 3 */



#define DDRB_UNUSED 0xf3    /* bit mask of unused bits in DDRB */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRB2 2   /* Data Direction for Port B Data Bit 2 */

#define B_DDRB3 3   /* Data Direction for Port B Data Bit 3 */



/*--------------------------------------

*   TSCR

*   Timer Status and Control Register

*   

*   Address: 0x0008

*   Manual Page: 97

*

*    RO  RO  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- RT0   - Real-Time Interrupt Select Bit 0

*     |   |   |   |   |   |   |-------- RT1   - Real-Time Interrupt Select Bit 1

*     |   |   |   |   |   |------------ RTIFR - Real-Time Interrupt Flag Reset

*     |   |   |   |   |---------------- TOFR  - Timer Overflow Flag Reset       

*     |   |   |   |-------------------- RTIE  - Real-Time Interrupt Enable

*     |   |   |------------------------ TOIE  - Timer Overflow Interrupt Enable

*     |   |---------------------------- RTIF  - Real-Time Interrupt Flag        

*     |-------------------------------- TOF   - Timer Overflow Flag                         

*     

*   Note:   Changing RT1 and RT0 when a COP timeout is imminent can cause a real-time

*           interrupt request to be missed or an additional real-time interrupt request

*           to be generated.  To prevent this occurrence, clear the COP timer before

*           changing RT1 and RT0.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define RT0     0x01    /* Real-Time Interrupt Select Bit 0 */

#define RT1     0x02    /* Real-Time Interrupt Select Bit 1 */

#define RTIFR   0x04    /* Real-Time Interrupt Flag Reset */

#define TOFR    0x08    /* Timer Overflow Flag Reset */

#define RTIE    0x10    /* Real-Time Interrupt Enable */

#define TOIE    0x20    /* Timer Overflow Interrupt Enable */

#define RTIF    0x40    /* Real-Time Interrupt Flag */

#define TOF     0x80    /* Timer Overflow Flag */



/*------------------

*   Bit Positions

------------------*/

#define B_RT0     0   /* Real-Time Interrupt Select Bit 0 */

#define B_RT1     1   /* Real-Time Interrupt Select Bit 1 */

#define B_RTIFR   2   /* Real-Time Interrupt Flag Reset */

#define B_TOFR    3   /* Timer Overflow Flag Reset */

#define B_RTIE    4   /* Real-Time Interrupt Enable */

#define B_TOIE    5   /* Timer Overflow Interrupt Enable */

#define B_RTIF    6   /* Real-Time Interrupt Flag */

#define B_TOF     7   /* Timer Overflow Flag */



/*--------------------------------------

*   TCR

*   Timer Counter Register

*   

*   Address: 0x0009

*   Manual Page: 99

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- TMR0 - Timer Bit 0

*     |   |   |   |   |   |   |-------- TMR1 - Timer Bit 1

*     |   |   |   |   |   |------------ TMR2 - Timer Bit 2

*     |   |   |   |   |---------------- TMR3 - Timer Bit 3      

*     |   |   |   |-------------------- TMR4 - Timer Bit 4

*     |   |   |------------------------ TMR5 - Timer Bit 5

*     |   |---------------------------- TMR6 - Timer Bit 6  

*     |-------------------------------- TMR7 - Timer Bit 7                  

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TMR0     0x01    /* Timer Bit 0 */

#define TMR1     0x02    /* Timer Bit 1 */

#define TMR2     0x04    /* Timer Bit 2 */

#define TMR3     0x08    /* Timer Bit 3 */

#define TMR4     0x10    /* Timer Bit 4 */

#define TMR5     0x20    /* Timer Bit 5 */

#define TMR6     0x40    /* Timer Bit 6 */

#define TMR7     0x80    /* Timer Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_TMR0     0    /* Timer Bit 0 */

#define B_TMR1     1    /* Timer Bit 1 */

#define B_TMR2     2    /* Timer Bit 2 */

#define B_TMR3     3    /* Timer Bit 3 */

#define B_TMR4     4    /* Timer Bit 4 */

#define B_TMR5     5    /* Timer Bit 5 */

#define B_TMR6     6    /* Timer Bit 6 */

#define B_TMR7     7    /* Timer Bit 7 */



/*--------------------------------------

*   ISCR

*   IRQ Status and Control Register

*   

*   Address: 0x000a

*   Manual Page: 91

*

*    RW  RO  RO  RW  RO  RO  RW  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- IRQR - Interrupt Request Reset

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- IRQF - External Interrupt Request Flag

*     |   |   |   |-------------------- Reserved

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- Unimplemented

*     |-------------------------------- IRQE - External Interrupt Request Enable

*

*     The STOP and WAIT insturctions set the IRQE bit so that an external interrupt can

*     bring the MCU out of these low-power modes.  In addition, reset sets the I bit which

*     masks all sources.      

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define IRQR    0x02    /* Interrupt Request Reset */

#define IRQF    0x08    /* External Interrupt Request Flag */

#define IRQE    0x80    /* External Interrupt Request Enable */



#define ISCR_UNUSED 0x75    /* bit mask of the unused bits in ISCR */



/*------------------

*   Bit Positions

------------------*/

#define B_IRQR  1   /* Interrupt Request Reset */

#define B_IRQF  3   /* External Interrupt Request Flag */

#define B_IRQE  7   /* External Interrupt Request Enable */



/*--------------------------------------

*   PDRA

*   Pull Down Register Port A

*   

*   Address: 0x0010

*   Manual Page: 75

*

*    WO  WO  WO  WO  WO  WO  WO  WO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PDIA0 - Pulldown Inhibit A Bit 0

*     |   |   |   |   |   |   |-------- PDIA1 - Pulldown Inhibit A Bit 1

*     |   |   |   |   |   |------------ PDIA2 - Pulldown Inhibit A Bit 2

*     |   |   |   |   |---------------- PDIA3 - Pulldown Inhibit A Bit 3

*     |   |   |   |-------------------- PDIA4 - Pulldown Inhibit A Bit 4

*     |   |   |------------------------ PDIA5 - Pulldown Inhibit A Bit 5

*     |   |---------------------------- PDIA6 - Pulldown Inhibit A Bit 6

*     |-------------------------------- PDIA7 - Pulldown Inhibit A Bit 7

*     

*   These write only bits control port A pulldown device.  Reset clears PDIA[7:0],

*   configuring all port A pins with active pulldowns.

*       1 = corresponding port A pin configured as pulldown disabled

*       0 = corresponding port A pin configured as pulldown not disabled

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PDIA0   0x01    /* Pulldown Inhibit A Bit 0 */

#define PDIA1   0x02    /* Pulldown Inhibit A Bit 1 */

#define PDIA2   0x04    /* Pulldown Inhibit A Bit 2 */

#define PDIA3   0x08    /* Pulldown Inhibit A Bit 3 */

#define PDIA4   0x10    /* Pulldown Inhibit A Bit 4 */

#define PDIA5   0x20    /* Pulldown Inhibit A Bit 5 */

#define PDIA6   0x40    /* Pulldown Inhibit A Bit 6 */

#define PDIA7   0x80    /* Pulldown Inhibit A Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PDIA0 0   /* Pulldown Inhibit A Bit 0 */

#define B_PDIA1 1   /* Pulldown Inhibit A Bit 1 */

#define B_PDIA2 2   /* Pulldown Inhibit A Bit 2 */

#define B_PDIA3 3   /* Pulldown Inhibit A Bit 3 */

#define B_PDIA4 4   /* Pulldown Inhibit A Bit 4 */

#define B_PDIA5 5   /* Pulldown Inhibit A Bit 5 */

#define B_PDIA6 6   /* Pulldown Inhibit A Bit 6 */

#define B_PDIA7 7   /* Pulldown Inhibit A Bit 7 */



/*--------------------------------------

*   PDRB

*   Pull Down Register Port B

*   

*   Address: 0x0011

*   Manual Page: 78

*

*    --  --  WO  WO  WO  WO  WO  WO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ PDIB2 - Pulldown Inhibit B Bit 2

*     |   |   |   |   |---------------- PDIB3 - Pulldown Inhibit B Bit 3

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

*   These write only bits control port B pulldown device.  Reset clears PDIB[3:2],

*   configuring all port B pins with active pulldowns.

*       1 = corresponding port B pin configured as pulldown disabled

*       0 = corresponding port B pin configured as pulldown not disabled

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PDIB2   0x04    /* Pulldown Inhibit B Bit 2 */

#define PDIB3   0x08    /* Pulldown Inhibit B Bit 3 */



#define PDRB_UNUSED 0xf3    /* bit mask of unsused bits in PDRB */



/*------------------

*   Bit Positions

------------------*/

#define B_PDIB2 2   /* Pulldown Inhibit B Bit 2 */

#define B_PDIB3 3   /* Pulldown Inhibit B Bit 3 */



/*--------------------------------------

*   EPROG

*   EPROM Programming Register

*   

*   Address: 0x0018

*   Manual Page: 26

*

*    RO  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- EPGM - EPROM Programming Bit

*     |   |   |   |   |   |   |-------- MPGM - MOR Programming Bit

*     |   |   |   |   |   |------------ ELAT - EPROM Bus Latch Bit

*     |   |   |   |   |---------------- Reserved

*     |   |   |   |-------------------- Reserved

*     |   |   |------------------------ Reserved

*     |   |---------------------------- Reserved

*     |-------------------------------- Unimplemented

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define EPGM    0x01    /* EPROM Programming Bit */

#define MPGM    0x02    /* MOR Programming Bit */

#define ELAT    0x04    /* EPROM Bus Latch Bit */



#define EPROG_UNUSED    0xf8    /* bit mask of unused bits in EPROG */



/*------------------

*   Bit Positions

------------------*/

#define B_EPGM  0   /* EPROM Programming Bit */

#define B_MPGM  1   /* MOR Programming Bit */

#define B_ELAT  2   /* EPROM Bus Latch Bit */



/*--------------------------------------

*   COPR

*   COP Register

*   

*   Address: 0x07f0

*   Manual Page: 83

*

*    --  --  --  --  --  --  --  WO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- COPC - COP Clear

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define COPC    0x01    /* COP Clear */



#define COPR_UNUSED 0xfe    /* bit mask of unused bits in COPR */



/*------------------

*   Bit Positions

------------------*/

#define B_COPC  0   /* COP Clear */



/*--------------------------------------

*   MOR

*   Mask Option Register

*   

*   Address: 0x07f1

*   Manual Page: 28

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- COPEN  - COP Enable Bit

*     |   |   |   |   |   |   |-------- LEVEL  - External Interrupt Sensitivity Bit

*     |   |   |   |   |   |------------ PIRQ   - Port A External Interrupt Bit

*     |   |   |   |   |---------------- SWPDI  - Software Pulldown Inhibit Bit

*     |   |   |   |-------------------- SWAIT  - Stop to Wait Conversion Bit

*     |   |   |------------------------ OSCRES - Oscillator Internal Resistor Bit

*     |   |---------------------------- EPMSEC - EPROM Security Bit

*     |-------------------------------- SOSCD  - Short Oscillator Delay Bit

*     

*   Note:  This register is implemented as EPROM cells so in order to write a new

*          value to this register, you must follow the procedure for programming

*          the EPROM.  Typically this register is only written at EPROM program time.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define COPEN   0x01    /* COP Enable Bit */

#define LEVEL   0x02    /* External Interrupt Sensitivity Bit */

#define PIRQ    0x04    /* Port A External Interrupt Bit */

#define SWPDI   0x08    /* Software Pulldown Inhibit Bit */

#define SWAIT   0x10    /* Stop to Wait Conversion Bit */

#define OSCRES  0x20    /* Oscillator Internal Resistor Bit */

#define EPMSEC  0x40    /* EPROM Security Bit */

#define SOSCD   0x80    /* Short Oscillator Delay Bit */



/*------------------

*   Bit Positions

------------------*/

#define B_COPEN     0   /* COP Enable Bit */

#define B_LEVEL     1   /* External Interrupt Sensitivity Bit */

#define B_PIRQ      2   /* Port A External Interrupt Bit */

#define B_SWPDI     3   /* Software Pulldown Inhibit Bit */

#define B_SWAIT     4   /* Stop to Wait Conversion Bit */

#define B_OSCRES    5   /* Oscillator Internal Resistor Bit */

#define B_EPMSEC    6   /* EPROM Security Bit */

#define B_SOSCD     7   /* Short Oscillator Delay Bit */



/*--------------------------------------

*   Register Manipulation Macros

--------------------------------------*/



/* PUT_REG writes an immediate value into the register */



#define PUT_REG(reg, value) reg = value



/* GET_REG reads the current value of a register and stores it in a variable */



#define GET_REG(reg, var)   var = reg



/* SET_REG_BITS or's the mask value into the register*/



#define SET_REG_BITS(reg, mask) reg |= mask



/* CLR_REG_BITS and's the inverse of the mask value into the register*/



#define CLR_REG_BITS(reg, mask) reg &= ~(mask)

