/*
 * File:        uart.c
 * Purpose:     Provide common ColdFire UART routines for serial IO
 *
 * Notes:       
 *              
 * License:     All software covered by license agreement in -
 *              docs/Freescale_Software_License.pdf
 */

#include "common.h"
#include "uart/uart.h"

/********************************************************************/
/*
 * Initialize the UART for 8N1 operation, interrupts disabled, and
 * no hardware flow-control
 *
 * Parameters:
 *  uartch      UART channel to initialize
 *  sysclk      UART System Clock (in KHz)
 *  baud        UART baud rate
 */
void 
uart_init (int uartch, int sysclk, int baud)
{
    register uint16 ubgs;

    /* 
     * Reset Transmitter 
     */
    MCF_UART_UCR(uartch) = MCF_UART_UCR_RESET_TX;

    /* 
     * Reset Receiver 
     */
    MCF_UART_UCR(uartch) = MCF_UART_UCR_RESET_RX;

    /* 
     * Reset Mode Register 
     */
    MCF_UART_UCR(uartch) = MCF_UART_UCR_RESET_MR;

    /* 
     * No parity, 8-bits per character 
     */
    MCF_UART_UMR(uartch) = (0
        | MCF_UART_UMR_PM_NONE
        | MCF_UART_UMR_BC_8);

    /* 
     * No echo or loopback, 1 stop bit 
     */
    MCF_UART_UMR(uartch) = (0
        | MCF_UART_UMR_CM_NORMAL
        | MCF_UART_UMR_SB_STOP_BITS_1);

    /* 
     * Set Rx and Tx baud by timer 
     */
    MCF_UART_UCSR(uartch) = (0
        | MCF_UART_UCSR_RCS_SYS_CLK
        | MCF_UART_UCSR_TCS_SYS_CLK);

    /* 
     * Mask all UART interrupts 
     */
    MCF_UART_UIMR(uartch) = 0;
                 
    /* 
     * Calculate baud settings 
     */
    ubgs = (uint16)((sysclk*1000)/(baud * 32));

    MCF_UART_UBG1(uartch) = (uint8)((ubgs & 0xFF00) >> 8);
    MCF_UART_UBG2(uartch) = (uint8)(ubgs & 0x00FF);

    /* 
     * Enable receiver and transmitter 
     */
    MCF_UART_UCR(uartch) = (0
        | MCF_UART_UCR_TX_ENABLED
        | MCF_UART_UCR_RX_ENABLED);
}
/********************************************************************/
/*
 * Wait for a character to be received on the specified UART
 *
 * Return Values:
 *  the received character
 */
char
uart_getchar (int channel)
{
    /* Wait until character has been received */
    while (!(MCF_UART_USR(channel) & MCF_UART_USR_RXRDY))
        ; 
    return (char)MCF_UART_URB(channel);
}
/********************************************************************/
/*
 * Wait for space in the UART Tx FIFO and then send a character
 */ 
void
uart_putchar (int channel, char ch)
{
    /* Wait until space is available in the FIFO */
    while (!(MCF_UART_USR(channel) & MCF_UART_USR_TXRDY))
        ;
    /* Send the character */
    MCF_UART_UTB(channel) = (uint8)ch;
}
/********************************************************************/
/*
 * Check to see if a character has been received
 *
 * Return values:
 *  0       No character received
 *  1       Character has been received
 */
int
uart_getchar_present (int channel)
{
    return (MCF_UART_USR(channel) & MCF_UART_USR_RXRDY);
}
/********************************************************************/
    
