/*
 * File:    m5211demo.c
 * Purpose: Board specific routines
 *
 * Notes:   
 * 
 * License:     All software covered by license agreement in -
 *              docs/Freescale_Software_License.pdf
 */

#include "common.h"
#include "uart/uart.h"
#include "clock/clock.h"

/********************************************************************/

/* Actual system clock frequency */
int sys_clk_khz = SYS_CLK_KHZ;

/********************************************************************/
/*
 * System Initialization
 * Called by asm_startmeup (mcf5213_lo.s)
 */
void
board_sysinit (void)
{
    /* Initialize the PLL */
    if (REF_CLK_KHZ == SYS_CLK_KHZ)
        clock_pll(REF_CLK_KHZ, SYS_CLK_KHZ, PLL_DISABLE);
    else
        clock_pll(REF_CLK_KHZ, SYS_CLK_KHZ, 0);

    /* Enable the proper UART pins */
    switch (TERMINAL_PORT)
    {
        case 0:
        default:
            MCF_GPIO_PUAPAR = 0
                | MCF_GPIO_PUAPAR_URXD0_URXD0
                | MCF_GPIO_PUAPAR_UTXD0_UTXD0;
    }

    /* Enable the default UART terminal port */
    uart_init(TERMINAL_PORT, SYS_CLK_KHZ, TERMINAL_BAUD);
    
    /* 
     * Allow interrupts from SW1 and SW2 (IRQ[4,7]) 
     */
     
    /* Enable IRQ signals on the port */
    MCF_GPIO_PNQPAR = 0
        | MCF_GPIO_PNQPAR_IRQ4_IRQ4
        | MCF_GPIO_PNQPAR_IRQ7_IRQ7;
    
    /* Set EPORT to look for rising edges */
    MCF_EPORT_EPPAR = 0
        | MCF_EPORT_EPPAR_EPPA4_RISING
        | MCF_EPORT_EPPAR_EPPA7_RISING;
        
    /* Clear any currently triggered events on the EPORT */
    MCF_EPORT_EPIER = 0
        | MCF_EPORT_EPIER_EPIE4
        | MCF_EPORT_EPIER_EPIE7;
        
    /* Enable interrupts in the interrupt controller */
    MCF_INTC_IMRL &= ~(0
        | MCF_INTC_IMRL_INT_MASK4 
        | MCF_INTC_IMRL_INT_MASK7 
        | MCF_INTC_IMRL_MASKALL);

}
/********************************************************************/
void
board_handle_interrupt (int vector)
{
    switch (vector)
    {
        case 68: /* Eport Interrupt 4 */
            printf("SW1\n");
            MCF_EPORT_EPFR = (uint8)(0x01 << 4);
            break;
        case 71: /* Eport Interrupt 7 */
            printf("SW2\n");
            MCF_EPORT_EPFR = (uint8)(0x01 << 7);
            break;
        case 65: /* Eport Interrupt 1 */
        case 66: /* Eport Interrupt 2 */
        case 67: /* Eport Interrupt 3 */
        case 69: /* Eport Interrupt 5 */
        case 70: /* Eport Interrupt 6 */
        default:
            MCF_EPORT_EPFR = (uint8)(0x01 << (vector - 64));
            printf("Edge Port Interrupt #%d\n",vector - 64);
            break;
    }
}
/********************************************************************/
/* 
 * Display the lower 4 bits of 'number' on the 4 LEDs connected to 
 * TIN[3:0]
 *
 *  LED: LED4 LED3 LED2 LED1
 *  PIN: TIN3 TIN2 TIN1 TIN0
 *  BIT:    3    2    1    0
 */
void
board_led_display(int number)
{
    /* Enable signals as GPIO */
    MCF_GPIO_PTCPAR = 0
        | MCF_GPIO_PTCPAR_DTIN3_GPIO
        | MCF_GPIO_PTCPAR_DTIN2_GPIO
        | MCF_GPIO_PTCPAR_DTIN1_GPIO
        | MCF_GPIO_PTCPAR_DTIN0_GPIO;
    
    /* Set output values */
    MCF_GPIO_PORTTC = (uint8)number;
    
    /* Enable signals as digital outputs */
    MCF_GPIO_DDRTC = 0
        | MCF_GPIO_DDRTC_DDRTC3
        | MCF_GPIO_DDRTC_DDRTC2
        | MCF_GPIO_DDRTC_DDRTC1
        | MCF_GPIO_DDRTC_DDRTC0;
}
/********************************************************************/
