/*
 * @brief PCx85063 example code using LPC54114
 * Version 1.0
 * @note
 * Copyright(C) NXP Semiconductors, 2020
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"
#include "PCx85063.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/* I2C transfer record for master */
static I2CM_XFER_T  i2cmXferRec;
#define I2C0_FLEXCOMM   4 

/* I2C clock is set to 2MHz */
#define I2C_CLK_DIVIDER     (6)

/* 100KHz I2C bit-rate - going too fast may prevent the salev from responding
   in time */
#define I2C_BITRATE         (100000)

/* I2C information */
#define LPC_I2C_PORT_0       LPC_I2C0
#define LPC_I2C_INTHAND_0    I2C0_IRQHandler
#define LPC_IRQNUM_0         I2C0_IRQn


/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* Initializes pin muxing for I2C interface - note that SystemInit() may
   already setup your pin muxing at system startup */
static void Init_I2C_PinMux(void)
{
	/* Connect the I2C1_SDA and I2C1_SCL signals to port pins */
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 25, (IOCON_FUNC1 | IOCON_DIGITAL_EN));
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 26, (IOCON_FUNC1 | IOCON_DIGITAL_EN));
}

/* Setup I2C */
static void setupI2CMaster(void)
{
	/* Setup clock rate for I2C */
	Chip_I2C_SetClockDiv(LPC_I2C_PORT_0, I2C_CLK_DIVIDER);

	/* Setup I2CM transfer rate */
	Chip_I2CM_SetBusSpeed(LPC_I2C_PORT_0, I2C_BITRATE);

	/* Enable I2C master interface */
	Chip_I2CM_Enable(LPC_I2C_PORT_0);
}


/* Function to wait for I2CM transfer completion */
static void WaitForI2cXferComplete(I2CM_XFER_T *xferRecPtr)
{
	/* Test for still transferring data */
	while (xferRecPtr->status == I2CM_STATUS_BUSY) {
		/* Sleep until next interrupt */
		__WFI();
	}
}

/* Function to setup and execute I2C transfer request */
static void SetupXferRecAndExecute(uint8_t devAddr,
								   uint8_t *txBuffPtr,
								   uint16_t txSize,
								   uint8_t *rxBuffPtr,
								   uint16_t rxSize)
{
	// Setup I2C transfer record 
	i2cmXferRec.slaveAddr = devAddr;
	i2cmXferRec.status = 0;
	i2cmXferRec.txSz = txSize;
	i2cmXferRec.rxSz = rxSize;
	i2cmXferRec.txBuff = txBuffPtr;
	i2cmXferRec.rxBuff = rxBuffPtr;

	// Wait for master to go pending - needed in mixed master/slave mode on single I2C bus 
	while (Chip_I2CM_IsMasterPending(LPC_I2C_PORT_0) == false) {}

	Chip_I2CM_Xfer(LPC_I2C_PORT_0, &i2cmXferRec);
	// Enable Master Interrupts 
	Chip_I2C_EnableInt(LPC_I2C_PORT_0, I2C_INTENSET_MSTPENDING | I2C_INTENSET_MSTRARBLOSS | I2C_INTENSET_MSTSTSTPERR);
	// Wait for transfer completion 
	WaitForI2cXferComplete(&i2cmXferRec);
	// Disable all Interrupts 
	Chip_I2C_DisableInt(LPC_I2C_PORT_0, I2C_INTENSET_MSTPENDING | I2C_INTENSET_MSTRARBLOSS | I2C_INTENSET_MSTSTSTPERR);

	if (i2cmXferRec.status != I2CM_STATUS_OK) {
		DEBUGOUT("\r\nI2C error: %d\r\n", i2cmXferRec.status);
	}
}


/*****************************************************************************
 * Public functions
 ****************************************************************************/

/**
 * @brief	Handle I2C1 interrupt by calling I2CM interrupt transfer handler
 * @return	Nothing
 */

void LPC_I2C_INTHAND_0(void)
{
	uint32_t state = Chip_I2C_GetPendingInt(LPC_I2C_PORT_0);

	/* Error handling */
	if (state & (I2C_INTENSET_MSTRARBLOSS | I2C_INTENSET_MSTSTSTPERR)) {
		Chip_I2CM_ClearStatus(LPC_I2C_PORT_0, I2C_STAT_MSTRARBLOSS | I2C_STAT_MSTSTSTPERR);
	}

	/* Call I2CM ISR function with the I2C device and transfer rec */
	if (state & I2C_INTENSET_MSTPENDING) {
		Chip_I2CM_XferHandler(LPC_I2C_PORT_0, &i2cmXferRec);
	}
}

/*****************************************************************************
 * @brief	main routine for NXP PCx85063A RTC device example
 * @return	Function should not exit.
******************************************************************************/

int main(void)
{
	/* Reading RTC timer looping */
	uint8_t loop = 1;	
	
	/* I2C rx/tx buffer, size and RTC slave address */
	uint8_t rx_buffer[8], tx_buffer[8];
	uint16_t tx_Size;
	uint16_t rx_Size;
	uint8_t RTC_Addr = 0x51;

	/* Generic Initialization */
	SystemCoreClockUpdate();
	Board_Init();

	/* Setup I2C pin muxing */
	Init_I2C_PinMux();

	/* Enable I2C clock and reset I2C peripheral */
	Chip_I2C_Init(LPC_I2C_PORT_0);

	/* Setup I2C master */
	setupI2CMaster();
	
	NVIC_EnableIRQ(LPC_IRQNUM_0);

	printf("Starting PCx85063 example code...\r\n\r\n");

  /* Setting 12 hour mode and select option for 12.5 pF quartz */
	Control_1_reg = 0x03;      
	tx_buffer[0] = Control_1;
	tx_buffer[1] = Control_1_reg;
	SetupXferRecAndExecute(RTC_Addr, tx_buffer, 2, rx_buffer, 0);
	
	delay_ms(1);
	
	/* Setting the clock to 3:45pm February 20, 2020 */
	Seconds_reg = 0x00;
	Minutes_reg = 0x45;
	Hours_reg = 0x13;
	Days_reg = 0x20;
	Weekdays_reg = 0x04;
	Months_reg = 0x02;
	Years_reg = 0x20;
	
	tx_buffer[0] = Seconds;
	tx_buffer[1] = Seconds_reg;
	tx_buffer[2] = Minutes_reg;
	tx_buffer[3] = Hours_reg;
	tx_buffer[4] = Days_reg;
	tx_buffer[5] = Weekdays_reg;
	tx_buffer[6] = Months_reg;
	tx_buffer[7] = Years_reg;
	
	tx_Size = 8;
	rx_Size = 0;
	
	SetupXferRecAndExecute(RTC_Addr, tx_buffer, tx_Size, rx_buffer, rx_Size);
	

	while (loop) {
		
		/* Setting address pointer to Seconds register */
		tx_buffer[0] = Seconds;
		
		tx_Size = 1;
	  rx_Size = 0;
		
		SetupXferRecAndExecute(RTC_Addr, tx_buffer, tx_Size, rx_buffer, rx_Size);
		
		delay_ms(1);
		
		/* Reading in RTC time/date */
		tx_Size = 0;
	  rx_Size = 7;
		
		SetupXferRecAndExecute(RTC_Addr, tx_buffer, tx_Size, rx_buffer, rx_Size);
		
		Seconds_reg = rx_buffer[0];
		Minutes_reg = rx_buffer[1];
		Hours_reg = rx_buffer[2];
		Days_reg = rx_buffer[3];
		Weekdays_reg = rx_buffer[4];
		Months_reg = rx_buffer[5];
		Years_reg = rx_buffer[6];
		
		printf("Seconds = %x \r\n", Seconds_reg);
		printf("Minutes = %x \r\n", Minutes_reg );
		printf("Hours = %x \r\n", Hours_reg);
		printf("Days = %x \r\n", Days_reg);
		printf("Weekdays = %x \r\n", Weekdays_reg);
		printf("Months = %x \r\n", Months_reg);
		printf("Years = %x \r\n\r\n", Years_reg);
		
		delay_ms(1000);
	}
	return 0;
}
