/*
*         Copyright (c), NXP Semiconductors Gratkorn / Austria
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
*particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

/** \file
* BAL Stub Component of Reader Library Framework.
* $Author: jenkins_ cm (nxp92197) $
* $Revision: 4184 $ (NFCRDLIB_V4.010.01.001603 : 3735 )
* $Date: 2016-01-22 18:04:59 +0530 (Fri, 22 Jan 2016) $
*
* History:
*  CHu: Generated 19. May 2009
*
*/

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include "phLpc1768_Build.h"

#ifdef NXPBUILD__PHBAL_REG_STUB

#include <lpc17xx_ssp.h>
#include <lpc17xx_i2c.h>
#include <lpc17xx_pinsel.h>
#include <lpc17xx_gpio.h>

phStatus_t phbalReg_Stub_Init(
                              phbalReg_Stub_DataParams_t * pDataParams,
                              uint16_t wSizeOfDataParams
                              )
{
    if (sizeof(phbalReg_Stub_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }

    PH_ASSERT_NULL (pDataParams);

    pDataParams->wId          = PH_COMP_BAL | PHBAL_REG_STUB_ID;
    pDataParams->bMultiRegRW  = PH_OFF;

    return PH_ERR_SUCCESS;
}

phStatus_t phbalReg_Stub_GetPortList(
                                     phbalReg_Stub_DataParams_t * pDataParams,
                                     uint16_t wPortBufSize,
                                     uint8_t * pPortNames,
                                     uint16_t * pNumOfPorts
                                     )
{
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Stub_SetPort(
                                 phbalReg_Stub_DataParams_t * pDataParams,
                                 uint8_t * pPortName
                                 )
{
    return PH_ERR_SUCCESS;
}

phStatus_t phbalReg_Stub_OpenPort(
                                  phbalReg_Stub_DataParams_t * pDataParams
                                  )
{
#ifdef SPI_USED
    PINSEL_CFG_Type PinCfg;
    SSP_CFG_Type SSP_ConfigStruct;
    volatile uint32_t delay;

    /* Disable SSP */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        SSP_Cmd(LPC_SSP0, DISABLE);
    }
    else
    {
        SSP_Cmd(LPC_SSP1, DISABLE);
    }

    /* EMVCo: High Speed SPI require for EMVCo */
    uint32_t ClockRate = 6400000;

    PinCfg.Pinmode = PINSEL_PINMODE_PULLUP;
    PinCfg.OpenDrain = PINSEL_PINMODE_NORMAL;

    /* NSS has to be driven by hand because RC523 requires the NSS to be
     * LOW to be able to send several bytes in one data stream (see MFRC523
     * page 86) and LPC17XX SSP0 Port in mode CPHA = 0 and CPOL = 0 always
     * sets NSS to HIGH after byte transmission.
     * */

    /* SSP MOSI */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        PinCfg.Pinnum = MOSI_PINNUM_SSP0;
    }
    else
    {
        PinCfg.Pinnum = MOSI_PINNUM_SSP1;
    }
    PinCfg.Portnum = 0;
    PinCfg.Funcnum = PINSEL_FUNC_2;
    PINSEL_ConfigPin(&PinCfg);

    /* SSP MISO */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        PinCfg.Pinnum = MISO_PINNUM_SSP0;
    }
    else
    {
        PinCfg.Pinnum = MISO_PINNUM_SSP1;
    }
    PinCfg.Portnum = 0;
    PinCfg.Funcnum = PINSEL_FUNC_2;
    PINSEL_ConfigPin(&PinCfg);

    /* SSP SCK */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        PinCfg.Pinnum = SCK_PINNUM_SSP0;
    }
    else
    {
        PinCfg.Pinnum = SCK_PINNUM_SSP1;
    }
    PinCfg.Portnum = 0;
    PinCfg.Funcnum = PINSEL_FUNC_2;
    PINSEL_ConfigPin(&PinCfg);

    /* SSP SSEL */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        PinCfg.Pinnum = SSEL_PINNUM_SSP0;
    }
    else
    {
        PinCfg.Pinnum = SSEL_PINNUM_SSP1;
    }
    PinCfg.Portnum = 0;
    PinCfg.Funcnum = PINSEL_FUNC_0;
    PINSEL_ConfigPin(&PinCfg);

    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
    {
        /* PIN_SSEL */
        GPIO_SetDir(0, 0x00000040, SET_OUT);
        GPIO_SetValue(0, 0x00000040);
    }
    else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
    {
        /* PIN_SSEL */
        GPIO_SetDir(0, 0x00000400, SET_OUT);
        GPIO_SetValue(0, 0x00000400);
    }
    else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        /* PIN_BUSY */
        PinCfg.Portnum = 2;
        PinCfg.Pinnum = 11;
        PinCfg.Funcnum = PINSEL_FUNC_0;
        PINSEL_ConfigPin(&PinCfg);

        GPIO_SetDir(2, 0x00000800, SET_IN);

        /* PIN_SSEL */
        GPIO_SetDir(0, 0x00010000, SET_OUT);
        GPIO_SetValue(0, 0x00010000);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    SSP_ConfigStruct.CPHA = SSP_CPHA_FIRST;
    SSP_ConfigStruct.CPOL = SSP_CPOL_HI;

    SSP_ConfigStruct.ClockRate = ClockRate;
    SSP_ConfigStruct.Databit = SSP_DATABIT_8;
    SSP_ConfigStruct.Mode = SSP_MASTER_MODE;
    SSP_ConfigStruct.FrameFormat = SSP_FRAME_SPI;

    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        SSP_Init(LPC_SSP0, &SSP_ConfigStruct);
        SSP_Cmd(LPC_SSP0, ENABLE);
    }
    else
    {
        SSP_Init(LPC_SSP1, &SSP_ConfigStruct);
        SSP_Cmd(LPC_SSP1, ENABLE);
    }

    /* Wait Startup time */
    for(delay=0;delay<10000;delay++){}
#endif /* SPI_USED */

#ifdef I2C_USED
    PINSEL_CFG_Type PinCfg;

    PinCfg.OpenDrain = PINSEL_PINMODE_OPENDRAIN;
    PinCfg.Pinmode = PINSEL_PINMODE_TRISTATE;
    PinCfg.Funcnum = 2;
    PinCfg.Portnum = 0;

    PinCfg.Pinnum = 10;
    PINSEL_ConfigPin(&PinCfg);

    PinCfg.Pinnum = 11;
    PINSEL_ConfigPin(&PinCfg);

    I2C_Init(LPC_I2C2, 1000000);

    I2C_Cmd(LPC_I2C2, ENABLE);

    /* Wait Startup time */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
    {
        volatile uint32_t delay;

        for(delay=0;delay<10000;delay++){}
    }
#endif /* I2C_USED */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Stub_ClosePort(
                                   phbalReg_Stub_DataParams_t * pDataParams
                                   )
{
    return PH_ERR_SUCCESS;
}

phStatus_t phbalReg_Stub_WaitForIrq(
                                    phbalReg_Stub_DataParams_t * pDataParams,
                                    uint32_t dwTimeoutMs
                                    )
{
    if (dwTimeoutMs != 0)
    {
        while (!(GPIO_ReadValue(2) & 0x00001000));
    }

    return PH_ERR_SUCCESS;
}

#ifdef SPI_USED
static void phbalReg_Stub_Wait4BusyIdle(void)
{
    while ((GPIO_ReadValue(2) & 0x00000800));
}
static void phbalReg_Stub_Wait4Busy(void)
{
    while (!((GPIO_ReadValue(2) & 0x00000800)));
}
#endif /* SPI_USED */

phStatus_t phbalReg_Stub_MultiRegWrite(
                                       phbalReg_Stub_DataParams_t * pDataParams,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint8_t * pRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
    uint16_t xferLen;
    uint8_t index;
    SSP_DATA_SETUP_Type xferConfig;

    index = 0;
    xferLen = 0;
    xferConfig.length = 2;
    xferConfig.rx_data = pRxBuffer;
    xferConfig.tx_data = pTxBuffer;

    while(index < wTxLength)
    {
        /* Chip SELECT reader IC */
        if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
        {
            LPC_GPIO0->FIOCLR = 0x00000040;
        }
        else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
        {
            LPC_GPIO0->FIOCLR = 0x00000400;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }

        xferConfig.rx_data = pRxBuffer;
        xferConfig.tx_data = pTxBuffer;
        xferLen = SSP_ReadWrite (LPC_SSP1, &xferConfig, SSP_TRANSFER_POLLING);
        pTxBuffer++;
        pRxBuffer++;

        /* Chip DESELECT reader IC */
        if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
        {
            LPC_GPIO0->FIOSET = 0x00000040;
        }
        else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
        {
            LPC_GPIO0->FIOSET = 0x00000400;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }
        index ++;
     }

    *pRxLength = xferLen;

    return PH_ERR_SUCCESS;
}

phStatus_t phbalReg_Stub_MultiRegRead(
                                      phbalReg_Stub_DataParams_t * pDataParams,
                                      uint8_t * pTxBuffer,
                                      uint16_t wTxLength,
                                      uint8_t * pRxBuffer,
                                      uint16_t * pRxLength
                                      )
{
    uint16_t xferLen;
    SSP_DATA_SETUP_Type xferConfig;

    xferConfig.length = wTxLength;
    xferConfig.rx_data = pRxBuffer;
    xferConfig.tx_data = pTxBuffer;

    /* Chip SELECT reader IC */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
    {
        LPC_GPIO0->FIOCLR = 0x00000040;
    }
    else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
    {
        LPC_GPIO0->FIOCLR = 0x00000400;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    xferLen = SSP_ReadWrite (LPC_SSP1, &xferConfig, SSP_TRANSFER_POLLING);

    /* Chip DESELECT reader IC */
    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
    {
        LPC_GPIO0->FIOSET = 0x00000040;
    }
    else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
    {
        LPC_GPIO0->FIOSET = 0x00000400;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    *pRxLength = xferLen;

    return PH_ERR_SUCCESS;
}

phStatus_t phbalReg_Stub_Exchange(
                                  phbalReg_Stub_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  uint8_t * pTxBuffer,
                                  uint16_t wTxLength,
                                  uint16_t wRxBufSize,
                                  uint8_t * pRxBuffer,
                                  uint16_t * pRxLength
                                  )
{
    phStatus_t status = PH_ERR_SUCCESS;

#ifdef SPI_USED
    uint16_t xferLen = 0;
    SSP_DATA_SETUP_Type xferConfig;
    uint8_t bIsWriteOpe;

    xferConfig.length = wTxLength;
    xferConfig.tx_data = pTxBuffer;

    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
    {
        xferConfig.rx_data = NULL;
    }
    else
    {
        xferConfig.rx_data = pRxBuffer;
    }

    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
    {
        bIsWriteOpe = 0x01;
    }

    if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
    {
        bIsWriteOpe = 0x80;
    }

    /* EMVCo: To meet the timing constraints defined by EMVCo for EMD related
     * scenarios separate function is defined to perform Reading and Writing of
     * multiple registers.
     */
    if(pDataParams->bMultiRegRW != PH_OFF)
    {
        if(!(pTxBuffer[0] & bIsWriteOpe))
        {
            status = phbalReg_Stub_MultiRegWrite(
                pDataParams,
                pTxBuffer,
                wTxLength,
                pRxBuffer,
                pRxLength);
            return status;
        }
        else
        {
            status = phbalReg_Stub_MultiRegRead(
                pDataParams,
                pTxBuffer,
                wTxLength,
                pRxBuffer,
                pRxLength);
            return status;
        }
    }
    else
    {
        /* Chip SELECT reader IC */
        if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
        {
            LPC_GPIO0->FIOCLR = 0x00000040;
        }
        else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
        {
            LPC_GPIO0->FIOCLR = 0x00000400;
        }
        else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
        {
             phbalReg_Stub_Wait4BusyIdle();

             /* Chip SELECT reader IC */
             LPC_GPIO0->FIOCLR = 0x00010000;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }

        /* Data exchange */
        if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
        {
            xferLen = SSP_ReadWrite (LPC_SSP0, &xferConfig, SSP_TRANSFER_POLLING);
        }
        else
        {
            xferLen = SSP_ReadWrite (LPC_SSP1, &xferConfig, SSP_TRANSFER_POLLING);
        }

        /* Chip DESELECT reader IC */
        if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC523)
        {
            LPC_GPIO0->FIOSET = 0x00000040;
        }
        else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_RC663)
        {
            LPC_GPIO0->FIOSET = 0x00000400;
        }
        else if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
        {
            LPC_GPIO0->FIOSET = 0x00010000;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }

        if (xferLen != wTxLength)
        {
            status =  PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        if( (pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180) &&
        		*pRxLength != 0	)
        {
            if ((wRxBufSize) && (PH_ERR_SUCCESS == status))
            {
                xferConfig.length = wRxBufSize;

                /* Use also RX buffer for transmission - it's anyhow ignored in this case */
                xferConfig.rx_data = pRxBuffer;
                xferConfig.tx_data = NULL;

                if(*pRxLength  != 0)
                	phbalReg_Stub_Wait4Busy();

                /* Chip select reader IC */
                LPC_GPIO0->FIOCLR = 0x00010000;

                /* Data exchange */
                if(pDataParams->wHalType == PHBAL_REG_HAL_HW_PN5180)
                {
                    xferLen = SSP_ReadWrite (LPC_SSP0, &xferConfig, SSP_TRANSFER_POLLING);
                }
                else
                {
                    xferLen = SSP_ReadWrite (LPC_SSP1, &xferConfig, SSP_TRANSFER_POLLING);
                }

                /* Chip deselect reader IC */
                LPC_GPIO0->FIOSET = 0x00010000;

                if (xferLen != wRxBufSize)
                {
                    status = PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }
            }
            /* Update the length of the received data */
            *pRxLength = xferLen;
        }


    }
#endif /* SPI_USED */

#ifdef I2C_USED
    I2C_M_SETUP_Type transferMCfg;

    /* Only 7Bits of slave address should be initialized as Last bit(R/W) will
     * be appended automatically based on the operation done. */
    transferMCfg.sl_addr7bit = 0x28;
    transferMCfg.tx_data     = pTxBuffer;
    transferMCfg.tx_length   = (uint32_t)wTxLength;
    transferMCfg.rx_data     = pRxBuffer;
    transferMCfg.rx_length   = (uint32_t)wRxBufSize;
    transferMCfg.retransmissions_max = 3;

    if (SUCCESS != I2C_MasterTransferData(LPC_I2C2, &transferMCfg, I2C_TRANSFER_POLLING))
    {
        status = PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    if((pRxLength != NULL) && (PH_ERR_SUCCESS == status))
    {
        *pRxLength = (uint16_t)(transferMCfg.rx_count & 0x0000FFFF);
    }
#endif /* I2C_USED */

    return status;
}

phStatus_t phbalReg_Stub_SetConfig(
                                   phbalReg_Stub_DataParams_t * pDataParams,
                                   uint16_t wConfig,
                                   uint16_t wValue
                                   )
{
    switch (wConfig)
    {
        /* EMVCo: Configure to use Multiple Register Read/Write Functionality */
        case PHBAL_CONFIG_RW_MULTI_REG:
        {
            pDataParams->bMultiRegRW = wValue;
            break;
        }

        case PHBAL_REG_CONFIG_HAL_HW_TYPE:
        {
            switch(wValue)
            {
                case PHBAL_REG_HAL_HW_RC523:
                    pDataParams->wHalType = wValue;
                    break;

                case PHBAL_REG_HAL_HW_RC663:
                    pDataParams->wHalType = wValue;
                    break;

                case PHBAL_REG_HAL_HW_PN5180:
                    pDataParams->wHalType = wValue;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
            break;
        }

        default:
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phbalReg_Stub_GetConfig(
                                   phbalReg_Stub_DataParams_t * pDataParams,
                                   uint16_t wConfig,
                                   uint16_t * pValue
                                   )
{
    return PH_ERR_SUCCESS;
}

#endif /* NXPBUILD__PHBAL_REG_STUB */
