/*
*         Copyright (c), NXP Semiconductors Gratkorn / Austria
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
*particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

/** \file
* Secure Firmware Update Library.
* $Author: Vinay Singh (nxp74831) $
* $Revision: 2917 $
* $Date: 2016-03-02 15:17:26 +0200 (Mi, 02 Mar 2016) $
*
* History:
*
*/

#include <phDownload_lib.h>

#include <ph_TypeDefs.h>

#include <phDlHalConfig_LPC1769.h>






/********************************************************************************************************/
/*   PRIVATE DEFINES FOR THE DOWNLOAD LIBRARY                                                         */
/********************************************************************************************************/

#define PH_BAL_REG_OPTION_DOWNLOAD_MODE            0x81		/**< Download mode options. */

#define PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_POS    0	/**<Very first byte is direction byte. */
#define PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN    1	/**<direction byte is 1 byte. */

#define PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS      1	/**<First byte of header. */
#define PH_DLHAL_HW_PN5180_DL_HEADER_BYTE1_POS      2	/**<Second byte of header. */
#define PH_DLHAL_HW_PN5180_DL_HEADER_LEN            2	/**<DL header is 2 bytes. */

#define PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET        (PH_DLHAL_HW_PN5180_DL_HEADER_LEN)	/**<Payload starts after header */

#define PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS     0/*first byte of payload is opcode byte */
#define PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_LEN     1/**<Opcode is 1 byte. */

#define PH_DLHAL_HW_PN5180_DL_CRC_LEN                2	/**<CRC is 2 bytes. */

#define PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD         (PH_DLHAL_HW_PN5180_DL_HEADER_LEN + PH_DLHAL_HW_PN5180_DL_CRC_LEN)	/**< bytes apart from payload data.*/


#define PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS        (PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN)

#define PH_DLHAL_HW_PN5180_DL_FW_MINOR_VERS_POS		4
#define PH_DLHAL_HW_PN5180_DL_FW_MAJOR_VERS_POS		5

#define PH_DLHAL_HW_PN5180_DL_CMD_FIRST_BYTE			0x7F

/* OPCODES */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_RESET              0xF0	/**<This command resets the IC. */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GETVERSION         0xF1	/**<This command provides the firmware version. */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GETDIEID           0xF4	/**<The command returns the die ID. */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GETSESSIONSTATE    0xF2	/**<This command provides the session state and lifecycle. */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GHECKINTEGRITY     0xE0	/**<This command checks the integrity of components. */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_SEC_WRITE          0xC0	/**<This command  writes a chunk of data. */
#define PH_DLHAL_HW_PN5180_DL_OPCODE_DL_READ               0xA2	/**<This command reads the EEPROM User Area Address specified. */

/* Return codes */
#define PH_DLHAL_HW_PN5180_DL_OK                      0x00 /**<.   command passed 			*/
#define PH_DLHAL_HW_PN5180_DL_INVALID_ADDR            0x01 /**<.   address not allowed 		*/
#define PH_DLHAL_HW_PN5180_DL_UNKNOW_CMD              0x0B /**<.   unknown command 			*/
#define PH_DLHAL_HW_PN5180_DL_ABORTED_CMD             0x0C /**<.   chunk sequence is large	*/
#define PH_DLHAL_HW_PN5180_DL_PLL_ERROR               0x0D /**<.   flash not activated 		*/
#define PH_DLHAL_HW_PN5180_DL_ADDR_RANGE_OFL_ERROR    0x1E /**<.   address out of range 	*/
#define PH_DLHAL_HW_PN5180_DL_BUFFER_OFL_ERROR        0x1F /**<.   the buffer is too small 	*/
#define PH_DLHAL_HW_PN5180_DL_MEM_BSY                 0x20 /**<.   no RSA key access 		*/
#define PH_DLHAL_HW_PN5180_DL_SIGNATURE_ERROR         0x21 /**<.   signature mismatch 		*/
#define PH_DLHAL_HW_PN5180_DL_FIRMWARE_VERSION_ERROR  0x24 /**<.   already up to date 		*/
#define PH_DLHAL_HW_PN5180_DL_PROTOCOL_ERROR          0x28 /**<.   protocol error 			*/
#define PH_DLHAL_HW_PN5180_DL_SFWU_DEGRADED           0x2A /**<.   eeprom corruption 		*/
#define PH_DLHAL_HW_PN5180_PH_STATUS_DL_FIRST_CHUNK   0x2D /**<.   first chunk received 	*/
#define PH_DLHAL_HW_PN5180_PH_STATUS_DL_NEXT_CHUNK    0x2E /**<.   wait for the next chunk 	*/
#define PH_DLHAL_HW_PN5180_PH_STATUS_INTERNAL_ERROR_5 0xC5 /**<.   length mismatch 			*/


/* GetVersion */
#define PH_DLHAL_HW_PN5180_DL_GETVERSION_CMD_LEN    4	/**< get version has a fixed command length of 4. */
#define PH_DLHAL_HW_PN5180_DL_GETVERSION_RESP_LEN  10	/**< get version response length. */

#define PH_DLHAL_HW_PN5180_DL_GETVERSION_FMxV_POS1  8  /**<  Minor Version Number. */
#define PH_DLHAL_HW_PN5180_DL_GETVERSION_FMxV_POS2  9  /**<  Major Version Number. */

/* GetDieId */
#define PH_DLHAL_HW_PN5180_DL_GETDIEID_CMD_LEN      4	/**< fixed command length of getDieID. */
#define PH_DLHAL_HW_PN5180_DL_GETDIEID_RESP_LEN    20	/**< fixed response length of GetDieID. */

#define PH_DLHAL_HW_PN5180_DL_GETDIEID_DIEID_POS    4	/**< Position of the DieID in response. */
#define PH_DLHAL_HW_PN5180_DL_GETDIEID_DIEID_LEN   16	/**< Length of the DieID. */

/* SoftReset */
#define PH_DLHAL_HW_PN5180_DL_SOFTRESET_CMD_LEN     4	/**< Command length of soft reset. */
#define PH_DLHAL_HW_PN5180_DL_SOFTRESET_RESP_LEN    0	/**< No response for soft reset. */


/* CheckSessionState */
#define PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_CMD_LEN     4	/**< GetSessionState command length. */
#define PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_RESP_LEN    4	/**< Response length of get session state */

#define PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_SSTA_POS    1  /**< position of Session state byte: 0..closed, 1..open */
#define PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_SSTA_LEN    1  /**< Session state is 1 byte . */
#define PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_LIFE_POS    3  /**< Position of lifeCycle byte. */
#define PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_LIFE_LEN    1  /**< Life cycle is represented in 1 byte */



/* CheckIntegrity */

#define PH_DLHAL_HW_PN5180_DL_USER_E2AREA_PROTECTED_DATA_OFFSET    0x2BC
#define PH_DLHAL_HW_PN5180_DL_USER_E2AREA_LENGTH                   0xC00

#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_CMD_LEN      8		/**< Command length of check integrity. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_LEN    32		/**< Command length of soft reset. */

#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_UOFF_POS     4
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_ULEN_POS     6

#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_STAT_POS     0
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_POS     1

#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_FUNC_TABLE_BITPOS      6  /**< Function table CRC is OK. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_PATCH_TABLE_BITPOS     5  /**< Patch table CRC is OK. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_FUNC_CODE_BITPOS       4  /**< Function code CRC is OK. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_PATCH_CODE_BITPOS      3  /**< Patch code CRC is OK. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_PROT_DATA_BITPOS       2  /**< Protected data CRC is OK. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_TRIM_DATA_BITPOS       1  /**< Trim data CRC is OK. */
#define PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_USER_DATA_BITPOS       0  /**< User data CRC is OK. */


/* Read command */
#define PH_DLHAL_HW_PN5180_DL_READ_CMD_LEN          8	/**< read command length. */
#define PH_DLHAL_HW_PN5180_DL_READ_RESP_LEN         4	/**< fixed response part of read command. */
#define PH_DLHAL_HW_PN5180_DL_MAX_READ_LEN          546	/**< Maximum of 546 bytes can be readout. */
#define PH_DLHAL_HW_PN5180_DL_START_ADDR			0x201380 /**< Start address for the READ function. */
#define PH_DLHAL_HW_PN5180_DL_END_ADDR				0x201F80 /**< End address for the READ function. */

/* Secure write command*/
#define PH_DLHAL_HW_PN5180_DL_SWRITE_RESP_LEN           4	/**<respose length for chunk write. */
#define PH_DLHAL_HW_PN5180_DL_SWRITE_PAYLOD_OFFSET      2	/**<Chunk data starts after 2nd byte. */
#define PH_DLHAL_HW_PN5180_DL_SWRITE_MAX_CHUNK_SIZE	   256	/**<Chunk size due to BAL buffer limitations. */

/********************************************************************************************************/
/*   PRIVATE VARIABLES FOR THE DOWNLOAD LIBRARY                                                         */
/********************************************************************************************************/
static uint16_t phHal_Host_CalcCrc16(uint8_t* p, uint32_t dwLength)
{
    uint32_t i;
    uint16_t crc_new;
    uint16_t crc = 0xffffU;

    for (i = 0; i < dwLength; i++)
    {
        crc_new = (uint8_t)(crc >> 8) | (crc << 8);
        crc_new ^= p[i];
        crc_new ^= (uint8_t)(crc_new & 0xff) >> 4;
        crc_new ^= crc_new << 12;
        crc_new ^= (crc_new & 0xff) << 5;
        crc = crc_new;
    }

    return crc;
}

static phStatus_t phhalHw_ReturnCodesMapping(uint8_t bStatusCode)
{
    phStatus_t status = PH_ERR_SUCCESS;

    switch (bStatusCode)
    {
        case PH_DLHAL_HW_PN5180_DL_INVALID_ADDR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_INVALID_ADDR;
            break;
        case PH_DLHAL_HW_PN5180_DL_UNKNOW_CMD:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_UNKNOWN_CMD;
            break;
        case PH_DLHAL_HW_PN5180_DL_ABORTED_CMD:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_ABORTED_CMD;
            break;
        case PH_DLHAL_HW_PN5180_DL_PLL_ERROR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PLL_ERROR;
            break;
        case PH_DLHAL_HW_PN5180_DL_ADDR_RANGE_OFL_ERROR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_ADDRESS_RANGE_OVERFLOW;
            break;
        case PH_DLHAL_HW_PN5180_DL_BUFFER_OFL_ERROR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_BUFFER_OVERFLOW;
            break;
        case PH_DLHAL_HW_PN5180_DL_MEM_BSY:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_MEMORY_BUSY;
            break;
        case PH_DLHAL_HW_PN5180_DL_SIGNATURE_ERROR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_SIGNATURE_ERROR;
            break;
        case PH_DLHAL_HW_PN5180_DL_FIRMWARE_VERSION_ERROR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_FIRMWARE_VERSION_ERROR;
            break;
        case PH_DLHAL_HW_PN5180_DL_PROTOCOL_ERROR:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR;
            break;
        case PH_DLHAL_HW_PN5180_DL_SFWU_DEGRADED:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_DEGRADED;
            break;

        default:
            status = PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_GENERIC_ERROR;
            break;
    }

    return status;
}


static phStatus_t phDlhalHw_Pn5180_DLCmd_Transceive(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
        uint8_t  *pCommand,
        uint16_t wCommandLength,
        uint8_t  **ppResponse,
        uint16_t *pResponseLength
        )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxLen = 0;
    uint16_t wTxLen = 0;
    uint16_t wCRC = 0;
    uint16_t wRespLenInHeader = 0;
    uint16_t wCrcPos = 0;

    do
    {
    	/* First Byte required in DownLoadMode */
    	pDlDataParams->pTxBufferStart[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_POS] = PH_DLHAL_HW_PN5180_DL_CMD_FIRST_BYTE;

    	/*Length fields updation to Tx Frame*/
    	pDlDataParams->pTxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS] = (uint8_t)( (wCommandLength >> 8 ) & 0x03);
    	pDlDataParams->pTxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE1_POS] = (uint8_t)(wCommandLength & 0xFF);

    	/*CRC calculation*/
		wCRC = phHal_Host_CalcCrc16(&pDlDataParams->pTxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS],
										PH_DLHAL_HW_PN5180_DL_HEADER_LEN + wCommandLength       );

		/*CRC field updation to Tx Frame*/
		pDlDataParams->pTxBufferStart[wCommandLength + PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN] = (uint8_t) ((wCRC >> 8) & 0xFF);
		pDlDataParams->pTxBufferStart[wCommandLength + PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN + 1] = (uint8_t) (wCRC & 0xFF);

		/*Total Frame = Frame Header + Data + CRC*/
		wTxLen = PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD + wCommandLength +  PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;

		/* No response available for SOFTRESET command */
		if ( pDlDataParams->pTxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_LEN + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN] != PH_DLHAL_HW_PN5180_DL_OPCODE_DL_RESET )
		{
			wRxLen = *pResponseLength + PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;
		}

		/*Perform a BAL exchange*/
		status = phbalReg_Exchange((pDlDataParams->pBalDataParams), 0, (pDlDataParams->pTxBufferStart), wTxLen, wRxLen, (pDlDataParams->pRxBufferStart), &wRxLen);
		CHECK_STATUS(status);

		/*Response length cannot be 0*/
		if ( wRxLen == 0 )
		{
			break;
		}

		/* Check length of the response against the expected length */
		if ((wRxLen - PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD - PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN)  != (*pResponseLength) )
		{
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}


		/* Check the length of frame against the length in the header */
		wRespLenInHeader = (  ( (uint16_t) (pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE1_POS]  +
				((uint16_t)(pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS]) << 8) ))  & 0x03FF);

		if ( *pResponseLength != wRespLenInHeader )
		{
			/* Received length and expected length mismatch */
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}


		/* Check CRC of the response */
		wCRC = phHal_Host_CalcCrc16( &(pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS]),
									(wRxLen - PH_DLHAL_HW_PN5180_DL_CRC_LEN - PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN) );

		wCrcPos = wRxLen - (PH_DLHAL_HW_PN5180_DL_CRC_LEN);

		if ( (pDlDataParams->pRxBufferStart[wCrcPos] != (uint8_t) ((wCRC >> 8) & 0xFF) ) ||
			 (pDlDataParams->pRxBufferStart[wCrcPos + 1] != (uint8_t) (wCRC & 0xFF)    )   )
		{
			/* CRC mismatch */
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}



		/* Check status of the response */
		if (pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS ] !=  PH_DLHAL_HW_PN5180_DL_OK)
		{
			/* Error status received */
			status = phhalHw_ReturnCodesMapping(pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS ] );
			status = PH_ADD_COMPCODE(status, PH_COMP_GENERIC);
			break;
		}

		/* Setting the return buffer */
		*ppResponse = &(pDlDataParams->pRxBufferStart)[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN];
		*pResponseLength = wRxLen - PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD - PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;

    }while(0);

    return status;
}

static phStatus_t phDlhalHw_Pn5180_Download_Transceive(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
        uint8_t  *pCommand,
        uint16_t wCommandLength,
        uint8_t  **ppResponse,
        uint16_t *pResponseLength
        )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxLen = 0;
    uint16_t wTxLen = 0;
    uint16_t wCRC = 0;
	uint16_t wChunkCmdLength = 0;
	uint16_t wChunkRespLength = 0;
	uint8_t aChunkBuffer[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN + PH_DLHAL_HW_PN5180_DL_SWRITE_MAX_CHUNK_SIZE + PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD ] = {0};
	uint8_t bLastChunk = 0;
	uint16_t bchunkcounter = 0;
	uint16_t wRespLenInHeader = 0;
	uint16_t wCrcPos = 0;

    do
    {
    	/* chunking the length when the command length is greater than MAX Size supported by BAL */
    	if( wCommandLength >  PH_DLHAL_HW_PN5180_DL_SWRITE_MAX_CHUNK_SIZE)
    	{
			wChunkCmdLength = PH_DLHAL_HW_PN5180_DL_SWRITE_MAX_CHUNK_SIZE;
			wChunkRespLength = PH_DLHAL_HW_PN5180_DL_SWRITE_RESP_LEN;
    	}
    	else
    	{
    		wChunkCmdLength = wCommandLength;
    		wChunkRespLength = *pResponseLength;
    		bLastChunk = 1;
    	}

    	/*Copy the chunk for transmission*/
    	memcpy( &(aChunkBuffer[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN + PH_DLHAL_HW_PN5180_DL_HEADER_LEN]), &(pCommand[bchunkcounter]), wChunkCmdLength );/* PRQA S 3200 */

    	/* Forming the final frame with header*/
		if( !bLastChunk )
		{
			/* Chunk of a Frame transmitted with bit10 set Required for chunking */
			aChunkBuffer[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_POS] = PH_DLHAL_HW_PN5180_DL_CMD_FIRST_BYTE;
			aChunkBuffer[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS]   = (uint8_t)( ((wChunkCmdLength >> 8) | 0x4 ) & 0x07);
			aChunkBuffer[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE1_POS]   = (uint8_t)(wChunkCmdLength & 0xFF);
	    }
		else
		{
			aChunkBuffer[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_POS] = PH_DLHAL_HW_PN5180_DL_CMD_FIRST_BYTE;
			aChunkBuffer[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS]   = (uint8_t)( (wChunkCmdLength >> 8 ) & 0x03);
			aChunkBuffer[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE1_POS]   = (uint8_t)(wChunkCmdLength & 0xFF);
		}


		/* calculate and  Add the CRC to the transmit data chunk */
		wCRC = phHal_Host_CalcCrc16(&aChunkBuffer[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS],
		                                PH_DLHAL_HW_PN5180_DL_HEADER_LEN + wChunkCmdLength       );

		aChunkBuffer[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN + wChunkCmdLength + PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET] = (uint8_t) ((wCRC >> 8) & 0xFF);
		aChunkBuffer[PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN + wChunkCmdLength + PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET + 1] = (uint8_t) (wCRC & 0xFF);

		 /*Total Frame = Frame Header + Data + CRC*/
		wTxLen = PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN + PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD + wChunkCmdLength;

		/* Response length */
		wRxLen = wChunkRespLength + PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;

		/*Perform a BAL exchange*/
		status = phbalReg_Exchange((pDlDataParams->pBalDataParams),
									0,
									aChunkBuffer,
									wTxLen,
									wRxLen,
									pDlDataParams->pRxBufferStart,
									&wRxLen);
		CHECK_STATUS(status);

		/*Response length cannot be 0*/
		if ( wRxLen == 0 )
		{
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}

		wCommandLength -= wChunkCmdLength;
		bchunkcounter += wChunkCmdLength;


		/* Check length of the response against the expected length */
		if ((wRxLen - PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD - PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN)  != (*pResponseLength) )
		{
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}



		/* Check the length of frame against the length in the header */
		wRespLenInHeader = (  ( (uint16_t) (pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE1_POS]  +
				((uint16_t)(pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS]) << 8) ))  & 0x03FF);

		if ( wChunkRespLength != wRespLenInHeader )
		{
			//!Attention: normal situation in case of Read, and Reset commands
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}



		/* Check CRC of the response */
		wCRC = phHal_Host_CalcCrc16( &(pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_HEADER_BYTE0_POS]),
									(wRxLen - PH_DLHAL_HW_PN5180_DL_CRC_LEN - PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN) );

		wCrcPos = wRxLen - PH_DLHAL_HW_PN5180_DL_CRC_LEN;

		if ( (pDlDataParams->pRxBufferStart[wCrcPos] != (uint8_t) ((wCRC >> 8) & 0xFF) ) ||
				(pDlDataParams->pRxBufferStart[wCrcPos + 1] != (uint8_t) (wCRC & 0xFF)    )   )
		{
			//!Attention: normal situation in case of Read, and Reset commands
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR, PH_COMP_GENERIC);
			break;
		}



		/* Check status of the response */
		if (pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS ] !=  PH_DLHAL_HW_PN5180_DL_OK &&
			pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS ] !=  PH_DLHAL_HW_PN5180_PH_STATUS_DL_FIRST_CHUNK &&
			pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS ] !=  PH_DLHAL_HW_PN5180_PH_STATUS_DL_NEXT_CHUNK )
		{
			//!Attention: normal situation in case of Reset command (if successful)
			status = phhalHw_ReturnCodesMapping(pDlDataParams->pRxBufferStart[PH_DLHAL_HW_PN5180_DL_STATUS_BYTE_POS] );
			status = PH_ADD_COMPCODE(status, PH_COMP_GENERIC);
			break;
		}

		/* Setting the return buffer */
		*ppResponse = &(pDlDataParams->pRxBufferStart)[PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN];
		*pResponseLength = wRxLen - PH_DLHAL_HW_PN5180_DL_FRAME_OVERHEAD - PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;

		/* Identifying the Last Chunk */
		if( (wCommandLength - wChunkCmdLength) <= 0)
		{
			bLastChunk = 1;
		}

    }while(wCommandLength > 0);

    return status;
}

static phStatus_t phDlhalHw_Pn5180_Download_PerformSecureFirmwareUpdateInt(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
    uint8_t * pImage,
    uint32_t dwSizeOfImage
    )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint32_t dwCurrBlockPtr = 0;
    uint16_t wCurrBlockLen = 0;
    uint16_t wRxBytes = PH_DLHAL_HW_PN5180_DL_SWRITE_RESP_LEN;
    uint8_t* pRxBuff = NULL;
    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
	PH_CHECK_NULL(pImage);
	if(dwSizeOfImage == 0)
	{
	return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_GENERIC);
	}
    /* Get current block length (stored bigendian) */
    wCurrBlockLen = (uint16_t)(((uint16_t)(pImage[dwCurrBlockPtr]) << 8) + pImage[dwCurrBlockPtr + 1]);
    while ( dwCurrBlockPtr < dwSizeOfImage )
    {
    	/*The entire firmware is transmitted frame by frame*/
       memcpy( pDlDataParams->pTxBuffer, &pImage[dwCurrBlockPtr + PH_DLHAL_HW_PN5180_DL_SWRITE_PAYLOD_OFFSET], wCurrBlockLen );	/* PRQA S 3200 */
       /*Send command and get response*/
       status = phDlhalHw_Pn5180_Download_Transceive(pDlDataParams,
    		   	   	   	   	   	   	   	   	   	   pDlDataParams->pTxBuffer,
                                                   wCurrBlockLen,
                                                   &pRxBuff,
                                                   &wRxBytes );
       CHECK_STATUS(status);
       /*advance pointer to next frame*/
       dwCurrBlockPtr += wCurrBlockLen + PH_DLHAL_HW_PN5180_DL_SWRITE_PAYLOD_OFFSET;
       if( dwCurrBlockPtr >= dwSizeOfImage)
       {
    	   /*whole image downloaded*/
    	   break;
       }
       /*calculate the next block length*/
       wCurrBlockLen = (uint16_t)(((uint16_t)(pImage[dwCurrBlockPtr]) << 8) & (0xFF00));
       wCurrBlockLen += pImage[dwCurrBlockPtr + 1];
       /*This check  is to ensure that we donot read beyond the array of image. We should
        * strictly limit our read  to what the higher layer says is the size of image*/
       if(( dwCurrBlockPtr + wCurrBlockLen) > dwSizeOfImage)
       {
    	   status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_BUFFER_OVERFLOW, PH_COMP_GENERIC);
       	   /*whole image downloaded*/
       	   break;
       }
    }
    return status;
}


/********************************************************************************************************/
/*   API FUNCTIONS FOR THE DOWNLOAD LIBRARY                                                         */
/********************************************************************************************************/

phStatus_t phDlhalHw_Pn5180_Download_Init(
	phDlhalHw_Pn5180_DL_DataParams_t *pDlDataParams,
	void * pbalparms,
    uint8_t * pTxBuffer,
    uint16_t wTxBufSize,
    uint8_t * pRxBuffer,
    uint16_t wRxBufSize
    )
{

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
	PH_CHECK_NULL(pTxBuffer);
	PH_CHECK_NULL(pRxBuffer);
    PH_CHECK_NULL(pbalparms);

    /*Assign the already opened BAL pointer*/
	pDlDataParams->pBalDataParams = (void *)pbalparms;

    /* Initialize the Buffers */
   	pDlDataParams->pTxBufferStart  = pTxBuffer;
   	pDlDataParams->pRxBufferStart  = pRxBuffer;

   	pDlDataParams->pTxBuffer     = pDlDataParams->pTxBufferStart + PH_DLHAL_HW_PN5180_DL_PAYLOAD_OFFSET + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;
   	pDlDataParams->pRxBuffer     = pDlDataParams->pRxBufferStart + PH_DLHAL_HW_PN5180_DL_DIRECTION_BYTE_LEN;

   	/*Initialize their sizes*/
   	pDlDataParams->wTxBufSize = wTxBufSize;
   	pDlDataParams->wRxBufSize = wRxBufSize;

   	/*Initialize the buffers to 0*/
   	memset(pDlDataParams->pTxBuffer, 0x00, pDlDataParams->wTxBufSize);	 /* PRQA S 3200 */
    memset(pDlDataParams->pRxBuffer, 0x00, pDlDataParams->wRxBufSize);	 /* PRQA S 3200 */

    return PH_ERR_SUCCESS;
}


phStatus_t phDlhalHw_Pn5180_Download_Read(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
    uint16_t wLength,
    uint32_t dwAddress,
    uint8_t** ppReadBuffer
    )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxBytes = wLength + PH_DLHAL_HW_PN5180_DL_READ_RESP_LEN;

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
	PH_CHECK_NULL(ppReadBuffer);

	if(wLength == 0)                                                      \
	 {                                                                   \
	     return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_GENERIC);  \
	 }


    if( wLength > PH_DLHAL_HW_PN5180_DL_MAX_READ_LEN )
    {
    	status = PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_GENERIC);
    	return status;
    }

    /* Return error if the address to read is out of range. */
    if( (dwAddress < PH_DLHAL_HW_PN5180_DL_START_ADDR) || (dwAddress > PH_DLHAL_HW_PN5180_DL_END_ADDR) )
    {
    	status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_ADDRESS_RANGE_OVERFLOW, PH_COMP_GENERIC);
     	return status;
    }

    pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS] = PH_DLHAL_HW_PN5180_DL_OPCODE_DL_READ;
    pDlDataParams->pTxBuffer[1] = 0x00;

    /* Setting length (little endian) */
    pDlDataParams->pTxBuffer[2] = (uint8_t)(wLength & 0xFF);
    pDlDataParams->pTxBuffer[3] = (uint8_t)((wLength >> 8) & 0xFF);

    /* Setting address (little endian) */
    pDlDataParams->pTxBuffer[4] = (uint8_t)(dwAddress & 0xFF);
    pDlDataParams->pTxBuffer[5] = (uint8_t)((dwAddress >>  8) & 0xFF);
    pDlDataParams->pTxBuffer[6] = (uint8_t)((dwAddress >> 16) & 0xFF);
    pDlDataParams->pTxBuffer[7] = (uint8_t)((dwAddress >> 24) & 0xFF);

    /*Send command and get response*/
    status = phDlhalHw_Pn5180_Download_Transceive(pDlDataParams,
    											pDlDataParams->pTxBuffer,
                                                PH_DLHAL_HW_PN5180_DL_READ_CMD_LEN,
                                                ppReadBuffer,
                                                &wRxBytes );
    CHECK_STATUS(status);

    return status;
}


phStatus_t phDlhalHw_Pn5180_Download_PerformSecureFirmwareUpdate(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
    uint8_t * pImage,
    uint32_t dwSizeOfImage
    )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint8_t bMajorVersion = 0;
    uint8_t bMinorVersion = 0;
    phDlhalHw_Pn5180_Download_IntegrityInformation_t sIntegrityInfo;
    phDlhalHw_Pn5180_Download_SessionStateInformation_t sStateInfo;

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
	PH_CHECK_NULL(pImage);

	if(dwSizeOfImage == 0)
	{
	return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_GENERIC);
	}


    /* If Integrity fails we go ahead and update with the latest version.
     * If integrity succeeds we check for the existing version and then update
     * Added to overcome download halted due to power failure */
    status = phDlhalHw_Pn5180_Download_CheckIntegrity(pDlDataParams ,&sIntegrityInfo);
	if( (sIntegrityInfo.bFunctionCodeOk) && (sIntegrityInfo.bPatchCodeOk) &&
			(sIntegrityInfo.bPatchTableOk) && (sIntegrityInfo.bUserDataOk) )
	{
	    /* Check for the current FIRMWARE VERSION */
	    status = phDlhalHw_Pn5180_Download_GetFirmwareVersion(pDlDataParams, &bMajorVersion, &bMinorVersion);
	    PH_CHECK_SUCCESS(status);

		/* Firmware Version checking rules to decide whether to start the download or exit download */
		if( (bMinorVersion == pImage[PH_DLHAL_HW_PN5180_DL_FW_MINOR_VERS_POS] ) &&
				(bMajorVersion == pImage[PH_DLHAL_HW_PN5180_DL_FW_MAJOR_VERS_POS] ) )
		{
			status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_FIRMWARE_VERSION_ERROR, PH_COMP_GENERIC);
			return status;
		}
	}

	/*Perform the actual firmware update*/
	status = phDlhalHw_Pn5180_Download_PerformSecureFirmwareUpdateInt(
			pDlDataParams,
		    pImage,
		    dwSizeOfImage);

	PH_CHECK_SUCCESS(status);

    /* Perform a integrity checking After download */
    status = phDlhalHw_Pn5180_Download_CheckIntegrity(pDlDataParams ,&sIntegrityInfo);
    if( !(sIntegrityInfo.bFunctionCodeOk) || !(sIntegrityInfo.bPatchCodeOk) ||
    		!(sIntegrityInfo.bPatchTableOk) || !(sIntegrityInfo.bUserDataOk) )
    {
    	status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_GENERIC_ERROR, PH_COMP_GENERIC);
    	return status;
    }

    /*Firmware download is considered complete once the session is closed*/
	status = phDlhalHw_Pn5180_Download_CheckSessionState(pDlDataParams, &sStateInfo);
	PH_CHECK_SUCCESS(status);

	if(sStateInfo.bSessionState != 0)
	{
		/*Session is open so the download has failed*/
		status = PH_ADD_COMPCODE(PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_GENERIC_ERROR, PH_COMP_GENERIC);
		return status;
	}

    return status;
}

phStatus_t phDlhalHw_Pn5180_Download_CheckIntegrity(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
    phDlhalHw_Pn5180_Download_IntegrityInformation_t * pIntegrityCheckResult
    )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxBytes = PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_LEN;
    uint8_t* pRxBuff = NULL;

    uint16_t wUOFF = PH_DLHAL_HW_PN5180_DL_USER_E2AREA_PROTECTED_DATA_OFFSET;
    uint16_t wULEN = PH_DLHAL_HW_PN5180_DL_USER_E2AREA_LENGTH;
	uint8_t bCRCBitmap = 0;

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
	PH_CHECK_NULL(pIntegrityCheckResult);

    /*Form the command*/
	pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS] = PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GHECKINTEGRITY;
	pDlDataParams->pTxBuffer[1] = 0x00;
	pDlDataParams->pTxBuffer[2] = 0x00;
	pDlDataParams->pTxBuffer[3] = 0x00;

    /* UOFF and ULEN stored little endian */
	pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_UOFF_POS] = (uint8_t)(wUOFF & 0xFF);
	pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_UOFF_POS + 1] = (uint8_t)((wUOFF >> 8) & 0xFF);
	pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_ULEN_POS] = (uint8_t)(wULEN & 0xFF);
	pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_ULEN_POS + 1] = (uint8_t)((wULEN >> 8) & 0xFF);

    /*Send command and get response*/
    status = phDlhalHw_Pn5180_DLCmd_Transceive( pDlDataParams,
    											pDlDataParams->pTxBuffer,
                                                PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_CMD_LEN,
                                                &pRxBuff,
                                                &wRxBytes );
    CHECK_STATUS(status);

    bCRCBitmap = pRxBuff[PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_POS];

    /*Trim Data is not part of firmware and PN5180 does not use function table */
    pIntegrityCheckResult->bFunctionCodeOk = (bCRCBitmap >> PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_FUNC_CODE_BITPOS)   & 0x01;
    pIntegrityCheckResult->bPatchCodeOk    = (bCRCBitmap >> PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_PATCH_CODE_BITPOS)  & 0x01;
    pIntegrityCheckResult->bPatchTableOk   = (bCRCBitmap >> PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_PATCH_TABLE_BITPOS) & 0x01;
    pIntegrityCheckResult->bUserDataOk     = (bCRCBitmap >> PH_DLHAL_HW_PN5180_DL_CHECKINTEGRITY_RESP_CRCS_USER_DATA_BITPOS)   & 0x01;

    return status;
}


phStatus_t phDlhalHw_Pn5180_Download_CheckSessionState(phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
													 phDlhalHw_Pn5180_Download_SessionStateInformation_t* sSSTAInfo)
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxBytes = PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_RESP_LEN;
    uint8_t* pRxBuff = NULL;

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
    PH_CHECK_NULL(sSSTAInfo);

    /*Form the command*/
    pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS] = PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GETSESSIONSTATE;
    pDlDataParams->pTxBuffer[1] = 0x00;
    pDlDataParams->pTxBuffer[2] = 0x00;
    pDlDataParams->pTxBuffer[3] = 0x00;

    /*Send command and get response*/
    status = phDlhalHw_Pn5180_DLCmd_Transceive( pDlDataParams,
    											pDlDataParams->pTxBuffer,
                                                PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_CMD_LEN,
                                                &pRxBuff,
                                                &wRxBytes );
    CHECK_STATUS(status);

    /*
     * Session State	- 0: close  - 1: open*/
    sSSTAInfo->bSessionState = pRxBuff[PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_SSTA_POS];

    /*
     * Life cycle
				- 11d: creation
				- 13d: initializing
				- 17d: operational
				- 19d: termination */
    sSSTAInfo->bLifeCycle =  pRxBuff[PH_DLHAL_HW_PN5180_DL_SESSIONSTATE_LIFE_POS];

    return status;
}


phStatus_t phDlhalHw_Pn5180_Download_PerformSoftReset(phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams)
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxBytes =  PH_DLHAL_HW_PN5180_DL_SOFTRESET_RESP_LEN;
    uint8_t* pRxBuff = NULL;

    /*Form the command*/
    pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS] = PH_DLHAL_HW_PN5180_DL_OPCODE_DL_RESET;
    pDlDataParams->pTxBuffer[1] = 0x00;
    pDlDataParams->pTxBuffer[2] = 0x00;
    pDlDataParams->pTxBuffer[3] = 0x00;

    /*Send command and get response*/
    /*If there is no response, then reset is successful else in case of command error, it will provide the error response*/
    status = phDlhalHw_Pn5180_DLCmd_Transceive( pDlDataParams,
    											pDlDataParams->pTxBuffer,
                                                PH_DLHAL_HW_PN5180_DL_SOFTRESET_CMD_LEN,
                                                &pRxBuff,
                                                &wRxBytes );
    CHECK_STATUS(status);

    return status;
}


phStatus_t phDlhalHw_Pn5180_Download_GetFirmwareVersion(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
        uint8_t *pMajorVersion,
        uint8_t *pMinorVersion
        )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxBytes = PH_DLHAL_HW_PN5180_DL_GETVERSION_RESP_LEN;
    uint8_t* pRxBuff = NULL;

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
    PH_CHECK_NULL(pMajorVersion);
    PH_CHECK_NULL(pMinorVersion);

    /*Form the command*/
    pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS] = PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GETVERSION;
    pDlDataParams->pTxBuffer[1] = 0x00;
    pDlDataParams->pTxBuffer[2] = 0x00;
    pDlDataParams->pTxBuffer[3] = 0x00;

    /*Send command and get response*/
    status = phDlhalHw_Pn5180_DLCmd_Transceive( pDlDataParams,
    		pDlDataParams->pTxBuffer,
                                                PH_DLHAL_HW_PN5180_DL_GETVERSION_CMD_LEN,
                                                &pRxBuff,
                                                &wRxBytes );
    CHECK_STATUS(status);
    /*Copy the major and minor version*/
    *pMajorVersion = pRxBuff[PH_DLHAL_HW_PN5180_DL_GETVERSION_FMxV_POS2];
    *pMinorVersion = pRxBuff[PH_DLHAL_HW_PN5180_DL_GETVERSION_FMxV_POS1];

    return status;
}


phStatus_t phDlhalHw_Pn5180_Download_GetDieId(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,
        uint8_t *pDieId
        )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wRxBytes = PH_DLHAL_HW_PN5180_DL_GETDIEID_RESP_LEN;
    uint8_t* pRxBuff = NULL;

    /*Input parameter validation*/
	PH_CHECK_NULL(pDlDataParams);
    PH_CHECK_NULL(pDieId);

    /*Form the command*/
    pDlDataParams->pTxBuffer[PH_DLHAL_HW_PN5180_DL_PAYLOAD_OPCODE_POS] = PH_DLHAL_HW_PN5180_DL_OPCODE_DL_GETDIEID;
    pDlDataParams->pTxBuffer[1] = 0x00;
    pDlDataParams->pTxBuffer[2] = 0x00;
    pDlDataParams->pTxBuffer[3] = 0x00;

    /*Send command and get response*/
    status = phDlhalHw_Pn5180_DLCmd_Transceive( pDlDataParams,
    											pDlDataParams->pTxBuffer,
                                                PH_DLHAL_HW_PN5180_DL_GETDIEID_CMD_LEN,
                                                &pRxBuff,
                                                &wRxBytes );

    CHECK_STATUS(status);

    memcpy(pDieId,			/* PRQA S 3200 */
           &pRxBuff[PH_DLHAL_HW_PN5180_DL_GETDIEID_DIEID_POS],
           PH_DLHAL_HW_PN5180_DL_GETDIEID_DIEID_LEN );

    return PH_ERR_SUCCESS;
}
