/*
*         Copyright (c), NXP Semiconductors Gratkorn / Austria
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
*particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

/** \file
* Secure Firmware Upload Library
* $Author: Gergely Csontos (nxp95312) $
* $Revision: $
* $Date:  $
*
*/

#ifndef DOWNLOAD_LIB_H
#define DOWNLOAD_LIB_H

#include <phbalReg.h>
#include <ph_Status.h>

/** \defgroup phDownloadLib Component : Pn5180Download Library
* \brief Library for Pn5180 secure firmware download.
* @{
*/

/**
* \brief Integrity information structure.
this structure will hold the integrity information of all the applicable blocks.
*/

typedef struct phDlhalHw_Pn5180_Download_IntegrityInformation
{
    uint8_t bPatchTableOk;		/**< 1: means patch table integrity is good. */
    uint8_t bFunctionCodeOk;	/**< 1: means function code integrity is good. */
    uint8_t bPatchCodeOk;		/**< 1: means patch code integrity is good. */
    uint8_t bUserDataOk;		/**< 1: means User Data integrity is good. */
} phDlhalHw_Pn5180_Download_IntegrityInformation_t;

/**
* \brief Download session state information structure.
This structure will contain the session state and lifecycle information.
*/
typedef struct phDlhalHw_Pn5180_Download_SessionStateInformation
{
    uint8_t bSessionState;	/**< Holds the session state information.
*  	 	 	 	 	 	 	 	- 0: close <br>
*							 	- 1: open <br> */

    uint8_t bLifeCycle;		/**< Holds the lifecycle information.
*     	 	 	 	 	 	 	- 11d: creation
*								- 13d: initializing <br>
*								- 17d: operational <br>
*								- 19d: termination <br> */

} phDlhalHw_Pn5180_Download_SessionStateInformation_t;

/**
 * \brief PN5180 Download main structure.
 * This structure holds all the data that are required for the PN5180 Download library operations.
 */
typedef struct phDlhalHw_Pn5180_DL_DataParams
{
	uint16_t wId;                                       /**< Layer ID for this component, NEVER MODIFY! */
	uint8_t bBalConnectionType;                         /**< Type of the underlying BAL connection. Refer to #PH_DLHAL_HW_CONFIG_BAL_CONNECTION. */

	void * pBalDataParams;                              /**< Points to the underlying layer over which HAL is going to communicate */

	uint8_t * pTxBuffer;                                /**< Pointer to global transmit buffer used by the Exchange() function. */
	uint8_t * pTxBufferStart;
	uint16_t  wTxBufSize;                                /**< Size of the global transmit buffer. */

	uint8_t * pRxBuffer;                                /**< Pointer to global receive buffer used by the Exchange() function. */
	uint8_t * pRxBufferStart;
	uint16_t  wRxBufSize;                                /**< Size of the global receive buffer. */

		uint32_t dwAdditionalInfo;                           /**< Storage for additional error information. */

} phDlhalHw_Pn5180_DL_DataParams_t;


#define PH_DLHAL_HW_PN5180_DOWNLOAD_MODE_ENABLE       1		/**< Download mode enable. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_MODE_DISABLE      0		/**< Download mode disable. */


/**
* \name Possible returning error codes   
*/
/*@{*/


#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_INVALID_ADDR            (PH_COMP_GENERIC + 0x01)	/**< Trying to read or write an invalid address. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_GENERIC_ERROR           (PH_COMP_GENERIC + 0x02)	/**< Generic error occurred. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_UNKNOWN_CMD             (PH_COMP_GENERIC + 0x0B)	/**< Unknown Command. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_ABORTED_CMD             (PH_COMP_GENERIC + 0x0C)	/**< Chunk sequence is large . */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PLL_ERROR               (PH_COMP_GENERIC + 0x0D)	/**< PLL configuration is wrong, flash not activated. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_ADDRESS_RANGE_OVERFLOW  (PH_COMP_GENERIC + 0x1E)	/**< Write/read performed outside of address range. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_BUFFER_OVERFLOW         (PH_COMP_GENERIC + 0x1F)	/**< Buffer overflow error. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_MEMORY_BUSY             (PH_COMP_GENERIC + 0x20)	/**< EEPROM Memory operation in progress. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_SIGNATURE_ERROR         (PH_COMP_GENERIC + 0x21)	/**< Signature mismatch. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_FIRMWARE_VERSION_ERROR  (PH_COMP_GENERIC + 0x24)	/**< trying to write same firmware version. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR          (PH_COMP_GENERIC + 0x28)	/**< Download protocol error. */
#define PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_DEGRADED                (PH_COMP_GENERIC + 0x2A)	/**< EEPROM corruption. */
/*@}*/

#define PH_CHECK_NULL(a)                                             \
if((a) == NULL)                                                      \
 {                                                                   \
     return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_GENERIC);  \
 }



#define CHECK_STATUS(x)                         \
if ((x) != PH_ERR_SUCCESS)                      \
{                                               \
    return (x);                                   \
}

/* prints if error is detected */
#define CHECK_SUCCESS(x)              \
	if ((x) != PH_ERR_SUCCESS)        \
{                                     \
	DEBUG_PRINTF("\nLine: %d   Error - (0x%04X) has occurred : 0xCCEE CC-Component ID, EE-Error code. Refer-ph_Status.h\n ", __LINE__, (x)); \
	return (x);                       \
}

/**
 * \brief Initialization of the PN5180 Download Library.
 * This function does the following. <br>  
 * 1. Input parameter validation. <br>
 * 2. Assigns the BAL pointer. <br>
 * 3. Enters into download mode. <br>  
 * 4. Initialize the transmit and receive buffers pointers. <br>  
 * 5. Initialize the buffer sizes. <br>  
 * 6. Initialize the buffers to 0. <br>  
 *
 * \return Status Code
 * \retval #PH_ERR_SUCCESS when the Initialization is successful.
 */
phStatus_t phDlhalHw_Pn5180_Download_Init(
	phDlhalHw_Pn5180_DL_DataParams_t	*pDlDataParams,	/**<[In] DataParams to this layer. */
	void  *pbalparams,								/**<[In] Pointer to the underlying BAL component structure. */
    uint8_t * pTxBuffer,							/**<[In] Pointer to buffer used for sending download commands to PN5180. */
    uint16_t wTxBufSize,							/**<[In] Size of \p pTxBuffer.  Depends on the BAL buffer size. */
    uint8_t * pRxBuffer,							/**<[In] Pointer to buffer used for receiving download responses from PN5180. */
    uint16_t wRxBufSize								/**<[In] Size of \p pRxBuffer.  Depends on the BAL buffer size.  */
    );

/**
 * \brief Perform a Secure Firmware Update and checks the integrity of updated firmware.
 * This function does the following. <br>
 * 1. Check Firmware version to ascertain we are not reloading the same version. <br>
 * 2. Perform the actual firmware update. <br>
 * 3. Perform a integrity checking After download. <br>
 * 4. Check if the session state is closed. <br>
 
 * \return Status Code
 * \retval #PH_ERR_SUCCESS Register successfully written.
 * \retval #PH_ERR_INVALID_PARAMETER - Invalid parameter passed.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_FIRMWARE_VERSION_ERROR - the Firmware version on IC is same as being updated.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_GENERIC_ERROR - when integrity check fails or the Session is not closed.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR -Invalid response received.
 */
phStatus_t phDlhalHw_Pn5180_Download_PerformSecureFirmwareUpdate(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,	/**<[In] DataParams to this layer. */
    uint8_t * pImage,									/**<[In] Pointer to the download sequence image. */
    uint32_t dwSizeOfImage								/**<[In] Size of the Image. */
    );

/**
 * \brief Perform a read.
 *	This function can be used to read the User Area from EEPROM.The address must be within range 0x201380- 0x201F80.
 * \return Status Code
 * \retval #PH_ERR_SUCCESS Register successfully written.
 * \retval #PH_ERR_INVALID_PARAMETER - Invalid parameter passed.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR - Invalid response received.
 */
phStatus_t phDlhalHw_Pn5180_Download_Read(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,	/**<[In] DataParams to this layer. */
    uint16_t wLength,								/**<[In] Length of the data to read. */
    uint32_t dwAddress,								/**<[In] Address to read FROM. */
    uint8_t** ppReadBuffer							/**<[In] Pointer to the buffer pointer.Contains First two bytes of length and then data read. */
    );

/**
 * \brief Perform an Integrity Check of the updated Image.
 * This fuction will return the integrity information of the existing firmware and return the data in the phDlhalHw_Pn5180_Download_IntegrityInformation_t structure.
  * \return Status Code
 * \retval #PH_ERR_SUCCESS if the integrity information is read back successfully.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR - Invalid response received.
 */
phStatus_t phDlhalHw_Pn5180_Download_CheckIntegrity(
	phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,	/**<[In] DataParams to this layer. */
    phDlhalHw_Pn5180_Download_IntegrityInformation_t * pIntegrityCheckResult	/**<[Out] Pointer to phDlhalHw_Pn5180_Download_IntegrityInformation_t structure where the result of the Integrity Check will be stored. */
    );

/**
 * \brief Check and return current download session state.
 *  An internal state is maintained in the firmware and the session is open when the firmware update is in progress. The session is closed only after the complete firmware is updated.
  * \return Status Code
 * \retval #PH_ERR_SUCCESS if the session state is successfully read-back.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR - Invalid response received.
 */
phStatus_t phDlhalHw_Pn5180_Download_CheckSessionState(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,	/**<[In] DataParams to this layer. */
		phDlhalHw_Pn5180_Download_SessionStateInformation_t* sSSTAInfo	/**<[Out] SessionState result . */
		);

/**
 * \brief Issues a soft reset command.
 * \param[In] pDlDataParams Dataparams to this layer.
 * \note This API just sends the reset command but doesn't check for proper execution of the command.
  * \return Status Code
 * \retval #PH_ERR_SUCCESS if the reset was successfully performed.
 */
phStatus_t phDlhalHw_Pn5180_Download_PerformSoftReset(phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams);

/**
 * \brief Reads out version of current firmware.
 * this function will return the major and minor version.
  * \return Status Code
 * \retval #PH_ERR_SUCCESS if the firmware version is successfully read.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR - Invalid response received.
 *
 */
phStatus_t phDlhalHw_Pn5180_Download_GetFirmwareVersion(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,	/**<[In] DataParams to this layer. */
        uint8_t *pMajorVersion,								/**<[Out] Firmware Major version output. */
        uint8_t *pMinorVersion								/**<[Out] Firmware Minor version output. */
        );

/**
 * \brief Returns Die-ID.
 * This function will return the Die-ID read out from the IC.
 * \return Status Code
 * \retval #PH_ERR_SUCCESS when the Die-ID is successfully read-out.
 * \retval #PH_DLHAL_HW_PN5180_DOWNLOAD_ERR_PROTOCOL_ERROR - Invalid response received.
 */
phStatus_t phDlhalHw_Pn5180_Download_GetDieId(
		phDlhalHw_Pn5180_DL_DataParams_t * pDlDataParams,	/**<[In] DataParams to this layer. */
        uint8_t *pDieId	/**<[Out] Pointer to a buffer where the Die ID shall be stored. uint8_t[16]. */
        );

#endif /* DOWNLOAD_LIB_H */
