/***************************************************************************
* Functions to read and write the PCF8576 LCD controller
***************************************************************************/
#include "i2c.h"
#include "lcd.h"
#include "touch_task.h"

extern const char ascii[];

uint8_t  service_lcd_timer = 0;

static I2C_MSG lcdMsg;                                  // PCF8576 message
static uint8_t lcdBuf[14];                              // PCF8576 message buffer

static uint8_t C5_mask = 0x00;                          // 
static uint8_t C6_mask = 0x00;                          // 

void LCD_LED(uint8_t lednr, uint8_t onoff)              // controls the 4 LEDS at the LCD display
{                                                       // LED 0/1 are SEGI/L of 5th character
    switch (lednr)                                      // LED 2/3 are SEGI/L of 6th character
    {
        case 0  :   if (onoff)                          // LED0 ON / OFF ?
                    {
                        C5_mask    |=  0x01;            // LED0 mask to ON
                        lcdBuf[11] |=  0x01;            // repeat last character and switch LED0 ON
                    }
                    else
                    {
                        C5_mask    &= ~0x01;            // LED0 mask to OFF
                        lcdBuf[11] &= ~0x01;            // repeat last character and switch LED0 OFF
                    }
                    break;
        case 1  :   if (onoff)                          // LED1 ON / OFF ?
                    {
                        C5_mask    |=  0x02;            // LED1 mask to ON
                        lcdBuf[11] |=  0x02;            // repeat last character and switch LED1 ON
                    }
                    else
                    {
                        C5_mask    &= ~0x02;            // LED1 mask to OFF
                        lcdBuf[11] &= ~0x02;            // repeat last character and switch LED1 OFF
                    }
                    break;
        case 2  :   if (onoff)                          // LED2 ON / OFF ?
                    {
                        C6_mask    |=  0x01;            // LED2 mask to ON
                        lcdBuf[13] |=  0x01;            // repeat last character and switch LED2 ON
                    }
                    else
                    {
                        C6_mask    &= ~0x01;            // LED2 mask to OFF
                        lcdBuf[13] &= ~0x01;            // repeat last character and switch LED2 OFF
                    }
                    break;
        case 3  :   if (onoff)                          // LED3 ON / OFF ?
                    {
                        C6_mask    |=  0x02;            // LED3 mask to ON
                        lcdBuf[13] |=  0x02;            // repeat last character and switch LED3 ON
                    }
                    else
                    {
                        C6_mask    &= ~0x02;            // LED3 mask to OFF
                        lcdBuf[13] &= ~0x02;            // repeat last character and switch LED3 OFF
                    }
                    break;
        default :   break;
    }
    // now resend last message with LED mask included
    lcdMsg.nrBytes = 14;                                // I2C message lenght
    I2C_Transfer(&lcdMsg);                              // transfer I2C message to PCF8576
}

void LCD_PrintFullString(char *str)                     // display string of 6 characters
{
  int8_t t;

	lcdBuf[0]      =  0x60                              // device-select
                   | (0x0 << 0)                         // 000: subaddress
                   | (1   << 7);                        // another command to follow
	lcdBuf[1]      =  0x00                              // load-data_pointer
                   | (0x0 << 0)                         // 00000: RAM address 0
                   | (0   << 7);                        // last command

    t          = *str++ - ' ';                          // get 1st character font
    lcdBuf[2]  = FontMatrix[t];
    lcdBuf[3]  = FontMatrix[t] >> 8;

    t          = *str++ - ' ';                          // get 2nd character font
    lcdBuf[4]  = FontMatrix[t];
    lcdBuf[5]  = FontMatrix[t] >> 8;

    t          = *str++ - ' ';                          // get 3rd character font
    lcdBuf[6]  = FontMatrix[t];
    lcdBuf[7]  = FontMatrix[t] >> 8;

    t          = *str++ - ' ';                          // get 4th character font
    lcdBuf[8]  = FontMatrix[t];
    lcdBuf[9]  = FontMatrix[t] >> 8;

    t          = *str++ - ' ';                          // get 5th character font
    lcdBuf[10] = FontMatrix[t];
    lcdBuf[11] = (FontMatrix[t] >> 8) | C5_mask;

    t          = *str - ' ';                            // get 6th character font
    lcdBuf[12] = FontMatrix[t];
    lcdBuf[13] = (FontMatrix[t] >> 8) | C6_mask;

    lcdMsg.nrBytes = 14;                                // I2C message lenght
    I2C_Transfer(&lcdMsg);                              // transfer I2C message to PCF8576
}

void LCD_Send(uint8_t *buf, uint8_t nr)                 // display nr of bytes
{
  char msg[6];

    if (par.touch_mode == ONE_DIMENSIONAL)              // 1D mode
    {
        msg[0] = '0';
        msg[1] = 'x';
        msg[2] = ascii[buf[2] >> 4];
        msg[3] = ascii[buf[2] & 0x0f];
        msg[4] = ascii[buf[3] >> 4];
        msg[5] = ascii[buf[3] & 0x0f];
    }
    else                                                // 2D mode, show X - Y position
    {
        msg[0] = ascii[(buf[2] >> 2) / 10];
        msg[1] = ascii[(buf[2] >> 2) % 10];

        msg[2] = ' ';

        msg[3] = ascii[(buf[3] >> 2) / 10];
        msg[4] = ascii[(buf[3] >> 2) % 10];

        msg[5] = ' ';
    }

    LCD_PrintFullString(msg);
}

void Init_LcdTask(void)
{
  uint32_t i;
  char msg[6];

    I2CMST_Init();                                      // initialize I2C bus

    lcdMsg.address = SLA_PCF8576;                       // I2C bus slave address
    lcdMsg.buf     = lcdBuf;                            // pointer to lcd message buffer
    lcdMsg.next    = 0;                                 // no linked message

	lcdBuf[0]      =  0x40                              // mode-set
                   | (0x0 << 0)                         // 1:4 multiplex
                   | (0   << 2)                         // LCD 1/3 bias
                   | (1   << 3)                         // enable display
                   | (1   << 7);                        // another command to follow

	lcdBuf[1]      =  0x70                              // blink-select
                   | (0x0 << 0)                         // blinking off
                   | (0   << 2)                         // blink mode normal
                   | (0   << 7);                        // last command

    lcdMsg.nrBytes = 2;                                 // I2C message lenght
    I2C_Transfer(&lcdMsg);                              // transfer I2C message to PCF8576

    LCD_PrintFullString("NXP   ");
    for (i = 0; i < 2000000; i++) ;

    msg[0] = 'V';
    msg[1] = ascii[sys.version >> 12];
    msg[2] = ascii[(sys.version >> 8) & 0x000f];
    msg[3] = ascii[(sys.version >> 4) & 0x000f];
    msg[4] = ascii[sys.version & 0x000f];
    msg[5] = ' ';
    LCD_PrintFullString(msg);
    for (i = 0; i < 2000000; i++) ;

    LCD_PrintFullString("Touch ");
}

void vLcdTask(void)
{
  static uint8_t trg  = 1;

    if (service_lcd_timer > 0)
        return;

    if (!(LPC_GPIO_PORT->PIN[0] & (1 << 12)) && trg)        // P0_12 (ISP button) high to low?
    {
        trg = 0;                                            // trigger low
        if (par.touch_mode == ONE_DIMENSIONAL)              // one dimensional mode
        {
            par.touch_mode = TWO_DIMENSIONAL;               // 2D position mode
            LCD_PrintFullString(" -2D- ");
        }
        else
        {
            par.touch_mode = ONE_DIMENSIONAL;               // 2D position mode
            LCD_PrintFullString(" -1D- ");
        }
    }
    else if ((LPC_GPIO_PORT->PIN[0] & (1 << 12)) && !trg)   // P0.12 low to high?
    {
        trg = 1;                                            // trigger high
    }

    service_lcd_timer = 10;                                 // 10 msec task pause
}
