/*
===============================================================================
 Name        :  touch_task.c
 Author      :  nlv11817
 Description :  handles all Touch Solution library API related functions
                + storage of Touch parameters in flash
===============================================================================
*/
#include "touch_task.h"
#include "string.h"
#include "comm_task.h"
#include "lcd.h"

#define FLASH_SECT              31                          // use sector 31 (last 1K of 32K part)
#define FLASH_PAGE              511                         // use sector 511 (last 64 byte page of 32K part)
#define FLASH_ADDR              (FLASH_PAGE * 64)           // address in flash of parameter storage

const TOUCH_CFG_T conf = {3, 3,                             // 3 rows and 3 columns
                         {{PIN_X0, PIN_X3, PIN_X6},
                          {PIN_X1, PIN_X4, PIN_X7},
                          {PIN_X2, PIN_X5, PIN_X8}},
                          PIN_YL, ACMP_I3, PIN_YH };

uint8_t StoreTouchParameters(void)
{
typedef void (*IAP)(uint32_t [], uint32_t []);

  IAP      iap_entry = (IAP) 0x1fff1ff1;
  uint32_t cmd[5];
  uint32_t res[5];

    __disable_irq();

    cmd[0] = 50;                                    // prepare sector
    cmd[1] = FLASH_SECT;                            // start sector
    cmd[2] = FLASH_SECT;                            // end sector
    iap_entry(cmd, res);

    cmd[0] = 59;                                    // erase page
    cmd[1] = FLASH_PAGE;                            // start page
    cmd[2] = FLASH_PAGE;                            // end page
    cmd[3] = SystemCoreClock/1000;                  // system clock in KHz
    iap_entry(cmd,res);                             // error checking recommended

    cmd[0] = 50;                                    // prepare sector
    cmd[1] = FLASH_SECT;                            // start sector
    cmd[2] = FLASH_SECT;                            // end sector
    iap_entry(cmd, res);

    cmd[0] = 51;                                    // copy ram to flash
    cmd[1] = FLASH_ADDR;                            // destination
    cmd[2] = (uint32_t) &par;                       // source
    cmd[3] = 64;                                    // nr of bytes, minimum size of a page
    cmd[4] = SystemCoreClock/1000;                  // system clock in KHz
    iap_entry(cmd,res);                             // error checking recommended

    __enable_irq();

    Touch_Update();
    return 0;
}

void TouchEventHandler(uint8_t event, uint8_t buf[4])
{
    switch (event)
    {
        case EV_START : if (send_pos_data)
                            UARTSend(buf, 4);              // send start message to GUI !
                        LCD_LED(1, 1);                     // LCD LED1 on
                        break;
        case EV_DATA  : if (send_pos_data)
                            UARTSend(buf, 4);              // send position message to GUI !
                        LCD_Send(buf, 4);                  // send position data to LCD
                        break;
        case EV_END   : if (send_pos_data)
                            UARTSend(buf, 4);               // send position message to GUI !
                        LCD_LED(1, 0);                      // LCD LED1 off
                        break;
    }
}

void Init_Touch(void)
{
    // Copy from parameter page
    memcpy (&par, (void *)FLASH_ADDR, sizeof(par));

    if (par.agc_min < 500 || par.agc_min > 7000)    // just check if one of the parameters is out of range
    {                                               // if yes, load all defaults
        par.touch_mode  = TWO_DIMENSIONAL;          // touch mode (one - or two dimensional)
        par.agc_mode    = USE_AGC;                  // automatic gain control on/off
        par.agc_min     = AGC_MIN_CYCLES;           // min value for Reference Integration Cycles in agc mode
        par.agc_max     = AGC_MAX_CYCLES;           // max value for Reference Integration Cycles in agc mode
        par.system_gain = SYSTEM_GAIN;              // analog comparator input gain
        par.dt_mode     = USE_DDT;                  // automatic or dynamic Touch Sensitivity Calibration Status
        par.dt_min      = DDT_MIN_LIMIT;            // minimum limit for difference between Reference & Raw touch
        par.dt_max      = DDT_MAX_LIMIT;            // maximum limit for difference between Reference & Raw touch
        par.fd_pos      = POS_FILTER;               // touch position averaging coefficient
        par.fd_ref      = REF_FILTER;               // touch reference cycles averaging coefficient
        par.fd_raw      = RAW_FILTER;               // touch raw cycles averaging coefficient
    }
    sys.cb_func = TouchEventHandler;
    par.system_gain = 21;                           // increase sensitivity because overlay changed
                                                    // from cello-tape to PC sheet
    Touch_Init(&conf);
}
