/***************************************************************************/
/*  USART0 (polling) driver for NXP LPC800 microcontrollers.               */
/***************************************************************************/
#include "LPC82x.h"                                         // LPC82x definitions
#include "uart.h"

#define OSR            (*(volatile uint32_t *)(0x40064028))

const char ascii[] = "0123456789ABCDEF";

uint8_t uart_buf[32];
uint8_t uart_ptr = 0;

void USART0_Init(uint32_t baudrate, uint8_t txdpin, uint8_t rxdpin)
/*****************************************************************/
{
  uint32_t brd = (SystemCoreClock / (baudrate << 4));

    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 7) |                 // enable clock to Switch Matrix
                                 (1 << 18);                 // enable AHB clock to the IOCON

    LPC_SWM->PINASSIGN[0] |= 0x0000FFFF;                    // 
    LPC_SWM->PINASSIGN[0] &= ((txdpin << 0) | 0xFFFFFF00);  // assign U0 TXD to P0_txdpin
    LPC_SWM->PINASSIGN[0] &= ((rxdpin << 8) | 0xFFFF00FF);  // assign U0 RXD to P0_rxdpin

    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 14);                 // Enable USART0 clock
    LPC_SYSCON->UARTCLKDIV     = 1;                         // UCLK divider

	LPC_USART0->BRG      = brd - 1;                         // set baud rate
    LPC_USART0->CFG      = (1 << 0) | (1 << 2);             // 8 bits, no Parity, 1 Stop bit

    LPC_USART0->STAT     = UART_STAT_DELTACTS
                         | UART_STAT_OVERRUNINT
                         | UART_STAT_DELTARXBRK
                         | UART_STAT_START
                         | UART_STAT_FRM_ERRINT
                         | UART_STAT_PAR_ERRINT
                         | UART_STAT_RXNOISEINT
                         | UART_STAT_ABERR;
    LPC_USART0->INTENSET = UART_INTEN_RXRDY;
    LPC_USART0->INTENCLR = UART_INTEN_TXRDY;
    LPC_USART0->CTRL    &= ~UART_CTRL_TXDIS;

    NVIC_EnableIRQ(UART0_IRQn);                             // enable IRQ for USART0
}

char ua_inchar(void)	                                    // look if there is a character available
/******************/
{
    if (LPC_USART0->STAT & (1<<0))
        return LPC_USART0->RXDATA & 0x7F;                   // yes, return character from input buffer
    else
        return 0xFF;                                        // no, return -1
}

char ua_getchar(void)                                       // wait for a character
/*******************/
{
    while (!(LPC_USART0->STAT & (1<<0))) ;                  // receiver ready?
    return LPC_USART0->RXDATA & 0x7F;                       // read character from input buffer
}

void ua_outchar(char c)		
/*********************/
{
    while (!(LPC_USART0->STAT & (1<<2))) ;                  // transmitter ready?
    LPC_USART0->TXDATA = c;
}

void PrintByte(unsigned char b)
/*****************************/
{
    ua_outchar(ascii[b >> 4]);
    ua_outchar(ascii[b & 0x0f]);
}

void PrintWord(unsigned short w)
/******************************/
{
    PrintByte((unsigned char)(w>>8));
    PrintByte((unsigned char)w);
}

void PrintString(const char *s)
/*****************************/
{
    while (*s)
    {
        if (*s == '\n')
            ua_outchar('\r');                               // output a '\r' first
        ua_outchar(*s);
        s++;
    }
}

void UART0_IRQHandler(void)
/*************************/
{
	if (LPC_USART0->STAT & UART_STAT_RXRDY && uart_ptr < 32)                 // Receiver Interrupt
	{
		uart_buf[uart_ptr] = (uint8_t)LPC_USART0->RXDATA;
		uart_ptr ++;
	}
/*	if(Chip_UART_GetStatus(LPC_USART0)&UART_STAT_TXRDY)     // TRansmitter Interrupt
	{
	}*/
}

/*****************************************************************************
** Function name:   UARTSend
**
** Description:     Send a block of data to the UART 0 port based
**                  on the data length
**
** parameters:      buffer pointer, and data length
** Return value:    None
** 
*****************************************************************************/
void UARTSend(uint8_t *buf, uint32_t length)
{
    while (length != 0)
    {
        while ((LPC_USART0->STAT & UART_STAT_TXRDY) == 0)
        {
		  // wait until previous byte is transferred
        }
        LPC_USART0->TXDATA = *buf;
        buf ++;
        length --;
    }
}
