#include "cortexm.h"

/* NVIC Registers */

#define NVIC_ISER(x) REG32(0xE000E100 + (4 * (x)))
#define NVIC_ICER(x) REG32(0xE000E180 + (4 * (x)))
#define NVIC_ISPR(x) REG32(0xE000E200 + (4 * (x)))
#define NVIC_ICPR(x) REG32(0xE000E280 + (4 * (x)))
#define NVIC_IABR(x) REG32(0xE000E300 + (4 * (x)))
#define NVIC_IPR(x)  REG32(0xE000E400 + (4 * (x)))
#define STIR         REG32(0xE000EF00 + 0x0E00)

/* SysTick Registers */

#define SYST_CSR   REG32(0xE000E010)
#define SYST_RVR   REG32(0xE000E014)
#define SYST_CVR   REG32(0xE000E018)
#define SYST_CALIB REG32(0xE000E01C)

void nvic_enable_irq(uint8_t irqn)
{
    NVIC_ISER(irqn / 32) = 1 << (irqn % 32);
}


void nvic_disable_irq(uint8_t irqn)
{
    NVIC_ICER(irqn / 32) = 1 << (irqn % 32);
}

int nvic_get_pending_irq(uint8_t irqn)
{
    return NVIC_ISPR(irqn / 32) & (1 << (irqn % 32)) ? 1 : 0;
}

void nvic_set_pending_irq(uint8_t irqn)
{
    NVIC_ISPR(irqn / 32) = 1 << (irqn % 32);
}

void nvic_clear_pending_irq(uint8_t irqn)
{
    NVIC_ICPR(irqn / 32) = 1 << (irqn % 32);
}

void nvic_set_priority(uint8_t irqn, uint8_t prio)
{
    uint32_t ipr_value = NVIC_IPR(irqn / 4);

    /* first, clear corresponding bits */
    ipr_value &= ~(0xFF << ((irqn % 4) * 8));

    /* next, set corresponding bits */
    ipr_value |= prio << ((irqn % 4) * 8);

    NVIC_IPR(irqn / 4) = ipr_value;
}
    
uint8_t nvic_get_priority(uint8_t irqn)
{
    uint32_t ipr_value = NVIC_IPR(irqn / 4);

    ipr_value >>= (irqn % 4) * 8;
    ipr_value &= 0xFF;

    return (uint8_t)ipr_value;
}

void systick_start(void)
{
    SYST_CSR = 0x7;
}

void systick_stop(void)
{
    SYST_CSR = 0;
}

uint32_t systick_reload(uint32_t reload)
{
    if (reload > 0) {
        SYST_RVR = reload;
    }

    return SYST_RVR;
}

uint32_t systick_current(uint32_t clear)
{
    if (clear) SYST_CVR = 1;
    return SYST_CVR;
}

void systick_clear_flag(void)
{
    SYST_CSR &= ~(1 << 16);
}
