package com.processorexpert.derivative.job;

/*******************************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM - Initial API and implementation
 *     John Cortell (Freescale) - code customized for Processor Expert
 *     
 * This is a derivative work created by Freescale and distributed as EPL. See 
 * description below 
 *******************************************************************************/

import java.util.LinkedList;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.WorkbenchPlugin;
import org.eclipse.ui.internal.misc.UIStats;
import org.eclipse.ui.internal.progress.ProgressMessages;

/**
 * This is a variation of eclipse's UIJob that ensures job A runs on the UI
 * thread before job B, if A is scheduled first, and both are scheduled from the
 * same thread.
 *
 * <p>
 * UIJob surprisingly does not guarantee order. I.e., if you do:
 * 
 * <pre>
 *    UIJob jobA = new UIJob("A"){...}
 *    UIJob jobB = new UIJob("B"){...}
 *    jobA.schedule(); 
 *    jobB.schedule();
 * </pre>
 * 
 * there is a pretty good chance that jobB will be put on the UI thread first.
 * Keep in mind this important distinction: A is guaranteed to actually be
 * scheduled before B, but B's runInUiThread() might get called first!
 * 
 * <p>
 * The reason is interesting. While Job A is guaranteed to be scheduled before
 * Job B, UIJob's implementation of
 * 
 * <pre>
 * org.eclipse.ui.progress.UIJob.run(IProgressMonitor)
 * </pre>
 * 
 * is unserialized. If you look at that method, you'll see that the code calls
 * Display.asyncExec() to invoke the runInUIThread() method. Since both jobs are
 * at that time running simultaneously and are neck in neck, there is no telling
 * which one will get their Runnable on the display queue first. A has a small
 * advantage over B, but experimentation shows B can overtake A quite frequently
 * and beat it to the Display.asyncExec() call.
 * 
 * <p>
 * This implementation basically ensures that Job B will wait for Job A to call
 * Display.asyncExec() first. Note that Display.asyncExec() does guarantee
 * order, so if we guarantee A calls that method first, we are guaranteed that
 * A's runInUiThread will run before's B, and without competition (since we are
 * talking about the UI thread, after all). This means that in the following:
 * 
 * <pre>
 *    OrderedUIJob jobA = new OrderedUIJob("A"){...}
 *    OrderedUIJob jobB = new OrderedUIJob("B"){...}
 *    jobA.schedule(); 
 *    jobB.schedule();
 * </pre>
 * 
 * jobA's runInUiThread() is guaranteed to be called before B's
 * 
 * <p>
 * Take care not to create and schedule these objects in a tight loop, since
 * that could cause a backup of many waiting Job objects
 */
@SuppressWarnings("restriction")
public abstract class OrderedUIJob extends Job {
	/**
	 * Every instance get a unique ID. This has the next ID. We need not worry
	 * about turnover.
	 */
	static int nextId;

	/** The unique ID of this instance */
	int id;

    /**
	 * A queue of outstanding instance IDs. An instance's ID is added when it's
	 * instantiated, and removed when the instance's
	 * {@link #run(IProgressMonitor)} method completes or the job is cancelled.
	 */
    private static LinkedList<Integer> queue = new LinkedList<Integer>();


    private Display cachedDisplay;

	/** Set to true after run() completes */
	private boolean alreadyRan;
    

    /**
     * Create a new instance of the receiver with the supplied name. The display
     * used will be the one from the workbench if this is available. UIJobs with
     * this constructor will determine their display at runtime.
     * 
     * @param name
     *            the job name
     *  
     */
    public OrderedUIJob(String name) {
        super(name);
        synchronized (OrderedUIJob.class) {
        	id = nextId++;
        }
    }

    /**
     * Create a new instance of the receiver with the supplied Display.
     * 
     * @param jobDisplay
     *            the display
     * @param name
     *            the job name
     * @see Job
     */
    public OrderedUIJob(Display jobDisplay, String name) {
        this(name);
        setDisplay(jobDisplay);
    }

	/* (non-Javadoc)
	 * @see org.eclipse.core.runtime.jobs.Job#shouldSchedule()
	 */
	public boolean shouldSchedule() {
		boolean doIt = super.shouldSchedule();
		if (doIt) {
			// We're being scheduled. Add our ID to the queue
			synchronized (OrderedUIJob.class) {
				queue.addLast(new Integer(id));
			}
		}
		return doIt;
	}
    /**
     * Convenience method to return a status for an exception.
     * 
     * @param exception
     * @return IStatus an error status built from the exception
     * @see Job
     */
	public static IStatus errorStatus(Throwable exception) {
        return WorkbenchPlugin.getStatus(exception); 
    }

    /**
     * @see org.eclipse.core.runtime.jobs.Job#run(org.eclipse.core.runtime.IProgressMonitor)
     *      Note: this message is marked final. Implementors should use
     *      runInUIThread() instead.
     */
    public final IStatus run(final IProgressMonitor monitor) {
    	if (alreadyRan) {
    		throw new IllegalStateException("An OrderedUIJob cannot be scheduled multiple times.");
    	}
    	try {
			// This code is what makes this class different than UIJob.
			// We yield if another instance was created first but has not yet
			// had a chance to call Display.asyncExec()
	    	boolean wait = false;
	    	while (true) {
		    	synchronized (OrderedUIJob.class) {
		    		assert queue.contains(new Integer(id));
	    			wait = (queue.size() > 1) && (id != queue.getFirst());
		    	}
		    	if (!wait) {
		    		break;
		    	}
				try {
					Thread.sleep(10);
				} catch (InterruptedException e) {
					return Status.CANCEL_STATUS;
				}
	    	}
	    	// end specialized code
	    	
	        if (monitor.isCanceled()) {
				return Status.CANCEL_STATUS;
			}
	    	
	        Display asyncDisplay = getDisplay();
	        if (asyncDisplay == null || asyncDisplay.isDisposed()) {
	            return Status.CANCEL_STATUS;
	        }
	        asyncDisplay.asyncExec(new Runnable() {
	            public void run() {
	                IStatus result = null;
	                Throwable throwable = null;
	                try {
	                    //As we are in the UI Thread we can
	                    //always know what to tell the job.
	                    setThread(Thread.currentThread());
	                    if (monitor.isCanceled()) {
							result = Status.CANCEL_STATUS;
						} else {
	                       	UIStats.start(UIStats.UI_JOB, getName());
	                        result = runInUIThread(monitor);
	                    }
	
	                } catch(Throwable t){
	                	throwable = t;
	                } finally {
	               		UIStats.end(UIStats.UI_JOB, OrderedUIJob.this, getName());
	                    if (result == null) {
							result = new Status(IStatus.ERROR,
	                                PlatformUI.PLUGIN_ID, IStatus.ERROR,
	                                ProgressMessages.InternalError,
	                                throwable);
						}
	                    done(result);
	                }
	            }
	        });
	        return Job.ASYNC_FINISH;
    	}
    	finally {
    		alreadyRan = true;
    		// more specialization
	        synchronized (OrderedUIJob.class) {
	        	boolean hadIt = queue.remove(new Integer(id));
	        	assert hadIt;
	        }
    	}
    }

    /* (non-Javadoc)
     * @see org.eclipse.core.runtime.jobs.Job#canceling()
     */
    protected void canceling() {
    	// This method is also unique to OrderedUIJob
        synchronized (OrderedUIJob.class) {
        	boolean hadIt = queue.remove(new Integer(id));
        	assert hadIt;
        }
    	super.canceling();
    }
    /**
     * Run the job in the UI Thread.
     * 
     * @param monitor
     * @return IStatus
     */
    public abstract IStatus runInUIThread(IProgressMonitor monitor);

    /**
     * Sets the display to execute the asyncExec in. Generally this is not'
     * used if there is a valid display available via PlatformUI.isWorkbenchRunning().
     * 
     * @param runDisplay
     *            Display
     * @see OrderedUIJob#getDisplay()
     * @see PlatformUI#isWorkbenchRunning()
     */
    public void setDisplay(Display runDisplay) {
        Assert.isNotNull(runDisplay);
        cachedDisplay = runDisplay;
    }

    /**
     * Returns the display for use by the receiver when running in an
     * asyncExec. If it is not set then the display set in the workbench
     * is used.
     * If the display is null the job will not be run.
     * 
     * @return Display or <code>null</code>.
     */
    public Display getDisplay() {
        //If it was not set get it from the workbench
        if (cachedDisplay == null && PlatformUI.isWorkbenchRunning()) {
			return PlatformUI.getWorkbench().getDisplay();
		}
        return cachedDisplay;
    }
}
