/*************************************************************************
*
* Freescale Confidential Proprietary
*
* Copyright (c) 2012-2013 Freescale Semiconductor;
* All Rights Reserved
*
**************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.    
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
* THE POSSIBILITY OF SUCH DAMAGE.
*
**************************************************************************/

#include <stdlib.h>
#include <EGL/egl.h>
#include <mqx.h>
#include <fio.h>
#include <hdmi.h>

#include "fbws.h"
#include "ovgapp.h"

#define TOTAL_FRAME     10000
#define SCREEN_ID_COUNT (2)

EGLNativeDisplayType fbdisp[SCREEN_ID_COUNT];
EGLNativeWindowType fbwin[SCREEN_ID_COUNT];
EGLDisplay disp[SCREEN_ID_COUNT];
EGLContext ctx[SCREEN_ID_COUNT];
EGLSurface surf[SCREEN_ID_COUNT];

static void ovgApp(uint32_t aPara);

const TASK_TEMPLATE_STRUCT  MQX_template_list[] = 
{ 
    /* Task Index,   Function,   Stack,  Priority, Name,     Attributes,                                    Param, Time Slice */
    { 5,             ovgApp,     0x8000,   10,    "ovgApp",  MQX_FLOATING_POINT_TASK | MQX_AUTO_START_TASK, 0,     0 },
    { 0 }
};

static void initialize_buffers(void)
{
    uint32_t i;
    
    for (i = 0; i < SCREEN_ID_COUNT; ++i)
    {
        fbdisp[i] = NULL;
        fbwin[i] = NULL;
        disp[i] = NULL;
        ctx[i] = NULL;
        surf[i] = NULL;
    }
}

static void create_native_window(uint8_t display_id, int32_t * width, int32_t * height)
{
    FbwsBufferInfo info;
    
    fbdisp[display_id] = fbws_open_display(display_id, NULL);
    fbws_get_display_info(fbdisp[display_id], &info, NULL);
    fbwin[display_id] = fbws_create_window(fbdisp[display_id], 0, 0, info.width, info.height);
    
    * width = info.width;
    * height = info.height;
}

static void create_egl_context(uint8_t display_id, EGLint cfgAttribs[], EGLConfig * chosenCfg)
{
    EGLint maj, min, num;
    EGLConfig cfg[8];
    EGLint r, g, b, a;
    EGLBoolean ret;
    int32_t i;
    
    EGL_ERR_CHK(EGL_NO_DISPLAY, disp[display_id], eglGetDisplay(fbdisp[display_id]));
    EGL_ERR_CHK(EGL_FALSE, ret, eglInitialize(disp[display_id], &maj, &min));
    
    /* Try to find out the exact configuration */
    EGL_ERR_CHK(EGL_FALSE, ret, eglChooseConfig(disp[display_id], cfgAttribs, cfg, sizeof(cfg) / sizeof(EGLConfig), &num));

    for (i = 0; i < num; ++i) 
	{
        eglGetConfigAttrib(disp[display_id], cfg[i], EGL_RED_SIZE, &r);
        eglGetConfigAttrib(disp[display_id], cfg[i], EGL_GREEN_SIZE, &g);
        eglGetConfigAttrib(disp[display_id], cfg[i], EGL_BLUE_SIZE, &b);
        eglGetConfigAttrib(disp[display_id], cfg[i], EGL_ALPHA_SIZE, &a);
		
        if ((r == cfgAttribs[1]) && (g == cfgAttribs[3]) && (b == cfgAttribs[5]) && (a == cfgAttribs[7]))
        {
            printf("Exact configuration found: %d/%d\n", i, num);
            break;
        }
    }

    if (i == num)
    {
        printf("Cannot find the exact configuration from total %d items, use the first configuration instead\n", num);
        i = 0;
    }
    
    EGL_ERR_CHK(EGL_NO_CONTEXT, ctx[display_id], eglCreateContext(disp[display_id], cfg[i], EGL_NO_CONTEXT, NULL));
    * chosenCfg = cfg[i];
}

static void initialize_egl_surface(uint8_t display_id, EGLConfig chosenCfg)
{
    EGLBoolean ret;
    
    EGL_ERR_CHK(EGL_NO_SURFACE, surf[display_id], eglCreateWindowSurface(disp[display_id], chosenCfg, fbwin[display_id], NULL));
    EGL_ERR_CHK(EGL_FALSE, ret, eglMakeCurrent(disp[display_id], surf[display_id], surf[display_id], ctx[display_id]));
}

static void egl_cleanup(uint8_t display_id)
{
    EGLBoolean ret;
    
    EGL_ERR_CHK(EGL_FALSE, ret, eglMakeCurrent(disp[display_id], EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT));
    EGL_ERR_CHK(EGL_FALSE, ret, eglDestroyContext(disp[display_id], ctx[display_id]));
    EGL_ERR_CHK(EGL_FALSE, ret, eglDestroySurface(disp[display_id], surf[display_id]));
    EGL_ERR_CHK(EGL_FALSE, ret, eglTerminate(disp[display_id]));
}

static void fbws_cleanup(uint8_t display_id)
{
    fbws_destroy_window(fbwin[display_id]);
    fbws_close_display(fbdisp[display_id]);
}

static void set_context(int * crt_id)
{
    EGLBoolean ret;
        
    * crt_id = 1 - (* crt_id);
    EGL_ERR_CHK(EGL_FALSE, ret, eglMakeCurrent(disp[* crt_id], surf[* crt_id], surf[* crt_id], ctx[* crt_id]));
}

static void ovgApp(uint32_t aPara)
{
    int32_t width, height;
    EGLConfig cfg;
    EGLBoolean ret;
    
    EGLint cfgAttribs[] = 
	{
        EGL_RED_SIZE,           8,
        EGL_GREEN_SIZE,         8,
        EGL_BLUE_SIZE,          8,
        EGL_ALPHA_SIZE,         8,
        EGL_DEPTH_SIZE,         0,
        EGL_RENDERABLE_TYPE,    EGL_OPENVG_BIT,
        EGL_NONE
    };
	int i;
    int current_id = 0;

    if (!HDMI_config(0)) {
        printf("OVG could not configure HDMI device\n");
    }
	
    if (!fbws_init())
    {
        printf("OVG initialize failed\n");
        return;
    }

    initialize_buffers();
    
    for (i = 0; i < SCREEN_ID_COUNT; ++i)
    {
        create_native_window(i, &width, &height);
    }
    
    EGL_ERR_CHK(EGL_FALSE, ret, eglBindAPI(EGL_OPENVG_API));
    
    for (i = 0; i < SCREEN_ID_COUNT; ++i)
    {
        create_egl_context(i, cfgAttribs, &cfg);
        initialize_egl_surface(i, cfg);
        OvgApp_Init(i, width, height);
    }

    for (i = 0; i < TOTAL_FRAME; i++)
    {
        set_context(&current_id);
        OvgApp_Draw(current_id, width, height);
        eglSwapBuffers(disp[current_id], surf[current_id]);
    }

    for (i = 0; i < SCREEN_ID_COUNT; ++i)
    {
        OvgApp_Close(i);
        egl_cleanup(i);
        fbws_cleanup(i);
    }
    
    fbws_fini();
}