package com.nxp.s32ds.cross.sdk.s32k1xx_2_9_0.pexconf;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.settings.model.CSourceEntry;
import org.eclipse.cdt.core.settings.model.ICConfigurationDescription;
import org.eclipse.cdt.core.settings.model.ICProjectDescription;
import org.eclipse.cdt.core.settings.model.ICSourceEntry;
import org.eclipse.cdt.managedbuilder.core.BuildException;
import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.IOption;
import org.eclipse.cdt.managedbuilder.core.IResourceConfiguration;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;

public abstract class APExConfigk1xxAdjuster {
	private static final String[] sourceFolders = new String[] { "SDK", "Generated_Code", "Sources" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	private static final String[] backupFolders = new String[] { "Project_Settings", "src", "include" };//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	private static final String[] backupFiles = new String[] { ".cproject" };//$NON-NLS-1$

	protected IProject project;

	protected APExConfigk1xxAdjuster(IProject project) {
		this.project = project;
	}

	void adjustPExConfiguration() {
		try {
			backupProjectResources();
			createSourceEntries();
			adjustOptions();
			cleanupFolders();
			excludeResources();
		} catch (Exception e) {
			Activator.log(e);
		}
	}

	protected void createSourceEntries() throws CoreException {
		Set<ICSourceEntry> additionalSourceEntrires = new HashSet<>();
		for (String sourceFolder : sourceFolders) {
			additionalSourceEntrires.add(new CSourceEntry(project.getFolder(sourceFolder), null, 0));
		}
		ICProjectDescription description = CCorePlugin.getDefault().getProjectDescription(project);
		for (ICConfigurationDescription config : description.getConfigurations()) {
			Set<ICSourceEntry> sourceEntries = new HashSet<>(Arrays.asList(config.getSourceEntries()));
			sourceEntries.addAll(additionalSourceEntrires);
			config.setSourceEntries(sourceEntries.toArray(new ICSourceEntry[sourceEntries.size()]));
		}
		CCorePlugin.getDefault().setProjectDescription(project, description, false, new NullProgressMonitor());
	}

	protected abstract void adjustOptions() throws BuildException;

	protected abstract void cleanupFolders() throws CoreException;

	protected void excludeResources() throws CoreException {
		IFolder linkedFilesFolder = project.getFolder("Project_Settings/Linker_Files"); //$NON-NLS-1$
		IConfiguration[] configs = ManagedBuildManager.getBuildInfo(project).getManagedProject().getConfigurations();
		for (IConfiguration config : configs) {
			IResourceConfiguration resourceConfig = config.getResourceConfiguration(linkedFilesFolder.getFullPath()
					.toString());
			if (resourceConfig == null) {
				resourceConfig = config.createFileInfo(linkedFilesFolder.getFullPath().removeFirstSegments(1));
			}
			if (resourceConfig != null) {
				resourceConfig.setExclude(true);
			}
		}
	}

	protected void addOptionValue(IOption opt, String value) throws BuildException {
		if (opt == null) {
			return;
		}
		List<String> curValueList = new ArrayList<>(Arrays.asList(opt.getBasicStringListValue()));
		curValueList.add("\"${ProjDirPath}/" + value + "\""); //$NON-NLS-1$ //$NON-NLS-2$
		opt.setValue(curValueList);
	}

	/**
	 * May be overriden by PExConfig*Adjuster classes if necessary.
	 */
	protected void backupProjectResources() throws CoreException {
		Stream.of(backupFiles)
				.map(f -> project.getFile(f))
				.filter(file -> (file.exists() && !file.isLinked()))
				.forEach(this::backupFile);

		Stream.of(backupFolders)
				.map(f -> project.getFolder(f))
				.filter(fld -> (fld.exists() && !fld.isLinked()))
				.forEach(this::backupFolder);
	}

	/**
	 * May be called from PExConfig*Adjuster classes for backup folders that are
	 * not defined in global list.
	 * 
	 * @param sourceFolder
	 *			folder to be backed up.
	 */
	protected void backupFolder(IFolder sourceFolder) {
		try {
			IFolder folder = project.getFolder(sourceFolder.getName() + ".bak");
			if (folder == null || folder.exists()) {
				return;
			}
			folder.create(IResource.FORCE, true, new NullProgressMonitor());
			IPath newFolderPath = folder.getFullPath();
			for (IResource resource : sourceFolder.members()) {
				IPath copyToPath = newFolderPath.append(resource.getFullPath().lastSegment());
				resource.copy(copyToPath, true, new NullProgressMonitor());
			}
			folder.accept(new IResourceVisitor() {
				@Override
				public boolean visit(IResource backupResource) throws CoreException {
					if (IResource.FILE == backupResource.getType()) {
						backupFile((IFile) backupResource, true);
					}
					return true;
				}
			});

		} catch (CoreException ce) {
			Activator.log(ce);
		}
	}

	/**
	 * May be called from PExConfig*Adjuster classes for backup files that are
	 * not defined in global list.
	 * 
	 * @param sourceFile
	 *			file to be backed up.
	 */
	protected void backupFile(IFile sourceFile) {
		backupFile(sourceFile, false);
	}

	protected void backupFile(IFile sourceFile, boolean isDeleteSource) {
		try {
			String pathStr = sourceFile.getFullPath().lastSegment() + ".bak";
			IFile file = project.getFile(pathStr);
			if (file.exists()) {
				return;
			}
			if (isDeleteSource) {
				sourceFile.move(new Path(pathStr), true, new NullProgressMonitor());
			} else {
				sourceFile.copy(new Path(pathStr), true, new NullProgressMonitor());
			}
		} catch (CoreException ce) {
			Activator.log(ce);
		}
	}

}
