#include "MAC57D54H.h"
#include "mc.h"

#define MC_ME_MODE_BASE     ((int)&MC_ME.RESET_MC)
#define MC_ME_PC_BASE       ((int)&MC_ME_RUN_PC0)
#define MC_ME_PCTL_BASE     ((int)&MC_ME_PCTL3)
#define MC_CGM_SYS_DIV_BASE ((int)&MC_CGM_SC_DC0)
#define MC_CGM_AUX_BASE     ((int)&MC_CGM_AC0_SC)
#define MC_CGM_AUX_OFFSET   0x0020

#define MC_ME_KEY 0x5AF0
#define MC_ME_INV_KEY 0xA50F

void mode_enable(enum mode mode) {
    MC_ME_ME |= (1 << (int)mode);
}

void mode_disable(enum mode mode) {
    MC_ME_ME &= ~(1 << (int)mode);
}

void mode_clock_enable(enum mode mode, enum clock clock) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << ((unsigned int)clock + 4));
    reg = (vuint32_t *)(MC_ME_MODE_BASE + (4 * (int)mode));

    *reg |= mask;
}

void mode_clock_disable(enum mode mode, enum clock clock) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << ((unsigned int)clock + 4));
    reg = (vuint32_t *)(MC_ME_MODE_BASE + (4 * (int)mode));

    *reg &= ~mask;
}
    
int mode_clock_is_enabled(enum mode mode, enum clock clock) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << ((unsigned int)clock + 4));
    reg = (vuint32_t *)(MC_ME_MODE_BASE + (4 * (int)mode));

    return (*reg & mask) != 0;
}

void mode_sysclk_src_set(enum mode mode, enum clock clock) {
    unsigned int mask;    
    vuint32_t * reg;

    mask = (unsigned int)clock;
    reg = (vuint32_t *)(MC_ME_MODE_BASE + (4 * (int)mode));

    mask |= *reg & 0xFFFFFFF0;

    *reg = mask;
}

enum clock mode_sysclk_src_get(enum mode mode) {
    vuint32_t * reg;

    reg = (vuint32_t *)(MC_ME_MODE_BASE + (4 * (int)mode));

    return (enum clock)(*reg & 0xF);
}

void enter_mode(enum mode mode) {
    /* mode & key */
    MC_ME_MCTL = ((unsigned int)mode << 28) | MC_ME_KEY;
    MC_ME_MCTL = ((unsigned int)mode << 28) | MC_ME_INV_KEY;
}

enum mode current_mode(void) {
    return (enum mode)(MC_ME_GS >> 28);
}

int mode_transition_ongoing(void) {
    return (MC_ME_GS & 0x08000000) != 0;
}

void run_percfg_mode_enable(int pc, enum mode mode) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << mode);
    reg = (vuint32_t *)((MC_ME_PC_BASE + (4 * (int)pc)));

    *reg |= mask;
}

void run_percfg_mode_disable(int pc, enum mode mode) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << mode);
    reg = (vuint32_t *)((MC_ME_PC_BASE + (4 * (int)pc)));

    *reg &= ~mask;
}

void lp_percfg_mode_enable(int pc, enum mode mode) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << mode);
    reg = (vuint32_t *)((MC_ME_PC_BASE + 0x20 + (4 * (int)pc)));

    *reg |= mask;
}

void lp_percfg_mode_disable(int pc, enum mode mode) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << mode);
    reg = (vuint32_t *)((MC_ME_PC_BASE + 0x20 + (4 * (int)pc)));

    *reg &= ~mask;
}

int run_percfg_mode_is_enabled(int pc, enum mode mode) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << mode);
    reg = (vuint32_t *)((MC_ME_PC_BASE + (4 * (int)pc)));

    return (*reg & mask) != 0;
}

int lp_percfg_mode_is_enabled(int pc, enum mode mode) {
    unsigned int mask;
    vuint32_t * reg;

    mask = (1 << mode);
    reg = (vuint32_t *)((MC_ME_PC_BASE + 0x20 + (4 * (int)pc)));

    return (*reg & mask) != 0;
}

void perctl_run_percfg_set(int perctl, int pc) {
    vuint8_t * reg;
    int word_offset = 4 * ((int)perctl / 4);
    int byte_offset = 3 - ((int)perctl % 4);

    reg = (vuint8_t *)(MC_ME_PCTL_BASE + word_offset + byte_offset);
    *reg = (*reg & 0x38) | (unsigned char)pc;
}

void perctl_lp_percfg_set(int perctl, int pc) {
    vuint8_t * reg;
    int word_offset = 4 * ((int)perctl / 4);
    int byte_offset = 3 - ((int)perctl % 4);

    reg = (vuint8_t *)(MC_ME_PCTL_BASE + word_offset + byte_offset);
    *reg = (*reg & 0x7) | ((unsigned char)pc << 3);
}

int perctl_run_percfg_get(int perctl) {
    vuint8_t * reg;
    int word_offset = 4 * ((int)perctl / 4);
    int byte_offset = 3 - ((int)perctl % 4);

    reg = (vuint8_t *)(MC_ME_PCTL_BASE + word_offset + byte_offset);
    return *reg & 0x7;
}

int perctl_lp_percfg_get(int perctl) {
    vuint8_t * reg;
    int word_offset = 4 * ((int)perctl / 4);
    int byte_offset = 3 - ((int)perctl % 4);

    reg = (vuint8_t *)(MC_ME_PCTL_BASE + word_offset + byte_offset);
    return (*reg & 0x38) >> 3;
}

int core_running(enum core core) {
    int running;

    switch (core) {
        case MC_CM4:
            running = MC_ME_CS & 0x1 ? 1 : 0;
            break;
        case MC_CA5:
            running = MC_ME_CS & 0x2 ? 1 : 0;
            break;
        case MC_CM0P:
            running = MC_ME_CS & 0x4 ? 1 : 0;
            break;
    }

    return running;
}

void core_mode_enable(enum core core, enum mode mode) {
    unsigned int mask;
    vuint16_t * reg;

    mask = (1 << (int)mode);

    switch (core){
        case MC_CM4:
            reg = (vuint16_t *)&MC_ME_CCTL0;
            break;
        case MC_CA5:
            reg = (vuint16_t *)&MC_ME_CCTL1;
            break;
        case MC_CM0P:
            reg = (vuint16_t *)&MC_ME_CCTL2;
            break;
    }

    *reg |= mask;
}

void core_mode_disable(enum core core, enum mode mode) {
    unsigned int mask;
    vuint16_t * reg;

    mask = (1 << (int)mode);

    switch (core){
        case MC_CM4:
            reg = (vuint16_t *)&MC_ME_CCTL0;
            break;
        case MC_CA5:
            reg = (vuint16_t *)&MC_ME_CCTL1;
            break;
        case MC_CM0P:
            reg = (vuint16_t *)&MC_ME_CCTL2;
            break;
    }

    *reg &= ~mask;
}

int core_mode_is_enabled(enum core core, enum mode mode) {
    unsigned int mask;
    vuint16_t * reg;

    mask = (1 << (int)mode);

    switch (core){
        case MC_CM4:
            reg = (vuint16_t *)&MC_ME_CCTL0;
            break;
        case MC_CA5:
            reg = (vuint16_t *)&MC_ME_CCTL1;
            break;
        case MC_CM0P:
            reg = (vuint16_t *)&MC_ME_CCTL2;
            break;
    }

    return (*reg & mask) != 0;
}

void core_boot_addr_set(enum core core, unsigned int addr) {
    int rmc = 0;
    
    switch (core) {
        case MC_CM4:
            rmc = MC_ME_CADDR0 & 0x1;
            MC_ME_CADDR0 = (addr & 0xFFFFFFFC);
            MC_ME_CADDR0 |= rmc;
            break;
        case MC_CA5:
            rmc = MC_ME_CADDR1 & 0x1;
            MC_ME_CADDR1 = (addr & 0xFFFFFFFC);
            MC_ME_CADDR1 |= rmc;
            break;
        case MC_CM0P:
            rmc = MC_ME_CADDR2 & 0x1;
            MC_ME_CADDR2 = (addr & 0xFFFFFFFC);
            MC_ME_CADDR2 |= rmc;
            break;
    }
}

unsigned int core_boot_addr_get(enum core core) {
    unsigned int addr;

    switch (core) {
        case MC_CM4:
            addr = MC_ME_CADDR0 & 0xFFFFFFFC;
            break;
        case MC_CA5:
            addr = MC_ME_CADDR1 & 0xFFFFFFFC;
            break;
        case MC_CM0P:
            addr = MC_ME_CADDR2 & 0xFFFFFFFC;
            break;
    }

    return addr;
}

void core_rmc_enable(enum core core) {
    vuint32_t * reg;

    switch (core) {
        case MC_CM4:
            reg = (vuint32_t *)&MC_ME_CADDR0;
            break;
        case MC_CA5:
            reg = (vuint32_t *)&MC_ME_CADDR1;
            break;
        case MC_CM0P:
            reg = (vuint32_t *)&MC_ME_CADDR2;
            break;
    }

    *reg |= 1;
}

void core_rmc_disable(enum core core) {
    vuint32_t * reg;

    switch (core) {
        case MC_CM4:
            reg = (vuint32_t *)&MC_ME_CADDR0;
            break;
        case MC_CA5:
            reg = (vuint32_t *)&MC_ME_CADDR1;
            break;
        case MC_CM0P:
            reg = (vuint32_t *)&MC_ME_CADDR2;
            break;
    }

    *reg &= ~1;
}

int core_rmc_is_enabled(enum core core) {
    vuint32_t * reg;

    switch (core) {
        case MC_CM4:
            reg = (vuint32_t *)&MC_ME_CADDR0;
            break;
        case MC_CA5:
            reg = (vuint32_t *)&MC_ME_CADDR1;
            break;
        case MC_CM0P:
            reg = (vuint32_t *)&MC_ME_CADDR2;
            break;
    }

    return *reg & 1;
}

void sysclk_div_set(int divider, int divn) {
    unsigned int mask;
    vuint32_t * reg;
    
    reg = (vuint32_t *)MC_CGM_SYS_DIV_BASE;
    reg += divider;

    mask = divn << 16;

    /* Keep the system clock enabled/disabled */
    if (sysclk_div_is_enabled(divider))
        mask |= 0x80000000;

    *reg = mask;
}

int sysclk_div_get(int divider) {
    vuint32_t * reg;
    
    reg = (vuint32_t *)MC_CGM_SYS_DIV_BASE;
    reg += divider;

    return (*reg >> 16) & 0xFF;
}

void sysclk_div_enable(int divider) {
    unsigned int mask;
    vuint32_t * reg;
    
    reg = (vuint32_t *)MC_CGM_SYS_DIV_BASE;
    reg += divider;

    mask = *reg;
    mask |= 0x80000000; /* OR in the DE bit */

    *reg = mask;
}

void sysclk_div_disable(int divider) {
    unsigned int mask;
    vuint32_t * reg;
    
    reg = (vuint32_t *)MC_CGM_SYS_DIV_BASE;
    reg += divider;

    mask = *reg;
    mask &= ~(0x80000000); /* AND out the DE bit */

    *reg = mask;
}

int sysclk_div_is_enabled(int divider) {
    vuint32_t * reg;
    
    reg = (vuint32_t *)MC_CGM_SYS_DIV_BASE;
    reg += divider;

    return (*reg & 0x80000000) != 0;
}

enum clock current_sysclk(void) {
    return (enum clock)(MC_CGM_SC_SS >> 24);
}

void auxclk_src_set(int auxclk, enum clock src) {
    unsigned int mask;
    vuint32_t * reg;
    
    mask = (unsigned int)src << 24;
    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    
    *reg = mask;
}

enum clock auxclk_src_get(int auxclk) {
    vuint32_t * reg;
    
    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    
    return (enum clock)(*reg >> 24);
}

void auxclk_div_set(int auxclk, int divider, int divn) {
    unsigned int mask;
    vuint32_t * reg;

    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    if (divider == 0) reg += 2;
    else reg += 3;

    mask = divn << 16;
    
    /* Keep the system clock enabled/disabled */
    if (auxclk_div_is_enabled(auxclk, divider))
        mask |= 0x80000000;

    *reg = mask;
}

int auxclk_div_get(int auxclk, int divider) {
    vuint32_t * reg;

    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    if (divider == 0) reg += 2;
    else reg += 3;

    return (*reg >> 16) & 0xFF;
}

void auxclk_div_enable(int auxclk, int divider) {
    unsigned int mask;
    vuint32_t * reg;

    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    if (divider == 0) reg += 2;
    else reg += 3;

    mask = *reg;
    mask |= 0x80000000; /* OR in the DE bit */

    *reg = mask;
}

void auxclk_div_disable(int auxclk, int divider) {
    unsigned int mask;
    vuint32_t * reg;

    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    if (divider == 0) reg += 2;
    else reg += 3;

    mask = *reg;
    mask &= ~(0x80000000); /* AND out the DE bit */

    *reg = mask;
}

int auxclk_div_is_enabled(int auxclk, int divider) {
    vuint32_t * reg;

    reg = (vuint32_t *)(MC_CGM_AUX_BASE + (MC_CGM_AUX_OFFSET * auxclk));
    if (divider == 0) reg += 2;
    else reg += 3;

    return (*reg & 0x80000000) != 0;
}

void update_clock_div(void) {
    MC_CGM_DIV_UPD_TRIG = 1;

    while (clock_div_updating()) {}
}

int clock_div_updating(void) {
    return MC_CGM_DIV_UPD_STAT != 0;
}

void pll_cfg(enum clock pll, int prediv, int mfd, int rfdphi) {
    volatile struct PLLDIG_tag * pllreg;

    switch (pll) {
        case MC_PLL0:
            pllreg = &PLLDIG_0;
            break;
        case MC_PLL1:
            pllreg = &PLLDIG_1;
            break;
        case MC_PLL2:
            pllreg = &PLLDIG_2;
            break;
        case MC_PLL3:
            pllreg = &PLLDIG_3;
            break;
        default:
            return;
    }

    pllreg->PLLDV.R = (prediv << 12) | (mfd << 0) | (rfdphi << 16);

    /* Need to set PLLCAL3 even if not using FM */
    pllreg->PLLCAL3.R = 0x09C3C000;
}
