#!/usr/bin/perl
use English;
use Getopt::Long;
use Cwd;
use strict;


# 
# Revision History:
# 1.0 - Initial release John Russo
# 1.1 - Added automark, netmark, etc. Report sorts the benchmark class.
# 1.2 - Add -twikifile <file> feature, ordering array for print out
# 1.3 - Warn if test fail
# 1.4 - Add spec2k6 support, change handling of missing data
# 1.5 - Not printing telecom-1 report fixed, compiler name no longer requires '-'
# 1.6 - Match compiler names exactly using grep.
# 1.7 - Match baseline compiler name exactly
# 1.8 - Generate "best of" summary information, normalized charts (now the default).
# 1.9 - Print output based on ValidDataTemplate for cases where some data is missing
#       Made baselining and comparisons work for bestof mode.
# 1.10- Added -writebestof option, to write the bestof results data to a file. This
#       is useful when you want to compare 2 or bestof runs (create 2 or more data files
#       from the bestof runs and then re-run bmksum.pl on the new data files.
# bmksum.pl - A summary report generator for benchmarks 
# Usage: perl bmksum.pl [ options ] <sumfile1> ... <sumfileN> 
#
#

# Predeclare globals
#
use vars '$Debug','$Version';
use vars '$CmdHelp', '$CmdWriteBestof', '$CmdLineWidth', '$CmdLeadColWidth', '$CmdColWidth', '$CmdCompare';
use vars '$CmdOldChart','@CmdBestofLists','%CmdOpts', '$CmdTwikiFile';
use vars '$Separator','$LineWidth','$LeadColWidth', '$ColWidth', '$Cumulative_padding';
use vars '%MarkNames', '%BmkWNames', '%Units';
use vars '@InputFileNames', '@Cores', '@Compilers', '@Flags', '@Dates', '@Classes', '@CompilerValues';
use vars '%Data', '%ValidDataTemplate';
use vars '%BmkClassDesc','@BmkClassDescOrder';

# Version
#
$Version = "1.10";

# Debug settings
#
$Debug = 0;

# Class name to mark name mapping
%MarkNames =  ( 'coremark' => 'coremark',
                'dhrystone' => 'dhrystone',
                'auto-1' => 'automark',
                'consumer-1' => 'conmark',
                'net-1' => 'netmark',
                'telecom-1' => 'telemark',
                'office-1' => 'officemark',
                'auto-1_lite' => 'automark',
                'consumer-1_lite' => 'conmark',
                'net-1_lite' => 'netmark',
                'telecom-1_lite' => 'telemark',
                'office-1_lite' => 'officemark',
                'spec2k' => 'spec2k total time',
                'consumer-2' => 'conmark2',
                'net-2' => 'netmark2',
                'office-2' => 'officemark2',
                'spec2k6' => 'spec2k6 total time'
    );

# Bmk class descriptors
# For printing in this order
@BmkClassDescOrder = ( 'coremark',
                       'dhrystone',
                       'auto-1',
                       'consumer-1',
                       'net-1',
                       'telecom-1',
                       'office-1',
                       'auto-1_lite',
                       'consumer-1_lite',
                       'net-1_lite',
                       'telecom-1_lite',
                       'office-1_lite',
                       'consumer-2',
                       'net-2',
                       'office-2',
                       'spec2k',
                       'spec2k6' );

# WikiWords used for anchoring              
%BmkWNames = ( 'coremark' => 'CoreMark',	   
               'dhrystone' => 'DhryStone',	   
               'auto-1' => 'Auto1N',	   
               'consumer-1' => 'Consumer1N',	   
               'net-1' => 'Net1N',		   
               'telecom-1' =>'Telecom1N',	   
               'office-1' => 'Office1N',	   
               'auto-1_lite' => 'Auto1Lite',	   
               'consumer-1_lite' => 'Consumer1Lite',  
               'net-1_lite' => 'Net1Lite',	   
               'telecom-1_lite' => 'Telecom1Lite',   
               'office-1_lite' => 'Office1Lite',	   
               'consumer-2' => 'Consumer2N',	   
               'net-2' => 'Net2N',		   
               'office-2' => 'Office2N',	   
               'spec2k' => 'Spec2K',	   
               'spec2k6' => 'Spec2K6' );        
                       
# For mapping individual tests to bmk class names
%BmkClassDesc = ( 
    'coremark' => ["coremark"],
    'dhrystone' => ["dhrystone"],
    'auto-1' => ["a2time01","aifftr01","aifirf01","aiifft01","basefp01","bitmnp01","cacheb01",
                 "canrdr01","idctrn01","iirflt01","matrix01","pntrch01","puwmod01","rspeed01",
                 "tblook01","ttsprk01"],
    'consumer-1' => ["cjpeg","djpeg","rgbcmy01","rgbhpg01","rgbyiq01"],
    'net-1' => ["ospf","pktflowb1m","pktflowb2m","pktflowb4m","pktflowb512k","routelookup"],
    'telecom-1' => ["autcor00data_1","conven00data_2","fbital00data_2","fft00data_2","viterb00data_3",
                    "autcor00data_2","conven00data_3","fbital00data_3","fft00data_3","viterb00data_4",
                    "autcor00data_3","fbital00data_6","viterb00data_1","conven00data_1","fft00data_1",
                    "viterb00data_2"],
    'office-1' => ["dither01","rotate01","text01"],
    'auto-1_lite' => ["a2time01_lite","aifftr01_lite","aifirf01_lite","aiifft01_lite","basefp01_lite","bitmnp01_lite",
                      "cacheb01_lite", "canrdr01_lite","idctrn01_lite","iirflt01_lite","matrix01_lite","pntrch01_lite",
                      "puwmod01_lite","rspeed01_lite", "tblook01_lite","ttsprk01_lite"],
    'consumer-1_lite' => ["cjpeg_lite","djpeg_lite","rgbcmy01_lite","rgbhpg01_lite","rgbyiq01_lite"],
    'net-1_lite' => ["ospf_lite","pktflowb1m_lite","pktflowb2m_lite","pktflowb4m_lite","pktflowb512k_lite","routelookup_lite"],
    'telecom-1_lite' => ["autcor00data_1_lite","conven00data_2_lite","fbital00data_2_lite","fft00data_2_lite",
                         "viterb00data_3_lite", "autcor00data_2_lite","conven00data_3_lite","fbital00data_3_lite",
                         "fft00data_3_lite","viterb00data_4_lite", "autcor00data_3_lite","fbital00data_6_lite",
                         "viterb00data_1_lite","conven00data_1_lite","fft00data_1_lite", "viterb00data_2_lite"],
    'office-1_lite' => ["dither01_lite","rotate01_lite","text01_lite"],

    'consumer-2' => ["mp2decodf32psnr1","mp2decodf32psnr2","mp2decodf32psnr3","mp2decodf32psnr4","mp2decodf32psnr5",
		     "mp2decodfixpsnr1","mp2decodfixpsnr2","mp2decodfixpsnr3","mp2decodfixpsnr4","mp2decodfixpsnr5",
		     "mp2decoddata1","mp2decoddata2","mp2decoddata3","mp2decoddata4","mp2decoddata5",
		     "mp2enfixdata1","mp2enfixdata2","mp2enfixdata3","mp2enfixdata4","mp2enfixdata5",
		     "mp4decodedata1","mp4decodedata2","mp4decodedata3","mp4decodedata4","mp4decodedata5",
		     "mp4decodepsnr1","mp4decodepsnr2","mp4decodepsnr3","mp4decodepsnr4","mp4decodepsnr5",
		     "mp4encodedata1","mp4encodedata2","mp4encodedata3","mp4encodedata4","mp4encodedata5",
		     "mp3playerfixeddata1","mp3playerfixeddata2","mp3playerfixeddata3","mp3playerfixeddata4","mp3playerfixeddata5",
		     "cjpegv2data1","cjpegv2data2","cjpegv2data3","cjpegv2data4","cjpegv2data5","cjpegv2data6","cjpegv2data7",
		     "djpegv2data1","djpegv2data2","djpegv2data3","djpegv2data4","djpegv2data5","djpegv2data6","djpegv2data7",
		     "rgbcmykv2data1","rgbcmykv2data2","rgbcmykv2data3","rgbcmykv2data4","rgbcmykv2data5","rgbcmykv2data6","rgbcmykv2data7",
		     "rgbhpgv2data1","rgbhpgv2data2","rgbhpgv2data3","rgbhpgv2data4","rgbhpgv2data5","rgbhpgv2data6","rgbhpgv2data7",
		     "rgbyiqv2data1","rgbyiqv2data2","rgbyiqv2data3","rgbyiqv2data4","rgbyiqv2data5","rgbyiqv2data6","rgbyiqv2data7",
		     "aes","huffde"],
    'net-2' => ["ip_pktcheckb4m","nat","routelookup","tcpmixed","ip_pktcheckb1m","ip_pktcheckb512k",
		"ospfv2","tcpbulk","ip_pktcheckb2m","ip_reassembly","qos","tcpjumbo"],
    'office-2' => ["bezierv2fixed_d2","bezierv2fixed_d3","bezierv2fixed_d4",
		   "bezierv2float_d2","bezierv2float_d3","bezierv2float_d4",
		   "textv2_d2","textv2_d3","textv2_d4",
		   "gs8_d2","gs8_d3","gs8_d4",
		   "ditherv2_d2","ditherv2_d3","ditherv2_d4","ditherv2_d5","ditherv2_d6","ditherv2_d7","ditherv2_d8","ditherv2_d9","ditherv2_d10","ditherv2_d11",
		   "rotatev2_d2","rotatev2_d3","rotatev2_d4","rotatev2_d5","rotatev2_d6","rotatev2_d7","rotatev2_d8","rotatev2_d9","rotatev2_d10","rotatev2_d11"],

    'spec2k' => ["gzip","vpr","crafty","parser","eon","gap","vortex","twolf","perlbmk","gcc",
		 "mcf","bzip2","wupwise","swim","mgrid","applu","mesa","galgel","art","equake",
		 "facerec","ammp","lucas","fma3d","sixtrack","apsi"],
    'spec2k6' => ["perlbench","bzip2","gcc","mcf","gobmk","hmmer","sjeng","libquantum","h264ref",
		  "omnetpp","astar","xalancbmk","bwaves","gamess","milc","zeusmp","gromacs","cactusADM",
		  "leslie3d","namd","dealII", "soplex", "povray","calculix","GemsFDTD",
		  "tonto","lbm","wrf","sphinx3"]
    );

# Options
#
$CmdHelp=0;
$CmdWriteBestof=0;
$CmdCompare="";
$CmdTwikiFile="";
$CmdOldChart=0;
$CmdLineWidth=100;
$CmdColWidth=22;
$CmdLeadColWidth=20;
@CmdBestofLists=();

%CmdOpts = (
	    'help'  => \$CmdHelp,
            'twikifile' => \$CmdTwikiFile, 
	    'compare' => \$CmdCompare,
            'bestof'  => \@CmdBestofLists,
            'writebestof' => \$CmdWriteBestof,
            'oldchart' => \$CmdOldChart,
	    'linewidth' => \$CmdLineWidth,
            'colwidth' => \$CmdColWidth,
            'leadcolwidth' => \$CmdLeadColWidth
	    );

# Miscellaneous globals
#
$Separator = "/";
$OSNAME =~ /MSWin32/ && do {
  $Separator = "\\";
};
		 
# Handle command line options
#
GetOptions( \%CmdOpts,
	    "help",
            "twikifile=s",
	    "compare=s",
            "bestof=s@",
            "writebestof",
            "oldchart",
	    "linewidth=i",
            "colwidth=i",
            "leadcolwidth=i"
	    );

$LineWidth = $CmdLineWidth;
$ColWidth = $CmdColWidth;
$LeadColWidth = $CmdLeadColWidth;

# Get command line arguments, i.e. the targets for which we will generate
# reports
#
@InputFileNames = @ARGV;


$CmdHelp && do {
  print "bmksum.pl: ",$Version,"\n";
  print<<'EOM';
bmksum.pl options filename filename .. filenameN
-help: print this help
-twikifile <filename> : create a twiki file report
-compare <compiler_name abbrev>: set baseline compiler, may be the minimum number 
  of characters to uniquely identify the compiler.
-bestof  <compid>,<compid2>,... : pick the best of the 
  results of the compilers indicated by the list of compids (compiler names).
  May be specified 0,1 or more times. The best data is given a 
  new compiler id (compiler name) taken from the compiler id name in the list. 
-writebestof : writes the bestof result to a data file 
-oldchart: use old chart format with absolute values instead of
  normalized chart.
-linewidth <value>: set line width of report 
-colwidth <value>: set data column width of report
-leadcolwidth <width>: set leading column width of report
EOM
  exit 0;
};


# Main entry point
#
run_summary();


##################
## Subroutines
##################

##
## run_summary - make a summary report
##
sub run_summary
{
  my $curdir = cwd();
  my $inputfile;
  my $cmp1;
  my $cmp2;

  $#InputFileNames == -1 &&  do {
    die "Expecting 1 or more file name arguments\n";
  };

  $CmdCompare ne ""  && do {
          print "Baseline is $CmdCompare (*)\n";
  };

  # Set top level directory for each target,platform
  #
  foreach $inputfile ( @InputFileNames ) {
      readdata ($inputfile)
  } 

  # Data for individual runs must have a <compiler_name>.target heading to identify data
  # for this compiler. Often different runs have data sets that don't overlap. For example,
  # one run may have data for Coremark and the other run does not. Here we create a list of
  # test that are common to all runs and only report data that is common to all compilers.
  create_validtestlist();


  # The -bestof option says, essentially, create a new compiler name
  # with the best of data populated from the compilers defined by the -bestof parameter.
  #
  create_bestofdata();

  # At this point we have all the data we need for each target,platform combination
  # saved. Time to print the report.
  #
  printreport();

  $CmdTwikiFile ne "" && do {
      create_twiki ();
  };

}

##
## readdata - read data from an input file
## Data consists of a stream of lines containing the following:
##
## "<compiler_name>.target_log" - indicates the following information
## is for <compiler_name>. This marker should only appear once in each 
## file read in. All files using the same compiler should have the
## same "<compiler_name>.target_log" heading, though different files
## may have data for different cores. A file can have multiple,
## distinct "<compiler_name>.target_log" headings, indicating
## data for different compilers. 
##
## "CORE=<core_name>" - May be seen more than once and indicates the
## following data is for <core_name>.
##
## "BMK=<bmark_name> VAL=<y/n>|EXEC=<y/n> ITER=<val>" - Will be seen 
## more than once and indicates the benchmark name and result. 
##
## "FLAGS=<flags>" - May appear once per compiler instance
##
## "DATE=yyyymmdd" - May appear once per compiler instance
##
sub readdata {
    my $file = shift @_;
    my $compiler_name="";
    my $core_name="";
    my $flags="";
    my $dates="";
    my $bmkstr, my $execstr, my $iterstr;
    my $bmk, my $bmkclass, my $exec, my $iter; 
    my $tmpstr;
    
    open (FILE, $file) || die "Cannot open $file\n";

    while (<FILE>) {
        # Ignore comments
        /^(\s*)\#/ && do {
            next;
        };

        # Note: exclude trailing core name in xxx.target_log string
        /^(.+)\.target_log$/ && do {
            $compiler_name = $1;
            if (!grep (/$compiler_name$/,@Compilers)) {
                push (@Compilers,$compiler_name);
            }

            # set default baseline
            if ($CmdCompare eq "") {
                $CmdCompare = $compiler_name;
            }
            next;
        };
        /^DATE/ && do {
            $compiler_name ne "" || die "There is no current compiler name\n";
            ($tmpstr,$dates) = split (/=/, $_);
            chomp($dates);
            push (@Dates,$dates);
            next;
        };
        /^FLAGS/ && do {
            $compiler_name ne "" || die "There is no current compiler name\n";
            ($tmpstr,$flags) = split (/=/, $_);
            chomp($flags);
            push (@Flags,$flags);
            next;
        };
        /^CORE/ && do {
            $compiler_name ne "" || die "There is no current compiler name\n";
            ($tmpstr,$core_name) = split (/=/, $_);
            chomp($core_name);
            # Save core_name if not already saved.
            if (!grep (/$core_name/, @Cores)) {
                push (@Cores,$core_name);
            }
            next;
        };
        /^BMK/ && do {
            $compiler_name ne "" || die "There is no current compiler name\n";
            $core_name ne "" || die "There is no current core name\n";
            ($bmkstr,$execstr,$iterstr) = split (/\s+/,$_);
            ($tmpstr,$bmk) = split (/=/, $bmkstr);

            # Some bmks don't have a class (e.g. coremark), so their class is the bmk name 
            ($bmkclass,$tmpstr) = split (/\./, $bmk);
            if ($bmkclass eq "") {
                $bmkclass = $bmk;
            }
            # Get the bmk name if there was a class.bmk identifier
            if ($tmpstr ne "") {
                $bmk  = $tmpstr;
            }
            
            # For eembc "_lite" bmks, we actually create make up a class for this
            if ($tmpstr =~ /_lite$/) {
                $bmkclass .= "_lite";
            }

            # Save unique benchmark classnames 
            if (!grep (/$bmkclass/,@Classes)) {
                push (@Classes,$bmkclass);
            }

            ($tmpstr,$exec) = split (/=/, $execstr);
            ($tmpstr,$iter) = split (/=/, $iterstr);


            # Save the units (Iterations or Seconds)
            if ($tmpstr =~ /TIME/) {
                $Units{$bmkclass}="Execution time in seconds (lower is better)";
            }
            elsif ($tmpstr =~ /ITER/) {
                $Units{$bmkclass}="Iterations per second (higher is better)";
            }
            else {
	        # Skip the line, no data "ITER" or "TIME" not found
		next;
            }

            if ( $exec eq "Y" ) {
                # Save the data in hash keyed by the "core_name:compiler_name:benchmark_class:benchmark_name"
                # The data is a 2 element array, exec ('Y'|'N') and the iteration value (float) or time value (float)
                $Data{"$core_name:$compiler_name:$bmkclass:$bmk"} = [$exec, $iter]; 
            }
            else {
                #$Data{"$core_name:$compiler_name:$bmkclass:$bmk"} = [$exec, 1.0]; 
                warn ( "Test failed for $core_name $compiler_name $bmkclass $bmk - dummy data substituted.\n");
            }
        };
    }
    close FILE;
}

## create_validtestlist
## Data for individual runs must have a <compiler_name>.target heading to identify data
## for this compiler. Often different runs have data sets that don't overlap. For example,
## one run may have data for Coremark and the other run does not. Here we create a list of
## test that are common to all runs and only report data that is common to all compilers.
sub create_validtestlist {
    my $key;
    my $key2;
    my $valid;

    foreach my $class_name ( @Classes ) {
        foreach my $core_name (@Cores) {
            foreach my $bmk_name (@{$BmkClassDesc{$class_name}}) {
                # Now for each compiler, if we have data for the test, it becomes
                # part of the valid list
                $valid=1;
                foreach my $compiler_name (@Compilers) {
                    $key = $core_name . ":" . $compiler_name . ":" . $class_name . ":" . $bmk_name;
                    if (!defined ($Data{$key})) {
                        $valid=0;
                    }
                }
                if ($valid) {
                    $key2 =$core_name . ":" .  $class_name . ":" . $bmk_name;
                    $ValidDataTemplate{$key2}=1;
                }
            }
        }
    }
}
    

##
## create_bestofdata - generate new "compilers" containing the best
## data from a group of compiler names.
##
sub create_bestofdata {
    my $bestspec;
    my @namelist;
    my $cnt=0;
    my $cnt2=0;
    my $datanamebase = "";
    my $first_bestof_compiler = $#Compilers + 1;

    # No -bestof specified 
    $#CmdBestofLists == -1 && do { 
        return;
    };


    for $bestspec ( @CmdBestofLists ) {
        @namelist = split(/,/, $bestspec);
        # Check the validity of each name specified by -bestof parameter
        for my $name (@namelist) {
            $datanamebase eq "" && do {
                $datanamebase = $name . "_";
            };
            if (!grep (/$name/,@Compilers)) {
                die ("Bad -bestof parameter, no such compiler name, \"$name\" found in data file\n");
            }
        }
        generate_bestofdata(\@namelist, $datanamebase . "best_$cnt" );
        # Add new name to Compilers list
        push(@Compilers,$datanamebase . "best_$cnt");
        $datanamebase = "";
        $cnt++;
    }

    # Tell the user what the bestof columns (new compilers) are composed of.
    for ($cnt=$first_bestof_compiler; $cnt <= $#Compilers; $cnt++) {
        print "$Compilers[$cnt] :";
        $bestspec = $CmdBestofLists[$cnt2++];
        @namelist = split(/,/, $bestspec);
        for my $name (@namelist) {
            print " $name"; 
        }
        print "\n";
    }

    # We now remove the individual compiler names specified by -bestof 
    # from the database so that we only report the bestof column.
    for $bestspec ( @CmdBestofLists ) {
        @namelist = split(/,/, $bestspec);
        for my $name (@namelist) {
            for (my $i=0; $i<=$#Compilers; $i++) {
                if ($name eq $Compilers[$i]) {
                    splice(@Compilers,$i,1);
                    last;
                }
            }
        }
    }
    # And we reset the CmdCompare option to the new first (bestof) of @Compilers
    # if the previous CmdCompare doesn't exist anymore
    if (!grep(/^$CmdCompare$/, @Compilers)) {
        $CmdCompare = $Compilers[0];
    }

    # If requested, write out the best of data to a file
    $CmdWriteBestof && do {
        write_all_bestof_data();
    };
}

##
## write_all_bestof_data
## Write the bestof data to a data file. The filename is the
## <compiler_name>.target_log
##
sub write_all_bestof_data {
    foreach my $compiler ( @Compilers ) {
        write_bestof_data ($compiler);
    }
}

##
## write_bestof_data - write the bestof data for a compiler
##
##
sub write_bestof_data {
    my $compiler = shift @_;
    my $fn = $compiler . ".target_log";
    my $units;
    my $exec_or_val;
    my $thevalue, my $data, my $key;
    my $core_name, my $bmkclass, my $bmkname;

    open (FILE, ">$fn") || die "in write_bestof_data: Cannot open $fn for writing\n";

    print FILE "$fn\n";
    foreach  $core_name (@Cores) {
        print FILE "CORE=$core_name\n";
        foreach  $bmkclass ( @Classes ) {
            # ITER value or TIME value
            $units = "TIME";
            $exec_or_val = "VAL";
            $Units{$bmkclass} =~ /^Iterations/ && do {
                $units = "ITER";
                $exec_or_val = "EXEC";
            };
            foreach  $bmkname (@{$BmkClassDesc{$bmkclass}}) {
                $key = $core_name . ":" . $compiler . ":" . $bmkclass . ":" . $bmkname;
                $data = $Data{$key};
                $thevalue = ${$data}[1];
		my $t_bmkclass = $bmkclass;
		$t_bmkclass =~ s/_lite//;
                printf FILE "BMK=%s.%s %s=Y %s=%-10.4f\n",$t_bmkclass,$bmkname,$exec_or_val,$units,$thevalue;
            }
        }
    }

    close FILE;
}

##
## generate_best_data - get the best data given a list of compiler names
##
##
sub generate_bestofdata {
    my $compiler_names_ref = shift @_;
    my $dataname = shift @_;

    foreach my $bmkclass ( @Classes ) {
        foreach my $core_name (@Cores) {
            foreach my $bmkname (@{$BmkClassDesc{$bmkclass}}) {
                get_bestofdata ($dataname, $core_name, $bmkclass, $bmkname, $compiler_names_ref);
            }
        }
    }
}

##
## get_bestofdata - calculate the best data from a group of compilers
##
##
sub get_bestofdata {
    my $dataname = shift @_;
    my $core = shift @_;
    my $class = shift @_;
    my $bmkname = shift @_;
    my $compiler_names_ref = shift @_;
    my $data;

    undef @CompilerValues;

    get_values_by_compiler ($core, $class, $bmkname, $compiler_names_ref, \@CompilerValues); 
    $data = calc_best_data(\@CompilerValues, $class);
    put_value_by_compiler ($dataname, $core, $class, $bmkname, $data);
}

##
## put_value_by_compiler - insert entry into Data hash
##
##
sub put_value_by_compiler {
    my $compiler_name = shift @_;
    my $core = shift @_;
    my $class = shift @_;
    my $bmkname = shift @_;
    my $data = shift @_;
    my $key;

    $key = $core . ":" . $compiler_name . ":" . $class . ":" . $bmkname;
    $Data{$key} = ['Y', $data];

} 

##
## calc_best_data - Return best data from a list
## average the two highest or lowest values in array
##
##
sub calc_best_data {
    my $array_ref = shift @_;
    my $class = shift @_;
    my $besttype="higher";
    my $minmax;
    my $idx=0;
    my $lastidx=0;
    my @vals; 
    my $rv;

    if ($Units{$class} =~ /lower is better/) {
        $besttype="lower";
    }

    $minmax=${$array_ref}[0];
    # Find min or max in array
    foreach my $value (@{$array_ref}) {
        if ($besttype eq "higher") {
            if ($value >= $minmax) {
                $lastidx = $idx;
                $minmax = $value;
            }
        }
        else {
            if ($value <= $minmax) {
                $lastidx = $idx;
                $minmax = $value;
            }
        }
        $idx++;
    }

    # Save the val
    push(@vals,$minmax);

    # Find next minmax (exclude the one already found)
    $idx=0;
    # Remove the previously found array element, so we find the next minmax
    splice(@{$array_ref},$lastidx,1);

    # if any values left in array
    $#{$array_ref} == -1 || do {
        $minmax=${$array_ref}[0];
        foreach my $value (@{$array_ref}) {
            if ($besttype eq "higher") {
                if ($value >= $minmax) {
                    $minmax = $value;
                }
            }
            else {
                if ($value <= $minmax) {
                    $minmax = $value;
                }
            }
            $idx++;
        }

        # Save the val
        push(@vals,$minmax);
    };

    $rv = compute_total (\@vals) / ($#vals + 1); 

    return $rv;
        
}

    

## 
## printreport - print nicely formatted report 
##
##
sub printreport {
    my $ordered_key, my $tmp;

    # Print the bmk data if it appeared in the file --
    # it's class name will be in the @Classes array
    foreach $ordered_key ( @BmkClassDescOrder ) {
        if (grep (/$ordered_key/,@Classes)) {
            printbmk($ordered_key);
            printseparator();
        }
    }
}

##
## printbmk - print benchmark data 
##
##
sub printbmk {
    my $classname = shift @_;
    my $compiler_name;
    my $core_name;
    my $key = ""; 
    my $key2;
    my $data;
    my $baseline;
    my $note=" ";
    my $date_name;
    my $flag_name;

    print "=== $classname === ($Units{$classname})\n";
    foreach $core_name (@Cores) {
        print "=== $core_name ===\n";
        # Format of reported line is:
        # <testname>     comp1 val (diff)  comp2 val (diff) ...

        # Leading column for test name
        printf ("%${LeadColWidth}s"," ");
        # Print baseline annotation
        foreach $compiler_name (@Compilers) {
            # Annotate the baseline compiler
            if ( $compiler_name =~ /^$CmdCompare$/ ) {
                $note = "baseline";
            }
            printf ("%-${ColWidth}s",$note);
            $note=" ";
        }
        printf ("\n");

        # Leading column 
        printf ("%${LeadColWidth}s"," ");
        # Print header of dates 
        foreach $date_name (@Dates) {
            printf ("%-${ColWidth}s",$date_name);
        }
        print "\n";

        # Leading column 
        printf ("%${LeadColWidth}s"," ");
        # Print header of flags 
        foreach $flag_name (@Flags) {
            printf ("%-${ColWidth}s",$flag_name);
        }
        print "\n";

        # Leading column
        printf ("%${LeadColWidth}s"," ");
        # Print header of compiler names
        foreach $compiler_name (@Compilers) {
            printf ("%-${ColWidth}s",$compiler_name);
        }
        print "\n";

        foreach my $bmk_name (@{$BmkClassDesc{$classname}}) {
            printf ("%-${LeadColWidth}s",$bmk_name);
            # Get the baseline value
            foreach $compiler_name (@Compilers) {
                if ( $compiler_name =~ /^$CmdCompare$/ ) {
                    $key = $core_name . ":" . $compiler_name . ":" . $classname . ":" . $bmk_name;
                    $key2 = $core_name . ":" . $classname . ":" . $bmk_name;
                    if (defined($ValidDataTemplate{$key2})) {
                        $data = $Data{$key};
                        $baseline = ${$data}[1];
                    }
                }
            }

            foreach $compiler_name (@Compilers) {
                $key = $core_name . ":" . $compiler_name . ":" . $classname . ":" . $bmk_name;
                $key2 = $core_name . ":" . $classname . ":" . $bmk_name;
                if (defined($ValidDataTemplate{$key2})) {
                    $data = $Data{$key};
                    printf ( "%-12.2f",${$data}[1]);
                    printf ( "(%-6.2f)", calcdiff (${$data}[1],$baseline));
                    printf ( " " x (${ColWidth}-20) );
                }
            }
            print "\n";
        }
        # Print marks or totals
        print "\n";
        printsum ($core_name, $classname);
        print "\n";
    }
}

## 
## calcdiff - calculate the difference between this value and the baseline
##
sub calcdiff {
    my $dataitem = shift @_;
    my $baseline = shift @_;
    my $val;

    if ( $baseline == 0 ) {
	$val = 0.0;
    } else {
	$val = (($dataitem/$baseline) - 1) * 100.0;
    }

    return $val;
}

##
## printsum - calculate a summary figure of merit for this benchmark class
## For eembc, this is the "mark" value, for others it is simply a total
## 
sub printsum {
    my $core_name = shift @_;
    my $bmkclass = shift @_;
    my $compiler_name;
    my $isbaseline=0;
    my $baseline;
    my $mark;
    my $total;
    my @valuearray;
    my $i=0;
    my $printmark=0;
    my $key, my $key2;
    my $data;
    my $mark;
    my $total;

    undef @CompilerValues;

    foreach $compiler_name (@Compilers) {
        $isbaseline=0;
        if ( $compiler_name =~ /^$CmdCompare$/ ) {
            $isbaseline=1;
        }

        get_values_by_bmkname ($core_name, $compiler_name, $bmkclass, \@CompilerValues); 

        if ($#CompilerValues != -1) {
            
            # Add the mark data (automark, etc.) to the data array here. The
            # last element in the key is the mark name, e.g. "automark", "coremark", etc. 
            $key = $core_name . ":" . $compiler_name . ":" . $bmkclass . ":" . $MarkNames{$bmkclass};
            $key2 = $core_name . ":" . $bmkclass . ":" . $MarkNames{$bmkclass};
            if ( is_eembc ($bmkclass) ) {
                $mark = compute_mark ($bmkclass, \@CompilerValues);
                $Data{$key} = ['Y', $mark]; 
                $ValidDataTemplate{$key2} = 1;
                push (@valuearray, $mark);
                if ($isbaseline) {
                    $baseline = $mark; 
                }
                $printmark=1;
            }
            elsif ( $#CompilerValues > 0 ){
                $total = compute_total (\@CompilerValues);
                $Data{$key} = ['Y', $total]; 
                $ValidDataTemplate{$key2} = 1;
                push (@valuearray, $total); 
                if ($isbaseline) {
                    $baseline = $total; 
                }
                $printmark=1;
            }
        }
    }

    if ($printmark) {
        printf ("%-${LeadColWidth}s",$MarkNames{$bmkclass});
        foreach $compiler_name (@Compilers) {
            printf ("%-12.2f",$valuearray[$i]);
            printf ( "(%-6.2f)", calcdiff ($valuearray[$i],$baseline));
            printf ( " " x (${ColWidth}-20) );
            $i++;
        }
        print "\n";
    }
}

##
## create_twiki - create a twiki file report
##
## Create a table of the form:
##    |target|targetname1|targetname2|...
##    |comp1 |markdata   |markdata   |...
##    |comp2 |markdata   |markdata   |...
##    ...
##
## Then generate a graph from the table
##
sub create_twiki {
    my $ordered_key, my $tmp;
    my $min=0, my $max=0;
    my $colcnt=0, my $rowcnt=0;
    my $normalized=1;

    $CmdOldChart && do { $normalized=0; }; 

    open (TWIKIFILE, "+>", $CmdTwikiFile) || warn "Cannot open $CmdTwikiFile for writing\n";

    # For each benchmark class in order
    foreach $ordered_key ( @BmkClassDescOrder ) {
        next if !has_mark_data($ordered_key);
	print TWIKIFILE "#$BmkWNames{$ordered_key}\n";

        ($colcnt, $rowcnt, $min, $max) = create_table ($ordered_key);
        print TWIKIFILE "\n";

        $normalized && do {
            ($colcnt, $rowcnt, $min, $max) = create_table_normalized ($ordered_key);
            print TWIKIFILE "\n";
        };

        create_chart ($normalized,$ordered_key, $colcnt, $rowcnt, $min, $max);
        print TWIKIFILE "\n---\n";

    }

    close TWIKIFILE;
}

##
## create_table - create a table for this benchmark class
##
##
sub create_table {
    my $bmk_class = shift @_;
    my $ccnt_ref = shift @_;
    my $core_name, my $compiler_name;
    my $val, my $colcnt=0, my $rowcnt=0, my $min, my $max;


    print TWIKIFILE "%TABLE{ name=\"$bmk_class\" }%\n";

    printf TWIKIFILE "|%-30s|","*$bmk_class*";

    foreach $core_name (@Cores) {
        $colcnt++;
        printf TWIKIFILE "%-12s|",$core_name;
    }
    print TWIKIFILE "\n";

    undef($max);
    undef($min);
    foreach $compiler_name (@Compilers) {
        $rowcnt++;
        printf TWIKIFILE "|%-30s|",$compiler_name;
        undef @CompilerValues;
        get_mark_values ($compiler_name,$bmk_class,\@CompilerValues);
        # Initialize min, max once
        defined ($max) || do { $max = $CompilerValues[0]; };
        defined ($min) || do { $min = $CompilerValues[0]; };
        foreach $val ( @CompilerValues ) {
            $val > $max && do { $max = $val };
            $val < $min && do { $min = $val }; 
            printf TWIKIFILE "%12.2f|",$val;
        }
        print TWIKIFILE "\n";
    }
    return ($colcnt,$rowcnt,$min,$max);
}

##
## create_table_normalized - create a table for this benchmark class
## Data is normalized to baseline compiler (baseline == 1)
##
##
sub create_table_normalized {
    my $bmk_class = shift @_;
    my $ccnt_ref = shift @_;
    my $core_name, my $compiler_name;
    my $val, my $colcnt=0, my $rowcnt=0, my $min, my $max;
    my $baseline=$Compilers[0];
    my @baseline_values=();
    my $table_name = $bmk_class . "_n";
    my $idx;


    print TWIKIFILE "%TABLE{ name=\"$table_name\" }%\n";

    printf TWIKIFILE "|%-30s|","*$bmk_class*";

    foreach $core_name (@Cores) {
        $colcnt++;
        printf TWIKIFILE "%-12s|",$core_name;
    }
    print TWIKIFILE "\n";

    # Get baseline compiler
    foreach $compiler_name (@Compilers) {
        if ( $compiler_name =~ /^$CmdCompare$/ ) {
            $baseline = $compiler_name;
        }
    }
    # Get baseline value
    get_mark_values ($baseline,$bmk_class,\@baseline_values);
    
    undef($max);
    undef($min);
    foreach $compiler_name (@Compilers) {
        $rowcnt++;
        printf TWIKIFILE "|%-30s|",$compiler_name;
        undef @CompilerValues;
        get_mark_values ($compiler_name,$bmk_class,\@CompilerValues);
        # Initialize min, max once
        defined ($max) || do { $max = $CompilerValues[0]/$baseline_values[0]; };
        defined ($min) || do { $min = $CompilerValues[0]/$baseline_values[0]; };
        $idx=0;
        foreach $val ( @CompilerValues ) {
            my $normal_val = $val/$baseline_values[$idx++];
            $normal_val > $max && do { $max = $normal_val };
            $normal_val < $min && do { $min = $normal_val }; 
            printf TWIKIFILE "%12.4f|",$normal_val;
        }
        print TWIKIFILE "\n";
    }
    return ($colcnt,$rowcnt,$min,$max);
}

##
## create_chart - create a chart for this benchmark class
##
##
sub create_chart {
    my $normalized = shift @_;
    my $classname = shift @_;
    my $colcnt = shift @_;
    my $rowcnt = shift @_;
    my $min = shift @_;
    my $max = shift @_;

    my $name="bar_$classname";
    my $title=uc($MarkNames{$classname}) . " [$Units{$classname}]";
    my $table=$classname;
    my $lastrow=$rowcnt+1;
    my $lastcol=$colcnt+1;
    my $data="R2:C2..R$lastrow:C$lastcol";
    my $xaxis="R1:C2..R1:C$lastcol";
    my $legend="R2:C1..R$lastrow:C1";
    my $ymin, my $ymax, my $yminstr, my $ymaxstr;
    my $width=750;
    my $height=300;

    $normalized && do {
        $table .= "_n";
    };
        
    if ($normalized) {
        if (($max-$min) < 0.001) {
            $ymin=0.95;
            $ymax=1.10
        }
        else {
            $ymin = $min - (($max-$min)*3);
            $ymax = $max + (($max-$min)*3);
        }
        $yminstr = sprintf("%4.2f",$ymin);
        $ymaxstr = sprintf("%4.2f",$ymax);
    }
    else {
        $ymin = $min - (($max-$min)*0.1);
        $ymax = $max + (($max-$min)*0.1);
        $yminstr = sprintf("%8.4f",$ymin);
        $ymaxstr = sprintf("%8.4f",$ymax);
    }

    print TWIKIFILE "%CHART{ type=\"bar\" name=\"$name\" title=\"$title\" colors=\"#FF0000, #FFCC00, #00CC00, #FF00FF, #33CCCC, #FF8000, #009900, #FF6666, #3333FF, #800080\" yaxis=\"on\" ygrid=\"on\" datalabel=\"box\" table=\"$table\" data=\"$data\" xaxis=\"$xaxis\" legend=\"$legend\" ymin=\"$yminstr\" ymax=\"$ymaxstr\" width=\"$width\" height=\"$height\"}%\n";
}    

##
## get_mark_values - get coremark, automark, etc. for a benchmark class/ compiler
## Fill array ref with number of values equal to number of cores for this benchmark class.
##
sub get_mark_values {
    my $compiler_name = shift @_;
    my $classname = shift @_;
    my $arrayref = shift @_;
    my $data, my $i, my $key, my $key2;

    foreach my $core_name (@Cores) {
        $key = $core_name . ":" . $compiler_name . ":" . $classname . ":" . $MarkNames{$classname};
        $key2 = $core_name . ":" .  $classname . ":" . $MarkNames{$classname};
        if (defined($ValidDataTemplate{$key2})) {
            $data = $Data{$key};
            ${$arrayref}[$i++] = ${$data}[1];
        }
    }
}

##
## get_values_by_bmkname - get values for a benchmark class/ compiler
##
sub get_values_by_bmkname {
    my $core_name = shift @_;
    my $compiler_name = shift @_;
    my $classname = shift @_;
    my $arrayref = shift @_;
    my $key;
    my $key2;
    my $data;
    my $i=0;

    foreach my $bmk_name (@{$BmkClassDesc{$classname}}) {
        $key = $core_name . ":" . $compiler_name . ":" . $classname . ":" . $bmk_name;
        $key2 = $core_name . ":" . $classname . ":" . $bmk_name;
        if (defined($ValidDataTemplate{$key2})) {
            $data = $Data{$key};
            ${$arrayref}[$i++] = ${$data}[1];
        }
    }
}

##
## get_values_by_compiler - get  values for all compiler names 
##
sub get_values_by_compiler {
    my $core = shift @_;
    my $class = shift @_;
    my $bmkname = shift @_;
    my $compiler_names_ref  = shift @_;
    my $arrayref = shift @_;
    my $key; 
    my $key2;
    my $data;
    my $i=0;

    foreach my $compiler_name (@{$compiler_names_ref}) {
        $key = $core . ":" . $compiler_name . ":" . $class . ":" . $bmkname;
        $key2 = $core . ":" . $class . ":" . $bmkname;
        if (defined($ValidDataTemplate{$key2})) {
            $data = $Data{$key};
            ${$arrayref}[$i++] = ${$data}[1];
        }
    }
}

##
## is_eembc - is class an eembc class
##
sub is_eembc {
    my $name = shift @_;
    if ( $name =~ /^auto/ 
         || $name =~ /^consumer/ 
         || $name =~ /^net/ 
         || $name =~ /^telecom/ 
         || $name =~ /^office/ ) {
        return 1;
    }

    return 0;
}

##
## has_mark_data - verify calculated mark for this bmk for each core, compiler
## return 1 is data exists for all, 0 otherwise
##
sub has_mark_data {
    my $bmkclass = shift @_;
    my $key;
    my $key2;

    foreach my $core (@Cores) {
        foreach my $compiler (@Compilers) {
            $key = $core . ":" . $compiler . ":" . $bmkclass . ":" . $MarkNames{$bmkclass} ;
            $key2 = $core . ":" .  $bmkclass . ":" . $MarkNames{$bmkclass} ;
            defined ($ValidDataTemplate{$key2}) || do { return 0; };
        }
    }
    return 1;
}
    
##
## compute_mark
##
sub compute_mark {
    my $bmkclass = shift @_;
    my $arrayref = shift @_;

    $bmkclass =~ /^auto/ && do {
        return ( geomean ($arrayref) / 307.455 );
    };
    $bmkclass =~ /^consumer/ && do {
        return ( geomean ($arrayref) / 2.192 );
    };
    $bmkclass =~ /^net/ && do {
        return ( geomean ($arrayref) / 395.184 );
    };
    $bmkclass =~ /^office/ && do {
        return ( geomean ($arrayref) / 1.139 );
    };
    $bmkclass =~ /^telecom/ && do {
        return ( geomean ($arrayref) / 785.138 );
    };

    return 0.0;
}

##
## compute_total
##
sub compute_total {
    my $arrayref = shift @_;
    my $val=0.0;
    my $sum=0.0;

    foreach $val ( @{$arrayref} ) {
        $sum += $val;
    }

    return $sum;
}

##
## geomean - compute geometric mean of an array of values
##
sub geomean {
    my $arrayref = shift @_;
    my $logsum = 0;
    my $value;
    my $n=0;

    if ($#{$arrayref} == -1) {
        return 0.0;
    }

    foreach $value ( @{$arrayref} ) {
	if ($value != 0) {
	    $logsum += log ($value);
	}
        $n++;
    }
    $logsum /= $n;
    return (exp $logsum);
}

##
## printseparator - Print line separator 
##
##
sub printseparator {
  printcomment( '=' x $LineWidth );
}

##
## printcomment - Print a comment line
##
##
sub printcomment {
  my $text = shift @_;

  print $text,"\n";
#  print ".        1         2         3         4         5          6         7          8         9         0\n" ;
#  print "1........0.........0.........0.........0.........0..........0.........0..........0.........0.........0\n" ;
}
