#!/bin/bash

get_rev () {
    # Function to get svn revision number from a working dir

    # Arguments are
    # working dir
    local WDIR=$1

    cd $WDIR
    REV=`svn info | grep Revision | awk '{print $2}'`
    cd - > /dev/null 2>&1
    echo "...$WDIR has revision $REV"
}

do_import () {
    # Function to do initial FSF import

    # Arguments are
    # target (gcc-trunk eglibc ...)
    local TARGET=$1
    # working dir
    local WDIR=$2
    # subversion tag (rev.999999 or snapshot.20110308)
    local TAG=$3

    echo "...Importing"
    svn --quiet import --no-auto-props $SVNOPTS -m "Import of $TARGET $TAG" \
	$WDIR \
	$REPO/FSF/$TARGET/current
    if [ $? != 0 ]; then exit 0; fi

    echo "...Importing - copy to FSF/$TARGET/$TAG"
    svn copy $SVNOPTS $REPO/FSF/$TARGET/current \
        $REPO/FSF/$TARGET/$TAG \
        -m "Tagging $TAG"
    if [ $? != 0 ]; then exit 0; fi

    echo "...Importing - copy to $TARGET"
    svn copy $SVNOPTS $REPO/FSF/$TARGET/$TAG \
        $REPO/$TARGET \
        -m "Bringing $TAG into Freescale $TARGET repository"
    if [ $? != 0 ]; then exit 0; fi
}

do_reimport () {
    # Function to do "svn vendor tag" on FSF

    # Arguments are
    # target (gcc-trunk eglibc ...)
    local TARGET=$1
    # working dir
    local WDIR=$2
    # subversion tag (rev.999999 or snapshot.20110308)
    local TAG=$3
    
    echo "...Re-importing"
    svn_load_dirs.pl $SVNLOADOPTS -no_user_input -no_auto_exe -t $TAG \
	$REPO/FSF/$TARGET \
	current \
	$WDIR >> svn_load_dirs.log 2>&1
    if [ $? != 0 ]; then echo "[svn_load_dirs.pl] failed]"; exit 0; fi
}

do_merge () {
    # Function to apply differences from two imports into Freescale repository

    # Arguments are
    # target (gcc-trunk eglibc ...)
    local TARGET=$1
    # previous tag
    local prevTAG=$2
    # current tag
    local currTAG=$3
    
    mkdir x
    cd x
    echo "...Merging new tag - checkout"
    svn --quiet co $REPO/$TARGET
    if [ $? != 0 ]; then exit 0; fi

    cd $TARGET
    echo "...Merging new tag - merge"
    svn --quiet merge $REPO/FSF/$TARGET/$prevTAG \
    	$REPO/FSF/$TARGET/$currTAG \
	.
    if [ $? != 0 ]; then exit 0; fi

    echo "...Merging new tag - commit"
    svn --quiet commit $SVNOPTS -m "Merging $currTAG into Freescale $TARGET repository"
    if [ $? != 0 ]; then exit 0; fi

    cd ../../
    rm -rf x
}

scriptdir=`dirname $0`

# Get user/password
if [ -f $scriptdir/svn_import.data ]; then
    echo "Reading authorization data"
    source $scriptdir/svn_import.data
    SVNOPTS="--username $USER --password $PASSWD"
    SVNLOADOPTS="-svn_username $USER -svn_password $PASSWD"
fi

# Get latest tags
if [ -f $scriptdir/svn_import.tags ]; then
    echo "Reading previous tags"
    source $scriptdir/svn_import.tags
fi

REPO="http://svn.freescale.net/svn/gcc"
#REPO="file:///local/build/svn_repo"
REV=""
TDY=`date +%Y%m%d`

# Execute
rm -f svn_load_dirs.log
for a in $@; do
    echo "Processing $a"
    echo "...Checking out"
    case $a in
	binutils)
	    #cvs -Q -z 9 -d :pserver:anoncvs@sourceware.org:/cvs/src co binutils
	    rm -rf binutils*
	    wget -q ftp://sourceware.org/pub/binutils/snapshots/binutils.weekly.tar.bz2
	    if [ $? != 0 ]; then echo "   ...Error"; continue; fi
	    tar xjf binutils.weekly.tar.bz2
	    mv binutils-* binutils-$TDY
	    if [ "$bin_prev_TAG" = "" ]; then
		do_import binutils binutils-$TDY snapshot.$TDY
	    elif [ "$bin_prev_TAG" != "snapshot.$TDY" ]; then
		do_reimport binutils binutils-$TDY snapshot.$TDY
		do_merge binutils $bin_prev_TAG snapshot.$TDY
	    else
		echo "...Nothing to do"
		continue
	    fi
	    echo "bin_prev_TAG=snapshot.$TDY" >> $scriptdir/svn_import.tags
	    ;;
	gcc-4.5-branch)
	    if [ -d gcc-4.5-branch ]; then
		cd gcc-4.5-branch
		svn -q update
		x=$?
		cd ../
	    else
		svn --quiet co http://gcc.gnu.org/svn/gcc/branches/gcc-4_5-branch gcc-4.5-branch
		x=$?
	    fi
	    if [ $x != 0 ]; then echo "   ...Error"; continue; fi
	    get_rev gcc-4.5-branch
	    if [ "$gcc45_prev_TAG" = "" ]; then
		do_import gcc-4.5-branch gcc-4.5-branch rev.$REV
	    elif [ "$gcc45_prev_TAG" != "rev.$REV" ]; then
		do_reimport gcc-4.5-branch gcc-4.5-branch rev.$REV
		do_merge gcc-4.5-branch $gcc45_prev_TAG rev.$REV
	    else
		echo "...Nothing to do"
		continue
	    fi
	    echo "gcc45_prev_TAG=rev.$REV" >> $scriptdir/svn_import.tags
	    ;;
	gcc-4.6-branch)
	    if [ -d gcc-4.6-branch ]; then
		cd gcc-4.6-branch
		svn -q update
		x=$?
		cd ../
	    else
		svn --quiet co http://gcc.gnu.org/svn/gcc/branches/gcc-4_6-branch gcc-4.6-branch
		x=$?
	    fi
	    if [ $x != 0 ]; then echo "   ...Error"; continue; fi
	    get_rev gcc-4.6-branch
	    if [ "$gcc46_prev_TAG" = "" ]; then
		do_import gcc-4.6-branch gcc-4.6-branch rev.$REV
	    elif [ "$gcc46_prev_TAG" != "rev.$REV" ]; then
		do_reimport gcc-4.6-branch gcc-4.6-branch rev.$REV
		do_merge gcc-4.6-branch $gcc46_prev_TAG rev.$REV
	    else
		echo "...Nothing to do"
		continue
	    fi
	    echo "gcc46_prev_TAG=rev.$REV" >> $scriptdir/svn_import.tags
	    ;;
	gcc-4.7-branch)
	    if [ -d gcc-4.7-branch ]; then
		cd gcc-4.7-branch
		svn -q update
		x=$?
		cd ../
	    else
		svn --quiet co http://gcc.gnu.org/svn/gcc/branches/gcc-4_6-branch gcc-4.7-branch
		x=$?
	    fi
	    if [ $x != 0 ]; then echo "   ...Error"; continue; fi
	    get_rev gcc-4.7-branch
	    if [ "$gcc47_prev_TAG" = "" ]; then
		do_import gcc-4.7-branch gcc-4.7-branch rev.$REV
	    elif [ "$gcc47_prev_TAG" != "rev.$REV" ]; then
		do_reimport gcc-4.7-branch gcc-4.7-branch rev.$REV
		do_merge gcc-4.7-branch $gcc47_prev_TAG rev.$REV
	    else
		echo "...Nothing to do"
		continue
	    fi
	    echo "gcc47_prev_TAG=rev.$REV" >> $scriptdir/svn_import.tags
	    ;;
	gcc-trunk)
	    if [ -d gcc-trunk ]; then
		cd gcc-trunk
		svn -q update
		x=$?
		cd ../
	    else
		svn --quiet co http://gcc.gnu.org/svn/gcc/trunk gcc-trunk
		x=$?
	    fi
	    if [ $x != 0 ]; then echo "   ...Error"; continue; fi
	    get_rev gcc-trunk
	    if [ "$gcc_prev_TAG" = "" ]; then
		do_import gcc-trunk gcc-trunk rev.$REV
	    elif [ "$gcc_prev_TAG" != "rev.$REV" ]; then
		do_reimport gcc-trunk gcc-trunk rev.$REV
		do_merge gcc-trunk $gcc_prev_TAG rev.$REV
	    else
		echo "...Nothing to do"
		continue
	    fi
	    echo "gcc_prev_TAG=rev.$REV" >> $scriptdir/svn_import.tags
	    ;;
	eglibc)
	    if [ -d eglibc ]; then
		cd eglibc
		svn -q update
		x=$?
		cd ../
	    else
		svn --quiet co http://www.eglibc.org/svn/trunk eglibc
		x=$?
	    fi
	    if [ $x != 0 ]; then echo "   ...Error"; continue; fi
	    get_rev eglibc
	    if [ "$eglibc_prev_TAG" = "" ]; then
		do_import eglibc eglibc rev.$REV
	    elif [ "$eglibc_prev_TAG" != "rev.$REV" ]; then
		do_reimport eglibc eglibc rev.$REV
		do_merge eglibc $eglibc_prev_TAG rev.$REV
	    else
		echo "...Nothing to do"
		continue
	    fi
	    echo "eglibc_prev_TAG=rev.$REV" >> $scriptdir/svn_import.tags
	    ;;
    esac
done
