#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <asm/types.h>
#include <signal.h>
#include <sys/time.h>
#include <fcntl.h>
#include <linux/input.h>
#include <errno.h>
#include <time.h>
#include <termios.h>
#include <sys/fcntl.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sched.h>

#include <FL/Fl.H>
#include <FL/Fl_Input.H>
#include <FL/filename.H>

#include "Release.h"
#include "Run.h"
#include "releasemain.h"

//TODO
// Implement save / restore to file

/*
 * Global variables and "environment" variables
 * ============================================
 */
#define NUM_TASK_BUTTONS 495
#define COLUMN_TASK_BUTTONS 11

FILE *flog;

int n_sets = 0;
Set *sets[64];
int n_maps = 0;
Map *maps[64];
int n_tasks = 0;
Task *tasks[1024];
Release rc;
Task *run_to_task[1024];

char *REPO = NULL;
char *SRC = NULL;
char *REV = NULL;
char *CDEuser = NULL;
char *DTuser = NULL;
char *FLGS = NULL;
char *DESC = NULL;

LL *L_ready = NULL;
LL *L_requires = NULL;
LL *L_running = NULL;

/*
 * Main functions
 * ==============
 */

void
alloc_set (char *id)
{
  Set *set;

  set = (Set *) malloc (sizeof(Set));
  set->id = strdup (id);
  set->len = 0;

  /* Push set to stack */
  sets[n_sets] = set;
  n_sets++;
}

void
alloc_set_list (char *elem)
{
  Set *cur;
  cur = sets[n_sets-1];

  cur->list[cur->len] = strdup(elem);
  cur->len++;
}

char *
do_list (char *id)
{
  int i;
  Set *s;
  char buf[4096];

  for (i = 0; i < n_sets; i++) {
    if (strcmp (sets[i]->id, id) == 0)
      break;
  }
  if (i == n_sets)
    return NULL;

  s = sets[i];
  buf[0] = '\0';
  for (i = 0; i < s->len; i++) {
    strcat (buf, s->list[i]);
    if (i < s->len - 1)
      strcat (buf, " ");
  }
  return strdup (buf);
}

Set *
find_set (char *id)
{
  char *p, *s;
  int i;

  p = id;
  while (*p != '[' && *p != '\0')
    p++;
  if (*p == '\0') {
    /* This is not a set */
    return NULL;
  }
  p++;
  s = p;
  while (*s != ']' && *s != '\0') // NOTE Since we modify the id bellow,
    s++;                          // it may happen that the ']' was already erased.
  *s = '\0';

  for (i = 0; i < n_sets; i++) {
    if (strcmp (sets[i]->id, p) == 0)
      break;
  }
  fprintf(flog, "[find_set]              Id: %s idx: %d\n", p, i);
  if (i == n_sets)
    return NULL;
  return sets[i];  
}

void
alloc_map (char *id)
{
  Map *map;

  map = (Map *) malloc (sizeof(Map));
  map->id = strdup (id);
  map->len = 0;

  /* Push map to stack */
  maps[n_maps] = map;
  n_maps++;
}

void
alloc_map_map (char *elem)
{
  Map *cur;
  char *saveptr;
  char *k, *m;

  cur = maps[n_maps-1];

  k = strtok_r (elem, "=", &saveptr);
  cur->key[cur->len] = strdup(k);
  m = strtok_r (NULL, "\n\t ", &saveptr);
  cur->map[cur->len] = strdup(m);
  fprintf (flog, "[alloc_map_map]         mapping %s to %s\n",
	   cur->key[cur->len], cur->map[cur->len]);
  cur->len++;
}

char *
do_map (char *id, char *key)
{
  int i;
  Map *m;

  for (i = 0; i < n_maps; i++) {
    if (strcmp (maps[i]->id, id) == 0)
      break;
  }
  if (i == n_maps)
    return NULL;

  m = maps[i];
  for (i = 0; i < m->len; i++) {
    if (strcmp (m->key[i], key) == 0)
      break;
  }
  if (i == m->len)
    return NULL;

  return m->map[i];
}

void
alloc_task (int type)
{
  Task *t;

  t = (Task *) malloc (sizeof (Task));
  memset (t, 0, sizeof (Task));
  t->type = type;
  tasks[n_tasks] = t;
  n_tasks++;
}

void
alloc_task_field (char *f, int lineno)
{
  Task *cur;
  char *saveptr;
  char *p1, *p2;

  cur = tasks[n_tasks - 1];

  /* All fileds in task descriptors are in form
     p1 = p2 */
  p1 = strtok_r (f, "=", &saveptr);
  p2 = strtok_r (NULL, "\n", &saveptr);
  fprintf (flog, "[alloc_task_field]      Task field defintion %s = %s\n", p1, p2);

  /* Now we process according to type of field */
  if (strcmp (p1, "job") == 0) {
    cur->job = strdup (p2);
  } else if (strcmp (p1, "ml") == 0) {
    cur->ml = strdup (p2);
  } else if (strcmp (p1, "user") == 0) {
    cur->user = strdup (p2);
  } else if (strcmp (p1, "priority") == 0) {
    cur->priority = atoi (p2);
  } else if (strcmp (p1, "wdir") == 0) {
    cur->wdir = strdup (p2);
  } else if (strcmp (p1, "bmk") == 0) {
    cur->bmk = strdup (p2);
  } else if (strcmp (p1, "log") == 0) {
    cur->log = strdup (p2);
  } else if (strcmp (p1, "requires") == 0) {
    cur->requires[cur->n_reqs] = strdup (p2);
    cur->n_reqs++;
  } else if (strcmp (p1, "provides") == 0) {
    cur->provides = strdup (p2);
  } else if (strcmp (p1, "exec") == 0) {
    cur->execs[cur->n_execs] = strdup (p2);
    cur->n_execs++;
  } else if (strcmp (p1, "result") == 0) {
    cur->result = strdup (p2);
  } else if (strcmp (p1, "tarball") == 0) {
    cur->tarball = safe_strdup (p2);
  } else if (strcmp (p1, "Wtarball") == 0) {
    cur->Wtarball = safe_strdup (p2);
  } else if (strcmp (p1, "machine") == 0) {
    cur->machine = strdup (p2);
  } else if (strcmp (p1, "target") == 0) {
    cur->target = strdup (p2);
  } else {
    fprintf (stderr, "[alloc_task_field] Error: Unknown task field on line %d: %s\n",
	     lineno, p1);
    exit (-1);
  }
}

char *
get_env (char *key)
{
  if (strcmp (key, "REPO") == 0)
    return REPO;
  else if (strcmp (key, "SRC") == 0)
    return SRC;
  else if (strcmp (key, "REV") == 0)
    return REV;
  else if (strcmp (key, "CDEuser") == 0)
    return CDEuser;
  else if (strcmp (key, "DTuser") == 0)
    return DTuser;
  else if (strcmp (key, "FLGS") == 0)
    return FLGS;
  else if (strcmp (key, "DESC") == 0)
    return DESC;
  return NULL;
}

char *
get_local (char *key, Task *t)
{
  if (strcmp (key, "job") == 0) {
    return t->job;
  } else if (strcmp (key, "ml") == 0) {
    return t->ml;
  } else if (strcmp (key, "user") == 0) {
    return t->user;
  } else if (strcmp (key, "log") == 0) {
    return t->log;
  } else if (strcmp (key, "bmk") == 0) {
    return t->bmk;
  } else if (strcmp (key, "tarname") == 0) {
    if (t->tarname == NULL)
      return cat3("<",key,">"); /* tarname local definition not available yet */
    else
      return t->tarname;
  } else if (strcmp (key, "tools") == 0) {
    if (t->tools == NULL)
      return cat3("<",key,">"); /* tools local definition not available yet */
    else
      return t->tools;
  } else if (strcmp (key, "machine") == 0) {
    return t->machine;
  } else if (strcmp (key, "target") == 0) {
    return t->target;
  }
  return NULL;
}

char *
get_map (char *key)
{
  int i;
  int len;
  char *p;
  Map *m;

  len = 0;
  for (i = 0; i < n_maps; i++) {
    len = strlen (maps[i]->id);
    if (strncmp (maps[i]->id, key, len) == 0)
      break;
  }
  p = key + len;
  m = maps[i];
  for (i = 0; i < m->len; i++) {
    if (strcmp (m->key[i], p) == 0)
      break;
  }
  return m->map[i];
}

char *
expand_helper (char *s, char delim1, char delim2, Task *t, int get_function)
{
  char *pin, *pout, *p1, *p2;
  char buf[4096];

  if (s == NULL)
    return NULL;

  /* Search for a keyword string to be expanded */
  pin = s;
  pout = buf;
  while (*pin != '\0') {
    /* Copy input until need expansion */
    while (*pin != delim1 && *pin != '\0')
      *pout++ = *pin++;
    if (*pin == '\0')
      break;
    /* Start of id */
    pin++;
    p1 = pin;
    while (*p1 != delim2)
      p1++;
    /* Terminate id */
    *p1 = '\0';
    /* Translate */
    if (get_function == 1)
      p2 = get_env (pin);
    else if (get_function == 2)
      p2 = get_local (pin, t);
    else if (get_function == 3)
      p2 = do_list (pin);
    else if (get_function == 4)
      p2 = get_map (pin);
    /* Copy translation to output */
    while (*p2 != '\0')
      *pout++ = *p2++;
    /* Re-start search from end of last id */
    pin = p1 + 1;
  }
  *pout = '\0';
  return strdup (buf);
}

char *
expand_env (char *s)
{
  return expand_helper (s, '(', ')', NULL, 1);
}

char *
expand_local (char *s, Task *t)
{
  return expand_helper (s, '<', '>', t, 2);
}

char *
expand_set (char *s)
{
  return expand_helper (s, '[', ']', NULL, 3);
}

char *
expand_map (char *s)
{
  return expand_helper (s, ':', ':', NULL, 4);
}

void
get_provider (Task *t)
{
  int i, j;

  for (j = 0; j < t->n_reqs; j++) {
    for (i = 0; i < n_tasks; i++) {
      if (tasks[i]->provides && strcmp (tasks[i]->provides, t->requires[j]) == 0)
	t->up[j] = tasks[i];
    }
  }
}

void
task_expand_locals (int first_t, int last_t)
{
  int i, j;
  Task *t;

  for (i = first_t; i <= last_t; i++) {
    t = tasks[i];
    t->user = expand_env (t->user);
    t->user = expand_local (t->user, t);
    t->user = expand_map (t->user);

    t->log = expand_env (t->log);
    t->log = expand_local (t->log, t);
    
    t->wdir = expand_env (t->wdir);
    t->wdir = expand_local (t->wdir, t);
    
    t->machine = expand_local (t->machine, t);
    t->machine = expand_map (t->machine);

    t->target = expand_env (t->target);
    t->target = expand_local (t->target, t);
    t->target = expand_map (t->target);

    for (j = 0; j < t->n_reqs; j++) {
      t->requires[j] = expand_env (t->requires[j]);
      t->requires[j] = expand_local (t->requires[j], t);
    }
    
    t->provides = expand_env (t->provides);
    t->provides = expand_local (t->provides, t);
    
    t->result = expand_env (t->result);
    t->result = expand_local (t->result, t);
    
    t->tarball = expand_env (t->tarball);
    t->tarball = expand_local (t->tarball, t);

    t->Wtarball = expand_env (t->Wtarball);
    t->Wtarball = expand_local (t->Wtarball, t);

    for (j = 0; j < t->n_execs; j++) {
      t->execs[j] = expand_env (t->execs[j]);
      t->execs[j] = expand_local (t->execs[j], t);
      t->execs[j] = expand_map (t->execs[j]);
      t->execs[j] = expand_set (t->execs[j]);
    }
    if (t->n_reqs != 0)
      get_provider (t);
  }
}

void
task_expand_parallel (void)
{
  int first_t;
  int last_t;
  int i, j;
  Task *cur;
  Set *set;

  first_t = n_tasks - 1;
  last_t = first_t;
  cur = tasks[first_t];
  /* There is only one field in build that can have parallel: either job or ml */
  if (cur->job != NULL && strncmp (cur->job, "parallel", 8) == 0) {
    set = find_set (cur->job);
    last_t += set->len - 1;
    for (i = 0; i < set->len - 1; i++)
      alloc_task (cur->type);
    tasks[first_t]->job = strdup (set->list[0]);
    for (i = first_t+1; i <= last_t; i++) {
      tasks[i]->job = strdup (set->list[i-first_t]);
      tasks[i]->user = safe_strdup (tasks[first_t]->user);
      tasks[i]->priority = tasks[first_t]->priority;
      tasks[i]->wdir = safe_strdup (tasks[first_t]->wdir);
      tasks[i]->bmk = safe_strdup (tasks[first_t]->bmk);
      tasks[i]->log = safe_strdup (tasks[first_t]->log);
      tasks[i]->n_reqs = tasks[first_t]->n_reqs;
      for (j = 0; j < tasks[i]->n_reqs; j++)
	tasks[i]->requires[j] = safe_strdup (tasks[first_t]->requires[j]);
      tasks[i]->provides = safe_strdup (tasks[first_t]->provides);
      tasks[i]->n_execs = tasks[first_t]->n_execs;
      for (j = 0; j < tasks[i]->n_execs; j++)
	tasks[i]->execs[j] = strdup (tasks[first_t]->execs[j]);
      tasks[i]->result = safe_strdup (tasks[first_t]->result);
      tasks[i]->tarball = safe_strdup (tasks[first_t]->tarball);
      tasks[i]->Wtarball = safe_strdup (tasks[first_t]->Wtarball);
      tasks[i]->machine = safe_strdup (tasks[first_t]->machine);
      tasks[i]->target = safe_strdup (tasks[first_t]->target);
    }
  } else if (cur->ml != NULL && strncmp (cur->ml, "parallel", 8) == 0) {
    set = find_set (cur->ml);
    last_t += set->len - 1;
    for (i = 0; i < set->len - 1; i++)
      alloc_task (cur->type);
    tasks[first_t]->ml = strdup (set->list[0]);
    for (i = first_t+1; i <= last_t; i++) {
      tasks[i]->job = safe_strdup (tasks[first_t]->job);
      tasks[i]->ml = strdup (set->list[i-first_t]);
      tasks[i]->user = safe_strdup (tasks[first_t]->user);
      tasks[i]->priority = tasks[first_t]->priority;
      tasks[i]->wdir = safe_strdup (tasks[first_t]->wdir);
      tasks[i]->bmk = safe_strdup (tasks[first_t]->bmk);
      tasks[i]->log = safe_strdup (tasks[first_t]->log);
      tasks[i]->n_reqs = tasks[first_t]->n_reqs;
      for (j = 0; j < tasks[i]->n_reqs; j++)
	tasks[i]->requires[j] = safe_strdup (tasks[first_t]->requires[j]);
      tasks[i]->provides = safe_strdup (tasks[first_t]->provides);
      tasks[i]->n_execs = tasks[first_t]->n_execs;
      for (j = 0; j < tasks[i]->n_execs; j++)
	tasks[i]->execs[j] = strdup (tasks[first_t]->execs[j]);
      tasks[i]->result = safe_strdup (tasks[first_t]->result);
      tasks[i]->tarball = safe_strdup (tasks[first_t]->tarball);
      tasks[i]->Wtarball = safe_strdup (tasks[first_t]->Wtarball);
      tasks[i]->machine = safe_strdup (tasks[first_t]->machine);
      tasks[i]->target = safe_strdup (tasks[first_t]->target);
    }
  }
  task_expand_locals (first_t, last_t);
}

char *
pick_best_host (char *user, char *id)
{
  Set *set;
  int n_cpus;
  double load;
  int i;
  char *best_name;
  double best_load;

  best_load = 99999.9;
  best_name = NULL;
  set = find_set (id);
  if (set == NULL) {
    if (! is_set_locked (id)) {
      /* This is a single element */
      n_cpus = get_host_cpus (user, id);
      load = get_host_load (user, id);
      load = load / n_cpus;
      fprintf (flog, "[pick_best_host]        Host %s has %d cpus and %f load\n",
	       id, n_cpus, load);
      if (n_cpus > 0 && load >= 0.0 && load < 0.85) {
	best_load = load;
	best_name = id;
      } else {
	/* This guy is too busy, don't waist time testing it again */
	lock_set (id);
      }
    }
  } else {
    for (i = 0; i < set->len; i++) {
      if (! is_set_locked (set->list[i])) {
	n_cpus = get_host_cpus (user, set->list[i]);
	load = get_host_load (user, set->list[i]);
	load = load / n_cpus;
	fprintf (flog, "[pick_best_host]        Host %s has %d cpus and %f load\n",
		 set->list[i], n_cpus, load);
	if (n_cpus > 0 && load >= 0.0 && load < 0.65) {
	  if (load < best_load) {
	    best_load = load;
	    best_name = set->list[i];
	  }
	} else {
	  /* This guy is too busy, don't waist time testing it again */
	  lock_set (set->list[i]);
	}
      }
    }
  }
  fprintf (flog, "[pick_best_host]        Best host is %s with load %f\n",
	   best_name, best_load);
  /* So, we don't repeadtedly pick the same one. Assumes log5-atx never locks */
  if (best_name != NULL && strcmp (best_name, "log5-atx"))
    lock_set (best_name);
  return best_name;
}

int
reset_one_build (Task *t)
{
  char buf[1024];
  int ret;

  if (t->log && t->mach) {
    sprintf (buf, "ssh -l %s %s \"true && %s && rm -rf `cat %s/logs/%s | grep '^#RESET_INFO' | awk '{print $2,$3,$4; exit}'`\"",
	     t->user, t->mach, t->wdir, REPO, t->log);
    ret = system (buf);
    fprintf (flog, "[reset_one_build]       Cleaning work area: %s\n", buf);
    if (ret == -1) {
      return -1;
    }
  }
  return 0;
}

int
retry_one_build (Task *t)
{
  char buf[1024];
  int ret;
  LL *l;

  reset_one_build (t);
  if (t->log) {
    sprintf (buf, "rm -f %s/logs/%s\n", REPO, t->log);
    fprintf (flog, "[retry_one_build]       Removing log: %s\n", buf);
    ret = system (buf);
    if (ret == -1) {
      fprintf (stderr, "[retry_one_build] Unexpected error reading buffer\n");
      return -1;
    }
    rc.task[t->task_num]->color (FL_WHITE);
    rc.window->redraw();
  }
  if (t->tarname != NULL) {
    sprintf (buf, "rm -f %s/releases/%s\n", REPO, t->tarname);
    fprintf (flog, "[retry_one_build]       Removing tarball: %s\n", buf);
    ret = system (buf);
    if (ret == -1) {
      fprintf (stderr, "[retry_one_build] Unexpected error reading buffer\n");
      return -1;
    }
  }
  if (t->Wtarname != NULL) {
    sprintf (buf, "rm -f %s/releases/%s\n", REPO, t->Wtarname);
    fprintf (flog, "[retry_one_build]       Removing tarball: %s\n", buf);
    ret = system (buf);
    if (ret == -1) {
      fprintf (stderr, "[retry_one_build] Unexpected error reading buffer\n");
      return -1;
    }
  }
  t->status = E_ready;
  l = LL_create_node (t);
  LL_add_node (&L_ready, l);
  
  return 0;
}

char *
get_tools_dir (char *tn)
{
  char *buf;
  char *p;

  buf = strdup (tn);
  p = strstr (buf, "-");
  p = strstr (p+1, "-");
  p = strstr (p+1, "-");
  p = strstr (p+1, "-");
  *p = '/';
  p = strstr (p+1, "-");
  p = strstr (p+1, "-");
  *p = '\0';
  return buf;
}

int
exec_one_build (void *ptr)
{
  int i, ret, len;
  char buf[1024];
  char *cmd, *p;
  FILE *fres;
  Task *t = (Task *)ptr;

  ret = 0;
  fres = NULL;
  t->status = E_running;
  sprintf (buf, "ssh -l %s %s \"true && %s && ", t->user, t->mach, t->wdir);
  /* Must go through all "exec" commands in sequence,
     then execute "result" and return */
  for (i = 0; i < t->n_execs; i++) {
    cmd = join3 (buf, t->execs[i], "\"");
    fprintf (flog, "[exec_one_build]        Starting build step    %s\n", cmd);
    ret = system (cmd);
    if (ret == -1) {
      t->status = E_error;
      return t->status;
    }
  }
  if (t->result) {
    fprintf (flog, "[exec_one_build]        Starting build step    %s\n",
	     t->result);
    fres = popen (t->result, "r");
    if (fres == NULL) {
      fprintf (stderr, "[exec_one_build] returned NULL\n%s\n", t->result);
      t->status = E_error;
      return t->status;
    }
    p = fgets (buf, 256, fres);
    pclose (fres);
    if (p == NULL || strncmp (p, "OK", 2) != 0) {
      fprintf (flog, "[exec_one_build]        Result of build was %s\n", p);
      t->status = E_fail;
      return t->status;
    }
  }
  if (t->tarball) {
    fprintf (flog, "[exec_one_build]        Starting build step    %s\n",
	     t->tarball);
    fres = popen (t->tarball, "r");
    if (fres == NULL) {
      fprintf (stderr, "[exec_one_build] returned NULL\n%s\n", t->tarball);
      t->status = E_error;
      return t->status;
    }
    p = fgets (buf, 256, fres);
    pclose (fres);
    if (p == NULL) {
      fprintf (stderr, "[exec_one_build] Unexpected error reading buffer\n");
      t->status = E_error;
      return t->status;
    }
    len = strlen (buf);
    if (buf[len-1] == '\n')
      buf[len-1] = '\0';
    fprintf (flog, "[exec_one_build]        Linux tarball %s\n", buf);
    t->tarname = strdup (buf);
    t->tools = get_tools_dir (t->tarname);
  }
  if (t->Wtarball) {
    fprintf (flog, "[exec_one_build]        Starting build step    %s\n",
	     t->Wtarball);
    fres = popen (t->Wtarball, "r");
    if (fres == NULL) {
      fprintf (stderr, "[exec_one_build] returned NULL\n%s\n", t->Wtarball);
      t->status = E_error;
      return t->status;
    }
    p = fgets (buf, 256, fres);
    pclose (fres);
    if (p == NULL) {
      fprintf (stderr, "[exec_one_build] Unexpected error reading buffer\n");
      t->status = E_error;
      return t->status;
    }
    len = strlen (buf);
    if (buf[len-1] == '\n')
      buf[len-1] = '\0';
    fprintf (flog, "[exec_one_build]        Windows tarball %s\n", buf);
    t->Wtarname = strdup (buf);
  }

  t->status = E_done;
  return t->status;
}

int
exec_one_test (void *ptr)
{
  int i, ret;
  char buf[1024];
  char *cmd, *p;
  FILE *fres;
  Task *t = (Task *)ptr;

  ret = 0;
  fres = NULL;
  t->status = E_running;
  sprintf (buf, "%s && ", t->wdir);
  /* Must go through all "exec" commands in sequence,
     then execute "result" and return */
  for (i = 0; i < t->n_execs; i++) {
    cmd = join2 (buf, t->execs[i]);
    fprintf (flog, "[exec_one_test]         Starting test step    %s\n", cmd);
    ret = system (cmd);
    if (ret == -1) {
      t->status = E_error;
      return t->status;
    }
  }
  if (t->result) {
    fprintf (flog, "[exec_one_test]         Starting test step    %s\n",
	     t->result);
    fres = popen (t->result, "r");
    if (fres == NULL) {
      fprintf (stderr, "[exec_one_test] Returned NULL\n%s\n", buf);
      t->status = E_error;
      return t->status;
    }
    p = fgets (buf, 256, fres);
    pclose (fres);
    if (p == NULL || strncmp (p, "OK", 2) != 0) {
      fprintf (flog, "[exec_one_test]         Result of test was %s\n", p);
      t->status = E_fail;
      return t->status;
    }
  }

  t->status = E_done;
  return t->status;
}

#define SIGCHLD 17
void
create_task_thread (Task *t)
{
  char *p;
  void *stack;
  int ret;

  p = (char *) malloc (1024*1024);
  stack = (void *) ((p + 1024*1024) - 16);

  if (t->type == T_build)
     ret = clone (exec_one_build, stack, CLONE_VM |  SIGCHLD, (void *)t);
  else
    ret = clone (exec_one_test, stack, CLONE_VM |  SIGCHLD, (void *)t);
  if (ret == -1) {
    fprintf (stderr, "[create_task_thread] Clone process failed\n");
    exit (1);
  }
}

void
dump_tasks (void)
{
  int i, j;
  Task *t;

  fprintf (flog, "Dump of all tasks expanded\n");
  fprintf (flog, "\n");
  for (i = 0; i < n_tasks; i++) {
    t = tasks[i];
                   fprintf (flog, "Task %s\n",
			    (t->type == T_build)?"Build":"Test");
                   fprintf (flog, "  job      %s\n", t->job);
  if (t->ml)       fprintf (flog, "  ml       %s\n", t->ml);
  if (t->user)     fprintf (flog, "  user     %s\n", t->user);
                   fprintf (flog, "  priority %d\n", t->priority);
  if (t->bmk)      fprintf (flog, "  bmk      %s\n", t->bmk);
  if (t->log)      fprintf (flog, "  log      %s\n", t->log);
  if (t->wdir)     fprintf (flog, "  wdir     %s\n", t->wdir);
                   fprintf (flog, "\n");
  if (t->n_reqs != 0)
    for (j = 0; j < t->n_reqs; j++)
                   fprintf (flog, "  requires %s\n", t->requires[j]);
  if (t->n_execs != 0)
    for (j = 0; j < t->n_execs; j++)
                   fprintf (flog, "  exec     %s\n", t->execs[j]);
  if (t->result)   fprintf (flog, "  result   %s\n", t->result);
  if (t->provides) fprintf (flog, "  provides %s\n", t->provides);
  if (t->tarball)  fprintf (flog, "  tarball  %s\n", t->tarball);
  if (t->Wtarball) fprintf (flog, "  Wtarball %s\n", t->Wtarball);
  if (t->machine)  fprintf (flog, "  machine  %s\n", t->machine);
  if (t->target)   fprintf (flog, "  target   %s\n", t->target);
                   fprintf (flog, "\n");
  }
  fprintf (flog, "Num of sets:      %d\n", n_sets);
  fprintf (flog, "Num of maps:      %d\n", n_maps);
  fprintf (flog, "Num of tasks:     %d\n", n_tasks);
}

int
load_release (FILE *f)
{
  int lineno;
  int state;
  char line [4096];
  char *s, *p;
  char *saveptr;

  lineno = 0;
  state = S_cmd;
  while ((s=fgets(line, 4096, f)) != NULL) {
    lineno++;
    if (state == S_build_list || state == S_test_list)
      p = strtok_r (s, "\n", &saveptr);
    else
      p = strtok_r (s, "\n\t ", &saveptr);

    /* Blank lines and comments and skipped */
    if (p == NULL)
      continue;
    while (*p == ' ')
      p++;
    if (*p == '#')
      continue;

    while (p != NULL) {
      switch (state) {
      default:
	fprintf (stderr, "Error: Undefined state at line %d: %d\n", lineno, state);
	return 0;
      case S_cmd:
	/* Must see one of valid commands: set, map, build, test, report */
	if (strcmp (p, "set") == 0) state = S_set;
	else if (strcmp (p, "map") == 0) state = S_map;
	else if (strcmp (p, "build") == 0) state = S_build;
	else if (strcmp (p, "test") == 0) state = S_test;
	else {
	  fprintf (stderr, "Error: Unknown command at line %d: %s\n", lineno, p);
	  return 0;
	}
	break;
      case S_set:
	/* This should be set id */
	fprintf (flog, "[load_release]          Set id: %s\n", p);
	alloc_set (p);
	state = S_set_begin;
	break;
      case S_map:
	/* This should be map id */
	fprintf (flog, "[load_release]          Map id: %s\n", p);
	alloc_map (p);
	state = S_map_begin;
	break;
      case S_build:
	fprintf (flog, "[load_release]          Build descriptor\n");
	if (strcmp (p, "{") != 0) {
	  fprintf (stderr, "[load_release] Error: Expected \"{\" at lineno: %d\n",
		   lineno);
	  return 0;
	}
	alloc_task (T_build);
	state = S_build_list;
	break;
      case S_test:
	fprintf (flog, "[load_release]          Test descriptor\n");
	if (strcmp (p, "{") != 0) {
	  fprintf (stderr, "[load_release] Error: Expected \"{\" at lineno: %d\n",
		   lineno);
	  return 0;
	}
	alloc_task (T_test);
	state = S_test_list;
	break;

      case S_set_begin:
	if (strcmp (p, "{") != 0) {
	  fprintf (stderr, "[load_release] Error: Expected \"{\" at lineno: %d\n",
		   lineno);
	  return 0;
	}
	state = S_set_list;
	break;
      case S_set_list:
	if (strcmp (p, "}") == 0) {
	  /* Ended list */
	  state = S_cmd;
	  break;
	}
	fprintf (flog, "[load_release]          List element: %s\n", p);
	alloc_set_list (p);
	break;

      case S_map_begin:
	if (strcmp (p, "{") != 0) {
	  fprintf (stderr, "[load_release] Error: Expected \"{\" at lineno: %d\n",
		   lineno);
	  return 0;
	}
	state = S_map_list;
	break;
      case S_map_list:
	if (strcmp (p, "}") == 0) {
	  /* Ended list */
	  state = S_cmd;
	  break;
	}
	fprintf (flog, "[load_release]          Map element: %s\n", p);
	alloc_map_map (p);
	break;

      case S_build_list:
	if (strcmp (p, "}") == 0) {
	  /* Ended list */
	  task_expand_parallel ();
	  state = S_cmd;
	  break;
	}
	fprintf (flog, "[load_release]          build element: %s\n", p);
	alloc_task_field (p, lineno);
	break;

      case S_test_list:
	if (strcmp (p, "}") == 0) {
	  /* Ended list */
	  task_expand_parallel ();
	  state = S_cmd;
	  break;
	}
	fprintf (flog, "[load_release]          test element: %s\n", p);
	alloc_task_field (p, lineno);
	break;

      }
      if (state == S_build_list || state == S_test_list)
	p = strtok_r (NULL, "\n", &saveptr);
      else
	p = strtok_r (NULL, "\n\t ", &saveptr);
    }
  }

  return 1;
}

void
show_usage (void)
{
  fprintf (stderr, "Usage: rc [options]\n");
  fprintf (stderr, "  All options are mandatory, but the program\n");
  fprintf (stderr, "  saves previous used ones. So there is no need\n");
  fprintf (stderr, "  for repetition\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "  You can also modify the values in the program GUI\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "  repo=<directory>    Path of the repository\n");
  fprintf (stderr, "  s=<source fragment> Fragment to be used in all builds\n");
  fprintf (stderr, "                      ex: F473, F480\n");
  fprintf (stderr, "  flgs=<options>      Passed along to test.sh\n");
  fprintf (stderr, "  cduser=<username>   Username for CDE machines\n");
  fprintf (stderr, "  dtuser=<username>   Username for DT machines\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "Previous options are saved in file ~/.rc/release.rc\n");
  fprintf (stderr, "Verbose execution are saved in file ~/.rc/release.log\n");
  fprintf (stderr, "\n");
}

/*
 * GUI functions
 * =============
 */

void
inherit_fields_and_ready (Task *t)
{
  t->user = strdup (t->up[0]->user);
  if (t->type == T_build)
    t->wdir = strdup (t->up[0]->wdir);
  /* We fixed which machine will be used but we still go through
     the host load process */
  t->machine = strdup (t->up[0]->mach);
  t->tarball = safe_strdup (t->up[0]->tarball);
  t->Wtarball = safe_strdup (t->up[0]->Wtarball);
  t->tarname = safe_strdup (t->up[0]->tarname);
  t->Wtarname = safe_strdup (t->up[0]->Wtarname);
  t->tools = safe_strdup (t->up[0]->tools);
  t->status = E_ready;
  rc.task[t->task_num]->color (FL_WHITE);
  rc.window->redraw();
}

void
input_cb (Fl_Input *w, void *data)
{
  char **par = (char **) data;

  *par = strdup (w->value());
}

void
ctrl_quit_cb (Fl_Button *w, void *data)
{
  FILE *fp;
  char buf[1024];

  fl_filename_expand (buf, "$HOME/.rc/release.rc");
  fp = fopen (buf, "w");
  if (fp != NULL) {
    fprintf (fp, "%s\n", REPO);
    fprintf (fp, "%s\n", SRC);
    fprintf (fp, "%s\n", FLGS);
    fprintf (fp, "%s\n", CDEuser);
    fprintf (fp, "%s\n", DTuser);
    fprintf (fp, "%s\n", DESC);
    fclose (fp);
  }
  fclose (flog);

  /* Kill every process in this grup */
  kill (0, SIGKILL);
  exit (0);
}

int
meet_all_requirements_p (Task *t, int status)
{
  int i;

  for (i = 0; i < t->n_reqs; i++) {
    if (t->up[i]->status < status)
      return 0;
  }
  return 1;
}

void
exec_task_cb (Fl_Button *w, void *data)
{
  Run *r = (Run *)data;
  Task *t = r->t;
  LL *l;

  if (w == r->w_reset) {
    reset_one_build (t);
  } else if (w == r->w_retry) {
    if (t->n_reqs != 0) {
      if (meet_all_requirements_p (t, E_done)) {
	l = LL_create_node (t);
	LL_add_node (&L_ready, l);
	inherit_fields_and_ready (t);
      }
    } else {
      retry_one_build (t);
    }
  }

  r->window->~Fl_Double_Window();
  r->~Run();
}

void
ask_task_cb (Fl_Button *w, void *data)
{
  int mbutton = Fl::event_button();
  Task *t = (Task *)data;
  char buf[32];

  if (mbutton == 1) {
    Run *r = new Run(t);
    r->window->position (w->x(), w->y());
    if (t->mach) r->mach->value (t->mach);
    sprintf (buf, "%d", t->status);
    r->status->value (buf);
    sprintf (buf, "%d", t->num_dependencies);
    r->num_dependencies->value (buf);
    r->window->show();
  } else if (mbutton == 3) {
    if (t->status == E_ready) {
      t->status = E_stopped;
      rc.task[t->task_num]->color (FL_DARK1);
      rc.window->redraw();
    } else if (t->status == E_stopped) {
      t->status = E_ready;
      rc.task[t->task_num]->color (FL_WHITE);
      rc.window->redraw();
    } else if (t->status == E_requires) {
      t->status = E_blocked;
      rc.task[t->task_num]->color (FL_DARK1);
      rc.window->redraw();
    } else if (t->status == E_blocked) {
      t->status = E_requires;
      rc.task[t->task_num]->color (FL_BLACK);
      rc.window->redraw();
    }
  }
}

void
ctrl_load_cb (Fl_Button *w, void *data)
{
  FILE *fd;
  char *fn;
  int i, j, tn;
  int cur_task;
  Task *t;
  char buf[1024];

  sprintf (buf, "cd %s/build_gnu && git rev-parse --short HEAD",
	   REPO);
  REV = exec_shell (buf);
  if (REV == NULL) {
    fprintf (stderr, "Could not access repository\n");
    exit (1);
  }

  n_sets = 0;
  n_maps = 0;
  n_tasks = 0;
  fn = cat3 (REPO, "/build_gnu/rc/", DESC);
  fd = fopen (fn, "r");
  if (fd == NULL) {
    fprintf (stderr, "Could not open descriptor file: %s\n", fn);
    show_usage ();
    exit (1);
  }
  if (!load_release (fd))
    exit (1);
  fclose (fd);
  dump_tasks ();

  /* Distribute build to the buttons */
  // FL_WHITE Ready to start
  // FL_BLACK Blocked waiting
  // FL_YELLOW running
  // FL_GREEN Finished ok
  // FL_RED Finished w/ error
  L_ready = NULL;
  L_running = NULL;
  L_requires = NULL;
  for (i = 0; i < NUM_TASK_BUTTONS; i++)
    rc.task[i]->hide();
  memset (run_to_task, 0, sizeof(run_to_task));

  cur_task = 0;
  for (i = 0; i < n_tasks; i++) {
    t = tasks[i];
    if (t->n_reqs == 0) {
      /* This build is imediately ready to execute */
      t->status = E_ready;
      t->task_num = cur_task;
      rc.task[cur_task]->set_visible();
      rc.task[cur_task]->color (FL_WHITE);
      run_to_task[cur_task] = t;

      /* Create tool tip */
      sprintf (buf, "%s\n%s\nrequires: --", t->job, t->log);
      rc.task[cur_task]->tooltip (strdup(buf));

      /* Install call back */
      rc.task[cur_task]->callback((Fl_Callback *)ask_task_cb, (void *)t);

      cur_task += COLUMN_TASK_BUTTONS;
    } else {
      t->status = E_requires;
      /* Get task that this depends on */
      for (j = 0; j < t->n_reqs; j++) {
	tn = t->up[j]->task_num;
	t->up[j]->num_dependencies++;
      }
      /* Find next available button */
      for (j = tn+1; (j % COLUMN_TASK_BUTTONS) != 0; j++)
	if (run_to_task[j] == NULL)
	  break;
      t->task_num = j;
      rc.task[j]->set_visible();
      rc.task[j]->color (FL_BLACK);
      run_to_task[j] = t;     

      /* Create tool tip */
      sprintf (buf, "%s\n%s\nrequires: %s", t->job, t->log, t->requires[0]);
      rc.task[j]->tooltip (strdup(buf));

      /* Install call back */
      rc.task[j]->callback((Fl_Callback *)ask_task_cb, (void *)t);
    }
  }
  rc.ctrl_auto->activate();
  rc.window->redraw();
}

int bkg_time = 0;
int tick_cnt = 0;
int sec_cnt = 0;
int in_progress = 0;
struct timeval t_start;
struct timeval t_now;

void
ctrl_auto_cb (Fl_Button *w, void *data)
{
  int i;
  Task *t;
  LL *l;

  for (i = 0; i < n_tasks; i++) {
    t = tasks[i];
    l = LL_create_node (t);
    if (t->n_reqs == 0) {
      LL_add_node (&L_ready, l);
    } else {
      LL_add_node (&L_requires, l);
    }
  }
  rc.ctrl_load->deactivate();
  rc.ctrl_auto->deactivate();
  in_progress = 1;
  rc.window->redraw();
}

void
main_bkg (void *data)
{
  LL *l, *lnext;
  Task *t;
  int j;
  char clk_buf[32];
  int minutes;

  Fl::repeat_timeout(1.0, main_bkg);
  sec_cnt++;
  if (sec_cnt == 60) {
    sec_cnt = 0;
    if (in_progress) {
      gettimeofday (&t_now, NULL);
      minutes = (t_now.tv_sec - t_start.tv_sec) / 60;
      sprintf (clk_buf, "%02d:%02d", minutes / 60, minutes % 60);
      rc.release_clock->value (clk_buf);
      rc.window->redraw();
      if (L_ready == NULL
	  && L_running == NULL
	  && L_requires == NULL) {
	in_progress = 0;
      }
    }
  }
  test_locked_tmo ();

  l = L_running;
  while (l) {
    lnext = l->next;
    t = l->task;
    if (t->status != E_running) {
      LL_remove_node (&L_running, l);
      if (t->status == E_done)
	rc.task[t->task_num]->color (FL_GREEN);
      else
	rc.task[t->task_num]->color (FL_RED);
      rc.window->redraw();
      Fl::wait();
      for (j = 0; j < t->n_reqs; j++)
	t->up[j]->num_dependencies--;
      if (t->type == T_build) {
	if (t->n_reqs == 0) {
	  if (t->num_dependencies == 0)
	    reset_one_build (t);
	} else {
	  reset_one_build (t->up[0]);
	}
      }
    }
    l = lnext;
  }

  l = L_requires;
  while (l) {
    lnext = l->next;
    t = l->task;
    if (t->status != E_requires) {
      l = lnext;
      continue;
    }
    if (meet_all_requirements_p (t, E_done)
	|| (t->type == T_test && t->n_reqs > 1
	    && meet_all_requirements_p (t, E_error))) {
      LL_remove_node (&L_requires, l);
      t->status = E_ready;
      LL_add_node (&L_ready, l);
      inherit_fields_and_ready (t);
      if (t->type == T_test) {
	/* One extra expansion of local "tarname", "tools" */
	for (j = 0; j < t->n_execs; j++) {
	  t->execs[j] = expand_local (t->execs[j], t);
	}
      }
    }
    l = lnext;
  }
  fflush (flog);

  if (bkg_time != 0) {
    bkg_time --;
    if (bkg_time != 0)
      return;
  }

  bkg_time = 30;
  tick_cnt++;
  fprintf (flog, "[main_bkg]              TICK %d\n", tick_cnt);

  /*
    For each L_ready
      try to get a host
      if (found) start a job, move node to L_running, change color
                 wait 30 sec (return)

    for each L_runnig
      Still running ? No, change color, remove from list
          if nobody depends on it do reset_one_build (b);

    for each L_requires
      if parent completed,
        clone some fields from parent
        move to L_ready, change color
  */
  l = L_ready;
  while (l) {
    lnext = l->next;
    t = l->task;
    if (t->status != E_ready) {
      l = lnext;
      continue;
    }
    if (t->priority < 1) {
      t->mach = pick_best_host (t->user, t->machine);
      if (t->mach != NULL) {
	LL_remove_node (&L_ready, l);
	t->status = E_running;
	LL_add_node (&L_running, l);
	rc.task[t->task_num]->color (FL_YELLOW);
	rc.window->redraw();
	Fl::wait();
	create_task_thread (t);
	fflush (flog);
      }
    }
    l = lnext;
  }
  l = L_ready;
  while (l) {
    lnext = l->next;
    t = l->task;
    if (t->status != E_ready) {
      l = lnext;
      continue;
    }
    if (t->priority <= 1) {
      t->mach = pick_best_host (t->user, t->machine);
      if (t->mach != NULL) {
	LL_remove_node (&L_ready, l);
	t->status = E_running;
	LL_add_node (&L_running, l);
	rc.task[t->task_num]->color (FL_YELLOW);
	rc.window->redraw();
	Fl::wait();
	create_task_thread (t);
	fflush (flog);
      }
    }
    l = lnext;
  }
  l = L_ready;
  while (l) {
    lnext = l->next;
    t = l->task;
    if (t->status != E_ready) {
      l = lnext;
      continue;
    }
    t->mach = pick_best_host (t->user, t->machine);
    if (t->mach != NULL) {
      LL_remove_node (&L_ready, l);
      t->status = E_running;
      LL_add_node (&L_running, l);
      rc.task[t->task_num]->color (FL_YELLOW);
      rc.window->redraw();
      Fl::wait();
      create_task_thread (t);
      fflush (flog);
    }
    l = lnext;
  }
}

/*
 * Main program
 * ============
 */

int
main (int argc, char **argv)
{
  int i;
  FILE *fp;
  char buf[1024];

  fl_filename_expand (buf, "$HOME/.rc/release.log");
  flog = fopen (buf, "w");
  if (flog == NULL) {
    fprintf (stderr, "Could not open log file\n");
    exit (1);
  }

  fl_filename_expand (buf, "$HOME/.rc/release.rc");
  fp = fopen (buf, "r");
  if (fp != NULL) {
    /* Preload parameters from last usage */
    fgets (buf, 256, fp);
    buf[strlen(buf)-1] = '\0';
    REPO = strdup (buf);

    fgets (buf, 256, fp);
    buf[strlen(buf)-1] = '\0';
    SRC = strdup (buf);

    fgets (buf, 256, fp);
    buf[strlen(buf)-1] = '\0';
    FLGS = strdup (buf);

    fgets (buf, 256, fp);
    buf[strlen(buf)-1] = '\0';
    CDEuser = strdup (buf);

    fgets (buf, 256, fp);
    buf[strlen(buf)-1] = '\0';
    DTuser = strdup (buf);

    fgets (buf, 256, fp);
    buf[strlen(buf)-1] = '\0';
    DESC = strdup (buf);

    fclose (fp);
  }

  for (i = 1; i < argc; i++) {
    if (strncmp (argv[i], "repo=", 5) == 0) {
      REPO = strdup (argv[i]+5);
    } else if (strncmp (argv[i], "s=", 2) == 0) {
      SRC = strdup (argv[i]+2);
    } else if (strncmp (argv[i], "flgs=", 5) == 0) {
      FLGS = strdup (argv[i]+5);
    } else if (strncmp (argv[i], "cdeuser=", 8) == 0) {
      CDEuser = strdup (argv[i]+8);
    } else if (strncmp (argv[i], "dtuser=", 7) == 0) {
      DTuser = strdup (argv[i]+7);
    } else if (strncmp (argv[i], "desc=", 5) == 0) {
      DESC = strdup (argv[i]+5);
    } else {
      fprintf (stderr, "Unknow command option: %s\n", argv[i]);
      show_usage ();
      return 1;
    }
  }

  if (REPO == NULL) {
    fprintf (stderr, "Must provide repository directory\n");
    show_usage ();
    return 1;
  }
  if (SRC == NULL) {
    fprintf (stderr, "Must provide source fragment\n");
    show_usage ();
    return 1;
  }
  if (FLGS == NULL) {
    fprintf (stderr, "Must provide optimization flags\n");
    show_usage ();
    return 1;
  }
  if (CDEuser == NULL) {
    fprintf (stderr, "Must provide CDE login name\n");
    show_usage ();
    return 1;
  }
  if (DTuser == NULL) {
    fprintf (stderr, "Must provide DT login name\n");
    show_usage ();
    return 1;
  }
  if (DESC == NULL) {
    fprintf (stderr, "Must provide description of release process\n");
    show_usage ();
    return 1;
  }

  /* GUI setup */
  rc.window->show();
  rc.w_repo->value (REPO);
  rc.w_src->value (SRC);
  rc.w_flgs->value (FLGS);
  rc.w_CDEuser->value (CDEuser);
  rc.w_DTuser->value (DTuser);
  rc.w_desc->value (DESC);
  Fl::add_timeout(1.0, main_bkg);
  rc.ctrl_auto->deactivate();
  gettimeofday (&t_start, NULL);
  sec_cnt = 59;

  while (Fl::wait()) {
  }
  return 0;
}
