#!/bin/bash

# Default parameters
USR=`id -nu`
BD=/local/$USR/tools_test
HN=localhost
HU=$USR
TN=localhost
TU=$USR
SIM="none"
BT=""
CT=""
PT=""
FLGS="-O3"
BOOTDIR=/proj/ppc/DT/labhome/benchmarks
TDY=`date +%Y%m%d`

# Other definitions
all_PT="spec2k spec2k6 coremark auto-1 consumer-1 net-1 telecom-1 office-1 consumer-2 net-2 office-2 dhrystone nbench"
all_CT="$all_PT whetstone stanford linpack nas perfect smalltests"
all_BT="$all_CT"
all_ML="Snof S32 S64 nof 32 64"
all_NML="NSnof NS32 NS64 Nnof N32 N64"
all_XML="Xnof X32 X64 XSnof XS32 XS64"
all_SML="vle vleLCF"
all_NT="linuxtst"
all_alternate="spec2k6int spec2k6fp"

report () {
    # Arguments are:
    # Function where error happend
    local fnc=$1
    # Error message
    local msg=$2

    echo "[$fnc] $msg"
    exit 1
}

check_arguments () {
    # Function to check presence of minimal command line options present
    if [ "$TAR" = "" -a "$HTOOLS" = "" ]; then
	report check_arguments "Either \"tar\" or \"Htools\" must be given"
    fi
    if [ "$TAR" != "" ]; then
	if [ ! -e $TAR ]; then report check_arguments "Could not open $TAR"; fi
	# Installation method. Assume tar ball will be installed/removed
	DOINSTALL="Y"
	[ "$HTOOLS" = "" ] && HTOOLS=$BD
	[ "$TLIBS" = "" ] && TLIBS=$HTOOLS
    fi

    if [ "$TOOLS" = "" ]; then report check_arguments "mandatory \"tools\" not present"; fi
    if [ "$CORE" = "" ]; then report check_arguments "mandatory \"core\" not present"; fi
    if [ "$ML" = "" ]; then report check_arguments "mandatory \"ml\" not present"; fi

    echo "Parameters used in this run:"
    echo "    bd      ${BD}${TREE}"
    echo "    hn      $HN"
    echo "    hu      $HU"
    echo "    tn      $TN"
    echo "    tu      $TU"
    echo "    sim     $SIM"
    echo ""
    echo "    tar     $TAR"
    echo "    tools   $TOOLS"
    echo "    prefix  \"$PREFIX\""
    echo "    core    \"$CORE\""
    echo ""
    echo "    ml      \"$ML\""
    echo "    bt      \"$BT\""
    echo "    ct      \"$CT\""
    echo "    pt      \"$PT\""
    echo ""
    echo "    flgs    \"$FLGS\""
    echo ""
}

copy_tar_host () {
    local btar gccfile specdir

    btar=`basename $TAR`
    echo -n "Installing $btar on host     "
    scp -q $TAR $HU@$HN:$BD/$btar
    if [ $? -ne 0 ]; then report copy_tar_host "Error copying $btar"
                     else echo "..OK"; fi

    echo -n "Expanding $btar on host      "
    ssh -l $HU $HN "true && cd $HTOOLS && tar xjf $BD/$btar"
    if [ $? -ne 0 ]; then report copy_tar_host "Error expanding $btar"
                     else echo "..OK"; fi

    echo "Removing specs file"
    gccfile=`ssh -l $HU $HN "true && $HTOOLS/$TOOLS/bin/${PREFIX}gcc -print-libgcc-file-name"`
    specdir=`dirname $gccfile`
    ssh -l $HU $HN "true && rm -f $specdir/specs"
}

copy_tar_target () {
    local btar

    btar=`basename $TAR`
    echo -n "Installing $btar on target   "
    scp -q $TAR $TU@$TN:$BD/$btar
    if [ $? -ne 0 ]; then report copy_tar_target "Error copying $btar"
                     else echo "..OK"; fi

    echo -n "Expanding $btar on target    "
    ssh -l $TU $TN "true && cd $TLIBS && tar xjf $BD/$btar"
    if [ $? -ne 0 ]; then report copy_tar_target "Error expanding $btar"
                     else echo "..OK"; fi
}

rem_tar_host () {
    local btar
    local dtools

    btar=`basename $TAR`
    dtools=`dirname $TOOLS`
    if [ "$dtools" = "." ]; then dtools=$TOOLS; fi
    echo -n "Removing $btar on host       "
    ssh -l $HU $HN "true && cd $BD && rm -rf $btar $HTOOLS/$dtools"
    if [ $? -ne 0 ]; then report rem_tar_host "Error removing $btar"
                     else echo "..OK"; fi
}

rem_tar_target () {
    local btar dtools

    btar=`basename $TAR`
    dtools=`dirname $TOOLS`
    if [ "$dtools" = "." ]; then dtools=$TOOLS; fi
    echo -n "Removing $btar on target     "
    ssh -l $TU $TN "true && cd $BD && rm -rf $btar $TLIBS/$dtools"
    if [ $? -ne 0 ]; then report rem_tar_target "Error removing $btar"
                     else echo "..OK"; fi
}

# Get fragments
TSTscript=`readlink -e $0`
TSTdir=`dirname $TSTscript`

scriptdir=`dirname $0`
hnfull=`hostname`
hn=`basename $hnfull .am.freescale.net`
source $scriptdir/build.env-$hn

# Parse command line
for a in "$@"; do
    case $a in
	-h)
	    cat <<EOF
Usage is:
    test.sh [<options>]

Where <options> is any of:
    -h                Print this help and exit
    bd=<dir>          base dir. Default is /local/`id -nu`/tools_test
    tree=<string>     A string do append to base dir name
    -keep             Keep temporary files and directories

    hn=<name>         Host machine name, default is localhost
    hu=<name>         Username for Host, default is `id -nu`
    tn=<name>         Target machine name, default is localhost
    tu=<user>         Username for Target, default is `id -nu`
    sim=<simulator>   Simulator program/script to run test

    install_host      Install all required directories on host machine
    install_target    Install all required directories on target machine
    boot_target       Install all required directories from tarballs

    tar=<filename>    tarball filename
    Htools=<path>     Tools installation dir on host
    Tlibs=<path>      Tools installation dir on target

Either tar or Htools must be given.
If tar is given, tools will be installed and them removed
If Htools is given, will assume tools are pre-installed. Optional Tlibs
can be used if directory on target is different
If both given, it will work as "tar" but the given installation dirs

    tools=<path>      Installation path for given tools
    prefix=<triplet>  Prefix for given tools
    core=<core id>    Core or cpu identification (eg: e300c2)
    ml=<type>         List of multilib types (32, 64, nof, S32, S64 Snof)
                      or native tools: N32, N64, Nnof, NS32, NS64 NSnof

    bt=<list>         List of tests to check build
    ct=<list>         List of tests to check build/correctness
    pt=<list>         List of tests to check build/correctness/performance
    x=<rep>           Repetitions for job server submission
    tag=<tag>         Tag to recotd job server results

    flgs=<list>       List of flags to add to every compilatiom
                        Default is "-O3"

And <list> is one or more of:
    spec2k          coremark	    dhrystone
    spec2k6	    auto-1  	    whetstone
       	            consumer-1      smalltests
    	            net-1  	    nbench
      	            telecom-1 	    stanford
		    office-1        nas
		    consumer-2      linpack
		    net-2           perfect
		    office-2
EOF
	    exit 0
	    ;;
	-keep)
	    KEEP="Y"
	    ;;
	bd=*)
	    BD=${a:3}
	    ;;
	tree=*)
	    TREE=${a:5}
	    ;;
	hn=*)
	    HN=${a:3}
	    ;;
	hu=*)
	    HU=${a:3}
	    ;;
	tn=*)
	    TN=${a:3}
	    ;;
	tu=*)
	    TU=${a:3}
	    ;;
	sim=*)
	    SIM=${a:4}
	    ;;
	tar=*)
	    TAR=${a:4}
	    ;;
	itar=*)
	    TAR=${a:5}
	    T_INSTALL="Y"
	    ;;
	etar=*)
	    TAR=${a:5}
	    T_EXEC="Y"
	    ;;
	rtar=*)
	    TAR=${a:5}
	    T_REMOVE="Y"
	    ;;
	Htools=*)
	    HTOOLS=${a:7}
	    [ "$TLIBS" = "" ] && TLIBS=$HTOOLS
	    ;;
	Tlibs=*)
	    TLIBS=${a:6}
	    ;;
	tools=*)
	    TOOLS=${a:6}
	    ;;
	prefix=*)
	    PREFIX=${a:7}
	    ;;
	core=*)
	    CORE=${a:5}
	    ;;
	x=*)
	    rep=${a:2}
	    ;;
	tag=*)
	    tag=${a:4}
	    ;;
	ml=*)
	    ML=${a:3}
	    if [ "$ML" = "all" ]; then
		ML=$all_ML
	    elif [ "$ML" = "Nall" ]; then
		ML=$all_NML
	    else
		for t in $ML; do
		    x=`echo $all_ML $all_NML $all_XML $all_SML | grep $t`
		    if [ "$x" = "" ]; then report main "Invalid multilib type $t"; fi
		done
	    fi
	    ;;
	bt=*)
	    BT=${a:3}
	    if [ "$BT" = "all" ]; then
		BT=$all_BT
	    else
		for t in $BT; do
		    BMK=`echo $t | awk -F "." '{print $1}'`
		    SUB=`echo $t | awk -F "." '{print $2}'`
		    x=`echo $all_BT $all_NT $all_alternate | grep $BMK`
		    if [ "$x" = "" ]; then report main "Invalid test name $t"; fi
		done
	    fi
	    ;;
	ct=*)
	    CT=${a:3}
	    if [ "$CT" = "all" ]; then
		CT=$all_CT
	    else
		for t in $CT; do
		    BMK=`echo $t | awk -F "." '{print $1}'`
		    SUB=`echo $t | awk -F "." '{print $2}'`
		    x=`echo $all_CT $all_NT $all_alternate | grep $BMK`
		    if [ "$x" = "" ]; then report main "Invalid test name $t"; fi
		done
	    fi
	    ;;
	pt=*)
	    PT=${a:3}
	    if [ "$PT" = "all" ]; then
		PT=$all_PT
	    else
		for t in $PT; do
		    BMK=`echo $t | awk -F "." '{print $1}'`
		    SUB=`echo $t | awk -F "." '{print $2}'`
		    x=`echo $all_PT $all_alternate | grep $BMK`
		    if [ "$x" = "" ]; then report main "Invalid test name $t"; fi
		done
	    fi
	    ;;
	flgs=*)
	    FLGS=${a:5}
	    ;;
	install_host)
	    echo "Installing required directories for host use on $HN"
	    ssh -l $HU $HN "true && rm -rf ${BD}${TREE} && mkdir -p ${BD}${TREE}/tmp"
	    scp -qrp $TSTdir/f-test $HU@$HN:${BD}${TREE}/f-test
	    for b in spec2k spec2k6 dhrystone eembc linpack whetstone \
	             smalltests nas stanford nbench perfect; do
		echo "   ... installing $b"
                ssh -l $HU $HN "true && test -L $HOME/bin/$b.sh"
                if [ $? -ne 0 ]; then
                    echo "Cannot find $b.sh, check $HOME/bin/$b.sh soft link to benchmark directory"
                    exit 1
                fi
		ssh -l $HU $HN "true && mkdir -p ${BD}${TREE}/installed_$b && cd ${BD}${TREE}/installed_$b && $b.sh install_src"
	    done
	    ssh -l $HU $HN "true && cd ${BD}${TREE}/installed_spec2k && rm -rf tools"
	    ssh -l $HU $HN "true && cd ${BD}${TREE}/installed_spec2k6 && rm -rf tools redistributable_sources"
	    exit 0
	    ;;
	install_target)
	    echo "Installing required directories for target use on $TN"
	    ssh -l $TU $TN "true && rm -rf ${BD}${TREE} && mkdir -p ${BD}${TREE}/tmp"
	    scp -qrp $TSTdir/f-test $TU@$TN:${BD}${TREE}/f-test
	    for b in spec2k spec2k6 dhrystone eembc linpack whetstone \
	             smalltests nas stanford nbench perfect; do
		echo "   ... installing $b"
                ssh -l $TU $TN "true && test -L $HOME/bin/$b.sh"
                if [ $? -ne 0 ]; then
                    echo "cannot find $b.sh, check $HOME/bin/$b.sh soft link to benchmark directory"
                    exit 1
                fi
		ssh -l $TU $TN "true && mkdir -p ${BD}${TREE}/installed_$b && cd ${BD}${TREE}/installed_$b && $b.sh install_src"
	    done
	    ssh -l $TU $TN "true && cd ${BD}${TREE}/installed_spec2k && rm -rf tools"
	    ssh -l $TU $TN "true && cd ${BD}${TREE}/installed_spec2k6 && rm -rf tools redistributable_sources"
	    for b in spec2k spec2k6; do
		echo "   ... installing $b"
		ssh -l $TU $TN "true && mkdir -p ${BD}${TREE}/exec_$b && cd ${BD}${TREE}/exec_$b && $b.sh install_src"
	    done
	    ssh -l $TU $TN "true && cd ${BD}${TREE}/exec_spec2k && rm -rf tools"
	    ssh -l $TU $TN "true && cd ${BD}${TREE}/exec_spec2k6 && rm -rf tools redistributable_sources"
	    exit 0
	    ;;
	save_target)
	    mkdir -p $BOOTDIR/$USR/
	    ssh -l $TU $TN "true && rm -f $BOOTDIR/$USR/t[12345].tar.*"
	    echo "   ...Saving new tarballs for target boot t1"
	    ssh -l $TU $TN "true && cd ${BD}${TREE} && tar cjf $BOOTDIR/$USR/${TREE}/t1.tar.bz2 installed_[delnpw]* installed_smalltests installed_stanford"
	    echo "   ...Saving new tarballs for target boot t2"
	    ssh -l $TU $TN "true && cd ${BD}${TREE} && tar cjf $BOOTDIR/$USR/${TREE}/t2.tar.bz2 installed_spec2k"
	    echo "   ...Saving new tarballs for target boot t3"
	    ssh -l $TU $TN "true && cd ${BD}${TREE} && tar cjf $BOOTDIR/$USR/${TREE}/t3.tar.bz2 installed_spec2k6"
	    echo "   ...Saving new tarballs for target boot t4"
	    ssh -l $TU $TN "true && cd ${BD}${TREE} && tar cjf $BOOTDIR/$USR/${TREE}/t4.tar.bz2 exec_spec2k"
	    echo "   ...Saving new tarballs for target boot t5"
	    ssh -l $TU $TN "true && cd ${BD}${TREE} && tar cjf $BOOTDIR/$USR/${TREE}/t5.tar.bz2 exec_spec2k6"
	    exit 0
	    ;;
	boot_target)
	    echo "Installing required directories for target use on $TN"
	    ssh -l $TU $TN "true && rm -rf ${BD}${TREE} && mkdir -p ${BD}${TREE}/tmp"
	    scp -qrp $TSTdir/f-test $TU@$TN:${BD}${TREE}/f-test
	    scp -q $BOOTDIR/$USR/${TREE}/t?.tar.bz2 $TU@$TN:${BD}${TREE}/f-test
	    for f in t1 t2 t3 t4 t5; do
		echo "   ... Expanding $f"
		ssh -l $TU $TN "true && cd ${BD}${TREE} && tar xjf ./f-test/$f.tar.bz2"
	    done
	    exit 0
	    ;;
	*)
	    echo "Unknown option $a"
	    exit 1
	    ;;
    esac
done

check_arguments
# Special case where I want to install tarballs only or remove only
if [ "$T_INSTALL" = "Y" ]; then
    [ "$DOINSTALL" = "Y" ] && copy_tar_host
    [ "$DOINSTALL" = "Y" -a "$TN" != "$HN" ] && copy_tar_target
    exit 0
fi
if [ "$T_REMOVE" = "Y" ]; then
    [ "$DOINSTALL" = "Y" -a "$TN" != "$HN" ] && rem_tar_target
    [ "$DOINSTALL" = "Y" ] && rem_tar_host
    exit 0
fi

[ "$T_EXEC" = "" -a "$DOINSTALL" = "Y" ] && copy_tar_host
[ "$T_EXEC" = "" -a "$DOINSTALL" = "Y" -a "$TN" != "$HN" ] && copy_tar_target

# Create an uniq id to collect test results
UniqueID=`echo $RANDOM`

for m in $ML; do
    echo "For ML: $m"
    ssh -l $HU $HN "echo CORE=$CORE.$m >> ${BD}${TREE}/${UniqueID}.host_log"

    for b in $BT; do
	echo -n "Building $b on host       "
	BMK=`echo $b | awk -F "." '{print $1}'`
	SUB=`echo $b | awk -F "." '{print $2}'`
	[ "$SUB" = "" ] && SUB=all
	ssh -l $HU $HN "TGURU_C=\"$TGURU_C\" TGURU_CPP=\"$TGURU_CPP\" TGURU_F=\"$TGURU_F\" TGURU_LD=\"$TGURU_LD\" TGURU_AS=\"$TGURU_AS\" TGURU_AR=\"$TGURU_AR\" ${BD}${TREE}/f-test/${BMK}_build.sh $m ${BD}${TREE} $HTOOLS $TLIBS $TOOLS \"$PREFIX\" $UniqueID $SUB \"$FLGS\""
    done

    for b in $CT; do
	echo -n "Building $b on host       "
	BMK=`echo $b | awk -F "." '{print $1}'`
	SUB=`echo $b | awk -F "." '{print $2}'`
	[ "$SUB" = "" ] && SUB=all
	ssh -l $HU $HN "TGURU_C=\"$TGURU_C\" TGURU_CPP=\"$TGURU_CPP\" TGURU_F=\"$TGURU_F\" TGURU_LD=\"$TGURU_LD\" TGURU_AS=\"$TGURU_AS\" TGURU_AR=\"$TGURU_AR\" ${BD}${TREE}/f-test/${BMK}_build.sh $m ${BD}${TREE} $HTOOLS $TLIBS $TOOLS \"$PREFIX\" $UniqueID $SUB \"$FLGS\""
	[ "$BMK" = "spec2k" -o "$BMK" = "spec2k6" -o "$BMK" = "spec2k6int" -o "$BMK" = "spec2k6fp" -o "$BMK" = "stanford" -o "$BMK" = "nas" -o "$BMK" = "nbench" ] \
	    && [ "$m" = "nof" -o "$m" = "Snof" -o "$m" = "Nnof" -o "$m" = "NSnof" ] && echo "Skipping execution of $b" && continue
	echo -n "Copying $b to target      "
	ssh -l $HU $HN "${BD}${TREE}/f-test/${BMK}_copy.sh ${BD}${TREE} $TU $TN $SUB"
	echo -n "Excuting $b on target     "
	ssh -l $TU $TN "${BD}${TREE}/f-test/${BMK}_exec.sh $CORE.$m ${BD}${TREE} $TU $TN $UniqueID $SUB $SIM"
	echo -n "Validating $b on target   "
	ssh -l $TU $TN "${BD}${TREE}/f-test/${BMK}_validate.sh ${BD}${TREE} $TU $TN $UniqueID $SUB $KEEP "
    done
    
    for b in $PT; do
	echo -n "Building $b on host       "
	BMK=`echo $b | awk -F "." '{print $1}'`
	SUB=`echo $b | awk -F "." '{print $2}'`
	[ "$SUB" = "" ] && SUB=all
	ssh -l $HU $HN "TGURU_C=\"$TGURU_C\" TGURU_CPP=\"$TGURU_CPP\" TGURU_F=\"$TGURU_F\" TGURU_LD=\"$TGURU_LD\" TGURU_AS=\"$TGURU_AS\" TGURU_AR=\"$TGURU_AR\" ${BD}${TREE}/f-test/${BMK}_build.sh $m ${BD}${TREE} $HTOOLS $TLIBS $TOOLS \"$PREFIX\" $UniqueID $SUB \"$FLGS\""
	echo -n "Submitting $b             "
	if [ "$tag" = "" ]; then tag2=${BMK}_${TDY}; else tag2=$tag; fi
	if [ "$rep" = "" ]; then rep=3; fi
	ssh -l $HU $HN "${BD}${TREE}/f-test/${BMK}_js.sh $CORE.$m ${BD}${TREE} $TN $tag2 $rep $SUB \"$FLGS\" $KEEP"
	echo "OK"
    done
done

ssh -l $HU $HN "cat ${BD}${TREE}/${UniqueID}.host_log && rm -f ${BD}${TREE}/${UniqueID}.host_log" >> test.host_log
[ "$CT" != "" ] && ssh -l $TU $TN "[ -e ${BD}${TREE}/${UniqueID}.target_log ] && cat ${BD}${TREE}/${UniqueID}.target_log && rm -f ${BD}${TREE}/${UniqueID}.target_log" >> test.target_log

[ "$T_EXEC" = "" -a "$DOINSTALL" = "Y" -a "$TN" != "$HN" ] && rem_tar_target
[ "$T_EXEC" = "" -a "$DOINSTALL" = "Y" ] && rem_tar_host
