#!/bin/bash
# V1 - initial release
# V2 - All descriptors with the same hn are grouped and executed in serial 
#      Removed silly <desc>:s attribute stuff
#      It's also quite impractical to install the tests in each /local/<user>/<desc> 
#      directory. For a given "hn", the default "bd" setting is /local/<user>/tools_test.
#      All descriptors with the same "hn" share this default "bd" setting, if "bd"
#      is not explicitly in the descriptor. Since all same "hn" desc are executed serially, this is
#      fine to share tools_test.
# V3 - Added -r <cnt>
#
# test_dispatch.sh - test.sh wrapper 
#  Usage: test_dispatch.sh
#          <descriptor>  [overrides] ... <descriptor> 
#          [-v] verbose mode 
#          [-n] generate commands but don't execute
#          [-s] dispatch all jobs serially
#          [-r <cnt>] repeat these descriptors <cnt> times
#          [-h] print help
#
# test_dispatch.sh is a wrapper script for the test.sh script. It provides
# pre-defined values for test.sh parameters in descriptor files  and creates
# test.sh command to be executed (dispatched) either serially or in parallel
# with other jobs. The descriptor files are found in the f-bmk directory.
#
# Test_dispatch.sh also provides a -i flag that scripts the test.sh install_host,
# install_target, save_target, boot_target commands to ease the process of
# installing tests on various targets.
#
# Furthermore, test_dispatch.sh provides a means to override the 
# descriptor-defined parameters on the command line. This provides 
# the flexibility to run the same descriptor with
# different parameters (if the descriptors are executed serially, more on that
# later). For example, the same compilers using different flags.
#
# Multiple descriptors may be submitted to test_dispatch.sh.
# The descriptors are grouped by their "hn" settings. Each group
# contains descriptors with the same "hn" setting. Each group is
# executed in parallel and each descriptor in each group is executed in serial.
# Note it's a problem if the "tn" setting in a group is used by another
# group. This would be a problem since the tests would try to run on the same
# tn 'tools_test' directory concurrently. No check is made for this case.
#
# The global -s flag will execute the groups serially, that is, all descriptors
# will execute in order, each waiting for the previous to complete.
#
# Generally, test_dispatch.sh should not be issued multiple times in the
# same directory if the previous command is still executing and if the same 
# descriptor is used in each invocation.
#
# The descriptor files are located in the f-bmk directory. The parameters 
# are saved in an associative array, i.e. keyed by name. See any of the
# files in the f-bmk directory for an example.
#
# USING test_dispatch.sh
#
# In its simplest invocation, test_dispatch.sh can be called with a single
# descriptor file:
#
# test_dispatch.sh <desc> 
#
# The script will read the descriptor file (found in f-bmk dir) and generate
# and execute the test.sh script with the parameters supplied from the 
# descriptor file. <desc> may be specified as many times as needed to generate
# performance/test data for each descriptor file. Null parameters (="") in the
# descriptor file are excluded from the generated test.sh command and 
# thus use the test.sh defaults.
#
# Each parameter in the descriptor file can be modified for a specific descriptor
# For example: 'test_dispatch.sh  dt-4.6.1-e5500 hn=perf3farm17' will change
# the "hn" parameter defined in dt-4.6.1-e5500 to perf3farm17 instead of the 
# descriptor's value of perf3farm19. The modified parameter applies only to the
# most recent descriptor. 
#
# This makes the following command possible:
#
# test_dispatch.sh  dt-4.6.1-e5500 flgs="-O3" dt-4.6.1-e5500 flgs="-O1"
#
# The command above will run dt-4.6.1-e5500 with flags=-O3 and run a case
# with the build commands flags set to -O1. The tests are executed serially since
# they are in the same group (same "hn" setting). 
#
# Each invocation of test_dispatch.sh creates a log file whose name is of the format:
#
# test_dispatch.sh.<desc>.log 
#
# in the directory where the script was invoked.
#
# The script executes each test.sh in a unique directory, './<descriptor>'.
# The data files created by test.sh (test.host_log, test.target_log) are saved in the
# unique ./<descriptor> directories. Note that if the same descriptor is specified
# it is serially executed and the results are appended to test.host_log and test.target_log
# in the ./<descriptor> directory. 
#
# OPTIONS
#
# The -n option generates the test.sh command but does not execute the command.
# This is normally useful when used with the -v (verbose) option to see the
# command to be executed.
#
# The -s option forces all groups to be executed serially. 
#
# The -i option installs the tests on the host and target. See INSTALLING
# THE TESTS below. 
#
# The -r <cnt> option repeats the descriptors <cnt> times. For example,
# test_dispatch.sh -r 2 <desc> <desc2> is equivalent to test_dispatch.sh <desc> <desc2> <desc> <desc2>
#
# SPECIAL PARAMETERS
#
# The parameter 'extopts=...' is unique to test_dispatch.sh, i.e. it does not
# have an equivalent in test.sh. In this case, the 'extopts' parameter is passed 
# directly to test.sh. For example,
# test_dispatch.sh <desc> -extopts="-keep" 
# -keep is passed to the generated test.sh command. 
#
# The parameter 'boot_src' is the host or target name that is used to populate
# a target machine test directory via a boot install. 'boot_src' is unique to
# test_dispatch.sh
#
# The parameter 'label' is unique to test_dispatch.sh and is used to add
# a "<compiler_name>.target_log" field to the test.target_log (test.sh output file)
# file. It would be common for multiple descriptors to share the same label,
# meaning that they represent the same compiler source base. 
#
# DEFAULT PARAMETERS
#
# The 'bd' parameter defines the build directory on both the host and the
# target machines. If this value is null, the default value is "/local/'user'/tools_test" 
# where 'user' the the 'hu' parameter value.
#
# The 'Tlibs' parameter defines the location of the target library. If this
# value is null, the default value becomes "/opt/freescale/'user'/'descriptor'"
# where 'user' is the 'hu' parameter value. 
#
# The "Htools' parameter defines the location of the host tools directory. If this
# value is null, the default value becomes "/local/'user'/'descriptor'"
# where 'user is the 'hu' parameter value.
#
# Other parameters, if null, are not issued the test.sh and test.sh uses its defaults.
#
# INSTALLING THE TESTS 
#
# Installation of the tests requires copying of test sources and f-test scripts
# to the applicable build and run target hosts. The test_dispatch.sh descriptors
# contain enough information to perform the complex installations. 
#
# Multiple descriptors are possible, but each descriptor designate unique build and 
# run target host as to not overwrite a previous install. All installations are
# performed serially for each descriptor. 
#
# The -i parameter installs tests on the build host, the target host, and possibly
# the remote target host. The remote target host does not have NFS access to the 
# build_gnu directory and installation must occurred via copied tarball files.
# For a normal installation, the test files are copied to the build host and target host
# at the build directory (BD) location. If the descriptor file contains an element,
# \"boot_src\" that is non-null, then the installation is performed on the build host,
# and the \"boot_src\" host. The \"boot_src\" host installation is tarball'd and copied
# to the target host (TN) where it is expanded and installed.
# 
# When -i is specified, only the install occurs and the script exits. Then the script
# may be run again to run the required tests. The installation checks for an existing
# directory $BD/f-test. If it finds an existing directory, it skips re-installation.
# 
#

# Extended regex pattern matching option is on
shopt -s extglob

TDY=`date +%Y%m%d`

# version
BMKVERSION=2
# bmk_regression invocation
BMKINVOKE=""
# script name
BMKSCRIPT=""
# Current bmk index
BMKIDX=0
# Array of cmd strings, one for each descriptor
BMKCMDS=()
# Boolean set if -v 
BMKVERBOSE=0
# Boolean set if -n
BMKDRYRUN=0
# Boolean all jobs are serial
BMKSERIAL=0
# Descriptor name by index
BMKDESCNAME=()
# Descriptor label by index
BMKDESCLABEL=()
# Do serial for descriptor
BMKDOSERIAL=0
# Do install only
BMKINSTALL=0
# List of group job issued pids
BMKGROUPJOBS=()
# Repeat count
BMKREPEAT=1

# List of BMKIDXes, keyed by 'hn'
declare -A BMKGROUPS

# from desc file for use as hash array
declare -A BMKDESC

# Current descriptor in process
BMKCURRENTDESC=""
TEST_SH_CMD="test.sh"

# Base Tlibs directory
BMKTLIBSBASEDIR="/opt/freescale"

# handle_override - handle opt=arg options
# input: opt=val opt=val2 ...
# output: the desc array for opt is modified
handle_override () {
    local i=$1

    if [ $BMKVERBOSE -eq 1 ]; then
        echo "Override: $i"
    fi
        case "$i" in 
	bd=*)
	    BMKDESC['bd']=${i:3}
	    ;;
	hn=*)
	    BMKDESC['hn']=${i:3}
	    ;;
	hu=*)
	    BMKDESC['hu']=${i:3}
	    ;;
	tn=*)
	    BMKDESC['tn']=${i:3}
	    ;;
	tu=*)
	    BMKDESC['tu']=${i:3}
	    ;;
	tar=*)
	    BMKDESC['tar']=${i:4}
	    ;;
	Htools=*)
	    BMKDESC['Htools']=${i:7}
	    ;;
	Tlibs=*)
	    BMKDESC['Tlibs']=${i:6}
	    ;;
	tools=*)
	    BMKDESC['tools']=${i:6}
	    ;;
	prefix=*)
	    BMKDESC['prefix']=${i:7}
	    ;;
	core=*)
	    BMKDESC['core']=${i:5}
	    ;;
	ml=*)
	    BMKDESC['ml']=${i:3}
	    ;;
	bt=*)
	    BMKDESC['bt']=${i:3}
	    ;;
	ct=*)
	    BMKDESC['ct']=${i:3}
	    ;;
	pt=*)
	    BMKDESC['pt']=${i:3}
	    ;;
	flgs=*)
	    BMKDESC['flgs']=${i:5}
	    ;;
	x=*)
	    BMKDESC['x']=${i:2}
	    ;;
	tag=*)
	    BMKDESC['tag']=${i:4}
	    ;;
        extopts=*)
            BMKDESC['extopts']=${i:8}
            ;;
        label=*)
            BMKDESC['label']=${i:6}
            ;;
        boot_src=*)
            BMKDESC['boot_src']=${i:9}
            ;;
        # Terminate the override options
        --)
            break
            ;;
        *)
            echo "Unknown option: $i"
            exit 1
            ;;
        esac
}

# Create directory if it doesn't exist
# Args: user host directory
#
create_if_not_exists () {
    local u="$1"
    local h="$2"
    local d="$3"

    ssh -l $u $h "true && mkdir -p $d"
}

process_desc () {

    # Save some descriptor info
    BMKDESCNAME[$BMKIDX]=${BMKCURRENTDESC}
    BMKDESCLABEL[$BMKIDX]=${BMKDESC['label']}
    if [ "${BMKDESCLABEL[$BMKIDX]}" = "" ]; then
        BMKDESCLABEL[$BMKIDX]="default_compiler";
    fi

    # Create the default build directory based on the user name. The
    # descriptor value must be "" for 'bd'. Otherwise, the overridden 'bd'
    # name will be the build directory.
    if [ "${BMKDESC['bd']}" = "" ]; then 
        BMKDESC['bd']="/local/${BMKDESC['hu']}/tools_test"
        # Create if doesn't exist
        create_if_not_exists ${BMKDESC['hu']} ${BMKDESC['hn']} ${BMKDESC['bd']}
	# Skip if using job server
    	if [ "${BMKDESC['pt']}" = "" ]; then 
        	create_if_not_exists ${BMKDESC['tu']} ${BMKDESC['tn']} ${BMKDESC['bd']}
	fi
    fi

    # The Tlibs directory is also base on user/descriptor name
    # as the default if none is specified in the descriptor or the command line.
    # You have to remember to install Tlibs on the target if not using 'tar=...'
    # since 'tar=...' option will install Tlibs but otherwise not installed automatically.
    # Skip if using job server
    if [ "${BMKDESC['pt']}" = "" ]; then 
    	if [ "${BMKDESC['Tlibs']}" = "" ]; then 
        	BMKDESC['Tlibs']="${BMKTLIBSBASEDIR}/${BMKDESC['hu']}/${BMKCURRENTDESC}"
	        create_if_not_exists ${BMKDESC['tu']} ${BMKDESC['tn']} ${BMKDESC['Tlibs']}
    	fi
    fi

    # Similarly, the Htools directory is given a unique user/descriptor name directory
    # unless specified in the descriptor or overridden on the command line.
    if [ "${BMKDESC['Htools']}" = "" ]; then 
        BMKDESC['Htools']="/local/${BMKDESC['hu']}/${BMKCURRENTDESC}"
        create_if_not_exists ${BMKDESC['hu']} ${BMKDESC['hn']} ${BMKDESC['Htools']}
    fi
    

    if [ $BMKINSTALL -eq 1 ]; then
        gen_install
    else
        gen_test
    fi
}

check_setup () {
    if [ "${BMKDESC['boot_src']}" != "" ]; then
        ssh -l ${BMKDESC['tu']} ${BMKDESC['boot_src']} "true && echo $PATH | grep /home/${BMKDESC['tu']}/bin > /dev/null"
        if [ $? -ne 0 ]; then
            echo "/home/${BMKDESC['tu']}/bin not in your path"
            exit 1
        fi
    fi

    ssh -l ${BMKDESC['tu']} ${BMKDESC['tn']} "true && echo $PATH | grep /home/${BMKDESC['tu']}/bin > /dev/null"
    if [ $? -ne 0 ]; then
        echo "/home/${BMKDESC['tu']}/bin not in your path"
        exit 1
    fi

    ssh -l ${BMKDESC['hu']} ${BMKDESC['hn']} "true && echo $PATH | grep /home/${BMKDESC['hu']}/bin > /dev/null"
    if [ $? -ne 0 ]; then
        echo "/home/${BMKDESC['hu']}/bin not in your path"
        exit 1
    fi
}

gen_install () {
    local test_cmd=""

    # Verify user setup
    check_setup 

    # Installs are done serially
    BMKSERIAL=1

    # If not a boot install, do a host install, target install if the tests are not previously
    # installed. We only check for an f-test directory to determine if previously installed.
    # You must wipe the previous sources manually for a clean install
    if [ "${BMKDESC['boot_src']}" = "" ]; then
        ssh -l ${BMKDESC['hu']} ${BMKDESC['hn']} "true && test -d ${BMKDESC['bd']}/f-test"
        # install if not exists previously
        if [ $? -ne 0 ]; then
            test_cmd="${TEST_SH_CMD} hu=${BMKDESC['hu']} hn=${BMKDESC['hn']} bd=${BMKDESC['bd']} install_host;"
        fi
        ssh -l ${BMKDESC['tu']} ${BMKDESC['tn']} "true && test -d ${BMKDESC['bd']}/f-test"
        # install if not exists previously
        if [ $? -ne 0 ]; then
            test_cmd="$test_cmd ${TEST_SH_CMD} tu=${BMKDESC['tu']} tn=${BMKDESC['tn']} bd=${BMKDESC['bd']} install_target;"
        fi
    # A boot src install. That is, do a target install unless one exists. Save the target
    # install, then boot the target install to the second target. The 'boot_src' element
    # is the boot target source (i.e. this target is copied to the new target).
    else
        # host install
        ssh -l ${BMKDESC['hu']} ${BMKDESC['hn']} "true && test -d ${BMKDESC['bd']}/f-test"
        # install if not exists previously
        if [ $? -ne 0 ]; then
            test_cmd="${TEST_SH_CMD} hu=${BMKDESC['hu']} hn=${BMKDESC['hn']} bd=${BMKDESC['bd']} install_host;"
        fi

        # boot target install
        ssh -l ${BMKDESC['tu']} ${BMKDESC['boot_src']} "true && test -d ${BMKDESC['bd']}/f-test"
        # install if not exists previously
        if [ $? -ne 0 ]; then
            test_cmd="$test_cmd ${TEST_SH_CMD} tu=${BMKDESC['tu']} tn=${BMKDESC['boot_src']} bd=${BMKDESC['bd']} install_target;"
        fi

        # save the target tarballs for copying
        test_cmd="$test_cmd ${TEST_SH_CMD} tu=${BMKDESC['tu']} tn=${BMKDESC['boot_src']} bd=${BMKDESC['bd']} save_target;"

        # install to the boot target machine
        test_cmd="$test_cmd ${TEST_SH_CMD} tu=${BMKDESC['tu']} tn=${BMKDESC['tn']} bd=${BMKDESC['bd']} boot_target;"
    fi
        
    echo "# Invoke Command:" > "${BMKSCRIPT}.${BMKDESCNAME[${BMKIDX}]}.log"
    echo "# ${BMKINVOKE}" >> "${BMKSCRIPT}.${BMKDESCNAME[${BMKIDX}]}.log"
    echo "# Build Command:" >> "${BMKSCRIPT}.${BMKDESCNAME[${BMKIDX}]}.log"
    echo "# ${test_cmd}" >> "${BMKSCRIPT}.${BMKDESCNAME[${BMKIDX}]}.log"

    # Groups are a string list of BMKIDXes for a given 'hn' setting
    hn=${BMKDESC['hn']}
    BMKGROUPS[$hn]="${BMKGROUPS[$hn]} $BMKIDX"

    BMKCMDS[$BMKIDX]="${test_cmd}"

    (( BMKIDX=$BMKIDX+1 ))

}

gen_test () {
    local i
    local j
    local test_cmd
    local hn

    test_cmd="${TEST_SH_CMD} "
    if [ "${BMKDESC['extopts']}" != "" ]; then test_cmd="$test_cmd ${BMKDESC['extopts']} "; fi 
    if [ "${BMKDESC['tar']}" != "" ]; then test_cmd="$test_cmd tar=${BMKDESC['tar']} "; fi 
    if [ "${BMKDESC['bd']}" != "" ]; then test_cmd="$test_cmd bd=${BMKDESC['bd']} "; fi 
    if [ "${BMKDESC['hn']}" != "" ]; then test_cmd="$test_cmd hn=${BMKDESC['hn']} "; fi 
    if [ "${BMKDESC['hu']}" != "" ]; then test_cmd="$test_cmd hu=${BMKDESC['hu']} "; fi 
    if [ "${BMKDESC['tn']}" != "" ]; then test_cmd="$test_cmd tn=${BMKDESC['tn']} "; fi 
    if [ "${BMKDESC['tu']}" != "" ]; then test_cmd="$test_cmd tu=${BMKDESC['tu']} "; fi 
    if [ "${BMKDESC['Htools']}" != "" ]; then test_cmd="$test_cmd Htools=${BMKDESC['Htools']} "; fi 
    if [ "${BMKDESC['Tlibs']}" != "" ]; then test_cmd="$test_cmd Tlibs=${BMKDESC['Tlibs']} "; fi 
    if [ "${BMKDESC['tools']}" != "" ]; then test_cmd="$test_cmd tools=${BMKDESC['tools']} "; fi 
    if [ "${BMKDESC['prefix']}" != "" ]; then test_cmd="$test_cmd prefix=${BMKDESC['prefix']} "; fi 
    if [ "${BMKDESC['core']}" != "" ]; then test_cmd="$test_cmd core=${BMKDESC['core']} "; fi 
    if [ "${BMKDESC['ml']}" != "" ]; then test_cmd="$test_cmd ml=\"${BMKDESC['ml']}\" "; fi 
    if [ "${BMKDESC['pt']}" != "" ]; then test_cmd="$test_cmd pt=\"${BMKDESC['pt']}\" "; fi 
    if [ "${BMKDESC['ct']}" != "" ]; then test_cmd="$test_cmd ct=\"${BMKDESC['ct']}\" "; fi 
    if [ "${BMKDESC['bt']}" != "" ]; then test_cmd="$test_cmd bt=\"${BMKDESC['bt']}\" "; fi 
    if [ "${BMKDESC['flgs']}" != "" ]; then test_cmd="$test_cmd flgs=\"${BMKDESC['flgs']}\" "; fi 
    if [ "${BMKDESC['x']}" != "" ]; then test_cmd="$test_cmd x=${BMKDESC['x']} "; fi
    if [ "${BMKDESC['tag']}" != "" ]; then test_cmd="$test_cmd tag=${BMKDESC['tag']} "; fi

    # Groups are a string list of BMKIDXes for a given 'hn' setting
    hn=${BMKDESC['hn']}
    BMKGROUPS[$hn]="${BMKGROUPS[$hn]} $BMKIDX"

    # Check shared lib directory on test target host machine if
    # if not using a tar file. If not using a tar file, the target libs
    # will have to be manually installed

    # Skip if using job server
    if [ "${BMKDESC['pt']}" = "" ]; then 
    	if [ "${BMKDESC['tar']}" = "" ]; then
        	echo "WARNING: Libs must be installed on ${BMKDESC['tn']} at ${BMKDESC['Tlibs']}"
    	fi
    fi


    echo "# Invoke Command:" > "${BMKSCRIPT}.${BMKDESCNAME[$BMKIDX]}.log"
    echo "# ${BMKINVOKE}" >> "${BMKSCRIPT}.${BMKDESCNAME[$BMKIDX]}.log"
    echo "# Build Command:" >> "${BMKSCRIPT}.${BMKDESCNAME[$BMKIDX]}.log"
    echo "# ${test_cmd}" >> "${BMKSCRIPT}.${BMKDESCNAME[$BMKIDX]}.log"

    BMKCMDS[$BMKIDX]="${test_cmd}"

    if [ $BMKVERBOSE -eq 1 ]; then
        echo "Benchmark: ${BMKIDX}" 
        echo "===================="
        # for each key
        for i in ${!BMKDESC[@]}; do
             echo "$i = ${BMKDESC[$i]}"
        done
        echo "===================="
    fi 

    (( BMKIDX=$BMKIDX+1 ))
}

trap kill_group_jobs INT

#
# kill group jobs on control-c
#
kill_group_jobs () {
    local i
    for (( i=0; i<${#BMKGROUPJOBS[@]}; i++ )); do
        [ -d /proc/${BMKGROUPJOBS[$i]} ] && kill -9 ${BMKGROUPJOBS[$i]}
        echo "(${BMKGROUPJOBS[$i]}) killed"
    done
}

#
# BMKGROUPJOBS is an array of PIDs for the 'sh' group jobs started in the background and
# are still running. Here, poll to see if the PID is still active. If it
# has ended, issue a message and remove the job PID from the PID array.
# Return if there are no more jobs to monitor.
# 
wait_for_groups () {
    local i
    local num_groupjobs

    if [ ${#BMKGROUPJOBS[@]} -ne 0 ]; then
        echo "Waiting for jobs to complete..."
    fi
    num_groupjobs=${#BMKGROUPJOBS[@]}
    while [ 1 ]; do
        for (( i=0; i<$num_groupjobs; i++ )); do
            # Check if process is done
            if [ ! -d /proc/${BMKGROUPJOBS[$i]} ]; then
                echo "Group $i completed at `date`"
                # remove job from array
                unset BMKGROUPJOBS[$i]
            fi
        done
        # Break if no more jobs
        if [ ${#BMKGROUPJOBS[@]} -eq 0 ]; then 
           break
        fi
	sleep 1
    done
}

#
# issue_group - execute a list of BMKCMDs serially
# input: $1=groupid, $2=string list of BMKCMD indexes
#
# Global test_dispatch.sh.<descriptor_name>.log files are created in the top level directory.
# These contain test_dispatch.sh log information.
# The test.target_log file is prepended with <desc>_<index>.target_log field (useful
# for bmksum.pl).
#
issue_group () {
    local i 
    local groupid=$1
    local bmkcmds=( $(echo $2) )
    local repcnt=$3

    # For each BMKIDX in group
    for i in ${bmkcmds[@]}; do
        base_directory=${BMKDESCNAME[$i]}
        mkdir -p "$base_directory" 
        pushd "$base_directory" > /dev/null

        echo "Test started at `date`" >> ../${BMKSCRIPT}.${BMKDESCNAME[$i]}.log
        echo "${BMKCMDS[$i]}" >> ../${BMKSCRIPT}.${BMKDESCNAME[$i]}.log

        if [ $BMKVERBOSE -eq 1 ]; then
            echo "Group $groupid: Idx: $i ${BMKCMDS[$i]}" 
        fi

        if [ $BMKDRYRUN -eq 0 ]; then
            # Insert <compiler_name> entry (use label for compiler name) 
            # append repeat cnt if multiple runs
            if [ $repcnt -gt 0 ]; then
                echo "${BMKDESCLABEL[$i]}_${repcnt}.target_log" >> test.target_log
            else
                echo "${BMKDESCLABEL[$i]}.target_log" >> test.target_log
            fi
                
            sh -c "${BMKCMDS[$i]} >> ../${BMKSCRIPT}.${BMKDESCNAME[$i]}.log >&1" 
        fi
        popd > /dev/null
    done
}

# Issue each group for execution. Groups are executed in parallel, so they
# must not share resources (host specs or target specs). Within each group
# each job is executed serially. Groups consists of a set of integer indexes.
# The indexes are used to identify the descriptor name from BMKDESCNAME and the
# test.sh command from the BMKCMDS array.
#  
# Jobs are executed in a unique directory named for the descriptor . That is,
# each descriptor executes the test.sh command in a unique directory. 
#
#
do_tests () {
    local i
    local j=0
    local k
    local r
    local base_directory
    local idx_string
    local group_pid
    local idx_array
    local group_members 

    # For repeat times 
    for (( r=0; r<$BMKREPEAT; r++ )); do
        # For each group
        for i in ${!BMKGROUPS[@]}; do
            group_members=""
            idx_string=${BMKGROUPS[$i]}
            idx_array=( $(echo $idx_string) )
            for k in ${idx_array[@]}; do
                group_members="$group_members ${BMKDESCNAME[$k]}"
            done
            echo "Issuing group job $j (\"$i\":$group_members) at `date`" 
            issue_group $j "$idx_string" $r & 
            group_pid=$!
            BMKGROUPJOBS[$j]=$group_pid
            if [ $BMKDOSERIAL -eq 1 ]; then 
                wait $group_pid 
            # remove group job when done
                unset BMKGROUPJOBS[$j]
            fi
            (( j=$j+1 ))
        done
        j=0
        wait_for_groups
    done
}


main ( ) {
        local arg
        local dorepeat=0

        if [ $# -eq 0 -o "$1" = "-h" ]; then
            echo "Usage: $BMKSCRIPT [-v] [-n] [-s] [-h] [-i] <descriptor> [overrides] ..."
            echo " "
            echo "  <descriptor> are parameter files found in the $REPODIR/build_gnu/f-bmk directory"
            echo "  <descriptor> must be specified 1 or more times"
            echo "  Multiple descriptors are grouped by their 'hn' setting. That is, all descriptors sharing"
            echo "  the same 'hn' setting make up a group. All groups are dispatched in parallel and"
            echo "  each group member is executed serially on the resources defined for that member"
            echo "  By default all groups are issued in parallel unless -s specified, in which case"
            echo "  the groups and members are serially executed."
            echo " [overrides] -- optional, override the parameters in the <descriptor> file" 
            echo "  These options override the default settings in the <descriptor> file"
            echo "  and are applied to the most recently specified <descriptor>"
            echo "  e.g. 'dt-e5500 hn=perf3farm17  dt-e500mc'  option hn= applies to most recent descriptor, dt-e5500"
            echo " [-v] -- verbose mode"
            echo " [-n] -- generate commands but do not execute"
            echo " [-s] -- all groups are issued serially"
            echo " [-r <cnt>] -- repeat descriptors <cnt> times"
            echo " [-h] -- print help"
            echo " [-i] -- install tests for build host and target host" 
            echo "  If -i is given the tests are installed serially and the script exits. Installation consists"
            echo "  of a host installation at the build directory (BD) location and similiar installation at the"
            echo "  target host location. For target hosts that do not or cannot mount the build_gnu directory,"
            echo "  installation consists of the build host/target host/remote target host installation. That is,"
            echo "  a target host installed is performed and tarball'ed and copied and installed at the remote"
            echo "  target host. The test_dispatch parameter, \"boot_src\" defines the target host install."
            exit 1
        fi
        for arg in "$@"; do
            case $arg in 
                 -i)
                    BMKINSTALL=1 
                    shift
                    ;;
                 -v)
                    BMKVERBOSE=1 
                    shift
                    ;;
                 -n)
                    BMKDRYRUN=1 
                    shift
                    ;;
                -s)
                    BMKDOSERIAL=1
                    shift
                    ;;
                -r)
                    shift
                    dorepeat=1
                    ;;
                [0-9][0-9])
                    shift
                    if [ $dorepeat -eq 1 ]; then
                        BMKREPEAT=$arg
                    else
                        echo "Unknown argument: $arg"
                        exit 1
                    fi
                    ;;
                [0-9])
                    shift
                    if [ $dorepeat -eq 1 ]; then
                        BMKREPEAT=$arg
                    else
                        echo "Unknown argument: $arg"
                        exit 1
                    fi
                    ;;
                        
                +(label|extopts|boot_src|bd|hn|hu|tn|tu|tar|Htools|Tlibs|tools|prefix|core|ml|bt|ct|pt|flgs|x|tag)=*)
                    if [ "${BMKCURRENTDESC}" = "" ]; then
                        echo "Must provide a <descriptor> before an override ($arg)"
                        exit 1
                    fi
                    handle_override "$arg"
                    shift
                    ;;
                *)
                    # Process pending desc before handling next one
                    if [ "${BMKCURRENTDESC}" != "" ]; then
                        process_desc
                    fi

                    BMKCURRENTDESC=$arg

                    if [ ! -e "${BMKDESCPATH}/${BMKCURRENTDESC}" ]; then
                        echo "File not found ${BMKDESCPATH}/${BMKCURRENTDESC}"
                        exit 1
                    fi
                    source ${BMKDESCPATH}/${BMKCURRENTDESC}
                    shift
                    ;;
            esac
        done

        process_desc
        do_tests
}

# Main entry point
#
scriptdir=`dirname $0`
hnfull=`hostname`
hn=`basename $hnfull .am.freescale.net`
hn=`basename $hn .mtwk.freescale.net`
source $scriptdir/build.env-$hn
BMKSCRIPT=`basename $0`
# Path to descriptor file
BMKDESCPATH="${REPODIR}/build_gnu/f-bmk"
BMKINVOKE="`pwd`/${BMKSCRIPT} $@"
main "$@"
