/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * Copyright (c) 2010, 2014 Freescale Semiconductor, Inc
 * Copyright 2023 NXP
 * 
 * SPDX-License-Identifier: EPL-1.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.nxp.sa.ui.common.epl;

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IActionBars;

/**
 * Represents a section.
 * <p>
 * The lifecycle of an ISection is as follows:
 * <ul>
 * <li><code>ISection.setInput()</code></li>
 * <li><code>ISection.createControls()</code></li>
 * <li><code>ISection.aboutToBeShown()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * <li><code>ISection.aboutToBeHidden()</code></li>
 * <li><code>ISection.dispose()</code></li>
 * </ul>
 * </p>
 * <p>
 * Implementors of this class should be aware that a section instance might be
 * reused for different input objects (as long as they are valid section
 * inputs). It means that <code>ISection.setInput</code> can be called at any
 * time between <code>ISection.createControls</code> and
 * <code>ISection.dispose</code>.
 * </p>
 * <p>
 * When an input change event occurs an ISection is sent:
 * <ul>
 * <li><code>ISection.setInput()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * </ul>
 * </p>
 * <p>
 * When an activation event occurs, such as the contributor activation
 * event, an ISection is sent:
 * <ul>
 * <li><code>ISection.setInput()</code></li>
 * <li><code>ISection.aboutToBeShown()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * <li><code>ISection.setInput()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * </ul>
 * </p>
 * <p>
 * This interface should not be extended or implemented. New section instances
 * should be created using <code>AbstractSection</code>.
 * </p>
 */
public interface ISection extends IRefreshableDisplay{

	/**
	 * Creates the controls for the section.
	 * <p>
	 * Clients should take advantage of the widget factory provided by the
	 * framework to achieve a common look between sections.
	 * </p>
	 * @see org.eclipse.ui.views.properties.tabbed.TabbedSheetPage#getWidgetFactory()
	 * 
	 * @param parent
	 *            the parent composite for the section.
	 */
	public void createControls(Composite parent);

	/**
	 * <p>Notifies the section that it should handle a new input.
	 * The section should update its UI to reflect that.</p>
	 * 
	 * @param input the active debug context 
	 */
	public void setInput(Object input);

	/**
	 * Returns the value passed into the most recent setInput() call.
	 */
	public Object getInput();
	
	/**
	 * Notifies the section that its controls are about to be shown. It is
	 * expected that sections enable domain related functions in this method,
	 * most commonly add listeners.
	 * <p>
	 * Since the controls are not visible, the section should wait for the
	 * refresh() before updating the section controls.
	 * </p>
	 */
	public void aboutToBeShown();

	/**
	 * Notifies the section that its controls are about to be hidden. It is
	 * expected that sections disable domain related functions in this method,
	 * most commonly remove listeners.
	 */
	public void aboutToBeHidden();

	/**
	 * Dispose this section.
	 */
	public void dispose();

	/**
	 * Returns the minimum height needed by this section. A return value of
	 * <code>SWT.DEFAULT</code> indicates that no minimum height is defined.
	 * 
	 * @return the minimum height needed by this section.
	 */
	public int getMinimumHeight();

	/**
	 * Determine whether this section would like extra height space in case
	 * there is some left. Normally this is true when the section is the last to
	 * be displayed on a tab or is the only section on a tab.
	 * @return <code>true</code> if this section would like extra height space.
	 */
	public boolean shouldUseExtraSpace();

	/**
	 * Refresh the contents of the controls displayed in this section.
	 * The typical implementation calls Viewer.refresh() on all jFace 
	 * viewers in its section.
	 */
	public void refresh();
	
	/**
	 * Refreshes the contents of the controls displayed in this section, using
	 * the data producer defined in the specified config file. If the specified
	 * file is null, then the default data producer is used.
	 * 
	 * @param configFile
	 *            the configuration file
	 */
	public void refreshTrace(IFile configFile); 

	/**
	 * Refreshes the contents of the controls displayed in this section, using
	 * whatever source was last used.
	 */
	public void refreshTrace();
	
	/**
	 * Invalidate the data exposed by this section. This does not directly 
	 * cause the section to gather new context, less display it. However, it 
	 * should do so next time it is asked to display the section content.
	 * To get the whole package (invalidate and force new content to be 
	 * gathered and shown), call refresh() instead.
	 * 
	 */
	public void invalidate();
	
	/**
	 * Allows a section to customize the action bars that will be available when
	 * the section is shown. This hook is called every time the section is about to show.
	 * 
	 * @param actionBars
	 *            the action bars for a part
	 */
	// TODO investigate how this could be made to work
	// the below method is a fallback and uses just a toolbar manager directly
	public void setActionBars(IActionBars actionBars);
	
	/**
	 * Allows a section to customize the action bars that will be available when
	 * the section is shown. This hook is called every time the section is about to show.
	 * 
	 * 
	 * @param toolbarMgr
	 *            the toolbar manager
	 */
	public void setActions(IToolBarManager toolbarMgr);

}
