/*******************************************************************************
 * Copyright 2023 NXP
 * NXP Confidential and Proprietary. This software is owned or controlled by NXP and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 *******************************************************************************/

package com.freescale.system.browser2;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.ui.progress.UIJob;


/**
 * A content provider that services a viewer in a System Browser section should
 * extend this class in order to automatically refresh when the process becomes
 * suspended (comes out of a step operation, or is suspended by a breakpoint or
 * the user). The refreshing is done lazily; if the user steps, but the viewer
 * is not in the foreground System Browser sheet, we don't hit the target,
 * instead deferring that activity until the tab is shown. This is very
 * important since a System Contributor may expose a lot of sheets. It would be
 * extremely wasteful for every sheet to update its content on every suspend
 * event.
 * 
 * <p>
 * The subclass needs to do two explicit things in order for this class to
 * successfully do its job:
 * 
 * <ul>
 * <li>call our elementsRequested() when its element-gathering method is called
 * <li>call our dispose() when its dispose() is called
 * </ul>
 * 
 * <p>
 * There is also a requirement on the System Browser section (ISection
 * implementation) containing the viewer. It must forward its ISection
 * aboutToBe[Shown|Hidden]() method invocations to us.
 * 
 * <p>
 * Finally, the code managing the viewer should call our invalidate() method
 * instead of directly calling the viewer's refresh() method. See invalidate()
 * comments for reasons why.
 */
public abstract class ContentProviderAutoUpdater2 { 	
	/**
	 * The viewer the content provider services
	 */
	protected final Viewer fViewer;

	/**
	 * State toggled by viewerAboutToBe[Shown|Hidden]()  
	 */
	private boolean fViewerIsHidden;
	
	/**
	 * If true, it means a state change in our context object occurred while the
	 * viewer was hidden. If and when we are told the viewer is about to be
	 * shown, we should call the viewer's refresh() method to ensure it does
	 * not expose stale information to the user.
	 * 
	 * Start out with 'true' to get proper behavior on first viewing
	 */
	private volatile boolean fRefreshPending = true;
	private volatile boolean fAcceptingCommands = false;
	private final Object fDsfEventListener;
	private DSFSessionTrail2 m_sessionTrail;
    
	/**
	 * Constructor
	 * @param viewer the viewer the content provider is servicing
	 */
	public ContentProviderAutoUpdater2(Viewer viewer) {
		//A content provider is responsible for telling its viewer to
		//refresh if the state of the input object (context) changes in
		//a way that may result in new content. So, we store away the
		//a reference to the viewer and set ourselves up to be notified 
		//whenever a debug event happens.
		fViewer = viewer;
		fDsfEventListener = new DSFEventListener2(this);		
	}
	
	/** */
	final public boolean getCommandState() {return fAcceptingCommands;}

	/**
	 * Call this method instead of directly calling the viewer's refresh()
	 * method. This has two advantages. First, we'll defer the viewer refresh()
	 * invocation if the tab is hidden; we'll invoke it when it becomes visible.
	 * Secondly, we'll automatically invoke the refresh on the UI thread; i.e.,
	 * the caller need not worry about creating a UIJob if he's on a secondary
	 * thread.
	 */
	synchronized public void invalidate() {
		SystemBrowserActivator2.log(1, "invalidate");//$NON-NLS-1$
		if (fViewerIsHidden) {
			// Optimization: no point telling the viewer to refresh when it's
			// not
			// showing. Defer the refresh
			fRefreshPending = true;
		} else {
			UIJob job = new UIJob("update tasks table") { //$NON-NLS-1$
				@Override
				public IStatus runInUIThread(IProgressMonitor monitor) {
					fViewer.refresh();
					return Status.OK_STATUS;
				}
			};
			job.schedule();
		}
	}

	/**
	 * The content provider must call this method when its called to retrieve the
	 * elements. Failure to do so will not allow us to effectively maintain our
	 * update state.
	 */
	protected void elementsRequested() {
		fRefreshPending = false;
	}

	/**
	 * The content provider should call this method when its dispose() is called.
	 * Failure to do so will leave a debug listener in place that adds unneeded
	 * overhead to the debugger.
	 */
	public void dispose() {
		//Remove ourselves as a listener for debug events or we'll be a leak.
		//Worse, we'll indefinitely and needlessly burden the debug platform
		//for debug events,
		//DebugPlugin.getDefault().removeDebugEventListener(this);
		setDebugSession(null);
	}

	/**
	 * This needs to be called whenever the ISection the viewer lives in has 
	 * its aboutToBeShown() method called.
	 */
	synchronized public void viewerAboutToBeShown() {
		fViewerIsHidden = false;		
		if (fRefreshPending) {
			fViewer.refresh(); //will reset m_refreshPending
		}
	}

	/**
	 * This needs to be called whenever the ISection the viewer lives in has 
	 * its aboutToBeHidden() method called.
	 */
	synchronized public void viewerAboutToBeHidden() {
		fViewerIsHidden = true;
	}
	
	/** */
	protected boolean setDebugSession(final String sessionId) {
		boolean changed = false;

		if (m_sessionTrail != null &&
				! m_sessionTrail.getSessionID().equals(sessionId))
		{
			m_sessionTrail.removeServiceEventListener(fDsfEventListener);
			m_sessionTrail.dispose();
			m_sessionTrail = null;
			changed = true;
		}

		if (m_sessionTrail == null &&
				sessionId != null)
		{
			m_sessionTrail = new DSFSessionTrail2(sessionId);
			m_sessionTrail.addServiceEventListener(fDsfEventListener);
			changed = true;
		}	
		return changed;
	}
	
}
