/*******************************************************************************
 * Copyright 2023 NXP
 * NXP Confidential and Proprietary. This software is owned or controlled by NXP and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 *******************************************************************************/

package com.freescale.system.browser2;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.RejectedExecutionException;

import org.eclipse.cdt.dsf.concurrent.DsfRunnable;
import org.eclipse.cdt.dsf.service.DsfServicesTracker;
import org.eclipse.cdt.dsf.service.DsfSession;

/**
 * DSF session state object.
 * Encapsulates and manages DsfSession we're currently tracking.
 */
public class DSFSessionTrail2
{
	// --- members ---
	/** Current session ID. */
	private final String m_sessionId;

	/** Current set of session event listeners. */
	private final List<Object> m_sessionListeners;
	
	/** Services tracker, used to access services. */
	private final DsfServicesTracker m_servicesTracker;

	// --- constructors/destructors ---
		
	public DSFSessionTrail2(final String sessionId) {
		m_sessionId = sessionId;
		m_sessionListeners = new ArrayList<>();
		m_servicesTracker = new DsfServicesTracker(SystemBrowserActivator2.getBundleContext(), m_sessionId);
	}
	
	/** Dispose method. */
	public void dispose() {
		if (m_sessionId != null) {
			removeAllServiceEventListeners();
		}
		
		if (m_servicesTracker != null) {
			m_servicesTracker.dispose();				
		}
	}
		
	// --- accessors ---
	
	/** Returns session ID. */
	public String getSessionID() {
		return m_sessionId;
	}
	
	
	// --- listener management ---

	/** Adds a service event listener. */
	public void addServiceEventListener(final Object listener) {
		//SystemBrowserActivator.log(1, "add_Listener "+listener.toString());// //$NON-NLS-1$
		final Object listener_f = listener;
		final DsfSession session_f = getDsfSession();
		if (session_f != null) {
			try {
				session_f.getExecutor().execute(new DsfRunnable() {
					@Override
					public void run() {
						session_f.addServiceEventListener(listener_f, null);
						m_sessionListeners.add(listener_f);
					}
				});
    		} catch (RejectedExecutionException e) {
    			SystemBrowserActivator2.log(2, e.getMessage());
    		}
		}
	}
	
	/** Removes a service event listener. */
	public void removeServiceEventListener(final Object listener) {
		//SystemBrowserActivator.log(1, "remove_Listener "+listener.toString());// //$NON-NLS-1$
		final Object listener_f = listener;
		final DsfSession session_f = getDsfSession();
		if (session_f != null) {
			try {
				session_f.getExecutor().execute(new DsfRunnable() {
					@Override
					public void run() {
						if (m_sessionListeners != null) {
							session_f.removeServiceEventListener(listener_f);
							m_sessionListeners.remove(listener_f);
						}
					}
				});
    		} catch (RejectedExecutionException e) {
    			SystemBrowserActivator2.log(2, e.getMessage());
    		}
		}
	}
	
	/** Removes all service event listeners. */
	public void removeAllServiceEventListeners() {
		final DsfSession session_f = getDsfSession();
		if (session_f != null) {
			try {
				session_f.getExecutor().execute(new DsfRunnable() {
					@Override
					public void run() {
						if (m_sessionListeners != null) {
							for (final Object listener : m_sessionListeners) {
								session_f.removeServiceEventListener(listener);
							}
							m_sessionListeners.clear();
						}
					}
				});
    		} catch (RejectedExecutionException e) {
    			SystemBrowserActivator2.log(2, e.getMessage());
    		}
		}
	}
	// --- methods ---	
	/** Gets current DsfSession, if it's still active. */
	protected DsfSession getDsfSession() {
		return DsfSession.getSession(m_sessionId);
	}
}
