"use strict";

/* Copyright 2015: Freescale Semiconductor, Inc. All Rights Reserved. */
/* Script to generate code for Pin-connect component */


var chFileName = "iomux_config";
var moduleName = "iomux_config";
//
var dtsSkeletonName = "skeleton";

/* c indentation 2 */
var cIndent2 = '  ';
/* c indentation 4 */
var cIndent4 = '    ';
/* c indentation 6 */
var cIndent6 = '      ';
/* c indentation 8 */
var cIndent8 = '        ';
/* c indentation 12 */
var cIndent12 = '            ';
/* c indentation 16 */
var cIndent16 = '                ';


var cCodeCommentColumn = 40;
var defineNameColumn = 7;
var defineValueEndColumn = 37;
var defineCommentColumn = 40;

/* PE DEBUG mode flag */
var PE_DEBUG = PExScript.getMacroSymbol("PE_DEBUG");

var pinMuxControlBitName = "MUX_MODE";

// Register database object
var registerDatabaseObject = PExProcessor.getRegistersDB();

// Is it running in test mode. The test mode can be launched from the tool with internal development support
var testMode = true;
if (PExScript.getMacroSymbol('GENERATE_ALL_CONFIGURATIONS') == null) {
    testMode = false;
}

// Preparation of other information ...
var allComponents = PExProject.getAllComponents();
if (registerDatabaseObject != null) {
  var peripherals = registerDatabaseObject.getPeripherals();
}

/** Copyright used for C code files - corresponds to Kinetis */
var headerCopyright = [
" * Copyright (c) 2015, Freescale Semiconductor, Inc.",
" * All rights reserved.",
" *",
" * Redistribution and use in source and binary forms, with or without modification,",
" * are permitted provided that the following conditions are met:",
" *",
" * o Redistributions of source code must retain the above copyright notice, this list",
" *   of conditions and the following disclaimer.",
" *",
" * o Redistributions in binary form must reproduce the above copyright notice, this",
" *   list of conditions and the following disclaimer in the documentation and/or",
" *   other materials provided with the distribution.",
" *",
" * o Neither the name of Freescale Semiconductor, Inc. nor the names of its",
" *   contributors may be used to endorse or promote products derived from this",
" *   software without specific prior written permission.",
" *",
" * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS \"AS IS\" AND",
" * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED",
" * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE",
" * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR",
" * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES",
" * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;",
" * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON",
" * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT",
" * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS",
" * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.",
" *"
]

/** Copyright used for DTS snippet file - corresponds to i.MX Linux examples */
var headerDtsCopyright = [
" * Copyright (c) 2015, Freescale Semiconductor, Inc.",
" *",
" * This program is free software; you can redistribute it and/or modify",
" * it under the terms of the GNU General Public License version 2 as",
" * published by the Free Software Foundation.",
" *"
]


/* Prints string array
 * array - string array (one item on one line)
 * stringAtBeginningOfEachLine - string which is placed at the beginning of each line
 * stringAtBeginningOfFirstLine - string which is placed at the beginning of the first line
 * return value - no data; just printing into output 
 */
function printStringArray(array, stringAtBegginningOfEachLine, stringAtBeginningOfFirstLine) {
  if (array.length > 0) {
    for (var lineIndex = 0; lineIndex < array.length; lineIndex++) {
      var line = array[lineIndex];
      if (lineIndex == 0) {
        line = stringAtBeginningOfFirstLine + line;  
      }
      line = stringAtBegginningOfEachLine + line;
      PExOut.gen(line);
    }
  }
}


/**
 * Gets value converted to hex format 
 * bitFieldValueX - number for conversion
 * returns converted value (e.g. 0x0Fu)
 */
function getNumberConvertedToHex(bitFieldValueX, useSuffix) {
  var hexNumberPart = bitFieldValueX.toString(16).toUpperCase();
  if(hexNumberPart.length % 2 == 1) {
    hexNumberPart = "0" + hexNumberPart;
  }
  while(hexNumberPart.length < 8) {
    hexNumberPart = "0" + hexNumberPart;
  }
  return ("0x" + hexNumberPart + (useSuffix == true ? "u" : ""));
}


/**
 * Gets reset value of register from database
 * dbReg - reference to database register
 * returns reset value
 */
function getResetValue(dbReg) {
  if (dbReg == null) return 0;
  //
  var value = java.math.BigInteger.valueOf(0);
  var dbBitFields = dbReg.getBitFields();
  for (var fi = 0; fi < dbBitFields.length; fi++) {
    var dbBitField = dbBitFields[fi];
    value |= dbBitField.getResetValue().doubleValue() << dbBitField.getOffset();   // Note: IRegBitFieldAPI::getResetValue() returns java.lang.Number object
  }
  return (value);
}


/**
 * Gets register mask of no read only bits
 * dbReg - reference to database register
 * returns non read only mask
 */
function getNonReadOnlyMask(dbReg) {
  var mask = java.math.BigInteger.valueOf(0);
  var dbBitFields = dbReg.getBitFields();
  for (var fi = 0; fi < dbBitFields.length; fi++) {
    var dbBitField = dbBitFields[fi];
    if (dbBitField.isReadOnly()) {
      mask |= dbBitField.getRegisterMask().doubleValue();   // Note: IRegBitFieldAPI::getRegisteMask() returns java.lang.Number object
    }
  }
  return (dbReg.getMask().doubleValue() ^ mask);   // Note: IRegisterAPI::getMask() returns java.lang.Number object
}

//--
var textOptions = null;
var componentCoreIds = new Array();
//
var configurationStrategies = new Array();
for (var pc = 0; pc < allComponents.length; pc++) {
  var funcNameItem = allComponents[pc].findItemBySymbol("FunctionName");
  if (funcNameItem != null) {
    configurationStrategies[pc] = funcNameItem.getText();
  }
  textOptions = allComponents[pc].getComponentOptions();
  if (textOptions != null) {
    var options = JSON.parse(textOptions);
    componentCoreIds.push(options["coreID"]);
    //PExOut.log("Component options: " + JSON.stringify(options));
  }
}
var configurationStrategy=null;


// get multi-core information
var coreIds = null;
var coreListTxt = PExProcessor.getCoresList();
if (coreListTxt != null) {
  coreIds = Object.keys(JSON.parse(coreListTxt));
}

// generate code only for cores with non-empty configurations
var notEmptyCoreIds = new Array();
for (var coreIndex in coreIds) {
  var coreId = coreIds[coreIndex];
  if (componentCoreIds.indexOf(coreId) > -1) {
    notEmptyCoreIds.push(coreId);
  }
}

/** 
 * Processes register configuration sequence for all pin function configurations to C code
 * coreId - cpu core id
 */
function processRegisterConfigurationSequence(coreId) {
  /* Iteration through the register settings and collecting constant definition data */
  for (var pc = 0; pc < allComponents.length; pc++) {                        // Pin configuration represented by tables in the UI of the tool
    if (coreId == componentCoreIds[pc]) {
      constantDefinitionList[pc] = new Object(); // Fixme: remove?
      configurationStrategy = configurationStrategies[pc];

      /* Getting register settings for pin muxing and functional pin properties (electrical features) from the tool. There are not included register settings for properties with default/reset values (text with cursive font) */
      var configurationSteps = PExProcessor.getRegisterConfigurationSequence(true,configurationStrategy,null);   // exclude automatic, given function, all steps

      /* Analyzing returned data. Iteration through all registers and finding and collecting clock gate configurations */
      registerList[pc] = new Object();
      var configurationRegisterList = registerList[pc];
      //
      for (var si = 0; si < configurationSteps.length; si++) {
        var configurationStepName = configurationSteps[si].getName(); 
        var configurationStepDescription = configurationSteps[si].getDescription(); 
        var configurationRegisters = configurationSteps[si].getRegistersConfigurations();
        for (var ri = 0; ri < configurationRegisters.length; ri++) {
          //var registerDesc = configurationRegisters[ri];
          var registerName = configurationRegisters[ri].getRegisterName();
          var dbPeripheral = registerDatabaseObject.getPeripheralByFullName(registerName);
          var dbRegister = registerDatabaseObject.getRegisterByFullName(registerName);
          if ((dbPeripheral != null) && (dbRegister != null)) {
            /* ***** Preparation of pin muxing, functional property register configuration lists ***** */
            configurationRegisterList[registerName] = new Object();
            configurationRegisterList[registerName].registerName = registerName;
            configurationRegisterList[registerName].registerMuxCtrl = dbRegister; // save DB register for mux control
            configurationRegisterList[registerName].registerPadCtrl = registerDatabaseObject.getRegisterByFullName(registerName.replace('SW_MUX_CTL','SW_PAD_CTL')); // save DB register for pad control
            configurationRegisterList[registerName].registerClrMask = configurationRegisters[ri].getClrRegMask();
            configurationRegisterList[registerName].registerSetMask = configurationRegisters[ri].getSetRegMask();
          }
        }
      }
	}
  }
}

/** 
 * Processes register configuration sequence for all pin function configurations to DTS
 */
function processDtsRegisterConfigurationSequence() {
  /* Iteration through the register settings and collecting constant definition data */
  for (var pc = 0; pc < allComponents.length; pc++) {                        // Pin configuration represented by tables in the UI of the tool
    //if (coreId == componentCoreIds[pc]) {
      configurationStrategy = configurationStrategies[pc];

      /* Getting register settings for pin muxing and functional pin properties (electrical features) from the tool. There are not included register settings for properties with default/reset values (text with cursive font) */
      var configurationSteps = PExProcessor.getRegisterConfigurationSequence(true,configurationStrategy,null);   // exclude automatic, given function, all steps

      /* Analyzing returned data. Iteration through all registers and finding and collecting clock gate configurations */
      registerList[pc] = new Object();
      var configurationRegisterList = registerList[pc];

      for (var si = 0; si < configurationSteps.length; si++) {
        var configurationStepName = configurationSteps[si].getName(); 
        var configurationStepDescription = configurationSteps[si].getDescription();
        var configurationRegisters = configurationSteps[si].getRegistersConfigurations();
        for (var ri = 0; ri < configurationRegisters.length; ri++) {
          //var registerDesc = configurationRegisters[ri];
          var registerName = configurationRegisters[ri].getRegisterName();
          var dbPeripheral = registerDatabaseObject.getPeripheralByFullName(registerName);
          var dbRegister = registerDatabaseObject.getRegisterByFullName(registerName);
          if ((dbPeripheral != null) && (dbRegister != null)) {
            /* ***** Preparation of pin muxing, functional property register configuration lists ***** */
            configurationRegisterList[registerName] = new Object();
            configurationRegisterList[registerName].registerMuxCtrl = dbRegister; // save DB register for mux control
            configurationRegisterList[registerName].registerPadCtrl = registerDatabaseObject.getRegisterByFullName(registerName.replace('SW_MUX_CTL','SW_PAD_CTL')); // save DB register for pad control
            configurationRegisterList[registerName].registerName = registerName;
            configurationRegisterList[registerName].registerClrMask = configurationRegisters[ri].getClrRegMask();
            configurationRegisterList[registerName].registerSetMask = configurationRegisters[ri].getSetRegMask();
          }
        }
      }
	//}
  }
}

/**
 * Prepare and enroll a register data for print in two formats: PEx and Pins that would be printed to output according to routed pins configuration
 * reg - register defined by name, clr mask and set mask
 * listPExData - array of output lines (string) for PEx formatted code 
 * listPinsData - array of output lines (string) for Pins formatted code 
 * return no value, just preparation of data for output
 */
function enrolOneRegisterConfigData(reg, listPExData, listPinsData) {
  // create set value for complete register 
  var regMask = 0xFFFFFFFF;
  var setValue = (getResetValue(reg.registerMuxCtrl) & (regMask ^ reg.registerClrMask)) | (reg.registerSetMask); // full value used for 'WR' macro
  //
  var regMacroNameWR = 'HW_' + reg.registerName + '_WR(IOMUXC_BASE,';
  var regMacroNameSET = 'HW_' + reg.registerName + '_SET(IOMUXC_BASE,';
  var regMacroNameCLR = 'HW_' + reg.registerName + '_CLR(IOMUXC_BASE,';
  //
  var regMacroStr = '';
  
  // process comments as simple writes of values into specified register(s), i.e. corresponding with code output from current PEx for i.MX v1.x
  if (reg.registerName.indexOf('SELECT_INPUT') >= 0 || reg.registerName.indexOf('SW_PAD_CTL') >= 0) {
      regMacroStr = cIndent2 + '// HW_' + reg.registerName + '_WR(IOMUXC_BASE, ' + getNumberConvertedToHex(0 + setValue, true) + ');';
      listPExData.push(regMacroStr + getCodeCommentPadded('/* ' + reg.registerName + ' register modification value */', regMacroStr.length, 80));
  } else {
      regMacroStr = cIndent2 + '// HW_' + reg.registerName + '_SET(IOMUXC_BASE, ' + getNumberConvertedToHex(0 + reg.registerSetMask, true) + ');';
      listPExData.push(regMacroStr + getCodeCommentPadded('/* ' + reg.registerName + ' register set mask value */', regMacroStr.length, 80));
      regMacroStr = cIndent2 + '// HW_' + reg.registerName + '_CLR(IOMUXC_BASE, ' + getNumberConvertedToHex(0 + reg.registerClrMask, true) + ');';
      listPExData.push(regMacroStr + getCodeCommentPadded('/* ' + reg.registerName + ' register clear mask value */', regMacroStr.length, 80));
  }
  
  //
  var maskValue = reg.registerClrMask | reg.registerSetMask;
     
  // prepare print data as expanded bit field value modififation writes into specified register(s), i.e. corresponding to legacy IOMUX tool output
  var regDbBitFields = reg.registerMuxCtrl.getBitFields();
  var regMacroData = new Array();  // register data object containing the lines of code and its comment, it will be printed into the output file
  // process all register bit fields
  for (var fi = 0; fi < regDbBitFields.length; fi++) {
      var bfName = regDbBitFields[fi].getName();
      if (bfName != '') {

          var bfMaskVal = regDbBitFields[fi].getRegisterMask().doubleValue();   // Note: IRegBitFieldAPI::getRegisterMask() returns java.lang.Number object
          var bfMaskHex = getNumberConvertedToHex(bfMaskVal, false);
          var bfLineStr = new Object(); //line data object containing the one line of code and comment for specific bit field 
          //
          if ((maskValue & bfMaskVal) != 0) {
              var bitField = reg.registerMuxCtrl.findBitFieldByName(bfName); // search for bit field by name
              var bitValDb = bitField.findValueByRegisterValue(reg.registerSetMask); // find bit field value written by set value
              //
              if (bitValDb != null) {
                  var bfValueName = 'BV_IOMUXC_' + bitValDb.getName();
                  // DB-EXCEPTION:
                  switch (bfName) {
                      case "SPEED":
                          switch (bitValDb.getValue()) {
                              case 0:
                                  bfValueName = 'BV_' + reg.registerName + '_SPEED_LOW';
                                  break;
                              case 1:
                                  bfValueName = 'BV_' + reg.registerName + '_SPEED_MEDIUM';
                                  break;
                              case 2:
                                  bfValueName = 'BV_' + reg.registerName + '_SPEED_MEDIUM';
                                  break;
                              case 3:
                                  bfValueName = 'BV_' + reg.registerName + '_SPEED_MAXIMUM';
                                  break;
                          }
                          break;
                      case "DSE":  // 240, 290
                          if (reg.registerName.indexOf('RGMII') != -1) {
                              switch (bitValDb.getValue()) {
                                  case 0:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_HIZ';
                                      break;
                                  case 1:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_287_OHM';
                                      break;
                                  case 2:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_121_OHM';
                                      break;
                                  case 3:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_76_OHM';
                                      break;
                                  case 4:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_57_OHM';
                                      break;
                                  case 5:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_45_OHM';
                                      break;
                                  case 6:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_37_OHM';
                                      break;
                                  case 7:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_31_OHM';
                                      break;
                              }
                          } else if (reg.registerName.indexOf('DRAM') != -1 | reg.registerName.indexOf('GRP') != -1) {
                              switch (bitValDb.getValue()) {
                                  case 0:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_HIZ';
                                      break;
                                  case 1:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_240_OHM';
                                      break;
                                  case 2:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_120_OHM';
                                      break;
                                  case 3:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_80_OHM';
                                      break;
                                  case 4:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_60_OHM';
                                      break;
                                  case 5:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_48_OHM';
                                      break;
                                  case 6:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_40_OHM';
                                      break;
                                  case 7:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_34_OHM';
                                      break;
                              }
                          } else {
                              switch (bitValDb.getValue()) {
                                  case 0:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_HIZ';
                                      break;
                                  case 1:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_260_OHM';
                                      break;
                                  case 2:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_130_OHM';
                                      break;
                                  case 3:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_90_OHM';
                                      break;
                                  case 4:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_60_OHM';
                                      break;
                                  case 5:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_50_OHM';
                                      break;
                                  case 6:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_40_OHM';
                                      break;
                                  case 7:
                                      bfValueName = 'BV_' + reg.registerName + '_DSE_33_OHM';
                                      break;
                              }
                          }
                          break;
                      default:
                          break;
                  }
                  //
                  bfLineStr.code = 'BF_' + reg.registerName + '_' + bfName + '(' + bfValueName + ')';
                  bfLineStr.comment = bitField.getDescription().replace(' Field.', ' Field') + ': ' + bitValDb.getDescription();
                  //
                  regMacroData.push(bfLineStr); // store bit field related line content for printing ...
              }
          }
      }
  }
  enrolRegisterMacroStatement(regMacroNameWR, regMacroData, listPinsData); // enroll complete register macro statement
}


/**
 *  Get padded comment string aligned to column
 *  commentString - string of code comment to be padded by spaces from left to match (at least) given column
 *  length - length of previous code string
 *  column - number of column to be aligned for
 */
function getCodeCommentPadded(commentString, length, column) {
  var str = commentString.toString();
  var len = length;
  while (len < (column-1)) {
    str = ' ' + str;
    len++;
  }
  return str;
}

/**
 *  Enroll register macro code from input enrolMacrodataArray for given macro name
 *  regMacroName - name of the register write macro
 *  enrolMacroDataArray - array object data containing register bit field macros content to enroll
 *  enrolLanes - array object for enrolled code statement
 */
function enrolRegisterMacroStatement(regMacroName, enrolMacroDataArray, enrolLanes) {
    // begins with register macro name
    enrolLanes.push(cIndent2 + regMacroName);
    // and continue by macro data enrolled earlier to enrol full register macro statement 
    for (var li = 0; li < enrolMacroDataArray.length; li++) {
        if (li != (enrolMacroDataArray.length - 1)) {
           enrolLanes.push(cIndent6 + enrolMacroDataArray[li].code + ' | ' + getCodeCommentPadded('/* ' + enrolMacroDataArray[li].comment + ' */', cIndent6.length + enrolMacroDataArray[li].code.length + 3, 110));
        }
        else {
           enrolLanes.push(cIndent6 + enrolMacroDataArray[li].code + '); ' + getCodeCommentPadded('/* ' + enrolMacroDataArray[li].comment + ' */', cIndent6.length + enrolMacroDataArray[li].code.length + 3, 110));
        }
    }
    // nothing to return, just enroll register macro statement modifications for output
}

/**
 * Print a register into DTS generated output
 * mux - mux register defined by its name, clr mask and set mask, mux register name and MUX_MODE bit field value defines the DTS macro generated into file output 
 * pad - pad register defined by its name, clr mask and set mask, might be null in case not defined value from UI - default after reset value is generated instead
 * returns no value, just print to output
 */
function printDtsRegister(mux, pad) {
  if (mux != null) {
    var bitFieldMux = mux.registerMuxCtrl.findBitFieldByName(pinMuxControlBitName); // search for 'MUX_MODE' bit field
    if (bitFieldMux != null) {
      if (mux.registerName.startsWith('IOMUXC_SW_MUX_CTL_PAD_')) {
        if (pad != null) {
          var setValue = (getResetValue(pad.registerPadCtrl) & (getNonReadOnlyMask(pad.registerPadCtrl) ^ pad.registerClrMask)) | (pad.registerSetMask);
          //PExOut.gen(cIndent16 + 'MX6Q_PAD_' + mux.registerName.split('IOMUXC_SW_MUX_CTL_PAD_')[1] + '__' + bitFieldMux.findValueByRegisterValue(mux.registerSetMask).getName() + ' ' + getNumberConvertedToHex(setValue, false));
          PExOut.gen(cIndent16 + getDtsMacroPrefixName(mux) + getDtsMacroSwitchFuncName(mux, bitFieldMux) + ' ' + getNumberConvertedToHex(setValue, false));
        }
        else {
          //PExOut.gen(cIndent16 + 'MX6Q_PAD_' + mux.registerName.split('IOMUXC_SW_MUX_CTL_PAD_')[1] + '__' + bitFieldMux.findValueByRegisterValue(mux.registerSetMask).getName() + ' ' + getNumberConvertedToHex(getResetValue(mux.registerPadCtrl), false));
          PExOut.gen(cIndent16 + getDtsMacroPrefixName(mux) + getDtsMacroSwitchFuncName(mux, bitFieldMux) + ' ' + getNumberConvertedToHex(getResetValue(mux.registerPadCtrl), false));
        }
      }
    }
  }
}

// Get DTS macro prefix including register name for a given register  
function getDtsMacroPrefixName(reg) {
    var cpuVariant = PExScript.getMacroSymbol('CPUvariant');
	var dtsMacroPrefix = "UNKNOWN_PAD_"; // default
	//
    if (cpuVariant.indexOf('IMX6Q') >= 0) {                                      // i.MX6 Quad
      dtsMacroPrefix = "MX6QDL_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX6QP') >= 0) {                                // i.MX6 QuadPlus
      dtsMacroPrefix = "MX6QP_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX6D') >= 0) {                                 // i.MX6 Dual
      dtsMacroPrefix = "MX6QDL_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX6DP') >= 0) {                                // i.MX6 DualPlus
      dtsMacroPrefix = "MX6DP_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX6U') >= 0) {                                 // i.MX6 DualLite
      dtsMacroPrefix = "MX6QDL_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX6S') >= 0) {                                 // i.MX6 Solo
      dtsMacroPrefix = "MX6QDL_PAD_";
    }
    else if (cpuVariant.indexOf('IMX6L') >= 0) {                                 // i.MX6 SoloLite
      dtsMacroPrefix = "MX6SL_PAD_";
    }
    else if (cpuVariant.indexOf('IMX6X') >= 0) {                                 // i.MX6 SoloX
      dtsMacroPrefix = "MX6SX_PAD_";
    }
    else if (cpuVariant.indexOf('IMX6G') >= 0) {                                 // i.MX6 UltraLite
      dtsMacroPrefix = "MX6UL_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX7D') >= 0) {                                 // i.MX7 Dual
      dtsMacroPrefix = "MX7D_PAD_";
    } 
    else if (cpuVariant.indexOf('IMX7S') >= 0) {                                 // i.MX7 Solo
      dtsMacroPrefix = "MX7D_PAD_";
    } 
    return dtsMacroPrefix + reg.registerName.split('IOMUXC_SW_MUX_CTL_PAD_')[1] + '__';
}


/**
 * Returns DTS function name string
 * regDb - reference to database register
 * bitFieldMux - bit field reference to database register field 'MUX_MODE'
 */
// NOTE: Currently there're no aliases defined for <bit_field_value> in the device register database, method getAlias(string) will return null!
function getDtsMacroSwitchFuncName(regDb, bitFieldMux) {
  var bitValDb = bitFieldMux.findValueByRegisterValue(regDb.registerSetMask);
  var dtsFuncName = '';
  //
  if (bitValDb != null) {
    var bitValName = bitValDb.getName(); // reflect the bit field value name used for C code, e.g. 'SW_MUX_CTL_PAD_<pad-name>_MUX_MODE_<alt-x>' as defined in the device register database
    var bitValDesc = bitValDb.getDescription(); // reflect the descr text as defined in the device register database, typically 'Select signal <name>.' that can be processed for getting the DTS macro switching name 
    // process from descriptiomn of the bit field found by value
    if (bitValName.startsWith('SW_MUX_CTL_PAD_')) {
      // need to process description for getting DTS macro switching name
      dtsFuncName = bitValDesc.split('.')[0].replace('Select signal ','');
    }
    else {
      // will take the bit field value name as DTS macro switching name
      dtsFuncName = bitValName; 
    }
  } 
  return dtsFuncName;
}


/** 
 * Prints register configurations for one configuration into DTS generated output
 * configuration - given configuration where constant definitions is added
 * return value - no data; just printing into output 
 */
function printOneDtsRegisterConfiguration(configuration) {
  /* ***** Pin muxing, functional property Register configurations ***** */
  var configurationRegisterList = registerList[configuration];
  var configurationRegMuxList = new Array();
  var configurationRegPadList = new Array();
  var configurationRegOtherList = new Array();
  //
  for (var r in configurationRegisterList) {
    var regNameStr = configurationRegisterList[r].registerName;
    if (regNameStr.startsWith('IOMUXC_SW_MUX_CTL_PAD_') || regNameStr.startsWith('IOMUXC_SW_PAD_CTL_PAD_')) {
      var regPadNameStr = null;
      if (regNameStr.startsWith('IOMUXC_SW_MUX_CTL_PAD_')) {
        configurationRegMuxList.push(configurationRegisterList[r]);
        // search if corresponding pad control reg value is modified as well
        regPadNameStr = configurationRegisterList[r].registerName.replace('SW_MUX_CTL', 'SW_PAD_CTL');
        if (configurationRegisterList[regPadNameStr] != null) {
          //PExOut.gen('// r.pad = ' + configurationRegisterList[regPadNameStr].registerName);
          configurationRegPadList.push(configurationRegisterList[regPadNameStr]);
        }
        else {
          configurationRegPadList.push(null);
        }
      }
      else if (regNameStr.startsWith('IOMUXC_SW_PAD_CTL_PAD_')) {
        // search if corresponding mux control reg value is modified as well
        regPadNameStr = configurationRegisterList[r].registerName.replace('SW_PAD_CTL', 'SW_MUX_CTL');
        if (configurationRegisterList[regPadNameStr] == null) {
          //PExOut.gen('// r.pad = ' + configurationRegisterList[regPadNameStr].registerName);
          configurationRegOtherList.push(configurationRegisterList[r]);
        }
      }
    }
    else if (regNameStr.search('SELECT_INPUT') == -1) {
      configurationRegOtherList.push(configurationRegisterList[r]);
    }
  }
  // inner group of pins config
  PExOut.gen(cIndent12 + 'fsl,pins = <')
  // process mux + pad control register pairs
  if (configurationRegMuxList.length == configurationRegPadList.length) {
    for (var i = 0; i < configurationRegMuxList.length; i++) {
      //PExOut.gen('// gen reg pair: ' + configurationRegMuxList[i].registerName + ' + ' + ((configurationRegPadList[i] != null) ? configurationRegPadList[i].registerName : 'null'));
      printDtsRegister(configurationRegMuxList[i], configurationRegPadList[i]); // DTS config reg pair
    }
  }
  PExOut.gen(cIndent12 + '>;');
  // process all other registers, initialization using scheme: reg = <address value>;
  if (configurationRegOtherList.length > 0) {
      PExOut.gen(''); // leave blank line after DTS config reg pairs 
  }
  //
  for (var i = 0; i < configurationRegOtherList.length; i++) {
    //PExOut.gen('// gen other reg: ' + configurationRegOtherList[i].registerName);
    var regDb = configurationRegOtherList[i].registerPadCtrl;
    var regMask = 0xFFFFFFFF;
	var iomuxcBase = 0x20E0000;
    //var setValue = (getResetValue(regDb) & (getNonReadOnlyMask(regDb) ^ configurationRegOtherList[i].registerClrMask)) | (configurationRegOtherList[i].registerSetMask);
    var setValue = (getResetValue(regDb) & (regMask ^ configurationRegOtherList[i].registerClrMask)) | (configurationRegOtherList[i].registerSetMask);
    PExOut.gen(cIndent12 + '/* Set register ' + configurationRegOtherList[i].registerName + ' to ' + getNumberConvertedToHex(setValue, false) + ' */');
    PExOut.gen(cIndent12 + 'reg = <' + getNumberConvertedToHex(iomuxcBase + regDb.getOffset(), false) + ' ' + getNumberConvertedToHex(setValue, false) + '>;');
  }
}


/** 
 * Prints register configurations for one configuration
 * configuration - given configuration where constant definitions is added
 * return value - no data; just printing into output 
 */
function printOneRegisterConfiguration(configuration) {
    /* ***** Print one configuration of Pin muxing, functional property register modifications ***** */
    var configurationRegisterList = registerList[configuration];
    // Register configuration data for print ...
    var configurationRegisterDataPExForm = new Array();
    var configurationRegisterDataPinsForm = new Array();
    // enroll register configuration data
    for (var r in configurationRegisterList) {
        enrolOneRegisterConfigData(configurationRegisterList[r], configurationRegisterDataPExForm, configurationRegisterDataPinsForm);
    }
    // PExOut.gen(cIndent2 + '// Configuration of  ... *TBD ... pad name [ball map location]'); // Fixme: check if possible to create description for specific pad !!!
    /* ***** Print code in PEx Format - simple register modifications using direct numbers as value literals ***** */
    for (r in configurationRegisterDataPExForm) {
        PExOut.gen(configurationRegisterDataPExForm[r]); // print line
    }
    PExOut.gen(cIndent2 + '// Note: The commented code above is generated by the tool in case of simplified register modification using direct values only are needed.');
    /* ***** Print code in Pins Format - well commented code for register modifications using symbolic bit-field value names & reg macros defined in header files in SDK format ***** */
    for (r in configurationRegisterDataPinsForm) {
        PExOut.gen(configurationRegisterDataPinsForm[r]); // print line
    }
}


// get device part name for DTS
function getDevPartName() {
    var cpuVariant = PExScript.getMacroSymbol('CPUvariant');
	var devPartName = "imq6"; // default
	//
    if (cpuVariant.indexOf('IMX6Q') >= 0) {                                      // i.MX6 Quad
      devPartName = "imx6q";
    } 
    else if (cpuVariant.indexOf('IMX6QP') >= 0) {                                // i.MX6 QuadPlus
      devPartName = "imx6qp";
    } 
    else if (cpuVariant.indexOf('IMX6D') >= 0) {                                 // i.MX6 Dual
      devPartName = "imx6d";
    } 
    else if (cpuVariant.indexOf('IMX6DP') >= 0) {                                // i.MX6 DualPlus
      devPartName = "imx6dp";
    } 
    else if (cpuVariant.indexOf('IMX6U') >= 0) {                                 // i.MX6 DualLite
      devPartName = "imx6dl";
    } 
    else if (cpuVariant.indexOf('IMX6S') >= 0) {                                 // i.MX6 Solo
      devPartName = "imx6s";
    }
    else if (cpuVariant.indexOf('IMX6L') >= 0) {                                 // i.MX6 SoloLite
      devPartName = "imx6sl";
    }
    else if (cpuVariant.indexOf('IMX6X') >= 0) {                                 // i.MX6 SoloX
      devPartName = "imx6sx";
    }
    else if (cpuVariant.indexOf('IMX6G') >= 0) {                                 // i.MX6 UltraLite
      devPartName = "imx6ul";
    } 
    else if (cpuVariant.indexOf('IMX7D') >= 0) {                                 // i.MX7 Dual
      devPartName = "imx7d";
    } 
    else if (cpuVariant.indexOf('IMX7S') >= 0) {                                 // i.MX7 Solo
      devPartName = "imx7s";
    } 
    return devPartName;
}

// get device description string for DTS
function getDevDescStr() {
    var cpuVariant = PExScript.getMacroSymbol('CPUvariant');
	var devPartName = "6Quad"; // default
	//
    if (cpuVariant.indexOf('IMX6Q') >= 0) {                                      // i.MX6 Quad
      devPartName = "6Quad";
    } 
    else if (cpuVariant.indexOf('IMX6QP') >= 0) {                                // i.MX6 QuadPlus
      devPartName = "6QuadPlus";
    } 
    else if (cpuVariant.indexOf('IMX6D') >= 0) {                                 // i.MX6 Dual
      devPartName = "6Dual";
    } 
    else if (cpuVariant.indexOf('IMX6DP') >= 0) {                                // i.MX6 DualPlus
      devPartName = "6DualPlus";
    } 
    else if (cpuVariant.indexOf('IMX6U') >= 0) {                                 // i.MX6 DualLite
      devPartName = "6DualLite";
    }
    else if (cpuVariant.indexOf('IMX6S') >= 0) {                                 // i.MX6 Solo
      devPartName = "6Solo";
    }
    else if (cpuVariant.indexOf('IMX6L') >= 0) {                                 // i.MX6 SoloLite
      devPartName = "6SoloLite";
    }
    else if (cpuVariant.indexOf('IMX6X') >= 0) {                                 // i.MX6 SoloX
      devPartName = "6SoloX";
    }
    else if (cpuVariant.indexOf('IMX6G') >= 0) {                                 // i.MX6 UltraLite
      devPartName = "6UltraLite";
    }
    else if (cpuVariant.indexOf('IMX7D') >= 0) {                                 // i.MX7 Dual
      devPartName = "7Dual";
    } 
    else if (cpuVariant.indexOf('IMX7S') >= 0) {                                 // i.MX7 Solo
      devPartName = "7Solo";
    } 
    return '\"Freescale i.MX ' + devPartName + ' User Board\"';
}

// get device board compatibility string description for DTS
function getDevCompatibleBoardStr() {
    return '\"fsl,' + getDevPartName() + '-board\", \"fsl,' + getDevPartName() + '\"';
}

//
var constantDefinitionList = new Object();    // List of constant definition for used bit field values in register setting // Fixme: remove?
var registerList = new Object();              // List of registers for pin muxing and pin functional properties; register is defined by name, clear mask and set mask 


/* Creates h and c file and prints theirs body.
 * configuration - given configuration where constant definitions is added
 * return value - no data; just printing into output 
 */
function printDriver() {
  for (var coreIndex in notEmptyCoreIds) {
    /** create module name suffix for multicore */
    var coreId = notEmptyCoreIds[coreIndex];
	if (coreId == 'singlecore') {
      var moduleNameSuffix = "";
    } else {
      var moduleNameSuffix = "_" + coreId;
    }
    //
    constantDefinitionList = new Object(); // Fixme: remove?
    registerList = new Object();

    /** pre-process register configuration sequence content(s) */
    processRegisterConfigurationSequence(coreId);
  
    // continue writing output files ...

    /** Generate header .H file - "chFileName".h */
    PExOut.setOutputFile(chFileName + moduleNameSuffix + ".h");
    PExOut.gen('/*');
    printStringArray(headerCopyright, '', '');
    PExOut.gen(' */');
    PExOut.gen('');
  
    PExOut.gen('#ifndef _' + (moduleName + moduleNameSuffix).toUpperCase() + '_H_');
    PExOut.gen('#define _' + (moduleName + moduleNameSuffix).toUpperCase() + '_H_');
    PExOut.gen('');
  
    var cpuVariant = PExScript.getMacroSymbol('CPUvariant');
    //
    if (cpuVariant.indexOf('IMX6Q') >= 0) {                                      // i.MX6 Quad
      PExOut.gen('#include \"iMX6Q_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6QP') >= 0) {                                // i.MX6 QuadPlus
      PExOut.gen('#include \"iMX6QP_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6D') >= 0) {                                 // i.MX6 Dual
      PExOut.gen('#include \"iMX6D_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMXDQP') >= 0) {                                // i.MX6 DualPlus
      PExOut.gen('#include \"iMX6QP_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6U') >= 0) {                                 // i.MX6 DualLite
      PExOut.gen('#include \"iMX6DL_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6S') >= 0) {                                 // i.MX6 Solo
      PExOut.gen('#include \"iMX6S_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6L') >= 0) {                                 // i.MX6 SoloLite
      PExOut.gen('#include \"iMX6SL_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6X') >= 0) {                                 // i.MX6 SoloX
      PExOut.gen('#include \"iMX6SX_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX6G') >= 0) {                                 // i.MX6 UltraLite
      PExOut.gen('#include \"iMX6UL_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX7D') >= 0) {                                 // i.MX7 Dual
      PExOut.gen('#include \"iMX7D_registers.h\"');
    } 
    else if (cpuVariant.indexOf('IMX7S') >= 0) {                                 // i.MX7 Solo
      PExOut.gen('#include \"iMX7S_registers.h\"');
    } 
  
    PExOut.gen('');
    PExOut.gen('/*!');
    PExOut.gen(' * @addtogroup ' + moduleName + moduleNameSuffix);
    PExOut.gen(' * @{');
    PExOut.gen(' */');
    PExOut.gen('');
  
    PExOut.gen('/*******************************************************************************');
    PExOut.gen(' * API');
    PExOut.gen(' ******************************************************************************/');
    PExOut.gen('');
    PExOut.gen('#if defined(__cplusplus)');
    PExOut.gen('extern \"C\" {');
    PExOut.gen('#endif');
    PExOut.gen('');
  
    for (var pc = 0; pc < allComponents.length; pc++) {                          // Pin configuration represented by tables in the UI of the tool
      if (coreId == componentCoreIds[pc]) {
        configurationStrategy = configurationStrategies[pc];
        PExOut.gen('/*!');
        printStringArray(allComponents[pc].getDescription(), ' * ', '@brief ');
//        PExOut.gen(' * @brief Initializes selected pins.');
//        PExOut.gen(' *');
//        PExOut.gen(' * This function executes configuration of muxing/routing and'); 
//        PExOut.gen(' * optionally electrical features of selected pins.');
        PExOut.gen(' *');
        PExOut.gen(' */');
        PExOut.gen('void ' + configurationStrategy + '(void);');
        PExOut.gen('');
      }
    }

    PExOut.gen('#if defined(__cplusplus)');
    PExOut.gen('}');
    PExOut.gen('#endif');
    PExOut.gen('');
    PExOut.gen('/*!');
    PExOut.gen(' * @}');
    PExOut.gen(' */');
    PExOut.gen('#endif /* _' + (moduleName + moduleNameSuffix).toUpperCase() + '_H_ */');
    PExOut.gen('');
    PExOut.gen('/*******************************************************************************');
    PExOut.gen(' * EOF');
    PExOut.gen(' ******************************************************************************/');

    /** Generate C code file - "chFileName".c */
    PExOut.setOutputFile(chFileName + moduleNameSuffix + ".c");
  
    PExOut.gen('/*');
    printStringArray(headerCopyright, '', '');
    PExOut.gen(' */');
    PExOut.gen('');
    PExOut.gen('/*');
    PExOut.gen(PExProject.getYamlState());
    PExOut.gen(' */');
    PExOut.gen('');

    PExOut.gen('#include \"' + chFileName + moduleNameSuffix + '.h\"');

    for (var pc = 0; pc < allComponents.length; pc++) {                          // Pin configuration represented by tables in the UI of the tool
      if (coreId == componentCoreIds[pc]) {
        configurationStrategy = configurationStrategies[pc];
        PExOut.gen('');
        PExOut.gen('/*');
        PExOut.gen(allComponents[pc].getYamlState());                            // Function configuration details as YAML description got from the tool
        PExOut.gen(' */'); 
        PExOut.gen('');
        PExOut.gen('/*FUNCTION**********************************************************************');
        PExOut.gen(' *');
        PExOut.gen(' * Function Name : ' + configurationStrategy);
        printStringArray(allComponents[pc].getDescription(), ' * ', 'Description   : ');
//        PExOut.gen(' * Description   : This function executes configuration of muxing/routing and'); 
//        PExOut.gen(' * optionally electrical features of selected pins.');
        PExOut.gen(' *');
        PExOut.gen(' *END**************************************************************************/');
        PExOut.gen('void ' + configurationStrategy + '(void) {');

        printOneRegisterConfiguration(pc);

        PExOut.gen('}');
        PExOut.gen('');
      }
    }
  
    PExOut.gen('/*******************************************************************************');
    PExOut.gen(' * EOF');
    PExOut.gen(' ******************************************************************************/');
  }
}


/* Creates dtsi file and prints their body.
 * return value - no data; just printing into output 
 */
function printDtsDriver() {
  /** Generate common dtsi skeleton file - e.g. pin_mux.dtsi */
  PExOut.setOutputFile(getDevPartName() + "-board.dtsi");
	
  PExOut.gen('/*');
  printStringArray(headerDtsCopyright, '', '');
  PExOut.gen(' */');
  PExOut.gen('');
  PExOut.gen('/*');
  PExOut.gen(PExProject.getYamlState());
  PExOut.gen(' */');
  PExOut.gen('');
  PExOut.gen('/dts-v1/;');
  PExOut.gen('');
  
  PExOut.gen('#include \"skeleton.dtsi\"');
  PExOut.gen('#include \"' + getDevPartName() + '-pinfunc.h\"');
  PExOut.gen('');
	
  PExOut.gen('/ {');
  PExOut.gen(cIndent4 + 'model = ' + getDevDescStr() + ';');
  PExOut.gen(cIndent4 + 'compatible = ' + getDevCompatibleBoardStr() + ';');
  PExOut.gen('');
  PExOut.gen(cIndent4 + 'soc {');
  PExOut.gen(cIndent8 + '#address-cells = <1>;');
  PExOut.gen(cIndent8 + '#size-cells = <1>;');
  PExOut.gen('');
  PExOut.gen(cIndent8 + 'iomuxc: iomuxc@020e0000 {');
  PExOut.gen(cIndent12 + 'compatible = "fsl,' + getDevPartName() + '-iomuxc";');
  PExOut.gen(cIndent12 + 'reg = <0x020e0000 0x4000>;');
  PExOut.gen(cIndent8 + '};');
  PExOut.gen(cIndent4 + '};');
  PExOut.gen('};');
  PExOut.gen('');

  /** pre-process register(s) configuration sequence content */
  registerList = new Object();
  processDtsRegisterConfigurationSequence();

  /** Generate iomux section start */
  PExOut.gen('&iomuxc {');
  PExOut.gen(cIndent4 + 'pinctrl-names = \"default\";');
  PExOut.gen(cIndent4 + 'pinctrl-0 = <&' + configurationStrategies[0] + '>;');
  PExOut.gen('');

  /** Generate device tree iomux element content - go through all user-defined UI functions and their configuration */
  PExOut.gen(cIndent4 + getDevPartName() + '-board {');

  for (var pc = 0; pc < allComponents.length; pc++) {                          // Pin configuration represented by tables in the UI of the tool
    configurationStrategy = configurationStrategies[pc];

    // Generate function config device tree elements 
    PExOut.gen('');
    PExOut.gen(cIndent8 + configurationStrategy + ': ' + configurationStrategy + 'grp {');
    // Function config summary
    PExOut.gen('/*');
    PExOut.gen(allComponents[pc].getYamlState());                           // Function configuration details as YAML description got from the tool
    PExOut.gen(' */');
    // Function config details
    printOneDtsRegisterConfiguration(pc);
    //
    PExOut.gen(cIndent8 + '};');
    PExOut.gen('');
  }
  
  /** Generate iomux section end */
  PExOut.gen(cIndent4 + '};');
  PExOut.gen('};');
  PExOut.gen('');
}

// calls main C code generation function
printDriver();

// calls main DTS generation function
printDtsDriver();

//eof.