/*******************************************************************************
 * Copyright 2020, 2023 NXP
 * NXP Confidential and Proprietary. This software is owned or controlled by NXP and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 *******************************************************************************/
package com.freescale.s32ds.cdt.core2;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.IOption;
import org.eclipse.cdt.managedbuilder.core.IToolChain;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.cdt.managedbuilder.envvar.IBuildEnvironmentVariable;
import org.eclipse.cdt.managedbuilder.macros.IBuildMacroProvider;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;

public class PathEnvironmentVariableEpl implements IBuildEnvironmentVariable {

	public static final String NAME = "PATH"; //$NON-NLS-1$
	public static final String START_BRACKET = "${"; //$NON-NLS-1$
	public static final String END_BRACKET = "}"; //$NON-NLS-1$

	private List<String> paths = new ArrayList<>();

	protected PathEnvironmentVariableEpl(List<String> paths) {
		this.paths.addAll(paths);
	}

	@Override
	public String getDelimiter() {
		return File.pathSeparator; 
	}

	@Override
	public String getName() {
		return NAME;
	}

	@Override
	public int getOperation() {
		return IBuildEnvironmentVariable.ENVVAR_PREPEND;
	}

	@Override
	public String getValue() {
		return paths.stream()
				.collect(Collectors.joining(getDelimiter()));
	}

	public static PathEnvironmentVariableEpl create(IConfiguration configuration,
			String variable) {
		return create(configuration, new String[] { variable });
	}

	public static PathEnvironmentVariableEpl create(IConfiguration configuration,
			String[] variables) {
		return new PathEnvironmentVariableEpl(getPaths(configuration, variables));
	}

	protected static List<String> getPaths(IConfiguration configuration,
			String[] variables) {
		IToolChain toolchain = configuration.getToolChain();
		IOption option = toolchain
				.getOptionBySuperClassId("cdt.managedbuild.option.gnu.cross.path"); //$NON-NLS-1$
		String toolchainPathString = (option == null) ? "" : (String) option.getValue(); //$NON-NLS-1$
		toolchainPathString = resolveMacro(toolchainPathString, configuration);
		File sysroot = new File(toolchainPathString);
		File bin = new File(sysroot, "bin"); //$NON-NLS-1$
		if (bin.isDirectory())
			sysroot = bin;

		List<String> paths = new ArrayList<>();
		paths.add(sysroot.getAbsolutePath());

		for (String variable : variables) {
			Path pathFromVariable = new Path(resolveMacro(variable, configuration));
			paths.add(pathFromVariable.toOSString());
		}
		return paths;
	}

	public static boolean isVar(String name) {
		// Windows has case insensitive env var names
		return Platform.getOS().equals(Platform.OS_WIN32) ? name
				.equalsIgnoreCase(PathEnvironmentVariableEpl.NAME) : name
				.equals(PathEnvironmentVariableEpl.NAME);
	}

	private static String resolveMacroRecursive(String value , IConfiguration configuration) {
		if (value.contains(START_BRACKET)) {
			/* Start with checking if this comes from an IDE variable definition */
			String returnedVal = null;
			try {
				returnedVal = ManagedBuildManager.getBuildMacroProvider().resolveValue(value, "", " ", //$NON-NLS-1$ //$NON-NLS-2$
						IBuildMacroProvider.CONTEXT_CONFIGURATION, configuration);
				if (returnedVal != null && !returnedVal.equals(value)) {
					return returnedVal;
				}
			} catch (CoreException e) {
				// silent catch
			}
			/* then check the SHELL */
			int start = value.indexOf(START_BRACKET);
			int end = value.indexOf(END_BRACKET);
			String name = value.substring(start + 2, end);
			String envVar = System.getenv(name);
			if (envVar == null)
				envVar = System.getenv(name.toUpperCase(Locale.getDefault()));
			if (envVar != null)
				return value.substring(0, start) + envVar
						+ value.substring(end + 1);
		}
		return value;
	}

	protected static String resolveMacro(String value, IConfiguration configuration) {
		String result = resolveMacroRecursive(value, configuration);
		while (!result.equals(value)) {
			value = result;
			result = resolveMacroRecursive(value, configuration);
		}
		return result;
	}
}