/*******************************************************************************
 * Copyright (c) 2018 NXP
 *******************************************************************************/
package com.freescale.s32ds.cdt.core;

import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.ErrorParserManager;
import org.eclipse.cdt.core.IErrorParser;
import org.eclipse.cdt.core.IErrorParserNamed;
import org.eclipse.cdt.core.errorparsers.ErrorParserNamedWrapper;
import org.eclipse.cdt.internal.core.XmlUtil;
import org.eclipse.cdt.internal.core.model.Util;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.osgi.service.prefs.BackingStoreException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.freescale.s32ds.cdt.core.internal.errorparsers.FSLRegexErrorParser;

@SuppressWarnings("restriction")
public class FSLErrorParserExtensionManager {

	private static final String DEFAULT_IDS_FOR_ERRORPARSER = "errorparser.default.ids"; //$NON-NLS-1$
	private static final String EMPTY_LINE = ""; //$NON-NLS-1$

	private static final String ELEMENT_PLUGIN = "plugin"; //$NON-NLS-1$

	private static final String S32DS_ERRORPARSER_EXTENSIONS_LOCATION = "model.extensions.xml"; //$NON-NLS-1$
	private static final String TEST_PLUGIN_ID = "org.eclipse.cdt.core.tests"; //$NON-NLS-1$
	private static final String DEPRECATED = CCorePlugin.getResourceString("CCorePlugin.Deprecated"); //$NON-NLS-1$

	private static final LinkedHashMap<String, IErrorParserNamed> extensionId2ExtensionErrorParsers = new LinkedHashMap<>();
	private static final LinkedHashMap<String, IErrorParserNamed> parserId2AvailableErrorParsers = new LinkedHashMap<>();
	private static final Map<String, Set<String>> fErrorParserContextsMap = new HashMap<>();
	private static LinkedHashMap<String, IErrorParserNamed> fUserDefinedErrorParsersMap = null;
	private static List<String> fDefaultErrorParserIds = null;

	@SuppressWarnings("serial")
	private static class ErrorParserComparator implements Comparator<IErrorParserNamed>, Serializable {

		@Override
		public int compare(IErrorParserNamed errorParser1, IErrorParserNamed errorParser2) {
			boolean plugin1 = errorParser1.getId().startsWith(TEST_PLUGIN_ID);
			boolean plugin2 = errorParser2.getId().startsWith(TEST_PLUGIN_ID);
			int more = 1;
			int less = -1;
			if (plugin1 && !plugin2) {
				return more;
			}
			if (!plugin1 && plugin2) {
				return less;
			}
			boolean deprecated1 = errorParser1.getName().contains(DEPRECATED);
			boolean deprecated2 = errorParser2.getName().contains(DEPRECATED);
			if (deprecated1 == true && deprecated2 == false) {
				return more;
			}
			if (deprecated1 == false && deprecated2 == true) {
				return less;
			}
			return errorParser1.getName().compareTo(errorParser2.getName());
		}
	}

	static {
		loadUserDefinedErrorParsers();
		loadDefaultErrorParserIds();
		loadErrorParserExtensions();
	}

	public static synchronized void loadUserDefinedErrorParsers() {
		fUserDefinedErrorParsersMap = null;
		Document document = null;
		try {
			document = XmlUtil.loadXml(getStoreURI(S32DS_ERRORPARSER_EXTENSIONS_LOCATION));
		} catch (Exception e) {
			CCorePlugin.log(Messages.FSLErrorParserExtensionManager_CantLoadPreferencesFrom + S32DS_ERRORPARSER_EXTENSIONS_LOCATION, e);
		}
		if (document != null) {
			Set<IErrorParserNamed> sortedParsers = new TreeSet<>(new ErrorParserComparator());
			FSLErrorParserExtensionManagerUtils.loadExtensions(document, sortedParsers);
			if (!sortedParsers.isEmpty()) {
				fUserDefinedErrorParsersMap = new LinkedHashMap<>();
				for (IErrorParserNamed errorParser : sortedParsers) {
					fUserDefinedErrorParsersMap.put(errorParser.getId(), errorParser);
				}
			}
		}
		refreshAvailableErrorParsers();
	}

	public static synchronized void loadDefaultErrorParserIds() {
		fDefaultErrorParserIds = null;
		IEclipsePreferences eclipsePreferences = InstanceScope.INSTANCE.getNode(CCorePlugin.PLUGIN_ID);
		String iDs = eclipsePreferences.get(DEFAULT_IDS_FOR_ERRORPARSER, EMPTY_LINE);
		if (iDs.equals(EMPTY_LINE)) {
			return;
		}
		fDefaultErrorParserIds = Arrays.asList(iDs.split(String.valueOf(ErrorParserManager.ERROR_PARSER_DELIMITER)));
	}

	private static void refreshAvailableErrorParsers() {
		parserId2AvailableErrorParsers.clear();
		List<String> iDs = new ArrayList<>();
		if (fDefaultErrorParserIds != null) {
			findeAvaliableParsers(iDs);
		}
		Set<IErrorParserNamed> sortedErrorParsers = new TreeSet<>(new ErrorParserComparator());
		if (fUserDefinedErrorParsersMap != null) {
			for (String id : fUserDefinedErrorParsersMap.keySet()) {
				if (!iDs.contains(id)) {
					IErrorParserNamed errorParser = fUserDefinedErrorParsersMap.get(id);
					sortedErrorParsers.add(errorParser);
				}
			}
		}
		for (String id : extensionId2ExtensionErrorParsers.keySet()) {
			if (!iDs.contains(id)) {
				IErrorParserNamed errorParser = extensionId2ExtensionErrorParsers.get(id);
				sortedErrorParsers.add(errorParser);
			}
		}
		for (IErrorParserNamed errorParser : sortedErrorParsers) {
			parserId2AvailableErrorParsers.put(errorParser.getId(), errorParser);
		}
	}

	private static void findeAvaliableParsers(List<String> iDs) {
		for (String id : fDefaultErrorParserIds) {
			IErrorParserNamed iErrorParserNamed = null;
			if (fUserDefinedErrorParsersMap != null) {
				iErrorParserNamed = fUserDefinedErrorParsersMap.get(id);
			}
			if (iErrorParserNamed == null) {
				iErrorParserNamed = extensionId2ExtensionErrorParsers.get(id);
			}
			if (iErrorParserNamed != null) {
				parserId2AvailableErrorParsers.put(id, iErrorParserNamed);
				iDs.add(id);
			}
		}
	}

	public static synchronized void loadErrorParserExtensions() {
		Set<IErrorParserNamed> errorParserNameds = new TreeSet<>(new ErrorParserComparator());
		FSLErrorParserExtensionManagerUtils.loadErrorParserExtensions(Platform.getExtensionRegistry(), errorParserNameds,
				fErrorParserContextsMap);
		extensionId2ExtensionErrorParsers.clear();
		for (IErrorParserNamed iErrorParserNamed : errorParserNameds) {
			extensionId2ExtensionErrorParsers.put(iErrorParserNamed.getId(), iErrorParserNamed);
		}
		refreshAvailableErrorParsers();
	}

	public static void serializeDefinedErrorParsers() throws CoreException {
		try {
			URI uri = getStoreURI(S32DS_ERRORPARSER_EXTENSIONS_LOCATION);
			String eol = Util.getLineSeparator(uri);
			if (eol == null) {
				eol = Util.getDefaultLineSeparator();
			}
			Document document = XmlUtil.newDocument();
			Element element = XmlUtil.appendElement(document, ELEMENT_PLUGIN);
			if (fUserDefinedErrorParsersMap != null) {
				for (Entry<String, IErrorParserNamed> entry : fUserDefinedErrorParsersMap.entrySet()) {
					IErrorParserNamed errorParser = entry.getValue();
					FSLErrorParserExtensionManagerUtils.addErrorParserExtension(element, errorParser);
				}
			}
			XmlUtil.serializeXml(document, uri, eol);
		} catch (Exception e) {
			throw new CoreException(CCorePlugin.createStatus(
					Messages.bind(Messages.FSLErrorParserExtensionManager_FailedSerializingToFile, S32DS_ERRORPARSER_EXTENSIONS_LOCATION),
					e));
		}
	}

	public static String[] getErrorParserAvailableIdsInContext(String context) {
		List<String> iDs = new ArrayList<>();
		Set<String> keySet = parserId2AvailableErrorParsers.keySet();
		for (String id : keySet) {
			if (getContextOfErrorParsers(id).contains(context)) {
				iDs.add(id);
			}
		}
		int size = iDs.size();
		return iDs.toArray(new String[size]);
	}

	public static void serializeDefaultErrorParserIds() throws BackingStoreException {
		IEclipsePreferences preferences = InstanceScope.INSTANCE.getNode(CCorePlugin.PLUGIN_ID);
		String ids = EMPTY_LINE;
		if (fDefaultErrorParserIds != null) {
			ids = ErrorParserManager.toDelimitedString(fDefaultErrorParserIds.toArray(new String[0]));
		}

		preferences.put(DEFAULT_IDS_FOR_ERRORPARSER, ids);
		preferences.flush();
	}

	private static URI getStoreURI(String store) {
		IPath location = CCorePlugin.getDefault().getStateLocation().append(store);
		return URIUtil.toURI(location);
	}

	public static IErrorParser getErrorParserInternal(String id) {
		IErrorParserNamed iErrorParserNamed = parserId2AvailableErrorParsers.get(id);
		if (iErrorParserNamed instanceof ErrorParserNamedWrapper) {
			return ((ErrorParserNamedWrapper) iErrorParserNamed).getErrorParser();
		}
		return iErrorParserNamed;
	}

	public static void setUserDefinedErrorParsers(IErrorParserNamed[] errorParsers) throws CoreException {
		setDefinedErrorParsersInternal(errorParsers);
		serializeDefinedErrorParsers();
	}

	public static void setDefinedErrorParsersInternal(IErrorParserNamed[] errorParsers) {
		if (errorParsers == null) {
			fUserDefinedErrorParsersMap = null;
		} else {
			Set<IErrorParserNamed> sortedErrorParsers = new TreeSet<>(new ErrorParserComparator());
			sortedErrorParsers.addAll(Arrays.asList(errorParsers));
			fUserDefinedErrorParsersMap = new LinkedHashMap<>();
			for (IErrorParserNamed errorParser : sortedErrorParsers) {
				fUserDefinedErrorParsersMap.put(errorParser.getId(), errorParser);
			}
		}
		refreshAvailableErrorParsers();
	}

	public static String[] getErrorParserAvailableIds() {
		return parserId2AvailableErrorParsers.keySet().toArray(new String[parserId2AvailableErrorParsers.size()]);
	}

	public static String[] getErrorParserExtensionIds() {
		return extensionId2ExtensionErrorParsers.keySet().toArray(new String[extensionId2ExtensionErrorParsers.size()]);
	}

	public static String[] getUserDefinedErrorParserIds() {
		if (fUserDefinedErrorParsersMap != null) {
			return fUserDefinedErrorParsersMap.keySet().toArray(new String[0]);
		}
		return null;
	}

	public static void setDefaultErrorParserIds(String[] ids) throws BackingStoreException {
		setDefaultErrorParserIdsInternal(ids);
		serializeDefaultErrorParserIds();
	}

	public static void setDefaultErrorParserIdsInternal(String[] ids) {
		if (ids != null) {
			fDefaultErrorParserIds = new ArrayList<>(Arrays.asList(ids));
		} else {
			fDefaultErrorParserIds = null;
		}
		refreshAvailableErrorParsers();
	}

	public static String[] getDefaultErrorParserIds() {
		if (fDefaultErrorParserIds == null) {
			Set<String> keySet = parserId2AvailableErrorParsers.keySet();
			int size = parserId2AvailableErrorParsers.size();
			return keySet.toArray(new String[size]);
		}
		int size = fDefaultErrorParserIds.size();
		return fDefaultErrorParserIds.toArray(new String[size]);
	}

	public static IErrorParserNamed getErrorParserCopy(String iD, boolean isExtension) {
		IErrorParserNamed errorParserNamed = getErrorParserNamed(iD, isExtension);
		try {
			if (errorParserNamed instanceof ErrorParserNamedWrapper) {
				return (ErrorParserNamedWrapper) ((ErrorParserNamedWrapper) errorParserNamed).clone();
			} else if (errorParserNamed instanceof FSLRegexErrorParser) {
				return (FSLRegexErrorParser) ((FSLRegexErrorParser) errorParserNamed).clone();
			}
		} catch (CloneNotSupportedException e) {
			CCorePlugin.log(e);
		}
		return errorParserNamed;
	}

	private static IErrorParserNamed getErrorParserNamed(String iD, boolean isExtension) {
		IErrorParserNamed errorParserNamed = isExtension ? extensionId2ExtensionErrorParsers.get(iD)
				: parserId2AvailableErrorParsers.get(iD);
		return errorParserNamed;
	}

	private static Collection<String> getContextOfErrorParsers(String id) {
		Set<String> contexts = fErrorParserContextsMap.get(id);
		if (contexts == null) {
			return Collections.singletonList(ErrorParserManager.BUILD_CONTEXT);
		}
		return contexts;
	}
}
