/*******************************************************************************
 * Copyright (c) 2018 NXP
 *******************************************************************************/
package com.freescale.s32ds.cdt.core;

import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.net.URI;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.ErrorParserManager;
import org.eclipse.cdt.core.IConsoleParser;
import org.eclipse.cdt.core.IErrorParser;
import org.eclipse.cdt.core.IErrorParser2;
import org.eclipse.cdt.core.IErrorParserNamed;
import org.eclipse.cdt.core.IMarkerGenerator;
import org.eclipse.cdt.core.ProblemMarkerInfo;
import org.eclipse.cdt.core.errorparsers.ErrorParserNamedWrapper;
import org.eclipse.cdt.core.language.settings.providers.IWorkingDirectoryTracker;
import org.eclipse.cdt.core.resources.ACBuilder;
import org.eclipse.cdt.internal.core.IErrorMarkeredOutputStream;
import org.eclipse.cdt.internal.core.resources.ResourceLookup;
import org.eclipse.cdt.utils.EFSExtensionManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.URIUtil;

import com.freescale.s32ds.cdt.core.internal.errorparsers.CompilationPathResolverManager;
import com.freescale.s32ds.cdt.core.internal.errorparsers.ErrorParserDelegateManager;

@SuppressWarnings("restriction")
public class FSLErrorParserManagerExt extends ErrorParserManager
		implements IConsoleParser, IWorkingDirectoryTracker, IErrorParserProcessor {
	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	private final StringBuilder currentLine = new StringBuilder();
	private String previousLine;

	// TODO: contribute incLineCounter and decLineCounter to CDT and remove
	private int lineCounter = 0;

	// TODO: contribute as protected to CDT and remove
	private IMarkerGenerator fMarkerGeneratorSuper;

	// TODO: contribute as protected to CDT and remove
	private List<ProblemMarkerInfo> listOfErorrs;

	private List<ProblemMarkerInfo> listOfPendingErrors;

	// TODO: contribute as protected to CDT and remove
	private List<IErrorParser> errorParsers = null;

	private IErrorParserManagerDelegate[] delegates;
	private ICompilationPathResolver[] resolvers;
	private OutputStream outputStreamSuper;

	private IFSLErrorParser3 previousErrorParser = null;

	private List<String> fErrorMsgLines = new ArrayList<>();
	private boolean fIsErrorMsgComplete;
	private int nOpens;
	private Vector<URI> directoryStackSuper;
	private boolean hasErrors = false;

	private final OutputLineQueue outputLineQueue = new OutputLineQueue();

	public FSLErrorParserManagerExt(ACBuilder builder) {
		this(builder.getProject(), builder);
	}

	public FSLErrorParserManagerExt(IProject project, IMarkerGenerator markerGenerator) {
		this(project, markerGenerator, null);
	}

	public FSLErrorParserManagerExt(IProject project, IMarkerGenerator markerGenerator, String[] parsersIDs) {
		this(project, (URI) null, markerGenerator, parsersIDs);
	}

	@Deprecated
	public FSLErrorParserManagerExt(IProject project, IPath workingDirectory, IMarkerGenerator markerGenerator, String[] parsersIDs) {
		this(project, (workingDirectory == null || workingDirectory.isEmpty()) ? null
				: org.eclipse.core.filesystem.URIUtil.toURI(workingDirectory), markerGenerator, parsersIDs);
	}

	public FSLErrorParserManagerExt(IProject project, URI baseDirectoryURI, IMarkerGenerator markerGenerator, String[] parsersIDs) {
		super(project, findBaseDirectoryURI(baseDirectoryURI, project), markerGenerator, parsersIDs);
		this.listOfErorrs = new ArrayList<>();
		listOfPendingErrors = new ArrayList<>();
		resolvers = CompilationPathResolverManager.getResolvers();
		delegates = ErrorParserDelegateManager.getDelegates();
		for (IErrorParserManagerDelegate delegate : delegates) {
			delegate.initialize(this);
		}
	}

	@Override
	public Collection<IErrorParser> getErrorParsers() {
		if (errorParsers == null) {
			errorParsers = new ArrayList<>();
			try {
				@SuppressWarnings("unchecked")
				Map<String, IErrorParser[]> parsersMap = (Map<String, IErrorParser[]>) getSuperField("collectionsOfErrorParsers"); //$NON-NLS-1$
				for (IErrorParser[] ps : parsersMap.values()) {
					errorParsers.addAll(Arrays.asList(ps));
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return errorParsers;
	}

	@Override
	public void pushDirectory(IPath dir) {
		if (dir != null) {
			URI uri;
			URI workingDirectoryURI = getWorkingDirectoryURI();
			if (!dir.isAbsolute()) {
				uri = URIUtil.append(workingDirectoryURI, dir.toString());
			} else {
				uri = toURI(dir);
				if (uri == null) {
					return;
				}
			}
			pushDirectoryURI(uri);
		}
	}

	public IMarkerGenerator getMarkerGenerator() {
		if (fMarkerGeneratorSuper == null) {
			fMarkerGeneratorSuper = (IMarkerGenerator) getSuperField("markerGenerator"); //$NON-NLS-1$
		}
		return fMarkerGeneratorSuper;
	}

	protected OutputStream getOutputStreamSuper() {
		if (outputStreamSuper == null) {
			outputStreamSuper = (OutputStream) getSuperField("outputStream"); //$NON-NLS-1$
		}

		return outputStreamSuper;
	}

	@Override
	public void setOutputStream(OutputStream os) {
		outputStreamSuper = os;
		super.setOutputStream(os);
	}

	@Override
	public boolean processLine(String line) {

		registerOutputLine(line);
		for (IErrorParserManagerDelegate delegate : delegates) {
			if (delegate.processLine(line, this))
				return true;
		}
		return processErrorParsers(line, getErrorParsers());
	}

	public void registerOutputLine(String line) {
		registerOutputLine(line, false);
	}

	public void registerOutputLine(String line, boolean duplicated) {
		outputLineQueue.add(new OutputLine(line, duplicated));
	}

	@Override
	public boolean processErrorParsers(String line, Collection<IErrorParser> parsers) {
		lineCounter++;
		String trimmedLine = line.trim();
		ProblemMarkerInfo problemMarkerInfo = null;
		int longLine = 1000;
		pointer: for (IErrorParser parser : parsers) {
			IErrorParser currentErrorParser = parser;
			if (parser instanceof ErrorParserNamedWrapper) {
				currentErrorParser = ((ErrorParserNamedWrapper) parser).getErrorParser();
			}
			int parserTypes = IErrorParser2.NONE;
			if (currentErrorParser instanceof IErrorParser2) {
				parserTypes = ((IErrorParser2) currentErrorParser).getProcessLineBehaviour();
			}
			if ((parserTypes & IErrorParser2.KEEP_LONGLINES) == 0) {
				if (trimmedLine.length() > longLine)
					continue;
			}
			String parsedLine = trimmedLine;
			if ((parserTypes & IErrorParser2.KEEP_UNTRIMMED) != 0) {
				parsedLine = line;
			} else {
				outputLineQueue.registerAlias(line, trimmedLine);
			}
			boolean result = false;
			try {
				try {
					result = currentErrorParser.processLine(parsedLine, this);
				} catch (Exception e) {
					String id = EMPTY_STRING;
					if (parser instanceof IErrorParserNamed) {
						id = ((IErrorParserNamed) parser).getId();
					}
					String message = MessageFormat.format("Errorparser {0} failed parsing line [{1}]", id, parsedLine); //$NON-NLS-1$
					CCorePlugin.log(message, e);
				}
				if (result) {
					if (previousErrorParser != null && previousErrorParser != currentErrorParser) {
						// Because previous error parser errors occurred before
						// the current parser,
						// and because some error parser will only report errors
						// inside the IErrorParser3.flush()
						// call, we need to store the current pending error list
						// so the the previous parser's
						// errors are listed before the current ones, if any.
						List<ProblemMarkerInfo> temporaryErrors = listOfPendingErrors;
						listOfPendingErrors = new ArrayList<>();
						previousErrorParser.flush(this);
						listOfPendingErrors.addAll(temporaryErrors);
					}
					previousErrorParser = (IFSLErrorParser3) (currentErrorParser instanceof IFSLErrorParser3
							? (IFSLErrorParser3) currentErrorParser
							: null);
					flushPendingErrors();
					break pointer;
				}
				flushPendingErrors();
			} finally {
				if (listOfErorrs.size() > 0) {
					if (problemMarkerInfo == null)
						problemMarkerInfo = listOfErorrs.get(listOfErorrs.size() - 1);
					listOfErorrs.clear();
				}
			}
		}

		if (parsers.isEmpty()) {
			flushPendingErrors();
		}

		if (fErrorMsgLines.isEmpty())
			outputLine(line, problemMarkerInfo); // output line to console
													// immediately if
		// it is not a multi-line error message
		else
			outputPendingErrorMessages(problemMarkerInfo);

		return false;
	}

	/**
	 * Conditionally output line to outputStream. If stream supports error
	 * markers, use it, otherwise use conventional stream
	 */
	private void outputLine(String line, ProblemMarkerInfo marker) {

		outputLineQueue.setDone(line, marker);

		outputDoneLines();
	}

	/**
	 * Output lines to console from the head of the queue till it meets first
	 * undone line
	 */
	private void outputDoneLines() {
		OutputLine line;
		while ((line = outputLineQueue.pollDone()) != null) {
			outputALine(line.getLine(), line.getInfo());
		}
	}

	/**
	 * Conditionally output line to outputStream. If stream supports error
	 * markers, use it, otherwise use conventional stream
	 * 
	 * TODO: contribute as protected into CDT and remove
	 */
	private void outputALine(String line, ProblemMarkerInfo marker) {
		String l = line + "\n"; //$NON-NLS-1$
		if (getOutputStreamSuper() == null) {
			return;
		}
		try {
			if (marker != null && getOutputStreamSuper() instanceof IErrorMarkeredOutputStream) {
				IErrorMarkeredOutputStream s = (IErrorMarkeredOutputStream) getOutputStreamSuper();
				s.write(l, marker);
			} else {
				byte[] b = l.getBytes();
				getOutputStreamSuper().write(b, 0, b.length);
			}
		} catch (IOException e) {
			CCorePlugin.log(e);
		}
	}

	/**
	 * Collect all lines of an error/warning/info message in case it spans
	 * multiple lines
	 * 
	 * @param line
	 *            - a single line of an error/warning/info message
	 */
	public void addErrorMsgLine(String line) {
		fErrorMsgLines.add(line);
	}

	/**
	 * Indicate that all lines of error/warning/info message have been processed
	 * 
	 * @param isComplete
	 *            - true if entire message has been processed, else false
	 */
	public void setErrorMsgComplete(boolean isComplete) {
		fIsErrorMsgComplete = isComplete;
	}

	/**
	 * @return counter counting processed lines of output
	 * @since 5.2
	 */
	@Override
	public int getLineCounter() {
		return lineCounter;
	}

	@Override
	protected IFile findFileInWorkspace(IPath path) {
		URI uri;
		if (path.isAbsolute()) {
			uri = toURI(path);
			if (uri == null) {
				return null;
			}
		} else {
			URI workingDirectoryURI = getWorkingDirectoryURI();
			uri = EFSExtensionManager.getDefault().append(workingDirectoryURI, path.toString());
		}
		return findFileInWorkspace(uri);
	}

	@Override
	protected IFile findFileInWorkspace(URI uri) {
		IFile file = null;
		boolean isAbsolute = !uri.isAbsolute();
		if (isAbsolute) {
			uri = URIUtil.makeAbsolute(uri, getWorkingDirectoryURI());
		}
		file = ResourceLookup.selectFileForLocationURI(uri, getProject());
		if (file != null && file.isAccessible()) {
			return file;
		}
		return file;
	}

	private void outputPendingErrorMessages(ProblemMarkerInfo marker) {
		if (fIsErrorMsgComplete == true) {
			// only output a multi-line message once all lines of the message
			// have been processed
			fErrorMsgLines.iterator().forEachRemaining(line -> outputLine(line, marker));
			fErrorMsgLines.clear();
			fIsErrorMsgComplete = false;
		}
	}

	@Override
	public void generateExternalMarker(IResource file, int lineNumber, String desc, int severity, String varName, IPath externalPath) {
		if (file == null) {
			file = getProject();
		}
		externalPath = findExternalPath(externalPath);
		ProblemMarkerInfo problemMarkerInfo = new ProblemMarkerInfo(file, lineNumber, desc, severity, varName, externalPath);
		outputPendingErrorMessages(problemMarkerInfo);
		listOfPendingErrors.add(problemMarkerInfo);
	}

	private IPath findExternalPath(IPath externalPath) {
		boolean fileIsExists = (externalPath != null) && !externalPath.toFile().exists();
		if (fileIsExists) {
			for (ICompilationPathResolver compilationPathResolver : resolvers) {
				IPath path = compilationPathResolver.resolve(externalPath, getProject());
				boolean pathIsExists = (path != null) && path.toFile().exists();
				if (pathIsExists) {
					externalPath = path;
					break;
				}
			}
		}
		return externalPath;
	}

	@Override
	public void addProblemMarker(ProblemMarkerInfo problemMarkerInfo) {
		listOfPendingErrors.add(problemMarkerInfo);
	}

	/**
	 * Persist any pending error marker to the marker generator, and remove any
	 * pending error message line for each pending error, if required.
	 */
	public void flushPendingErrors() {
		for (int i = 0; i < listOfPendingErrors.size(); i++) {
			ProblemMarkerInfo problemMarkerInfo = listOfPendingErrors.get(i);
			listOfErorrs.add(problemMarkerInfo);
			getMarkerGenerator().addMarker(problemMarkerInfo);
			if (problemMarkerInfo.severity == IMarkerGenerator.SEVERITY_ERROR_RESOURCE)
				hasErrors = true;
			if (fErrorMsgLines.size() > 0)
				outputLine(fErrorMsgLines.remove(0), problemMarkerInfo);
		}
		listOfPendingErrors.clear();
	}

	@Override
	public boolean hasErrors() {
		return hasErrors;
	}

	/**
	 * @see java.io.OutputStream#close() Note: don't rely on this method to
	 *      close underlying OutputStream, close it explicitly
	 */
	@Override
	public synchronized void close() throws IOException {
		if (previousErrorParser != null) {
			previousErrorParser.flush(this);
			previousErrorParser = null;
		}
		flushPendingErrors();
		if (nOpens > 0 && --nOpens == 0) {
			checkLine(true);
			getDirectoryStack().removeAllElements();
		}

		// set all lines done at the end in order to output them to console
		outputLineQueue.setAllDone();
		outputDoneLines();
	}

	@SuppressWarnings("unchecked")
	private Vector<URI> getDirectoryStack() {
		if (directoryStackSuper == null) {
			directoryStackSuper = ((Vector<URI>) getSuperField("directoryStack")); //$NON-NLS-1$
		}
		return directoryStackSuper;
	}

	@Override
	public synchronized void write(int b) throws IOException {
		char charFromString = (char) b;
		currentLine.append(charFromString);
		checkLine(false);
	}

	@Override
	public synchronized void write(byte[] bytes, int off, int len) throws IOException {
		if (bytes == null) {
			throw new NullPointerException();
		} else if (off != 0 || (len < 0) || (len > bytes.length)) {
			throw new IndexOutOfBoundsException();
		} else if (len == 0) {
			return;
		}
		String str = new String(bytes, 0, len);
		currentLine.append(str);
		checkLine(false);
	}

	private void checkLine(boolean flush) {
		String buffer = currentLine.toString();
		int i = 0;
		while ((i = buffer.indexOf('\n')) != -1) {
			String line = buffer.substring(0, i);
			String suffix = "\r"; //$NON-NLS-1$
			if (line.endsWith(suffix)) {
				line = line.substring(0, line.length() - 1);
			}
			if (!line.equals(previousLine)) {
				processLine(line);
			} else {
				registerOutputLine(line, true);
				outputLine(line, null);
			}
			previousLine = line;
			buffer = buffer.substring(i + 1);
		}
		currentLine.setLength(0);
		if (flush) {
			if (buffer.length() > 0) {
				processLine(buffer);
				previousLine = buffer;
			}
		} else {
			currentLine.append(buffer);
		}
	}

	private URI toURI(IPath path) {
		URI baseURI = getWorkingDirectoryURI();
		String uriString = path.toString();
		boolean isCorrectPath = path.isAbsolute() && uriString.charAt(0) != IPath.SEPARATOR;
		if (isCorrectPath) {
			uriString = IPath.SEPARATOR + uriString;
		}
		return EFSExtensionManager.getDefault().createNewURIFromPath(baseURI, uriString);
	}

	/**
	 * @since 5.4
	 */
	@Override
	public void shutdown() {
		getErrorParsers().stream().filter(IFSLErrorParser3.class::isInstance).map(IFSLErrorParser3.class::cast)
				.forEach(IFSLErrorParser3::shutdown);
	}

	public IErrorParser getPreviousErrorParser() {
		return previousErrorParser;
	}

	public void resetPreviousErrorParser() {
		previousErrorParser = null;
	}

	public static URI findBaseDirectoryURI(URI baseDirectoryURI, IProject project) {
		if (baseDirectoryURI != null)
			return baseDirectoryURI;
		else if (project != null)
			return project.getLocationURI();
		else
			return org.eclipse.core.filesystem.URIUtil.toURI(System.getProperty("user.dir")); // CWD //$NON-NLS-1$
	}

	protected Object getSuperField(String superFieldName) {
		try {
			Field field = ErrorParserManager.class.getDeclaredField(superFieldName);
			field.setAccessible(true);
			Object obj = field.get(this);
			field.setAccessible(false);
			return obj;
		} catch (NoSuchFieldException e) {
			throw new RuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}
}