/**
 * Copyright 2017-2021 NXP
 */
package com.nxp.swtools.periphs.gui;

import java.util.Objects;

import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IStartup;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PerspectiveAdapter;
import org.eclipse.ui.PlatformUI;

import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.handler.MigrationOfferDialogHelper;
import com.nxp.swtools.periphs.gui.perspective.PeripheralsPerspective;
import com.nxp.swtools.periphs.gui.view.componentsettings.CollapsibleSettingsStorageHelper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingViewHelper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ProvidableSettingsSelectionStorageHelper;
import com.nxp.swtools.utils.profiler.Profiler;

/**
 * Class implementing startup of the peripherals tool
 * @author Juraj Ondruska
 */
public class PeripheralsStartup implements IStartup {
	/* (non-Javadoc)
	 * @see org.eclipse.ui.IStartup#earlyStartup()
	 */
	@Override
	public void earlyStartup() {
		if (SWToolsProperties.isPropertyTrue(SWToolsProperties.PROFILER_ENABLE_LOGGING)) {
			Profiler.getInstance(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL).enableLogging();
		}
		Controller.getInstance();
		if (!SWToolsProperties.isHeadlessOn()) {
			// add listener to refresh everything when switched to peripherals tool
			PerspectiveAdapter perspectiveListener = new PerspectiveAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.ui.PerspectiveAdapter#perspectiveActivated(org.eclipse.ui.IWorkbenchPage, org.eclipse.ui.IPerspectiveDescriptor)
				 */
				@Override
				public void perspectiveActivated(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
					if (perspective.getId().equals(PeripheralsPerspective.ID)) {
						// refresh when perspective switches
						Controller.getInstance().setReactToOtherToolChangesImmediately(true);
						Controller.getInstance().softRefresh(); // use only soft refresh because full refresh would drop all the model objects and re-create them, see MCUXCON-2882
						MigrationOfferDialogHelper.perspectiveActivated();
					}
				}

				/* (non-Javadoc)
				 * @see org.eclipse.ui.PerspectiveAdapter#perspectiveDeactivated(org.eclipse.ui.IWorkbenchPage, org.eclipse.ui.IPerspectiveDescriptor)
				 */
				@Override
				public void perspectiveDeactivated(@Nullable IWorkbenchPage page, @Nullable IPerspectiveDescriptor perspective) {
					super.perspectiveDeactivated(page, perspective);
					if (page == null) {
						return;
					}
					if (perspective == null) {
						return;
					}
					if (perspective.getId().equals(PeripheralsPerspective.ID)) {
						Controller.getInstance().setReactToOtherToolChangesImmediately(false);
					}
				}

				/* (non-Javadoc)
				 * @see org.eclipse.ui.PerspectiveAdapter#perspectiveChanged(org.eclipse.ui.IWorkbenchPage, org.eclipse.ui.IPerspectiveDescriptor, java.lang.String)
				 */
				@Override
				public void perspectiveChanged(IWorkbenchPage page, IPerspectiveDescriptor perspective, String changeId) {
					if (perspective.getId().equals(PeripheralsPerspective.ID)) {
						if (changeId.equals(IWorkbenchPage.CHANGE_RESET)) {
							ComponentSettingViewHelper.getInstance().setRemoveState(ComponentSettingViewHelper.REMOVE_DISABLED);
						}
						if (changeId.equals(IWorkbenchPage.CHANGE_RESET_COMPLETE)) {
							ComponentSettingViewHelper.getInstance().setRemoveState(ComponentSettingViewHelper.REMOVE_ENABLED);
							IViewPart viewPart = ComponentSettingViewHelper.getViewPart();
							if (viewPart == null) {
								// No active view
								return;
							}
							IViewSite viewSite = viewPart.getViewSite();
							if (viewSite == null) {
								return;
							}
						}
						if (changeId.equals(IWorkbenchPage.CHANGE_VIEW_SHOW)) {
							Controller.getInstance().setReactToOtherToolChangesImmediately(true);
						}
					}
				}
			};
			// Perspective listener
			IWorkbench workbench = PlatformUI.getWorkbench();
			ProvidableSettingsSelectionStorageHelper.getInstance();
			CollapsibleSettingsStorageHelper.getInstance();
			ComponentSettingViewHelper.getInstance();
			Objects.requireNonNull(workbench.getActiveWorkbenchWindow()).addPerspectiveListener(perspectiveListener);
			// Workbench listener
			workbench.addWorkbenchListener( new IWorkbenchListener()
			{
				@SuppressWarnings({ "null", "unused" }) //customer reported exception where the viewHelper was null
				@Override
				public boolean preShutdown(@Nullable IWorkbench activeWorkbench, boolean forced) {
					// Do not allow editors to save that they were closed
					ComponentSettingViewHelper viewHelper = ComponentSettingViewHelper.getInstance();
					if (viewHelper != null) {
						viewHelper.switchRemoveState();
						viewHelper.saveEditors();
					} else {
						assert false : "Null ComponentSettingViewHelper.getInstance()";  //$NON-NLS-1$
					}
					CollapsibleSettingsStorageHelper.getInstance().onShutdown();
					ProvidableSettingsSelectionStorageHelper.getInstance().onShutdown();
					return true;
				}
				@Override
				public void postShutdown(@Nullable IWorkbench activeWorkbench) {
					// Do nothing
				}
			});
		}
	}
}
