/**
 * Copyright 2021 NXP
 * Created: Mar 2 2021
 */
package com.nxp.swtools.periphs.gui.templates.component;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.swt.ControlDecorationUtils;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.data.mcu.IMcu;
import com.nxp.swtools.periphs.model.templates.component.ComponentTemplate;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.text.TextBoxHelper;
import com.nxp.swtools.utils.text.TextBoxHelper.Status;

/**
 * Dialog that serves for providing additional information before component template is created
 * @author Tomas Rudolf nxf31690
 */
public class ComponentTemplateCreationDialog extends Dialog {
	/** Percentage of parent shell width to be used as dialog's shell width */
	static final double SHELL_WIDTH_PERCENTAGE = 0.2;
	/** Minimal width of dialog's shell */
	static final int SHELL_MINIMAL_WIDTH = 400;
	/** Minimal height of dialog's shell */
	static final int SHELL_MINIMAL_HEIGHT = 400;
	/** Lines of description textbox */
	static final int DESCRIPTION_TEXTBOX_LINES = 5;

	/** Textbox with name */
	private @Nullable Text nameText;
	/** Textbox with description */
	private @Nullable Text descriptionText;
	/** Instance which should be converted into component template */
	private IComponentInstanceConfig instanceConfig;
	/** Controller wrapper */
	private IControllerWrapper controllerWrapper;

	/**
	 * Constructor
	 * @param parentShell shell in which the dialog should open
	 * @param instanceConfig instance which should be converted into component template
	 * @param controllerWrapper controller wrapper for access to Peripherals tool model and GUI
	 */
	protected ComponentTemplateCreationDialog(Shell parentShell, IComponentInstanceConfig instanceConfig, IControllerWrapper controllerWrapper) {
		super(parentShell);
		this.instanceConfig = instanceConfig;
		this.controllerWrapper = controllerWrapper;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText(Messages.get().ComponentTemplateCreationDialog_DialogTitle);
		Shell parentShell = getParentShell();
		if (parentShell != null) {
			int parentShellWidth = parentShell.getSize().x;
			int partOfParentWidth = (int) (parentShellWidth * SHELL_WIDTH_PERCENTAGE);
			int width = Math.max(partOfParentWidth, SHELL_MINIMAL_WIDTH);
			newShell.setSize(width, SHELL_MINIMAL_HEIGHT);
			newShell.setMinimumSize(width, SHELL_MINIMAL_HEIGHT);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		GridLayout layout = new GridLayout(1, false);
		composite.setLayout(layout);

		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText(Messages.get().ComponentTemplateCreationDialog_NameLabelText);
		nameLabel.setToolTipText(Messages.get().ComponentTemplateCreationDialog_NameLabelTooltip);
		nameLabel.setLayoutData(new GridData());

		Text nameTextLoc = nameText = new Text(composite, SWT.BORDER);
		nameTextLoc.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		ControlDecorationUtils.createErrorDecoration(nameTextLoc, SWT.LEFT | SWT.TOP);
		ControlDecorationUtils.setImage(nameTextLoc, ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR));
		SWTFactoryProxy.INSTANCE.setTestId(nameTextLoc, TestIDs.PERIPHS_TEMPLATE_NAME_TEXT);
		TextBoxHelper.attachModifyErrorListener(() -> UtilsText.EMPTY_STRING, nameTextLoc, (input) -> {
			if (templateExists(input)) {
				return Status.VALUE_ERROR;
			}
			return Status.OK;
		});
		TextBoxHelper.attachModifyListeners(nameTextLoc, (input) -> {
			boolean templateExists = templateExists(input);
			getButton(IDialogConstants.OK_ID).setEnabled(!templateExists);
			String errorDescription = templateExists ? Messages.get().ComponentTemplateCreationDialog_NameTextErrorTooltip : null;
			ControlDecorationUtils.updateControlDecoration(nameTextLoc, errorDescription);
		}, TextBoxHelper.TYPING_DELAY_50);

		Label descriptionLabel = new Label(composite, SWT.NONE);
		descriptionLabel.setText(Messages.get().ComponentTemplateCreationDialog_DescriptionLabelText);
		descriptionLabel.setToolTipText(Messages.get().ComponentTemplateCreationDialog_DescriptionLabelTooltip);
		descriptionLabel.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, false, false));

		Text descriptionTextLoc = descriptionText = new Text(composite, SWT.BORDER | SWT.MULTI);
		GridData descriptionTextLayoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		descriptionTextLayoutData.heightHint = DESCRIPTION_TEXTBOX_LINES * descriptionTextLoc.getLineHeight();
		descriptionTextLoc.setLayoutData(descriptionTextLayoutData);
		SWTFactoryProxy.INSTANCE.setTestId(descriptionTextLoc, TestIDs.PERIPHS_TEMPLATE_DESCRIPTION_TEXT);
		return composite;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}

	/**
	 * Checks whether the template with name from user input already exists or not
	 * @param input user entered name
	 * @return {@code true} when template with given name already exists, {@code false} otherwise
	 */
	private boolean templateExists(String input) {
		return controllerWrapper.getController().getMcu().getAvailableComponents().templateExists(input + UtilsText.UNDERSCORE + instanceConfig.getComponentTypeId());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		Text nameTextLoc = nameText;
		Text descriptionTextLoc = descriptionText;
		if (nameTextLoc != null && descriptionTextLoc != null) {
			IMcu mcu = controllerWrapper.getController().getMcu();
			if (ComponentTemplate.createAndSaveTemplateFiles(nameTextLoc.getText(), descriptionTextLoc.getText(), mcu.getPackage(), controllerWrapper.getController().getFunctionalGroup().getCore(), instanceConfig, mcu.getAvailableComponents())) {
				super.okPressed(); // Closes the dialog
				return;
			}
			MessageDialog.openError(getParentShell(), Messages.get().ComponentTemplateCreationDialog_CreationErrorTitle, Messages.get().ComponentTemplateCreationDialog_CreationErrorMessage);
			return; // Keep the creation dialog open
		}
		super.cancelPressed(); // When text inputs are not available then it should behave as cancel
	}

	/**
	 * Opens the dialog
	 * @param shell in which the dialog will open
	 * @param instanceConfig instance which should be stored in the new template
	 * @param controllerWrapper controller wrapper to access to Peripherals model and GUI
	 * @return {@code true} when dialog finished properly, {@code false} 
	 */
	public static boolean open(Shell shell, IComponentInstanceConfig instanceConfig, IControllerWrapper controllerWrapper) {
		ComponentTemplateCreationDialog dialog = new ComponentTemplateCreationDialog(shell, instanceConfig, controllerWrapper);
		dialog.setBlockOnOpen(true);
		return dialog.open() == OK;
	}

}
