/**
 * Copyright 2018-2022 NXP
 * Created: Jul 13, 2018
 */
package com.nxp.swtools.periphs.gui.view;

import java.io.File;
import java.io.IOError;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Objects;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.script.ScriptContext;
import javax.script.SimpleScriptContext;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.variables.VariablesPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.browser.LocationAdapter;
import org.eclipse.swt.browser.LocationEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.json.JSONException;
import org.json.JSONObject;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.files.UtilsFile;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.os.OSDetect;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.common.utils.xml.HtmlTags;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.data.AvailableComponents;
import com.nxp.swtools.provider.configuration.ConfigChangeReason;
import com.nxp.swtools.provider.configuration.ISharedConfiguration;
import com.nxp.swtools.provider.configuration.SharedConfigurationAdapter;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.Description;
import com.nxp.swtools.resourcetables.model.data.documentation.Contents;
import com.nxp.swtools.resourcetables.model.data.documentation.ISectionsProvider;
import com.nxp.swtools.resourcetables.model.data.documentation.Section;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.scripting.JavaScriptHelper;
import com.nxp.swtools.utils.support.markdown.MarkDownSupport;

/**
 * Displays component documentation in the markdown syntax
 * @author Viktar Paklonski
 * @author Tomas Rudolf - nxf31690
 */
public class DocumentationView extends APeriphsViewBase {
	/** Constant in javascript for logger */
	private static final String MENU_SCRIPT_DEBUG_LOGGER = "debugLogger"; //$NON-NLS-1$
	/** Constant in javascript for current component directory */
	private static final String MENU_SCRIPT_CURRENT_COMPONENT_DIRECTORY = "currentComponentDirectory"; //$NON-NLS-1$
	/** Constant in javascript for current page's file name */
	private static final String MENU_SCRIPT_CURRENT_PAGE = "currentPage"; //$NON-NLS-1$
	/** Constant in javascript for menu contents object */
	private static final String MENU_SCRIPT_CONTENTS = "contents"; //$NON-NLS-1$
	/** String "About:" */
	private static final @NonNull String ABOUT_COLON = "about:"; //$NON-NLS-1$
	/** Number of columns in header of this view */
	private static final int ITEMS_IN_HEADER_COUNT = 10;
	/** Separator used in links to other components. Example: other_component::some_page.md */
	public static final @NonNull String OTHER_COMPONENT_SEPARATOR = UtilsText.COLON + UtilsText.COLON;
	/** URI 'about:blank' */
	private static final @NonNull String ABOUT_BLANK = ABOUT_COLON + "blank"; //$NON-NLS-1$
	/** Prefix for files in URI */
	static final @NonNull String FILE_PREFIX = OSDetect.isWindows()? "file:///" : "file://"; //$NON-NLS-1$ //$NON-NLS-2$
	/** Extension used by markdown */
	private static final @NonNull String MARKDOWN_EXTENSION = "md"; //$NON-NLS-1$
	/** Logger of class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(DocumentationView.class);
	/** ID of the view */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.documentationView"; //$NON-NLS-1$
	/** markdown keyword */
	private static final @NonNull String RAW_DATA_PREFIX = "data:"; //$NON-NLS-1$
	/** file scheme name */
	private static final @NonNull String FILE_SCHEME = "file"; //$NON-NLS-1$
	/** The platform specific */
	transient @NonNull SWTFactoryProxy swtFactory = SWTFactoryProxy.INSTANCE;
	/** HTML viewer */
	@Nullable Composite viewer = null;
	/** Location of file with initial documentation content */
	@NonNull String contentLocation = UtilsText.EMPTY_STRING;
	/** HTML Document Tag */
	private static final @NonNull String HTML_DOC_TAG = "component-doc-link"; //$NON-NLS-1$
	/** TypeId of component that opened this view */
	@Nullable ConfigurationComponentTypeId componentTypeId;
	/** Stack of visited pages for navigation */
	LinkedList<@NonNull String> visitedPages = new LinkedList<>();
	/** Index of current page in stack */
	int currentPageIndex = 0;
	/** Content of menu */
	@Nullable Contents menuContent = null;

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.ViewPart#init(org.eclipse.ui.IViewSite)
	 */
	@Override
	public void init(IViewSite site) throws PartInitException {
		super.init(site);
		hideOthers();
		final String typeId = UtilsText.safeString(site.getSecondaryId());
		final AvailableComponents availComps = Controller.getInstance().getMcu().getAvailableComponents();
		ConfigurationComponentTypeId componentTypeIdLoc = availComps.getConfigCompTypeId(typeId);
		componentTypeId = componentTypeIdLoc;
		final boolean willBeClosed = updateCurrentView(typeId, availComps, componentTypeId);
		if (willBeClosed) {
			return;
		}
		if (componentTypeIdLoc == null) {
			return;
		}
		contentLocation = UtilsText.safeString(componentTypeIdLoc.getDocumentationLocation());
		menuContent = componentTypeIdLoc.getDocumentationMenuContent();
	}

	/**
	 * Checks that current view should still be open and closes it if not
	 * @param type of component
	 * @param availComps available components
	 * @param typeId of current component
	 * @return {@code true} if the view will be closed by this function or {@code false} if not
	 */
	boolean updateCurrentView(@Nullable String type, @NonNull AvailableComponents availComps, @Nullable ConfigurationComponentTypeId typeId) {
		if (type == null) {
			hideSelf();
			return true;
		}
		if (availComps.getConfigComps().isEmpty()) {
			hideSelf();
			return true;
		}
		if (typeId == null) {
			LOGGER.severe(String.format("[TOOL] A '%1s' with typeId '%2s' is missing among available components", ConfigurationComponentTypeId.class.getName(), type)); //$NON-NLS-1$
			hideSelf();
			return true;
		}
		if (!typeId.isDocumentationPresent()) {
			LOGGER.severe(String.format("[TOOL] No documentation found for '%1s' with typeId '%2s'", ConfigurationComponentTypeId.class.getName(), type)); //$NON-NLS-1$
			hideSelf();
			return true;
		}
		return false;
	}

	/**
	 * Hides (and closes a single instance of) the view.
	 */
	void hideSelf() {
		final Display display = Display.getCurrent();
		if (display != null) {
			display.asyncExec(new Runnable() {
				@Override
				public void run() {
					IViewSite site = getViewSite();
					assert site != null;
					IWorkbenchWindow workbenchWindow = site.getWorkbenchWindow();
					if (workbenchWindow == null) {
						return;
					}
					IWorkbenchPage activePage = workbenchWindow.getActivePage();
					if (activePage == null) {
						return;
					}
					activePage.hideView(DocumentationView.this);
				}
			});
		}
	}

	/**
	 * Hides other instances of this view.
	 */
	void hideOthers() {
		final Display display = Display.getCurrent();
		if (display != null) {
			display.asyncExec(new Runnable() {
				@Override
				public void run() {
					IViewSite site = getViewSite();
					assert site != null;
					IWorkbenchPage page = Objects.requireNonNull(Objects.requireNonNull(site.getWorkbenchWindow()).getActivePage());
					IViewReference[] views = page.getViewReferences();
					List<IViewReference> otherDocumentationViews = Arrays.asList(views).stream()
						.filter(x -> x.getId().equals(DocumentationView.ID))
						.filter(x -> !Objects.equals(x.getView(false), DocumentationView.this))
						.collect(CollectorsUtils.toList());
					otherDocumentationViews.forEach(x -> page.hideView(x));
				}
			});
		}
	}

	/**
	 * Returns labels of pages for page selector
	 * @param sectionsProvider provider of sections
	 * @param indent level of indentation
	 * @return list of strings that will be used in page selector
	 */
	private @NonNull List<@NonNull String> getPagesLabels(@Nullable ISectionsProvider sectionsProvider, int indent) {
		if (sectionsProvider == null) {
			return Collections.emptyList();
		}
		List<@NonNull String> result = new ArrayList<>();
		for (Section section : sectionsProvider.getSections()) {
			StringBuilder builder = new StringBuilder();
			for (int i = 0; i < indent; i++) {
				builder.append(UtilsText.SPACE);
				builder.append(UtilsText.SPACE);
				builder.append(UtilsText.SPACE);
			}
			if (indent != 0) {
				builder.append(UtilsText.MINUS_STRING);
			}
			builder.append(section.getName());
			result.add(builder.toString());
			if (section.hasSubSections()) {
				result.addAll(getPagesLabels(section, indent + 1));
			}
		}
		return result;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		assert parent != null;
		Composite contentComposite = createDefaultComposite(parent);
		contentComposite.setLayout(new GridLayout(ITEMS_IN_HEADER_COUNT, true));
		Button backButton = new Button(contentComposite, SWT.PUSH);
		backButton.setImage(ToolsImages.getImage(IToolsImages.ICON_LEFT));
		backButton.setToolTipText(Messages.get().DocumentationView_Back);
		backButton.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
		SWTFactoryProxy.INSTANCE.setTestId(backButton, TestIDs.PERIPHS_DOCUMENTATION_NAVIGATION_BACK_BUTTON);
		Button forwardButton = new Button(contentComposite, SWT.PUSH);
		forwardButton.setImage(ToolsImages.getImage(IToolsImages.ICON_RIGHT));
		forwardButton.setToolTipText(Messages.get().DocumentationView_Forward);
		forwardButton.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
		SWTFactoryProxy.INSTANCE.setTestId(forwardButton, TestIDs.PERIPHS_DOCUMENTATION_NAVIGATION_FORWARD_BUTTON);
		Button indexButton = new Button(contentComposite, SWT.PUSH);
		indexButton.setImage(ToolsImages.getImage(IToolsImages.ICON_OVERVIEW));
		indexButton.setToolTipText(Messages.get().DocumentationView_Home);
		indexButton.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
		SWTFactoryProxy.INSTANCE.setTestId(indexButton, TestIDs.PERIPHS_DOCUMENTATION_NAVIGATION_HOME_BUTTON);
		InstantSearchList pageSelector = new InstantSearchList(contentComposite, SWT.BORDER);
		GridData layoutData = new GridData(SWT.FILL, SWT.FILL, true, false, 5, 1);
		pageSelector.setLayoutData(layoutData);
		if (menuContent == null) {
			layoutData.exclude = true;
			pageSelector.setVisible(false);
		}
		List<@NonNull String> pagesLabels = getPagesLabels(menuContent, 0);
		pageSelector.setItems(pagesLabels.toArray(new @NonNull String[] {}));
		pageSelector.setSelection((!pagesLabels.isEmpty()) ? pagesLabels.get(0) : UtilsText.EMPTY_STRING);
		SWTFactoryProxy.INSTANCE.setTestId(pageSelector, TestIDs.PERIPHS_DOCUMENTATION_NAVIGATION_COMBOBOX);
		final Composite viewerLoc = swtFactory.createHtmlViewer(contentComposite, SWT.NONE);
		if (viewerLoc != null) {
			viewerLoc.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, ITEMS_IN_HEADER_COUNT, 1));
			viewer = viewerLoc;
			// FIXME TomasR v99 - Also support StyledTextWithImageHtml
			if (viewerLoc instanceof Browser) {
				Browser browser = (Browser) viewerLoc;
				pageSelector.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(@NonNull SelectionEvent e) {
						String selectionName = e.text;
						Section section = DocumentationViewHelper.findSectionByPredicate(menuContent, x -> selectionName.endsWith(x.getName()));
						if (section == null) {
							return;
						}
						String sectionFile = section.getFile();
						ConfigurationComponentTypeId componentTypeIdLoc = componentTypeId;
						if (componentTypeIdLoc == null) {
							return;
						}
						browser.setUrl(componentTypeIdLoc.getFileLocation().resolve(ConfigurationComponentTypeId.DOCUMENTATION_SUBFOLDER_NAME).resolve(sectionFile).toString());
					}
				});
				browser.addLocationListener(new LocationAdapter() {
					/* (non-Javadoc)
					 * @see org.eclipse.swt.browser.LocationAdapter#changing(org.eclipse.swt.browser.LocationEvent)
					 */
					@Override
					public void changing(LocationEvent event) {
						String location = UtilsText.safeString(event.location);
						if (location.endsWith(UtilsText.DOT + MARKDOWN_EXTENSION)) {
							if (location.startsWith(FILE_PREFIX)) {
								location = location.substring(FILE_PREFIX.length());
							}
							if (location.startsWith(ABOUT_COLON)) {
								location = location.substring(ABOUT_COLON.length());
							}
							handleContent(location);
							event.doit = false;
						}
						handleHistoryStack(location);
					}

					/**
					 * Handles showing of content
					 * @param location to file that should be shown
					 */
					private void handleContent(@NonNull String location) {
						ConfigurationComponentTypeId componentTypeIdLoc = componentTypeId;
						if (componentTypeIdLoc != null) {
							File pageFile = new File(location);
							String pageName = pageFile.getName();
							updatePageSelector(pageSelector, pageName);
							String fileContent = UtilsFile.readFileToString(pageFile.getAbsolutePath(), StandardCharsets.UTF_8);
							// Check if content should be opened externally
							String componentDescription = UtilsText.EMPTY_STRING;
							if ((fileContent != null)) {
								if (fileContent.contains(HTML_DOC_TAG)) {
									openDocInBrowser(fileContent, componentTypeIdLoc.getFileLocation().toAbsolutePath().toString());
								}
								Description description = componentTypeIdLoc.getConfigurationComponent().getDescription();
								if (description != null) {
									componentDescription = UtilsText.safeString(description.getDescription(controllerWrapper.getController().getProfile().getExpressionContext()));
								}
								browser.setText(getHtmlContentForBrowser(getMenuHtmlContent(pageName), fileContent, componentDescription));
							} else {
								browser.setText("<p>404 File not found</p>"); //$NON-NLS-1$
							}
						}
					}

					/**
					 * Handles history stack
					 * @param location that should be added to stack
					 */
					private void handleHistoryStack(@NonNull String location) {
						if (location.equals(ABOUT_BLANK)) {
							return; // Do not react on about blank
						}
						// Do not add the same page if it is already the last one
						if (visitedPages.size() > currentPageIndex) {
							String previousPage = visitedPages.get(currentPageIndex);
							if (previousPage.equals(location)) {
								return;
							}
						}
						// Remove future pages from history when user clicks on link
						if ((visitedPages.size() - 1) > currentPageIndex) { // Somewhere in history
							String nextPage = visitedPages.get(currentPageIndex + 1);
							if (!nextPage.equals(location)) { // Next page in history is not the same as currently loaded page
								ListIterator<@NonNull String> listIterator = visitedPages.listIterator(currentPageIndex + 1);
								while (listIterator.hasNext()) {
									listIterator.next();
									listIterator.remove();
								}
							} else {
								currentPageIndex++;
								return; // Already in history
							}
						}
						visitedPages.add(location);
						currentPageIndex++;
					}

					/* (non-Javadoc)
					 * @see org.eclipse.swt.browser.LocationAdapter#changed(org.eclipse.swt.browser.LocationEvent)
					 */
					@Override
					public void changed(LocationEvent event) {
						super.changed(event);
						backButton.setEnabled(currentPageIndex > 1);
						forwardButton.setEnabled(visitedPages.size() > (currentPageIndex + 1));
					}
				});
				if (!UtilsText.isEmpty(contentLocation)) { // MCUXCON-6929 - do not set empty URL
					browser.setUrl(contentLocation);
					visitedPages.add(contentLocation);
				}
				backButton.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(@NonNull SelectionEvent e) {
						if (currentPageIndex > 1) {
							currentPageIndex--;
							browser.setUrl(visitedPages.get(currentPageIndex));
						}
					}
				});
				forwardButton.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(@NonNull SelectionEvent e) {
						if (visitedPages.size() > (currentPageIndex + 1)) {
							currentPageIndex++;
							browser.setUrl(visitedPages.get(currentPageIndex));
						}
					}
				});
				indexButton.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(@NonNull SelectionEvent e) {
						browser.setUrl(contentLocation);
					}
				});
			}
		}
		SharedConfigurationFactory.getSharedConfigurationSingleton().addListener(new SharedConfigurationAdapter() {
			@Override
			public void configurationReloaded(ISharedConfiguration sharedConfig, ConfigChangeReason reason) {
				updateCurrentView((componentTypeId != null) ? componentTypeId.getTypeId() : UtilsText.EMPTY_STRING,
						Controller.getInstance().getMcu().getAvailableComponents(), componentTypeId);
				switch (reason) {
					case NEW_CONFIG: {
						hideSelf();
						break;
					}
					default: {
						// Do not react
					}
				}
			}
		});
	}

	/**
	 * Converts string with markdown notation to html
	 * @param markdown string to be converted
	 * @return string with markdown converted to HTML
	 */
	@NonNull String markdownToHTML(@Nullable String markdown) {
		ConfigurationComponentTypeId componentTypeIdLoc = componentTypeId;
		if (componentTypeIdLoc == null) {
			return UtilsText.EMPTY_STRING;
		}
		return MarkDownSupport.markDownToHtml(UtilsText.safeString(markdown), new Function<@NonNull String, @NonNull String>() {
			@Override
			public String apply(String address) {
				if (address.startsWith(RAW_DATA_PREFIX)) {
					// e.g. "data:image/png;base64,<...>"
					return address; // skip embedded data
				}
				// Check for link to another component
				ConfigurationComponentTypeId componentTypeIdForDocumentation = getTargetTypeId(address, controllerWrapper);
				if (componentTypeIdForDocumentation == null) {
					// Use current component
					componentTypeIdForDocumentation = componentTypeIdLoc;
				} else {
					// Address contained another component type id
					int indexOfDoubleColon = address.indexOf(OTHER_COMPONENT_SEPARATOR);
					address = address.substring(indexOfDoubleColon + OTHER_COMPONENT_SEPARATOR.length());
				}
				String typeId = componentTypeIdForDocumentation.getTypeId();
				// Local and online links or processed link to another component
				URI uri;
				try {
					uri = new URI(address);
				} catch (final URISyntaxException e) {
					LOGGER.log(Level.SEVERE, String.format("[TOOL] Invalid URI in '%1s' with typeId '%2s': %3s", //$NON-NLS-1$
							ConfigurationComponentTypeId.class.getName(), typeId, address), e);
					return address;
				}
				final String scheme = UtilsText.safeString(uri.getScheme());
				if (!scheme.isEmpty() && !scheme.equals(FILE_SCHEME)) {
					// e.g. https://www.nxp.com/
					return address; // don't know how to process non-file URIs
				}
				Path filePath = Paths.get(uri.getSchemeSpecificPart()); // cannot use Paths.get(uri); since it requires FILE_SCHEME
				if (!filePath.isAbsolute()) {
					try {
						filePath = componentTypeIdForDocumentation.resolvePath(filePath).toAbsolutePath();
					} catch (IOError e) {
						LOGGER.log(Level.SEVERE, String.format("[TOOL] Failed to resolve URI in '%1s' with typeId '%2s': %3s", //$NON-NLS-1$
							ConfigurationComponentTypeId.class.getName(), typeId, address), e);
					}
				}
				return filePath.toUri().toString();
			}
		});
	}

	/**
	 * Parses the given address to find component that contains the required page
	 * @param address to the page
	 * @param wrapper
	 * @return current component if address does not contain reference to another component,
	 */
	public static @Nullable ConfigurationComponentTypeId getTargetTypeId(@NonNull String address, @NonNull IControllerWrapper wrapper) {
		ConfigurationComponentTypeId componentTypeIdForDocumentation = null;
		// Link to another component
		int indexOfDoubleColon = address.indexOf(OTHER_COMPONENT_SEPARATOR);
		if (indexOfDoubleColon != -1) {
			String type = address.substring(0, indexOfDoubleColon);
			componentTypeIdForDocumentation = wrapper.getController().getProfile().getActiveComponents().getComponentTypeIdByType(type);
			if (componentTypeIdForDocumentation == null) {
				AvailableComponents availableComponents = wrapper.getController().getMcu().getAvailableComponents();
				List<@NonNull ConfigurationComponentTypeId> componentTypeIdsByType = availableComponents.getComponentTypeIdsByType(type);
				if (componentTypeIdsByType == null) {
					LOGGER.log(Level.SEVERE, String.format("[TOOL] Component with type %s is not available on current mcu or does not exist", type));  //$NON-NLS-1$
					return null;
				}
				componentTypeIdForDocumentation = availableComponents.getBestMatchingConfigCompType(componentTypeIdsByType);
				if (componentTypeIdForDocumentation == null) {
					LOGGER.log(Level.WARNING, String.format("[TOOL] No available component with type %s should be used. Using random version for documentation only!", type));  //$NON-NLS-1$
					return componentTypeIdsByType.get(0);
				}
			}
		}
		return componentTypeIdForDocumentation;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		final Composite viewerLoc = viewer;
		if (viewerLoc != null) {
			viewerLoc.setFocus();
		}
	}

	/**
	 * Reloads content of documentation view
	 */
	public void reload() {
		ConfigurationComponentTypeId componentTypeIdLoc = componentTypeId;
		if (componentTypeIdLoc != null) {
			menuContent = componentTypeIdLoc.getDocumentationMenuContent(); // Menu needs to be reloaded explicitly
		}
		if (viewer instanceof Browser) {
			Browser browser = (Browser) viewer;
			try {
				browser.setUrl(visitedPages.get(currentPageIndex));
			} catch (@SuppressWarnings("unused") IndexOutOfBoundsException ex) {
				LOGGER.log(Level.WARNING, "[TOOL] Problem in Documentation View occured when reload was issued due to IndexOutOfBoundException"); //$NON-NLS-1$
			}
		}
	}

	/**
	 * Goes to given page in the browser
	 * @param page to be opened
	 */
	public void goToPage(@NonNull String page) {
		ConfigurationComponentTypeId componentTypeIdLoc = componentTypeId;
		if (componentTypeIdLoc == null) {
			return;
		}
		ConfigurationComponentTypeId targetTypeId = getTargetTypeId(page, controllerWrapper);
		if (targetTypeId == null) {
			targetTypeId = componentTypeIdLoc;
		} else {
			// Address contained another component type id
			int indexOfDoubleColon = page.indexOf(OTHER_COMPONENT_SEPARATOR);
			page = page.substring(indexOfDoubleColon + OTHER_COMPONENT_SEPARATOR.length());
		}
		Composite viewerLoc = viewer;
		if (viewerLoc instanceof Browser) {
			Browser browser = (Browser) viewerLoc;
			browser.setUrl(targetTypeId.getFileLocation().resolve(ConfigurationComponentTypeId.DOCUMENTATION_SUBFOLDER_NAME).resolve(page).toString());
		}
	}

	/**
	 * Open documentation view.
	 * @param viewSite with binding to the view
	 * @param typeId of the component to show documentation for
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if the view was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@NonNull IViewSite viewSite, @NonNull String typeId, boolean activate) {
		return open(viewSite.getPage(), typeId, activate);
	}

	/**
	 * Open documentation view.
	 * @param workbenchPage with binding to the view
	 * @param typeId of the component to show documentation for
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if the view was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@Nullable IWorkbenchPage workbenchPage, @NonNull String typeId, boolean activate) {
		if (workbenchPage != null) {
			try {
				workbenchPage.showView(ID, typeId, activate ? IWorkbenchPage.VIEW_ACTIVATE : IWorkbenchPage.VIEW_VISIBLE);
				return true;
			} catch (@SuppressWarnings("unused") PartInitException e) {
				return false;
			}
		}
		return false;
	}

	/**
	 * Tries to refresh currently opened Documentation view if there is one
	 */
	public static void refresh() {
		DocumentationView instance = getInstance();
		if (instance != null) {
			instance.reload();
		}
	}

	/**
	 * Tries to open page in Documentation view
	 * @param page to be opened
	 */
	public static void openPage(String page) {
		DocumentationView instance = getInstance();
		if (instance != null) {
			instance.goToPage(page);
		} else {
			LOGGER.log(Level.SEVERE, "[TOOL] The Documentation view is not open now"); //$NON-NLS-1$
		}
	}

	/**
	 * Get instance of currently open Documentation view
	 * @return current view when is open now, otherwise returns {@code null}
	 */
	private static @Nullable DocumentationView getInstance() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench == null) {
			return null;
		}
		IWorkbenchWindow activeWorkbenchWindow = workbench.getActiveWorkbenchWindow();
		if (activeWorkbenchWindow == null) {
			return null;
		}
		IWorkbenchPage activePage = activeWorkbenchWindow.getActivePage();
		if (activePage == null) {
			return null;
		}
		IViewReference[] viewReferences = activePage.getViewReferences();
		for (IViewReference reference : viewReferences) {
			if (ID.equals(reference.getId())) {
				IViewPart view = reference.getView(false);
				if (view instanceof DocumentationView) {
					return (DocumentationView) view;
				}
			}
		}
		return null;
	}

	/**
	 * Try to open the documentation in browser
	 * The path to the .html file can be: ${ENV_PATH}/doc/html_${CPU}/group__flexcan__driver.html or
	 * a relative path to components folder. If ${CPU} is found, it will be replaced with current processor.
	 * @param docContent - component documentation
	 * @param componentLocation - full path to the .component file
	 */
	void openDocInBrowser(@NonNull String docContent, @NonNull String componentLocation) {
		try {
			JSONObject jsonOb = new JSONObject(docContent);
			String link = jsonOb.get(HTML_DOC_TAG).toString();
			// try to replace the &{CPU} with the mcu name
			link = link.replace("${CPU}", Controller.getInstance().getMcu().getMcuIdentification().getMcu()); //$NON-NLS-1$
			// try to replace the &{CPU_FAMILY} with the family of the MCU
			link = link.replace("${CPU_FAMILY}", Controller.getInstance().getMcu().getFamily()); //$NON-NLS-1$
			// replace the environment variables declared in the link. If the link doesn't contains env var, the link will be the same
			link = VariablesPlugin.getDefault().getStringVariableManager().performStringSubstitution(link, true);
			if (!Paths.get(link).isAbsolute()) {
				link = Paths.get(componentLocation, link).normalize().toString();
			}
			SWTFactoryProxy.INSTANCE.openUrlInExternalBrowser(UtilsText.safeString(link));
			hideSelf();
		} catch (JSONException | CoreException e) {
			LOGGER.info(e.getMessage());
		}
	}

	/**
	 * Returns content of menu in HTML
	 * @param currentPageFileName file name of current page
	 * @return string with HTML content of menu
	 */
	@NonNull String getMenuHtmlContent(String currentPageFileName) {
		String menu = UtilsText.EMPTY_STRING;
		Contents menuContentLoc = menuContent;
		ConfigurationComponentTypeId componentTypeIdLoc = componentTypeId;
		if ((menuContentLoc != null) && (componentTypeIdLoc != null)) {
			Object result = null;
			SimpleScriptContext simpleScriptContext = new SimpleScriptContext();
			simpleScriptContext.setAttribute(MENU_SCRIPT_DEBUG_LOGGER, LOGGER, ScriptContext.ENGINE_SCOPE);
			simpleScriptContext.setAttribute(MENU_SCRIPT_CONTENTS, menuContentLoc, ScriptContext.ENGINE_SCOPE);
			simpleScriptContext.setAttribute(MENU_SCRIPT_CURRENT_PAGE, currentPageFileName, ScriptContext.ENGINE_SCOPE);
			simpleScriptContext.setAttribute(MENU_SCRIPT_CURRENT_COMPONENT_DIRECTORY, componentTypeIdLoc.getFileLocation().toString(), ScriptContext.ENGINE_SCOPE);
			ConfigurationComponentTypeId systemComponent = controllerWrapper.getController().getSystemComponent();
			if (systemComponent != null) {
				File generatorFile = systemComponent.getDocumentationMenuGeneratorScript();
				if (generatorFile != null) {
					try {
						result = JavaScriptHelper.eval(generatorFile, simpleScriptContext);
					} catch (Exception e) {
						LOGGER.log(Level.SEVERE, "[TOOL] Generation of menu content failed. ", e); //$NON-NLS-1$
					}
				}
			}
			if (result instanceof String) {
				menu = (String) result;
			} else {
				LOGGER.log(Level.SEVERE, "[DATA] Result of menu generator is not a string, but : {0}", result); //$NON-NLS-1$
			}
		}
		return menu;
	}

	/**
	 * Updates page selector to show current page
	 * @param pageSelector page selector
	 * @param fileName file name of page which should be selected
	 */
	void updatePageSelector(InstantSearchList pageSelector, String fileName) {
		Section currentSection = DocumentationViewHelper.findSectionByPredicate(menuContent, x -> x.getFile().equals(fileName));
		if (currentSection != null) {
			String selection = CollectionsUtils.nullableOptionalGet(pageSelector.getItems().stream().filter(x -> x.endsWith(currentSection.getName())).findAny());
			if (selection != null) {
				pageSelector.select(selection);
			}
		}
	}

	/**
	 * Creates HTML content for browser
	 * @param menuHtmlContent content of menu in HTML
	 * @param fileMdContent content of page in MD
	 * @param title to be shown in header
	 * @return content as string with HTML
	 */
	@NonNull String getHtmlContentForBrowser(@NonNull String menuHtmlContent, @NonNull String fileMdContent, @NonNull String title) {
		StringBuilder builder = new StringBuilder(HtmlTags.HTML_BEG);
		builder.append(HtmlTags.HEAD_BEG);
		// FIXME TomasR v99 - consider creation of HTML head content
		builder.append(HtmlTags.HEAD_END);
		builder.append(HtmlTags.BODY_BEG);
		if (menuContent != null) {
			builder.append(HtmlTags.H1_BEG).append(title).append(HtmlTags.H1_END);
		}
		builder.append(menuHtmlContent).append(markdownToHTML(fileMdContent)).append(menuHtmlContent);
		builder.append(HtmlTags.BODY_END).append(HtmlTags.HTML_END);
		return builder.toString();
	}
}
