/**
 * Copyright 2019-2020 NXP
 * Created: 7 Aug 2019
 */
package com.nxp.swtools.periphs.gui.view;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IViewSite;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.utils.TestIDs;

/**
 * Dialog used for editing comment of instance
 * @author Tomas Rudolf - nxf31690
 */
public class EditCommentDialog extends Dialog {
	/** Associated view site */
	final @NonNull IViewSite viewSite;
	/** Instance from which the migration will be invoked */
	final @NonNull IComponentInstanceConfig instance;
	/** The controller wrapper */
	final @NonNull IControllerWrapper controllerWrapper;
	/** Reference to text box */
	private @Nullable Text textBox;
	/** Button that invokes save */
	private @Nullable Button saveButton;

	/**
	 * Constructor.
	 * @param viewSite associated view site
	 * @param instance from which to migrate
	 * @param controllerWrapper containing the generic controller
	 */
	private EditCommentDialog(@NonNull IViewSite viewSite, @NonNull IComponentInstanceConfig instance, @NonNull IControllerWrapper controllerWrapper) {
		super(viewSite.getWorkbenchWindow().getShell());
		this.viewSite = viewSite;
		this.instance = instance;
		this.controllerWrapper = controllerWrapper;
	}

	/**
	 * Open MigrateComponent dialog
	 * @param viewSite to open new instances view
	 * @param instance to migrate from
	 * @param controllerWrapper wrapper with controllers to work with
	 * @return dialog instance
	 */
	public static @NonNull EditCommentDialog open(@NonNull IViewSite viewSite, @NonNull IComponentInstanceConfig instance,
			@NonNull IControllerWrapper controllerWrapper) {
		EditCommentDialog dialog = new EditCommentDialog(viewSite, instance, controllerWrapper);
		dialog.setBlockOnOpen(true);
		dialog.updateSaveButtonState();
		dialog.open();
		return dialog;
	}

	/** Enables the "Save" button */
	protected void updateSaveButtonState() {
		Button saveButtonLoc = saveButton;
		if (saveButtonLoc != null) {
			saveButtonLoc.setEnabled(true);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite dialogComposite = (Composite) super.createDialogArea(parent);
		dialogComposite.setLayout(new GridLayout());
		dialogComposite.getShell().setText(Messages.get().EditCommentDialog_Title);
		Text text = new Text(dialogComposite, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL);
		SWTFactoryProxy.INSTANCE.setTestId(text, TestIDs.PERIPHS_EDIT_COMMENT_TEXTBOX);
		String comment = instance.getComment();
		text.setText(comment);
		text.setSelection(comment.length());
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.heightHint = 15 * text.getLineHeight();
		data.widthHint = 700;
		text.setLayoutData(data);
		textBox = text;
		return dialogComposite;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		saveButton = createButton(parent, IDialogConstants.OK_ID, Messages.get().EditCommentDialog_SaveButton, true);
		SWTFactoryProxy.INSTANCE.setTestId(saveButton, TestIDs.PERIPHS_ACTION_SAVE_EDIT_COMMENT);
		createButton(parent, IDialogConstants.CANCEL_ID, com.nxp.swtools.common.ui.utils.Messages.get().MessageBoxDialog_Cancel, false);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		if (textBox != null) {
			String comment = textBox.getText();
			APeriphController controller = controllerWrapper.getController();
			controller.setComment(instance, comment, EditCommentDialog.class);
		}
		super.okPressed();
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#cancelPressed()
	 */
	@Override
	protected void cancelPressed() {
		close();
	}
}
