/**
 * Copyright 2017-2020 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IViewSite;
import org.osgi.framework.Version;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.ComparatorHelpers;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.MigrationHelper;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.SWComponent;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Dialog used for migration from old to new component version.
 * @author Tomas Rudolf - nxf31690
 */
// FIXME TomasR v99 - Unify with AddComponentDialog from which this was copied
public class MigrateComponentDialog extends Dialog {
	/** Associated view site */
	final @NonNull IViewSite viewSite;
	/** Instance from which the migration will be invoked */
	final @NonNull IComponentInstanceConfig instance;
	/** The controller wrapper */
	final @NonNull IControllerWrapper controllerWrapper;
	/** Reference to table viewer */
	private @Nullable TableViewer tableViewer;
	/** Button that invokes migration */
	private @Nullable Button migrateButton;
	/** Result of this dialog */
	private @Nullable IChildProvidable result = null;
	/** This number is used to compute the table's height; number of rows that should be visible by default */
	private static final int DEFAULT_ROWS_NUM = 12;
	/** This number is used to compute the table's width given its height */
	private static final float DEFAULT_VIEWER_WIDTH_HEIGHT_RATIO = 16f / 9f;

	/**
	 * Constructor.
	 * @param viewSite associated view site
	 * @param instance from which to migrate
	 * @param controllerWrapper containing the generic controller
	 */
	private MigrateComponentDialog(@NonNull IViewSite viewSite, @NonNull IComponentInstanceConfig instance, @NonNull IControllerWrapper controllerWrapper) {
		super(viewSite.getWorkbenchWindow().getShell());
		this.viewSite = viewSite;
		this.instance = instance;
		this.controllerWrapper = controllerWrapper;
	}

	/**
	 * Open MigrateComponent dialog
	 * @param viewSite to open new instances view
	 * @param instance to migrate from
	 * @param controllerWrapper wrapper with controllers to work with
	 * @return dialog instance
	 */
	public static @NonNull MigrateComponentDialog open(@NonNull IViewSite viewSite, @NonNull IComponentInstanceConfig instance,
			@NonNull IControllerWrapper controllerWrapper) {
		MigrateComponentDialog dialog = new MigrateComponentDialog(viewSite, instance, controllerWrapper);
		dialog.setBlockOnOpen(true);
		dialog.open();
		return dialog;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite dialogComposite = (Composite) super.createDialogArea(parent);
		dialogComposite.setLayout(new GridLayout());
		Composite optionsComposite = new Composite(dialogComposite, dialogComposite.getStyle());
		optionsComposite.setLayout(new GridLayout(2, true));  // 2 is SWT.CHECK button count (onlyPrjCompsCheckBox and allCompsCheckBox)
		dialogComposite.getShell().setText(Messages.get().ComponentsView_SelectComponentDialog_Title);
		// create list viewer
		final TableViewer viewer = new TableViewer(dialogComposite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
		tableViewer = viewer;
		Table table = viewer.getTable();
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(viewer);
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		SWTFactoryProxy.INSTANCE.setTestId(table, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_LIST);
		viewer.setContentProvider(ArrayContentProvider.getInstance());
		createTableColumns(viewer);
		viewer.setComparator(new ViewerComparator() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerComparator#compare(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public int compare(Viewer viewerLoc, Object e1, Object e2) {
				if ((e1 instanceof ConfigurationComponentTypeId) && (e2 instanceof ConfigurationComponentTypeId)) {
					ConfigurationComponentTypeId typeId1 = (ConfigurationComponentTypeId) e1;
					ConfigurationComponentTypeId typeId2 = (ConfigurationComponentTypeId) e2;
					String name1Lower = UtilsText.safeString(typeId1.getConfigurationComponent()
							.getUIName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext()).toLowerCase());
					String name2Lower = UtilsText.safeString(typeId2.getConfigurationComponent()
							.getUIName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext()).toLowerCase());
					int comparisonResult = ComparatorHelpers.compareSignalNames(name1Lower, name2Lower);
					if ((comparisonResult == 0) && (typeId1.getConfigurationComponent().getComponents().size() == 1)
							&& (typeId2.getConfigurationComponent().getComponents().size() == 1)) {
						Version ver1 = typeId1.getConfigurationComponent().getComponents().get(0).getVersion();
						Version ver2 = typeId2.getConfigurationComponent().getComponents().get(0).getVersion();
						if ((ver1 != null) && (ver2 != null)) {
							return ver1.compareTo(ver2);
						}
					}
					return comparisonResult;
				}
				return 0;
			}

		});
		List<@NonNull ConfigurationComponentTypeId> components = controllerWrapper.getController().getMcu().getAvailableComponents()
				.getReplacementsForTypeId(instance.getComponentTypeId());
		viewer.setInput(components);
		ViewerFilter[] filters = new ViewerFilter[1];
		filters[0] = new ViewerFilter() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public boolean select(@NonNull Viewer viewerWithFilter, @NonNull Object parentElement, @NonNull Object element) {
				if (element instanceof ConfigurationComponentTypeId) {
					ConfigurationComponentTypeId migratableComponent = (ConfigurationComponentTypeId) element;
					if (isMigrationAnUpgrade(migratableComponent, instance.getConfigCompTypeId().getTypeId())) {
						return hasNewerComponentHigherVersion(migratableComponent, instance.getConfigCompTypeId());
					}
					return true;
				}
				return false;
			}
		};
		viewer.setFilters(filters);
		GridData viewerLayoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewerLayoutData.heightHint = table.getItemHeight() * DEFAULT_ROWS_NUM;
		viewerLayoutData.widthHint = (int) (viewerLayoutData.heightHint * DEFAULT_VIEWER_WIDTH_HEIGHT_RATIO);
		viewer.getControl().setLayoutData(viewerLayoutData);
		if (table.getItemCount() > 0) {
			table.select(0);
		}
		viewer.addDoubleClickListener((e) -> okPressed());
		packColumns();
		return dialogComposite;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		migrateButton = createButton(parent, IDialogConstants.OK_ID, Messages.get().ComponentsView_MigrateComponentDialog_OK, true);
		SWTFactoryProxy.INSTANCE.setTestId(migrateButton, TestIDs.PERIPHS_MIGRATE_COMPONENT_SHELL_OK_BUTTON);
		createButton(parent, IDialogConstants.CANCEL_ID, com.nxp.swtools.common.ui.utils.Messages.get().MessageBoxDialog_Cancel, false);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		if (tableViewer != null) {
			ISelection selection = tableViewer.getSelection();
			APeriphController controller = controllerWrapper.getController();
			MigrateComponentDialog caller = MigrateComponentDialog.this;
			Runnable migrationTransaction = () -> {
				IComponentInstanceConfig newInstance = createComponentsFromSelection(selection, caller, controllerWrapper);
				if (newInstance != null) {
					// Migrate settings, show new instance, disable the old instance to avoid problems
					MigrationHelper.migrateMode(newInstance, instance, controller, caller);
					MigrationHelper.migrateSettingsValues(newInstance, instance, controller, caller);
					controllerWrapper.getGUIController().openViewOfInstance(viewSite, newInstance, true);
					controller.setComponentInstancesEnabled(Arrays.asList(instance), false, caller);
				}
				setResult(newInstance);
				close();
			};
			controller.runTransaction(migrationTransaction);
		}
		super.okPressed();
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#cancelPressed()
	 */
	@Override
	protected void cancelPressed() {
		close();
	}

	/**
	 * Create columns of the table.
	 * @param viewer to create columns in
	 */
	private void createTableColumns(@NonNull TableViewer viewer) {
		TableViewerColumn configCompcolumn = new TableViewerColumn(viewer, SWT.NONE);
		configCompcolumn.setLabelProvider(new ColumnLabelProvider() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				return ((ConfigurationComponentTypeId) element).getConfigurationComponent()
						.getLabel(controllerWrapper.getController().getFunctionalGroup().getExpressionContext());
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				if (canUseComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				return ToolsImages.getImage(IToolsImages.ICON_EMPTY_7X8P);
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				// FIXME TomasR v99 Should we show something here?
				return UtilsText.EMPTY_STRING;
			}
		});
		TableViewerColumn categoryColumn = new TableViewerColumn(viewer, SWT.NONE);
		categoryColumn.setLabelProvider(new ColumnLabelProvider() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				return configComp.getConfigurationComponent().getCategory();
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				if (canUseComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}
		});
		TableViewerColumn requiredSdkCompColumn = new TableViewerColumn(viewer, SWT.NONE);
		requiredSdkCompColumn.setLabelProvider(new ColumnLabelProvider() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				return configComp.getConfigurationComponent().getComponents().stream().map(c -> c.getName() + " [" + c.getVersion() + "]") //$NON-NLS-1$ //$NON-NLS-2$
						.collect(CollectorsUtils.joining(UtilsText.COMMA_SPACE));
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				return (canUseComponent(configComp)) ? null : SwToolsColors.getColor(SwToolsColors.DISABLED_FG);

			}
		});
		configCompcolumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Name);
		categoryColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Category);
		requiredSdkCompColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Version);
	}

	/**
	 * Pack the columns. This method packs the columns' widths based on their content (omitting their titles)
	 */
	private void packColumns() {
		TableViewer tableViewerLoc = tableViewer;
		if (tableViewerLoc != null) {
			for (TableColumn col : tableViewerLoc.getTable().getColumns()) {
				col.pack();
			}
		}
	}

	/**
	 * Check whether the component can be used for migration
	 * @param newComponent to check for
	 * @return {@code true} if the component can be used for migration, {@code false} otherwise
	 */
	static boolean canUseComponent(@NonNull ConfigurationComponentTypeId newComponent) {
		String compType = newComponent.getConfigurationComponent().getId();
		String configuredComponentTypeId = Controller.getInstance().getProfile().getConfiguredComponentTypeId(compType);
		if ((configuredComponentTypeId == null)) {
			// No component of this type was already used
			return true;
		}
		if (configuredComponentTypeId.equals(newComponent.getTypeId())) {
			// Same component and same typeId too
			return true;
		}
		// Same component with maybe different typeId was already used
		return isMigrationAnUpgrade(newComponent, configuredComponentTypeId);
	}

	/**
	 * Checks if wanted migration is meant as an upgrade
	 * @param newComponent that will replace the old one
	 * @param typeIdOfOldComponent
	 * @return {@code true} when this migrations is and upgrade or {@code false} when not
	 */
	static boolean isMigrationAnUpgrade(ConfigurationComponentTypeId newComponent, String typeIdOfOldComponent) {
		// This method could require rewrite in future as it just supports the first sw_com_ref in component file
		ConfigurationComponentTypeId oldComponent = Controller.getInstance().getMcu().getAvailableComponents().getConfigCompTypeId(UtilsText.safeString(typeIdOfOldComponent));
		if (oldComponent != null) {
			return hasNewerComponentHigherVersion(newComponent, oldComponent);
		}
		return false;
	}

	/**
	 * Checks if newComponent has higher version than oldComponent
	 * @param newComponent new component
	 * @param oldComponent old component
	 * @return {@code true} when new component has higher version than old component, {@code false} otherwise
	 */
	static boolean hasNewerComponentHigherVersion(ConfigurationComponentTypeId newComponent, ConfigurationComponentTypeId oldComponent) {
		Version oldComponentVersion = null;
		Version newComponentVersion = null;
		List<SWComponent> oldComponentSwCompRefs = oldComponent.getConfigurationComponent().getComponents();
		List<SWComponent> newComponentSwCompRefs = newComponent.getConfigurationComponent().getComponents();
		if (!oldComponentSwCompRefs.isEmpty()) {
			oldComponentVersion = oldComponentSwCompRefs.get(0).getVersion();
		}
		if (!newComponentSwCompRefs.isEmpty()) {
			newComponentVersion = newComponentSwCompRefs.get(0).getVersion();
		}
		return ((oldComponentVersion != null) && (newComponentVersion != null) && (oldComponentVersion.compareTo(newComponentVersion) < 0));
	}

	/**
	 * Create component instance from the {@link ISelection}.
	 * @param selection the selected component type to create
	 * @param caller the method caller which will be used as an originator of a subsequent UI event.
	 * @param controllerWrapperLoc containing the controller wrapper
	 * @return new instance if specified selection contains available ConfigurationComponentTypeId, otherwise returns {@code null}
	 */
	protected @Nullable IComponentInstanceConfig createComponentsFromSelection(@Nullable ISelection selection, @NonNull Object caller,
			@NonNull IControllerWrapper controllerWrapperLoc) {
		APeriphController controller = controllerWrapperLoc.getController();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Object data = structuredSelection.getFirstElement();
			if (data instanceof ConfigurationComponentTypeId) {
				ConfigurationComponentTypeId typeId = (ConfigurationComponentTypeId) data;
				if (canUseComponent(typeId)) {
					return MigrationHelper.tryToCreateNewInstance(typeId, instance, controller, caller);
				}
			}
		}
		return null;
	}

	/**
	 * Get result of dialog.
	 * @return Component or ComponentInstance created by this dialog
	 */
	public @Nullable IChildProvidable getResult() {
		return result;
	}

	/**
	 * Set result of dialog.
	 * @param result Component or ComponentInstance created by this dialog
	 */
	void setResult(@Nullable IChildProvidable result) {
		this.result = result;
	}
}
