/**
 * Copyright 2017-2021 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Logger;

import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;

import com.nxp.swtools.common.ui.utils.perspectives.PerspectivesHelper;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.os.OSDetect;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.core.service.scriptapi.db.IRegParentPeripheralAPI;
import com.nxp.swtools.core.service.scriptapi.db.IRegistersDatabaseAPI;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.dialogs.initializationorder.InitializationOrderDialog;
import com.nxp.swtools.periphs.gui.dialogs.initializationorder.InitializationOrderDialogProperties;
import com.nxp.swtools.periphs.gui.perspective.PeripheralsPerspective;
import com.nxp.swtools.periphs.gui.view.ComponentMenuCreator.ComponentMenuOptions;
import com.nxp.swtools.periphs.gui.view.ComponentMenuCreator.IComponentSelectionProvider;
import com.nxp.swtools.periphs.gui.view.ComponentMenuCreator.IPeripheralSelectionProvider;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingViewHelper;
import com.nxp.swtools.periphs.gui.view.provider.PeripheralInitResolveHandler;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.validation.ValidationHelper;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.text.TextBoxHelper;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableImplementation;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.engine.ProblemUtils;
import com.nxp.swtools.validation.engine.ValidationEngineFactory;
import com.nxp.swtools.validation.engine.ValidationProblemListenerAdapter;

/**
 * View responsible for displaying peripherals and their settings.
 * @author David Danaj
 * @author Juraj Ondruska
 */
public class PeripheralsView extends APeriphsViewBase {
	/** Identification of ToolItem action */
	private static final @NonNull String FILTER_USED_OR_REQUESTED_PERIPHERALS = "filterUsedOrRequestedPeripherals"; //$NON-NLS-1$
	/** Id of ToolItem action */
	private static final @NonNull String TOOL_ITEM_ACTION = "Action"; //$NON-NLS-1$
	/** ID of the view */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.peripheralsView"; //$NON-NLS-1$
	/** Logger of the class */
	protected static final @NonNull Logger LOGGER = LogManager.getLogger(PeripheralsView.class);
	/** Main composite of the view */
	@Nullable Composite mainComposite;
	/** Table viewer composite of the view */
	@Nullable Composite twComposite;
	/** Checkbox table viewer responsible for viewing content of peripherals view */
	@Nullable CheckboxTableViewer tw;
	/** Minimum width of columns inside a table */
	private static final int COLUMN_MIN_WIDTH = 120;
	/** Percentage of table width given to a "Peripheral" column */
	private static final int PERIPHERAL_COL_WEIGHT = 20;
	/** Percentage of table width given to an "Used in" column */
	private static final int USED_IN_COL_WEIGHT = 80;
	/** Index of "Peripheral" column in a table */
	private static final int PERIPHERAL_COL_IDX = 0;
	/** Index of "Used in" column in a table */
	public static final int USED_IN_COL_IDX = 1;
	/** String used for filtering rows in table */
	@NonNull String filterString = UtilsText.EMPTY_STRING;
	/** Text box with the filter text */
	private @Nullable Text filterBox;
	/** Show only used or requested peripherals */
	boolean filterShowUsedOrRequestedPeripherals = false;
	/** The filter tool bar item */
	private @Nullable ToolItem usedOrRequestedPeripheralToolItem;
	/** Listener for validations changes */
	private @NonNull ValidationProblemListenerAdapter validationsListener = new ValidationProblemListenerAdapter() {
		@Override
		public void validationProblemsChanged(Collection<@NonNull IValidationProblem> problems) {
			// Refresh when any dependency changes
			TableViewer viewer = tw;
			if ((viewer != null) && (!viewer.getTable().isDisposed())) {
				viewer.refresh();
			}
		}
	};

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		Composite composite = createDefaultComposite(parent);
		composite.setLayout(new GridLayout());
		mainComposite = composite;
		createFilterPart(composite);
		createTableViewer(composite);
		createContextMenu();
	}

	/**
	 * Creates table viewer part of view
	 * @param parent parent composite
	 */
	private void createTableViewer(@NonNull Composite parent) {
		twComposite = new Composite(parent, SWT.NONE);
		twComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		CheckboxTableViewer tableViewer = new CheckboxTableViewer(
				new Table(Objects.requireNonNull(twComposite), SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL | SWT.CHECK));
		final Table table = tableViewer.getTable();
		SWTFactoryProxy.INSTANCE.setTestId(table, TestIDs.PERIPHS_PERIPHERALS_TABLE);
		tw = tableViewer;
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(tableViewer);

		// set columns, layout, width
		setupTableProperties(table);

		// open editor when doubleclicked on a component instance config in "Used in" column
		addMouseDoubleClickListener(table);

		// set listeners
		setupTableViewer();
		ValidationEngineFactory.addListener(validationsListener);
	}

	/**
	 * Create context menu for the viewer.
	 */
	void createContextMenu() {
		IComponentSelectionProvider selectionProvider = new IComponentSelectionProvider() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.periphs.gui.view.ComponentsView.IChildSelectionProvider#getSelection()
			 */
			@Override
			public Collection<? extends @NonNull IChild> getSelection() {
				Object firstElement = (tw != null) ? tw.getStructuredSelection().getFirstElement() : null;
				if (firstElement instanceof TableViewerDataModel) {
					return ((TableViewerDataModel) firstElement).getComponentInstanceConfigs();
				}
				return CollectionsUtils.emptyList();
			}
		};
		IPeripheralSelectionProvider peripheralProvider = new IPeripheralSelectionProvider() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.periphs.gui.view.ComponentMenuCreator.IPeripheralSelectionProvider#getSelection()
			 */
			@Override
			public @Nullable String getSelection() {
				Object firstElement = (tw != null) ? tw.getStructuredSelection().getFirstElement() : null;
				if (firstElement instanceof TableViewerDataModel) {
					return ((TableViewerDataModel) firstElement).getPeripheral();
				}
				return null;
			}
		};
		Viewer viewer = tw;
		if (viewer != null) {
			// Right mouse button
			new ComponentMenuCreator(selectionProvider, peripheralProvider, new ComponentMenuOptions(/* ask what to remove */ true), controllerWrapper)
			.createMenu(viewer, getViewSite());
		}
	}

	/**
	 * Creates filter panel with left and right sections
	 * @param parent parent composite
	 */
	private void createFilterPart(@NonNull Composite parent) {
		Composite filterComposite = new Composite(parent, SWT.NONE);
		GridLayout filterCompositeLayout = new GridLayout(3, false);
		filterCompositeLayout.horizontalSpacing = 10;
		filterCompositeLayout.marginHeight = 0;
		filterCompositeLayout.marginWidth = 0;
		filterComposite.setLayout(filterCompositeLayout);
		filterComposite.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		createFilterButtons(filterComposite);
		createFilterTextBox(filterComposite);
		Button button = new Button(filterComposite, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_OPEN_REORDER_DIALOG);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().InitializationOrderDialog_OpenDialogButtonTooltip);
		button.setImage(ToolsImages.getImage(IToolsImages.IMAGE_REORDER));
		button.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, false));
		button.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				Shell shell = getSiteNonNull().getShell();
				if (shell != null) {
					InitializationOrderDialog.open(new InitializationOrderDialogProperties(shell, controllerWrapper));
				}
			}
		});
	}

	/**
	 * Creates right part of filter composite
	 * @param filterComposite to create in
	 */
	private void createFilterTextBox(@NonNull Composite filterComposite) {
		Text filterBoxLoc = new Text(filterComposite, SWT.SEARCH);
		filterBoxLoc.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		filterBoxLoc.setMessage(UtilsText.safeString(com.nxp.swtools.utils.Messages.get().RegistersView_TypeFilterText));
		SWTFactoryProxy.INSTANCE.setTestId(filterBoxLoc, TestIDs.PERIPHS_PERIPHERALS_FILTER_TEXT);
		TextBoxHelper.attachModifyListeners(filterBoxLoc, s -> {
			getFilterStringAndRefresh(filterBoxLoc);
		});
		filterBoxLoc.addFocusListener(new FocusAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
			 */
			@Override
			public void focusLost(FocusEvent e) {
				getFilterStringAndRefresh(filterBoxLoc);
			}
		});
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(filterBoxLoc, Messages.get().PeripheralsView_FilterTooltip);
		filterBox = filterBoxLoc;
	}

	/**
	 * Creates left part of filter composite
	 * @param filterComposite to create in
	 */
	private void createFilterButtons(@NonNull Composite filterComposite) {
		SelectionAdapter toolbarItemListener = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Object src = e.getSource();
				if (src instanceof ToolItem) {
					ToolItem toolItem = (ToolItem) src;
					boolean selected = toolItem.getSelection();
					String action = (String) Objects.requireNonNull(toolItem.getData(TOOL_ITEM_ACTION));
					switch (action) {
						case FILTER_USED_OR_REQUESTED_PERIPHERALS: {
							filterShowUsedOrRequestedPeripherals = selected;
							break;
						}
					}
				}
				TableViewer tableViewer = tw;
				if (tableViewer != null) {
					tableViewer.refresh();
				}
			}
		};

		ToolBar filterButtonsBar = new ToolBar(filterComposite, SWT.FLAT);
		filterButtonsBar.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));

		ToolItem usedOrRequestedPeripheralToolItemLoc = new ToolItem(filterButtonsBar, SWT.CHECK);
		SWTFactoryProxy.INSTANCE.setTestId(usedOrRequestedPeripheralToolItemLoc, TestIDs.PERIPHS_PERIPHERALS_FILTER_INITIALIZED_BUTTON);
		usedOrRequestedPeripheralToolItemLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_FILTER));
		usedOrRequestedPeripheralToolItemLoc.setToolTipText(Messages.get().PeripheralsView_ShowOnlyUsedOrRequestedPeripherals);
		usedOrRequestedPeripheralToolItemLoc.setSelection(filterShowUsedOrRequestedPeripherals);
		usedOrRequestedPeripheralToolItemLoc.setData(TOOL_ITEM_ACTION, FILTER_USED_OR_REQUESTED_PERIPHERALS);
		usedOrRequestedPeripheralToolItemLoc.addSelectionListener(toolbarItemListener);
		usedOrRequestedPeripheralToolItem = usedOrRequestedPeripheralToolItemLoc;
	}

	/**
	 * Set input, listeners, filter and check provider to a table viewer.
	 */
	private void setupTableViewer() {
		TableViewer tableViewer = tw;
		if (tableViewer != null) {
			tableViewer.setContentProvider(ArrayContentProvider.getInstance());
			// filter out peripherals that don't have usable components
			addFilter();
			setCheckStateProvider();
			addCheckStateListener();
			registerListener(EventTypes.CHANGE | EventTypes.SETTING_CHANGE, new IEventListener() {
				/*
				 * (non-Javadoc)
				 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
				 */
				@Override
				public void handle(@NonNull ToolEvent event) {
					tableViewer.setInput(getDataForTableViewer());
				}

				/*
				 * (non-Javadoc)
				 * @see com.nxp.swtools.utils.events.IEventListener#handle(java.util.Collection)
				 */
				@Override
				public void handle(@NonNull Collection<@NonNull ToolEvent> events) {
					boolean hasChangeEvent = events.stream().anyMatch(e -> e.isType(EventTypes.CHANGE | EventTypes.SETTING_CHANGE));
					if (hasChangeEvent) {
						tableViewer.setInput(getDataForTableViewer());
					} else {
						tableViewer.refresh();
					}
				}
			});
			// get the content for the viewer, setInput will call getElements in the
			// contentProvider
			tableViewer.setInput(getDataForTableViewer());
		}
	}

	/**
	 * Add check state listener to a table viewer. When checkbox is checked, creates new dialog to create new component instance config.
	 */
	private void addCheckStateListener() {
		CheckboxTableViewer tableViewer = tw;
		if (tableViewer != null) {
			tableViewer.addCheckStateListener(new ICheckStateListener() {
				/*
				 * (non-Javadoc)
				 * @see org.eclipse.jface.viewers.ICheckStateListener#checkStateChanged(org.eclipse.jface.viewers.CheckStateChangedEvent)
				 */
				@Override
				public void checkStateChanged(CheckStateChangedEvent event) {
					TableViewerDataModel row = (TableViewerDataModel) event.getElement();
					String peripheral = row.getPeripheral();
					List<@NonNull IComponentInstanceConfig> configs = row.getComponentInstanceConfigs();
					if (event.getChecked()) {
						if (configs.isEmpty()) {
							configurePeripheral(peripheral, Controller.getInstance().getFunctionalGroup(), PeripheralsView.this, controllerWrapper);
						} else {
							ProgressUtils.run(m -> controllerWrapper.getController().setComponentInstancesEnabled(configs, true, PeripheralsView.this));
						}
					} else {
						ProgressUtils.run(m -> controllerWrapper.getController().setComponentInstancesEnabled(configs, false, PeripheralsView.this));
					}
					// in SWTBot there was exception org.eclipse.swt.SWTException: Widget is disposed
					if (!tableViewer.getTable().isDisposed()) {
						if (OSDetect.isMac()) {
							// MCUXCON-6173 select the (un)checked row - this does not work out-of-the-box on Mac
							tableViewer.setSelection(new StructuredSelection(row));
						}
						tableViewer.refresh();
					}
				}
			});
		}
	}

	/**
	 * Configure given peripheral (create component instance for given peripheral).
	 * @param peripheral instance to use in the newly created component instance
	 * @param group functional group in which to create component instance
	 * @param caller originator of the event
	 * @param controllerWrapper containing the generic controller
	 */
	public static void configurePeripheral(@NonNull String peripheral, @NonNull IFunctionalGroup group, @NonNull Object caller, @NonNull IControllerWrapper controllerWrapper) {
		APeriphController controller = controllerWrapper.getController();
		boolean openView = PerspectivesHelper.isPerspectiveActive(PeripheralsPerspective.ID)
				&& (controller.getFunctionalGroup() == group);
		IViewPart viewPart = ComponentSettingViewHelper.getViewPart();
		IViewSite viewSite = viewPart == null ? null : viewPart.getViewSite();
		if (viewSite == null) {
			LOGGER.severe("[TOOL] Could not configure peripheral due to no view found"); //$NON-NLS-1$
			return;
		}
		List<@NonNull ConfigurationComponentTypeId> components = controller.getComponentsOfPeripheral(peripheral);
		// remove component if its "hidden" flag is set to true and it is not used in project
		AddComponentDialog.filterNonDisplayableComponents(components, controller.getProfile());
		if ((components.size() == 1) && (AddComponentDialog.getProblemMessage(components.get(0), controllerWrapper, group.getName(), peripheral) == null) &&
				!AddComponentDialog.componentHasTemplates(components.get(0), controllerWrapper)) {
			// First option - it is only one usable component there, use it without showing the dialog when it has no templates
			AddComponentDialog.createComponent(components.get(0), peripheral, viewSite, group.getName(), openView, caller, controllerWrapper);
		} else {
			if (components.size() > 0) {
				final String typeOfFirstComponent = components.get(0).getType();
				boolean onlyOneComponentType = components.stream().allMatch(compTypeId -> compTypeId.getType().equals(typeOfFirstComponent));
				if (onlyOneComponentType) {
					// Second option - only one component type on this peripheral
					ConfigurationComponentTypeId alreadyUsedComponent = controller.getProfile().getActiveComponents().getComponentTypeIdByType(typeOfFirstComponent);
					if ((alreadyUsedComponent != null) && !AddComponentDialog.componentHasTemplates(alreadyUsedComponent, controllerWrapper)) {
						// Some component of this type is already used and this component does not have templates
						AddComponentDialog.createComponent(alreadyUsedComponent, peripheral, viewSite, group.getName(), openView, caller, controllerWrapper);
						return;
					}
				}
			}
			// Last option - show add component dialog with tool-chain filter off
			AddComponentDialog.open(viewSite, peripheral, group.getName(), openView, true, controllerWrapper);
		}
	}

	/**
	 * Add check state provider to a checkbox column in a table viewer.
	 */
	private void setCheckStateProvider() {
		CheckboxTableViewer tableViewer = tw;
		if (tableViewer != null) {
			tableViewer.setCheckStateProvider(new ICheckStateProvider() {
				/* (non-Javadoc)
				 * @see org.eclipse.jface.viewers.ICheckStateProvider#isGrayed(java.lang.Object)
				 */
				@Override
				public boolean isGrayed(Object element) {
					TableViewerDataModel row = (TableViewerDataModel) element;
					List<@NonNull IComponentInstanceConfig> configs = row.getComponentInstanceConfigs();
					if (!configs.isEmpty()) {
						final boolean firstEnabled = configs.get(0).isEnabled();
						boolean difference = configs.stream().anyMatch(c -> c.isEnabled() != firstEnabled);
						return difference;
					}
					return false;
				}

				/* (non-Javadoc)
				 * @see org.eclipse.jface.viewers.ICheckStateProvider#isChecked(java.lang.Object)
				 */
				@Override
				public boolean isChecked(Object element) {
					TableViewerDataModel row = (TableViewerDataModel) element;
					List<@NonNull IComponentInstanceConfig> configs = row.getComponentInstanceConfigs();
					return (configs.size() > 0) && configs.stream().anyMatch(IComponentInstanceConfig::isEnabled);
				}
			});
		}
	}

	/**
	 * Add filter to a table viewer.
	 * Filter out peripherals that do not have components that use them.
	 * Filter out peripherals that does not contain filterString
	 * Filter out instances with name that does not contain filterString
	 * Enable showing of row if user wants to show initialized or uninitialized peripherals
	 */
	private void addFilter() {
		TableViewer tableViewer = tw;
		if (tableViewer != null) {
			tableViewer.addFilter(new ViewerFilter() {
				/* (non-Javadoc)
				 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
				 */
				@Override
				public boolean select(Viewer viewer, Object parentElement, Object element) {
					TableViewerDataModel row = (TableViewerDataModel) element;
					boolean peripheralIsInitialized = !row.getComponentInstanceConfigs().isEmpty();
					boolean instancesNameMatchesFilterString = row.getComponentInstanceConfigs().stream()
							.map(instance -> instance.getUiName().toLowerCase())
							.anyMatch(name -> name.contains(filterString.toLowerCase()));
					boolean peripheralNameMatchesFilterString = row.getPeripheral().toLowerCase().contains(filterString.toLowerCase());
					boolean rowMatchesFilterString = peripheralNameMatchesFilterString || instancesNameMatchesFilterString;
					boolean peripheralSupported = !controllerWrapper.getController().getComponentsOfPeripheral(row.getPeripheral()).isEmpty();
					boolean rowPassedFiltering = peripheralSupported && rowMatchesFilterString;
					boolean peripheralRequested = getRequestedPeripherals().contains(row.getPeripheral());
					boolean showUsedOrRequested = filterShowUsedOrRequestedPeripherals && (peripheralIsInitialized || peripheralRequested);
					boolean showAllAvailable = !filterShowUsedOrRequestedPeripherals;
					boolean isVisible = rowPassedFiltering && (showUsedOrRequested || showAllAvailable);
					return isVisible;
				}
			});
		}
	}

	/**
	 * Get peripherals that should be initialized (but aren't)
	 * @return List of peripherals that should be initialized
	 */
	static @NonNull List<@NonNull String> getRequestedPeripherals() {
		List<@NonNull String> peripherals = new ArrayList<>();
		Collection<@NonNull IValidationProblem> problems = ValidationEngineFactory.getValidationProblemsCollection();
		for (IValidationProblem problem : problems) {
			if (PeripheralInitResolveHandler.INSTANCE.canHandle(problem)) {
				peripherals.add(problem.getDependency().getResourceId());
			}
		}
		return peripherals;
	}

	/**
	 * Add listener to mouse double click in a table. Open an editor when clicked in a "Used in" column on any component instance config name.
	 * @param table table which should listen to double click event
	 */
	private void addMouseDoubleClickListener(final Table table) {
		table.addListener(SWT.MouseDoubleClick, new Listener() {
			@Override
			public void handleEvent(Event event) {
				Point pt = new Point(event.x, event.y); // point where user clicked
				TableViewer tableViewer = tw;
				if (tableViewer != null) {
					ViewerCell viewerCell = tableViewer.getCell(pt);
					// If clicked on row
					if ((viewerCell != null)) {
						TableViewerDataModel row = (TableViewerDataModel) viewerCell.getElement();
						// if there is at least 1 component instance config configured
						List<@NonNull IComponentInstanceConfig> instances = row.getComponentInstanceConfigs();
						if (!instances.isEmpty()) {
							openView(instances.get(0));
						}
					}
				}
			}
		});
	}

	/**
	 * Create columns, set visibility, set layout and width of columns.
	 * @param table for which settings are applied
	 */
	private void setupTableProperties(Table table) {
		createColumns();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		TableColumnLayout tableColumnLayout = new TableColumnLayout(); // used for applying weights to columns
		Composite composite = twComposite;
		if (composite != null) {
			composite.setLayout(tableColumnLayout);
		}

		// set width of a peripheral column
		if (table.getColumn(PERIPHERAL_COL_IDX) != null) {
			tableColumnLayout.setColumnData(table.getColumn(PERIPHERAL_COL_IDX),
					new ColumnWeightData(PERIPHERAL_COL_WEIGHT, COLUMN_MIN_WIDTH, true));
		}
		// set width of a used in column
		if (table.getColumn(USED_IN_COL_IDX) != null) {
			tableColumnLayout.setColumnData(table.getColumn(USED_IN_COL_IDX),
					new ColumnWeightData(USED_IN_COL_WEIGHT, COLUMN_MIN_WIDTH, true));
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		Composite composite = twComposite;
		if (composite != null) {
			composite.setFocus();
		}
	}

	/**
	 * Create columns for the table.
	 */
	private void createColumns() {
		// first column is for the peripheral name
		TableViewerColumn col = createTableViewerColumn(UtilsText.safeString(Messages.get().PeripheralsView_PeripheralColumn));
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				TableViewerDataModel row = (TableViewerDataModel) element;
				return row.getPeripheral();
			}

			@Override
			public Image getImage(Object element) {
				// The warning should be also displayed for peripherals that have a dependency request for initialization from
				// other tools and are not used by any component.
				if (element instanceof TableViewerDataModel) {
					TableViewerDataModel model = (TableViewerDataModel) element;
					Map<@NonNull String, @NonNull List<@NonNull IValidationProblem>> problems = ValidationHelper.collectValidationProblemsOnPeripheralsTool();
					List<@NonNull IValidationProblem> problemsOfPeripheral = problems.get(model.getPeripheral());
					if (problemsOfPeripheral == null) {
						return null;
					}
					IValidationProblem problem = ProblemUtils.getHighestSeverityProblem(problemsOfPeripheral.stream());
					if (problem == null) {
						return null;
					}
					return ToolsImages.getStatusIcon(problem.getProblemLevel());
				}
				return null;
			}

			@Override
			public String getToolTipText(Object element) {
				if (element instanceof TableViewerDataModel) {
					TableViewerDataModel model = (TableViewerDataModel) element;
					Map<@NonNull String, @NonNull List<@NonNull IValidationProblem>> problems = ValidationHelper.collectValidationProblemsOnPeripheralsTool();
					ToolTipableImplementation tooltip = new ToolTipableImplementation();
					StringBuilder errorBuilder = new StringBuilder();
					StringBuilder warningBuilder = new StringBuilder();
					StringBuilder infoBuilder = new StringBuilder();
					List<@NonNull IValidationProblem> problemsOfPeripheral = problems.get(model.getPeripheral());
					List<@NonNull String> usedErrorDescriptions = new ArrayList<>();
					List<@NonNull String> usedWarningDescriptions = new ArrayList<>();
					List<@NonNull String> usedInfoDescriptions = new ArrayList<>();
					if (problemsOfPeripheral != null) {
						for(IValidationProblem problem : problemsOfPeripheral) {
							String dependencyText = createDependencyText(problem);
							switch (problem.getProblemLevel()) {
								case ErrorLevels.LEVEL_FATAL: // fall-through
								case ErrorLevels.LEVEL_ERROR:
									if (usedErrorDescriptions.contains(dependencyText)) {
										continue;
									}
									usedErrorDescriptions.add(dependencyText);
									errorBuilder.append(dependencyText);
									break;
								case ErrorLevels.LEVEL_WARNING:
									if (usedWarningDescriptions.contains(dependencyText)) {
										continue;
									}
									usedWarningDescriptions.add(dependencyText);
									warningBuilder.append(dependencyText);
									break;
								case ErrorLevels.LEVEL_INFORMATION:
									if (usedInfoDescriptions.contains(dependencyText)) {
										continue;
									}
									usedInfoDescriptions.add(dependencyText);
									infoBuilder.append(dependencyText);
									break;
							}
						}
					}
					tooltip.setError(errorBuilder.toString());
					tooltip.setWarning(warningBuilder.toString());
					tooltip.setInfo(infoBuilder.toString());
					ToolTipableMarkdownDescriptionDecorator decoratedTooltip = new ToolTipableMarkdownDescriptionDecorator(tooltip);
					String toolTipText = ToolTipableFormatter.getToolTipText(decoratedTooltip);
					String id = model.getPeripheral();
					String name = model.getFullNameOfPeripheral();
					// Finish and use the tooltip only when there is problem or full name is set
					if (!toolTipText.isEmpty() || (!name.equals(id))) {
						tooltip.setUiName(name);
						tooltip.setId(id);
					}
					toolTipText = ToolTipableFormatter.getToolTipText(decoratedTooltip);
					return toolTipText.isEmpty() ? null : toolTipText;
				}
				return null;
			}

			/**
			 * Creates unified info text of given problem and appends it to given builder
			 * @param problem Problem to get info text from
			 */
			private @NonNull String createDependencyText(@NonNull IValidationProblem problem) {
				StringBuilder builder = new StringBuilder(200);
				builder.append(problem.getDependency().getDescription());
				builder.append(UtilsText.SPACE);
				builder.append(UtilsText.LEFT_PARENTHESIS);
				builder.append(Messages.get().PeripheralsView_Dependency_RequiredBy);
				builder.append(UtilsText.SPACE);
				builder.append(problem.getDependency().getSource());
				builder.append(UtilsText.RIGHT_PARENTHESIS);
				builder.append(UtilsText.CRLF);
				return UtilsText.safeString(builder.toString());
			}
		});
		// second column is for component instance configs that are using peripheral from first column
		col = createTableViewerColumn(UtilsText.safeString(Messages.get().PeripheralsView_UsedInColumn));
		col.setLabelProvider(new ColumnLabelProvider() {
			/**
			 * @param instanceConfig of the component
			 * @return component name including component state
			 */
			private @NonNull String fullTextLabel(@NonNull IComponentInstanceConfig instanceConfig) {
				return instanceConfig.isEnabled() ? instanceConfig.getName() : MessageFormat.format(UtilsText.safeString(
						Messages.get().PeripheralsView_DisabledComponentLabel), instanceConfig.getName());
			}

			@Override
			public String getText(Object element) {
				TableViewerDataModel row = (TableViewerDataModel) element;
				String instanceNames = row.getComponentInstanceConfigs() // get all component instance configs from a list
						.stream()
						.map(this::fullTextLabel) // get their names
						.collect(CollectorsUtils.joining(UtilsText.COMMA_SPACE)); // join names in a way: name1, name2, name3
				if (controllerWrapper.getController().isPeripheralMarkedAsUsed(row.getPeripheral())) {
					return MessageFormat.format("{0}[{1}]", instanceNames, Messages.get().PeripheralsView_UserInitSuffix); //$NON-NLS-1$
				}
				return instanceNames;
			}

			@Override
			public String getToolTipText(Object element) {
				if (element instanceof TableViewerDataModel) {
					TableViewerDataModel model = (TableViewerDataModel) element;
					List<@NonNull IComponentInstanceConfig> instances = model.getComponentInstanceConfigs();
					String result =  UtilsText.join(instances.stream()
							.map(x -> UtilsText.safeString(ComponentsView.getTooltipText(x, controllerWrapper.getController().getCategories()))),
							UtilsText.XHTML_BR + UtilsText.XHTML_BR);
					return result.isEmpty() ? null : result;
				}
				return null;
			}

			@Override
			public Image getImage(Object element) {
				Image image = null;
				if (element instanceof TableViewerDataModel) {
					TableViewerDataModel model = (TableViewerDataModel) element;
					List<@NonNull IComponentInstanceConfig> instances = model.getComponentInstanceConfigs();
					for(IComponentInstanceConfig instance : instances) {
						String comment = instance.getComment();
						if (!comment.isEmpty()) {
							image = ToolsImages.getImage(IToolsImages.ICON_LEGEND);
						}
						if (!ValidationHelper.getDriverValidationProblems(instance, ErrorLevels.LEVEL_ERROR).isEmpty()) {
							return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
						}
						final int problemLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(instance);
						if ((instance.getError() != null) || (problemLevel >= ErrorLevels.LEVEL_ERROR)) {
							image = ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
						} else if ((instance.getWarning() != null) || (problemLevel == ErrorLevels.LEVEL_WARNING)) {
							image = ToolsImages.getStatusIcon(ErrorLevels.LEVEL_WARNING);
						} else if ((instance.getInfo() != null) || (problemLevel == ErrorLevels.LEVEL_INFORMATION)) {
							image = ToolsImages.getStatusIcon(ErrorLevels.LEVEL_INFORMATION);
						}
					}
				}
				return image;
			}
		});
	}

	/**
	 * Creates a resizable table viewer column.
	 * @param title Name of a column to be displayed in the column header
	 * @return new instance of TableViewerColumn with given name
	 */
	private @NonNull TableViewerColumn createTableViewerColumn(@NonNull String title) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tw, SWT.SINGLE);
		final TableColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setResizable(true);
		return viewerColumn;
	}

	/**
	 * Get data used to fill table in Peripherals view.
	 * @return List of TableViewerDataModel instances representing data for rows in a table
	 */
	protected @NonNull List<@NonNull TableViewerDataModel> getDataForTableViewer() {
		List<@NonNull TableViewerDataModel> dataCollection = new ArrayList<>();
		Collection<@NonNull String> peripheralInstances = controllerWrapper.getController().getActiveCorePeripherals();
		IRegistersDatabaseAPI registersDb = controllerWrapper.getController().getMcu().getRegistersDb();
		// for each peripheral instance
		for (String peripheralInstanceName : peripheralInstances) {
			String fullPeripheralName = peripheralInstanceName;
			if (registersDb != null) {
				IRegParentPeripheralAPI peripheral = registersDb.getPeripheral(peripheralInstanceName);
				if (peripheral != null) {
					fullPeripheralName = peripheral.getFullName();
				}
			}
			dataCollection.add(new TableViewerDataModel(peripheralInstanceName, fullPeripheralName,
					controllerWrapper.getController().getComponentInstanceConfigs(peripheralInstanceName)));
		}
		return dataCollection;
	}

	/**
	 * Class used to store data for table inside Peripherals View
	 * @author David Danaj (b57899/nxa30572)
	 */
	private static class TableViewerDataModel {
		/** Name of peripheral */
		private @NonNull String peripheral;

		/** Full name of peripheral */
		private @NonNull String fullNameOfPeripheral;

		/** List of component instance configs configured for this peripheral */
		private @NonNull List<@NonNull IComponentInstanceConfig> componentInstanceConfigs;

		/**
		 * Creates instance representing one row in a table.
		 * @param peripheral name of peripheral
		 * @param fullNameOfPeripheral full unabbreviated name of the peripheral
		 * @param compConfigs list of component instance configs configured for this peripheral
		 */
		TableViewerDataModel(@NonNull String peripheral, @NonNull String fullNameOfPeripheral, @NonNull List<@NonNull IComponentInstanceConfig> compConfigs) {
			this.peripheral = peripheral;
			this.componentInstanceConfigs = compConfigs;
			this.fullNameOfPeripheral = fullNameOfPeripheral;
		}

		/**
		 * Get peripheral name.
		 * @return String with name of peripheral used in a row
		 */
		public @NonNull String getPeripheral() {
			return peripheral;
		}

		/**
		 * Get name of current peripheral
		 * @return String with full name of peripheral used in a row
		 */
		public @NonNull String getFullNameOfPeripheral() {
			return fullNameOfPeripheral;
		}

		/**
		 * Get list of component instance configs configured for this peripheral.
		 * @return list of component instance configs configured for this peripheral
		 */
		public @NonNull List<@NonNull IComponentInstanceConfig> getComponentInstanceConfigs() {
			return componentInstanceConfigs;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode() {
			return Objects.hash(peripheral);
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			TableViewerDataModel other = (TableViewerDataModel) obj;
			// comparing the peripheral only by an intent, the viewer uses equals method to match old input data with the new one, MCUXCON-6173
			return Objects.equals(peripheral, other.peripheral);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.view.ToolView#resetView()
	 */
	@Override
	protected void resetView() {
		super.resetView();
		filterString = UtilsText.EMPTY_STRING;
		if (filterBox != null) {
			filterBox.setText(UtilsText.EMPTY_STRING);
		}
		filterShowUsedOrRequestedPeripherals = false;
		if (usedOrRequestedPeripheralToolItem != null) {
			usedOrRequestedPeripheralToolItem.setSelection(false);
		}
		if (tw != null) {
			tw.refresh();
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		ValidationEngineFactory.removeListener(validationsListener);
		super.dispose();
	}

	/**
	 * Sets current content of textbox as filter string and refreshes content of table
	 * @param filterBoxLoc textbox
	 */
	void getFilterStringAndRefresh(@NonNull Text filterBoxLoc) {
		filterString = UtilsText.safeString(filterBoxLoc.getText());
		TableViewer tableViewer = tw;
		if (tableViewer != null) {
			tableViewer.refresh();
		}
	}

}
