/**
 * Copyright 2020-2021 NXP
 * Created: Nov 6, 2020
 */
package com.nxp.swtools.periphs.gui.view;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;

import com.nxp.swtools.common.ui.utils.GcUtils;
import com.nxp.swtools.common.utils.NonNullByDefault;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlFactory;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;

/**
 * Class representing the dialog in which the sub-settings should be displayed.
 * @author Ionut-Cristian Tudose - nxf64017
 */
@NonNullByDefault
public class SubSettingsDialog extends TitleAreaDialog {

	/** The control of sub-setting */
	private @Nullable IChildControl childControl;
	/** The sub-setting configuration */
	private IChild child;
	/** The controller */
	private IControllerWrapper controllerWrapper;
	/** UI options for sub-settings */
	private ControlOptions controlOptions;
	/** Minimal width of the dialog in ems */
	private static final int MIN_WIDTH_EMS = 70;
	/** Number of columns for the new component */
	private static final int NUM_COLUMNS = 1;
	/** if the sub-settings are not represented by a structure, 2 columns are needed(label + control) */
	private static final int NUM_COLUMNS_SCALAR_CONFIG = 2;
	/** Column span needed for sub-setting */
	private int noOfCols = 1;

	/**
	 * @param parentShell - to create the dialog inside
	 * @param child - sub-settings to display in dialog
	 * @param controllerWrapper - controller
	 * @param controlOptions - UI options for sub-settings inside dialog
	 */
	public SubSettingsDialog(Shell parentShell, IChild child, IControllerWrapper controllerWrapper, ControlOptions controlOptions) {
		super(parentShell);
		this.child = child;
		this.controllerWrapper = controllerWrapper;
		this.controlOptions = controlOptions;
		setShellStyle(getShellStyle() | SWT.RESIZE);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	@Override
	protected void configureShell(Shell shell) {
		super.configureShell(shell);
		shell.setText(Messages.get().SubSettingsDialog_Title + child.getUiName());
		shell.layout(true);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#create()
	 */
	@Override
	public void create() {
		super.create();
		setTitle(child.getUiName());
		setMessage(child.getDescription());
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.TitleAreaDialog#createDialogArea(org.eclipse.swt.
	 * widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(final Composite parent) {
		Composite mainComposite = (Composite) super.createDialogArea(parent);
		ScrolledComposite scrolledComposite = new ScrolledComposite(mainComposite, SWT.V_SCROLL | SWT.H_SCROLL);
		scrolledComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		scrolledComposite.setExpandVertical(true);
		scrolledComposite.setExpandHorizontal(true);
		if (child instanceof ScalarConfig) {
			noOfCols = NUM_COLUMNS_SCALAR_CONFIG;
		}
		Composite contentComposite = new Composite(scrolledComposite, SWT.NONE);
		contentComposite.setLayout(new GridLayoutComponents(noOfCols, false));
		contentComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		scrolledComposite.setContent(contentComposite);
		scrolledComposite.setMinWidth(GcUtils.getEmWidth(contentComposite, true) * MIN_WIDTH_EMS);
		scrolledComposite.setMinHeight(contentComposite.computeSize(contentComposite.getClientArea().width, SWT.DEFAULT).y);

		Listener resizeListener = new Listener() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
			 */
			@Override
			public void handleEvent(Event event) {
				scrolledComposite.setMinHeight(contentComposite.computeSize(contentComposite.getClientArea().width, SWT.DEFAULT).y);
			}
		};
		contentComposite.addListener(SWT.Resize, resizeListener);
		controlOptions.controlHidden(false);

		IChildControl childUI = ChildControlFactory.create(child, controlOptions, controllerWrapper);
		if (childUI != null) {
			setChildControl(childUI);
			childUI.create(contentComposite, NUM_COLUMNS);
			childUI.update(UpdateType.SUB_SETTINGS_DIALOG);
		}
		return scrolledComposite;
	}

	/**
	 * Setter for childControl
	 * @param childControl
	 */
	public void setChildControl(IChildControl childControl) {
		this.childControl = childControl;
	}

	/**
	 * Getter for childControl member
	 * @return childControl
	 */
	public @Nullable IChildControl getChildControl() {
		return this.childControl;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButton(org.eclipse.swt.widgets.Composite, int, java.lang.String, boolean)
	 */
	@Override
	protected @Nullable Button createButton(@Nullable Composite parent, int id, @Nullable String label, boolean defaultButton) {
		// do not create Cancel button since it is not needed.
		if (id == IDialogConstants.CANCEL_ID) {
			return null;
		}
		return super.createButton(parent, id, label, defaultButton);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#close()
	 */
	@Override
	public boolean close() {
		IChildControl childControlLoc = childControl;
		if ((childControlLoc != null) && !childControlLoc.isDisposed()) {
			childControlLoc.dispose();
		}
		return super.close();
	}
}