/**
 * Copyright 2017-2021 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.List;
import java.util.Objects;

import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal;
import com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular;
import com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlInternalFactory;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;

/**
 * Class representing control of an array setting configuration.
 * @author Juraj Ondruska
 * @author Tomas Rudolf - nxf31690
 */
public class ArrayControl extends ChildProvidableControlBase {
	/** The delegate control */
	private @NonNull AArrayControlInternal delegate;

	/**
	 * @param childProvidable to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ArrayControl(@NonNull ArrayConfig childProvidable, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(childProvidable, controlOptions, controllerWrapper, false);
		this.delegate = ArrayControlInternalFactory.create(childProvidable, controlOptions, controllerWrapper);
		this.delegate.setParentArrayControl(this);
	}

	@Override
	public void setParentControl(IChildControl parentChildControl) {
		delegate.setParentControl(parentChildControl);
	}

	@Override
	public @Nullable IChildControl getParentChildControl() {
		return delegate.getParentChildControl();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int, boolean)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		parent = composite;
		this.currentColSpan = colSpan;
		delegate.create(composite, colSpan);
	}

	/**
	 * Finds the scroll offset to a problematic row in array
	 * @param rowNr
	 * @return integer containing the scroll offset to a problematic row in the array
	 */
	public Integer getOffsetToRow(int rowNr) {
		int rowHeight = getTableItemHeight();
		if (rowHeight != 0) {
			Control control = getMainControl();
			if (control != null) {
				ScrolledComposite scrolledComposite = ScrolledCompositeHelper.findScrolledComposite(control);
				if (scrolledComposite != null) {
					// find the distance between the scrolledComposite's top and the table's top
					Rectangle rec = scrolledComposite.getDisplay().map(control, scrolledComposite, scrolledComposite.getBounds());
					return new Integer((rowNr * rowHeight) + rec.y);
				}
			}
		}
		return new Integer(rowNr);
	}

	/**
	 * @return the height of an {@link ArrayControl} table item
	 */
	private int getTableItemHeight() {
		if (delegate instanceof AArrayControlTabular) {
			AArrayControlTabular tabular = (AArrayControlTabular) delegate;
			return tabular.getItemHeight();
		}
		return 0;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createMainControl(@NonNull Composite composite) {
		// This is never called as create is passed to delegate
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		// This is never called as create is passed to delegate
		return null;
	}

    /* (non-Javadoc)
     * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
     */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		// This is never called as update is passed to delegate
	}

	/**
	 * Select given item in table
	 * @param control to be selected
	 */
	public void selectItem(IChildControl control) {
		delegate.selectItem(control);
	}

	/**
	 * @return the main control of the array
	 */
	private @Nullable Control getMainControl() {
		return delegate.mainControl;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		delegate.update(updateType);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildProvidableControlBase#getChild()
	 */
	@Override
	public @NonNull ArrayConfig getChild() {
		return (ArrayConfig) child;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#getChildren()
	 */
	@Override
	public @NonNull List<@NonNull IChildControl> getChildren() {
		return delegate.getChildren();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildProvidableControlBase#dispose()
	 */
	@Override
	public void dispose() {
		delegate.dispose();
		super.dispose();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#isDisposed()
	 */
	@Override
	public boolean isDisposed() {
		return delegate.isDisposed();
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#getControlOptions()
	 */
	@Override
	public @NonNull ControlOptions getControlOptions() {
		if (!Objects.isNull(delegate)) { // required because this method is called in super(), before delegate is initialized
			return delegate.getControlOptions();
		}
		// at the time when delegate is not yet initialized, it is ok to use controlOptions of this ArrayControl instead,
		// because it equals to controlOptions created later for delegate
		return controlOptions;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#setSelectedChild(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.IChildControl)
	 */
	@Override
	public void setSelectedChild(@Nullable IChildControl selectedChild) {
		delegate.setSelectedChild(selectedChild);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#selectChildTab(com.nxp.swtools.periphs.gui.view.componentsettings
	 * .IChildControl)
	 */
	@Override
	public void selectChildTab(@NonNull IChildControl selectedChild) {
		delegate.selectChildTab(selectedChild);
	}

	/**
	 * Recreates the delegate controls
	 * @param controlBefore control before which the child controls should be created
	 */
	public void recreateDelegate(@Nullable Control controlBefore) {
		createDelegate();
		Composite parentCompositeLoc = parent;
		if (parentCompositeLoc != null) {
			delegate.create(parentCompositeLoc, currentColSpan);
			if (controlBefore != null) {
				delegate.moveBefore(controlBefore);
			}
			delegate.update(UpdateType.NORMAL);
			ScrolledCompositeHelper.updateScrollSize(parentCompositeLoc);
		}
	}

	/**
	 * Creates new delegate from current state
	 */
	private void createDelegate() {
		this.delegate = ArrayControlInternalFactory.create(getChild(), controlOptions, controllerWrapper);
		this.delegate.setParentArrayControl(this);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#setEnabledStateToMainControl(boolean)
	 */
	@Override
	protected void setEnabledStateToMainControl(boolean enabled) {
		delegate.setEnabledStateToMainControl(enabled);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#onExpandGroupVisibilityChange()
	 */
	@Override
	protected void onExpandGroupVisibilityChange() {
		delegate.onExpandGroupVisibilityChange();
	}
}
