/**
 * Copyright 2018-2021 NXP
 * Created: Jul 18, 2018
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils.IdentitySet;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ConfigSetConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IConfigSetConfig;

/**
 * Class representing the GUI elements of component instance configuration.
 * @author Viktar Paklonski
 */
public class ComponentInstanceControl extends ChildProvidableControlBase {

	/** controls of all displayed config sets */
	private final @NonNull Map<@NonNull IConfigSetConfig, @Nullable ConfigSetControl> configSetControls = new HashMap<>();
	/** content of the editor */
	private Composite contentComposite;

	/**
	 * @param instanceConfig to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ComponentInstanceControl(@NonNull IComponentInstanceConfig instanceConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(instanceConfig, controlOptions, controllerWrapper);
		onCreateConfigSetsContent();
	}

	/**
	 * @return either {@link ChildControlBase#mainControl} or {@link #contentComposite}
	 */
	private @NonNull Composite getMainComposite() {
		final Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			return (Composite) mainControlLoc;
		}
		final Composite contentCompositeLoc = contentComposite;
		assert contentCompositeLoc != null;
		return contentCompositeLoc;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		contentComposite = composite;
		if (!Objects.equals(ControlOptions.ShowContentAs.TABS, getControlOptions().getShowContentAs())) {
			// draw line
			final Label label = new Label(composite, SWT.SEPARATOR | SWT.HORIZONTAL);
			label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, ComponentSettingView.CONFIG_SET_COLS, 1));
		}
		super.create(composite, colSpan);
		final Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			labelControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, colSpan, 1));
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			mainControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, colSpan, 1));
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			onUpdateConfigSetsContent();
		}
		super.update(updateType);
	}

	/**
	 * Update controls map for the given config sets.
	 */
	void onUpdateConfigSetsContent() {
		final List<@NonNull IConfigSetConfig> newChildren = ((IComponentInstanceConfig) child).getChildren();
		Set<@NonNull IConfigSetConfig> newChildrenIdentitySet = new IdentitySet<>(newChildren);
		Set<@NonNull IConfigSetConfig> currentConfigSets = new IdentitySet<>(configSetControls.keySet());
		boolean sameContent = (currentConfigSets.containsAll(newChildrenIdentitySet) && newChildrenIdentitySet.containsAll(currentConfigSets));
		if (!sameContent) {
			// dispose of old controls
			for (Map.Entry<@NonNull IConfigSetConfig, ConfigSetControl> entry : configSetControls.entrySet()) {
				final ConfigSetControl value = entry.getValue();
				if (value == null) {
					LOGGER.log(Level.SEVERE, "[TOOL] Config set with ID:{0} does not have control", entry.getKey()); //$NON-NLS-1$
					continue;
				}
				if (!value.isDisposed()) {
					value.dispose();
				}
			}
			configSetControls.clear();
		}
		final List<@NonNull IChildControl> newControls = new ArrayList<>();
		// re-create GUI for new and re-used controls
		for (IConfigSetConfig key : newChildren) {
			IChildControl control = configSetControls.get(key);
			if (control == null) {
				final ControlOptions overrideOptions = createControlOptionsForChild(key);
				control = ChildControlFactory.create(key, overrideOptions, controllerWrapper);
				if (control != null) {
					control.create(getMainComposite(), ComponentSettingView.CONFIG_SET_COLS);
					configSetControls.put(key, (ConfigSetControl) control);
				}
			} else if (control.isDisposed()) {
				control.create(getMainComposite(), ComponentSettingView.CONFIG_SET_COLS);
			}
			if (control != null) {
				newControls.add(control);
			}
		}
		children.clear();
		children.addAll(newControls);
	}

	/**
	 * Create controls map for the given config sets.
	 */
	private void onCreateConfigSetsContent() {
		// associate config sets with controls
		for (IChildControl control : children) {
			final IChild controlChild = control.getChild();
			if (controlChild instanceof IConfigSetConfig) {
				configSetControls.put((ConfigSetConfig) controlChild, (ConfigSetControl) control);
			}
		}
	}

}
