/**
 * Copyright 2017-2022 NXP
 * Created: Aug 28, 2017
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;
import org.w3c.dom.DOMException;

import com.nxp.swtools.common.ui.utils.perspectives.PerspectivesHelper;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionMap;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.lang.CollectionsUtils.IdentitySet;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.kex.MfactConstants;
import com.nxp.swtools.kex.selector.IMcuIdentification;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.perspective.PeripheralsPerspective;
import com.nxp.swtools.periphs.gui.view.APeriphsExternalViewHandler;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.configuration.storage.periphs.AStoragePeriphsComponent;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsComponent;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsComponentInstance;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsFuncGroup;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.resourcetables.model.data.SettingUtils;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.storage.StorageHelper;

/**
 * Helper class for ComponentSettingView class
 * @author Tomas Rudolf - nxf31690
 */
public class ComponentSettingViewHelper {
	/** ID of extension to handle external views for Peripherals tool */
	private static final String ADDITIONAL_VIEWS_EXTENSION_ID = "com.nxp.swtools.periphs.gui.perspective.additionalViews"; //$NON-NLS-1$
	/** Attribute to obtain handler of the external view */
	private static final String ADDITIONAL_VIEWS_EXTENSION_ATTRIBUTE_HANDLER = "view_handler"; //$NON-NLS-1$
	/** Attribute which view is external */
	private static final String EXTENSION_POINT_ID_ATTRIBUTE = "id"; //$NON-NLS-1$
	/** ID of schematic view */
	private static final String SCHEMATIC_VIEW_ID = "com.nxp.swtools.plu.view.DiagramView"; //$NON-NLS-1$
	/** Logger of class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ComponentSettingViewHelper.class);
	/**
	 *  SetMap of opened editors. HashSet<FuncGroup, Component>
	 *  Each functional group contains set of components.
	 *  Each component is either instance config or global config
	 */
	private @NonNull CollectionMap<@NonNull String, @NonNull AStoragePeriphsComponent> editors = new CollectionMap<>(IdentityHashMap.class, IdentitySet.class);
	/**
	 *  SetMap of opened schematic views. HashSet<FuncGroup, Component>
	 *  Each functional group contains set of schematic views.
	 *  Each entry is schematic view of some component instance
	 */
	// FIXME TomasR v13 - Create unified storage of multiple open views for component instance's UUID
	private @NonNull CollectionMap<@NonNull String, @NonNull AStoragePeriphsComponent> schematicViews = new CollectionMap<>(IdentityHashMap.class, IdentitySet.class);
	/** Key for type */
	public static final @NonNull String TYPE = "type"; //$NON-NLS-1$
	/** Key for name */
	public static final @NonNull String UUID = "uuid"; //$NON-NLS-1$
	/** Key for global */
	public static final @NonNull String GLOBAL = "global"; //$NON-NLS-1$
	/** Key for group */
	public static final @NonNull String GROUP = "group"; //$NON-NLS-1$
	/** Name of editor node in memento */
	public static final @NonNull String EDITOR = "editor"; //$NON-NLS-1$
	/** editors contant */
	public static final @NonNull String EDITORS = "editors"; //$NON-NLS-1$
	/** Schematic constant */
	private static final String SCHEMATIC = "schematic"; //$NON-NLS-1$
	/** Storage ID prefix */
	public static final @NonNull String EDITORS_UNDERSCORE = EDITORS + UtilsText.UNDERSCORE;
	/** ID of empty editor */
	public static final @NonNull String EMPTY_EDITOR_ID = "org.eclipse.ui.internal.emptyEditorTab"; //$NON-NLS-1$
	/** Disable removing editors form list of opened editors */
	public static final boolean REMOVE_DISABLED = true;
	/** Enable removing editors form list of opened editors */
	public static final boolean REMOVE_ENABLED = false;
	/** Maximal occurrences of each view */
	private static final int MAXIMAL_OCCURRENCES_OF_VIEW = 100;
	/** Storage for XML memento */
	@NonNull StorageHelper storageHelper = new StorageHelper(MfactConstants.TOOL_PERIPHERALS_ID);
	/** Current MCU identification */
	@NonNull IMcuIdentification mcuIdentification = Controller.getInstance().getMcu().getMcuIdentification();
	/** Reference to current functional group */
	@NonNull StoragePeriphsFuncGroup functionalGroupReference = Controller.getInstance().getFunctionalGroup().getStorageFuncGroup();
	/** Last hash of functional group reference */
	int functionalGroupReferenceHash = functionalGroupReference.hashCode();
	/** Current state of removing editors from list of opened editors */
	boolean removeState = REMOVE_ENABLED;
	/** Map of permitted occurrence suffixes for each identification base */
	private Map<@NonNull String, Set<@NonNull Integer>> permittedOccurences = new HashMap<>();

	/** Path of the MEX file for the current project */
	String mexPath = null;

	/**
	 * Get singleton
	 * @return singleton of this class
	 */
	public static @NonNull ComponentSettingViewHelper getInstance() {
		try {
			return SWTFactoryProxy.INSTANCE.getSingletonInstance(ComponentSettingViewHelper.class);
		} catch (InstantiationException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot obtain instance of a component editor helper", e); //$NON-NLS-1$
		}
	}

	/**
	 * Switches remove state from disabled to enabled and back
	 */
	public void switchRemoveState() {
		removeState = !removeState;
	}

	/**
	 * Sets given remove state as active
	 * @param newRemoveState new state
	 */
	public void setRemoveState(boolean newRemoveState) {
		removeState = newRemoveState;
	}

	/**
	 * Constructor
	 */
	public ComponentSettingViewHelper() {
		// Reset temporary location before saving to mex
		storageHelper.saveString(EDITORS_UNDERSCORE + UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING);
		Controller.getInstance().addListener(EventTypes.CHANGE, new IEventListener() {
			@Override
			public void handle(@NonNull ToolEvent event) {
				String currentPerspectiveId = PerspectivesHelper.getActivePerspectiveId();
				if (currentPerspectiveId == null) {
					// Do nothing if no perspective is active
					return;
				}
				if (!currentPerspectiveId.equals(PeripheralsPerspective.ID)) {
					// Do nothing if current perspective is not Peripherals
					return;
				}
				IViewPart view = getViewPart();
				// No view
				if (view == null) {
					return;
				}
				IViewSite viewSite = view.getViewSite();
				// No viewSite
				if (viewSite == null) {
					// Should not happen
					return;
				}
				String mexPathNew = getMexFilePath();
				boolean configurationSwitched = false;
				if (!Objects.equals(mexPath, mexPathNew)) {
					configurationSwitched = true;
				}
				mexPath = mexPathNew;
				IMcuIdentification mcuIdentificationNew = Controller.getInstance().getMcu().getMcuIdentification();
				Display display = Display.getCurrent();
				assert display != null;
				if (!mcuIdentification.equals(mcuIdentificationNew)) {
					// Common config has changed
					mcuIdentification = mcuIdentificationNew;
					display.asyncExec(new Runnable() {
						@Override
						public void run() {
							// Save functional group after loading of MEX/newMCU
							functionalGroupReference = Controller.getInstance().getFunctionalGroup().getStorageFuncGroup();
							functionalGroupReferenceHash = functionalGroupReference.hashCode();
							// Save and erase everything from old config from memory
							restoreConfigurationEditors(viewSite);
						}
					});
				} else if (configurationSwitched) {
					// UCT-3094 when project is switched, display only the editors of components from MEX file.
					display.asyncExec(new Runnable() {
						/* (non-Javadoc)
						 * @see java.lang.Runnable#run()
						 */
						@Override
						public void run() {
							restoreConfigurationEditors(viewSite);
						}
					});
				} else {
					StoragePeriphsFuncGroup functionalGroupNew = Controller.getInstance().getFunctionalGroup().getStorageFuncGroup();
					// Functional group changed
					if (!functionalGroupReference.getUUID().equals(functionalGroupNew.getUUID())) {
						functionalGroupReference = functionalGroupNew;
						functionalGroupReferenceHash = functionalGroupReference.hashCode();
						display.asyncExec(new Runnable() {
							@Override
							public void run() {
								removeState = REMOVE_DISABLED;
								closeOpenedEditors();
								showOpenedEditors(viewSite);
								showOpenedSchematicViews(viewSite);
								removeState = REMOVE_ENABLED;
							}
						});
					} else {
						// React to changes of name, prefix and selected core
						if (functionalGroupReferenceHash != functionalGroupReference.hashCode()) {
							saveEditors();
						}
					}
				}
			}
		});
	}

	/**
	 * Returns first non null view part. If none is found then returns null.
	 * @return first non null view part. If none is found then returns null.
	 */
	public static @Nullable IViewPart getViewPart() {
		final IWorkbenchPage activePage = Objects.requireNonNull(PlatformUI.getWorkbench().getActiveWorkbenchWindow()).getActivePage();
		if (activePage != null) {
			IViewReference[] viewReferences = activePage.getViewReferences();
			for (IViewReference reference : viewReferences) {
				IViewPart view = reference.getView(false);
				if (view != null) {
					return view;
				}
			}
		}
		return null;
	}

	/**
	 * Clears list of opened editors
	 */
	public void clearEditors() {
		editors.clear();
	}

	/**
	 * Clears list of opened schematic views
	 */
	public void clearSchematicViews() {
		schematicViews.clear();
	}

	/**
	 * Removes editor from list of opened ones
	 * @param input for comparing which editor need to be closed
	 * @param funcGroupUuid for using as key
	 */
	public void removeEditor(@NonNull ComponentSettingViewInput input, @NonNull String funcGroupUuid) {
		// Do not remove anything in switching state. Otherwise you will close editor and delete it from persistent storage
		if (removeState == REMOVE_DISABLED) {
			return;
		}
		IdentitySet<@NonNull AStoragePeriphsComponent> originalSet = (IdentitySet<@NonNull AStoragePeriphsComponent>) editors.get(funcGroupUuid);
		IdentitySet<@NonNull AStoragePeriphsComponent> funcGroupEditors;
		if (originalSet == null) {
			funcGroupEditors = new IdentitySet<>();
		} else {
			funcGroupEditors = new IdentitySet<>(originalSet);
		}
		if (input.isGlobalConfig()) {
			for (@NonNull Entry<String, AStoragePeriphsComponent> entry : editors.flatEntrySet()) {
				AStoragePeriphsComponent component = entry.getValue();
				if (component instanceof StoragePeriphsComponent) {
					if (component.getName().equals(input.getComponentType())) {
						funcGroupUuid = UtilsText.safeString(entry.getKey());
						funcGroupEditors.add(component);
					}
				}
			}
		}
		if (funcGroupEditors.isEmpty()) {
			return;
		}
		@Nullable AStoragePeriphsComponent editorToBeRemoved =
			CollectionsUtils.nullableOptionalGet(funcGroupEditors.stream()
		.filter(x -> {
			if (input.isGlobalConfig()) {
				if (x instanceof StoragePeriphsComponent) {
					return x.getName().equals(input.getComponentType());
				}
			} else {
				if (x instanceof StoragePeriphsComponentInstance) {
					IComponentInstanceConfig componentSetting = Controller.getInstance().getComponentInstance(input.getComponentType(), input.getComponent());
					if (componentSetting == null) {
						componentSetting = Controller.getInstance().getComponentInstance(input.getUUID());
					}
					if (componentSetting != null) {
						return x.getUUID().equals(componentSetting.getUUID());
					}
				}
			}
			return false;
		})
		.findAny());
		if (editorToBeRemoved == null) {
			return;
		}
		if (editors.removeItem(funcGroupUuid, editorToBeRemoved)) {
			saveEditors();
		}
	}

	/**
	 * Removes schematic view from list of opened ones
	 * @param uuid of instance whose schematic view should be removed from opened ones
	 * @param funcGroupUuid UUID of functional group to have the list of opened schematic views updated
	 */
	public void removeSchematicView(@NonNull String uuid, @NonNull String funcGroupUuid) {
		// Do not remove anything in switching state. Otherwise you will close editor and delete it from persistent storage
		if (removeState == REMOVE_DISABLED) {
			return;
		}
		IdentitySet<@NonNull AStoragePeriphsComponent> originalSet = (IdentitySet<@NonNull AStoragePeriphsComponent>) schematicViews.get(funcGroupUuid);
		IdentitySet<@NonNull AStoragePeriphsComponent> funcGroupEditors = (originalSet == null) ? new IdentitySet<>() : new IdentitySet<>(originalSet);
		if (funcGroupEditors.isEmpty()) {
			return;
		}
		@Nullable AStoragePeriphsComponent editorToBeRemoved = CollectionsUtils.nullableOptionalGet(funcGroupEditors.stream()
			.filter(x -> {
				if (x instanceof StoragePeriphsComponentInstance) {
					IComponentInstanceConfig componentSetting = Controller.getInstance().getComponentInstance(uuid);
					if (componentSetting != null) {
						return x.getUUID().equals(componentSetting.getUUID());
					}
				}
				return false;
			})
			.findAny());
		if (editorToBeRemoved == null) {
			return;
		}
		if (schematicViews.removeItem(funcGroupUuid, editorToBeRemoved)) {
			saveEditors();
		}
	}

	/**
	 * Adds editor to set of opened ones
	 * @param funcGroup name of functional group to add editor to
	 * @param component to be stored as opened editor
	 * @return {@code true} when editor is successfully added, {@code false} otherwise
	 */
	public boolean addEditor(@NonNull StoragePeriphsFuncGroup funcGroup, @NonNull AStoragePeriphsComponent component) {
		if (component instanceof StoragePeriphsComponent) {
			// Skip when already in opened editors no matter in which group
			final Object comp = component;
			if (editors.containsValue(comp)) {
				return false;
			}
		}
		Object found = CollectionsUtils.findAny(editors.flatEntrySet(), entry -> funcGroup.getUUID().equals(entry.getKey()) && component.equalsUUID(entry.getValue()));
		// Do not add something that is already there
		if (found == null) {
			editors.add(funcGroup.getUUID(), component);
		}
		return saveEditors();
	}

	/**
	 * Adds schematic view to set of opened ones
	 * @param funcGroup name of functional group to add editor to
	 * @param component to be stored as opened editor
	 * @return {@code true} when schematic view was successfully added, {@code false} otherwise
	 */
	public boolean addSchematic(@NonNull StoragePeriphsFuncGroup funcGroup, @NonNull AStoragePeriphsComponent component) {
		if (component instanceof StoragePeriphsComponent) {
			// Skip when already in opened schematic views no matter in which group
			final Object comp = component;
			if (schematicViews.containsValue(comp)) {
				return false;
			}
		}
		Object found = CollectionsUtils.findAny(schematicViews.flatEntrySet(), entry -> funcGroup.getUUID().equals(entry.getKey()) && component.equalsUUID(entry.getValue()));
		// Do not add something that is already there
		if (found == null) {
			schematicViews.add(funcGroup.getUUID(), component);
		}
		return saveEditors();
	}

	/**
	 * Looks up for MEX location
	 * @return path to MEX file
	 */
	private static String getMexFilePath() {
		@Nullable IPath locationPath = SharedConfigurationFactory.getSharedConfigurationSingleton().getLocationPath();
		String projectLocation;
		if (locationPath == null) {
			projectLocation = UtilsText.EMPTY_STRING;
		} else {
			projectLocation = locationPath.toOSString();
		}
		return projectLocation;
	}

	/**
	 * Shows currently opened editors from editors
	 * @param viewSite to create editors in
	 */
	public void showOpenedEditors(@NonNull IViewSite viewSite) {
		// Open new ones
		Iterator<Entry<@NonNull String, @NonNull AStoragePeriphsComponent>> iterator = editors.flatEntrySet().stream()
				.filter(e -> {
						Controller controller = Controller.getInstance();
						String fgUuid = e.getKey();
						StoragePeriphsFuncGroup storageFuncGroupLoc = controller.getFunctionalGroup().getStorageFuncGroup();
						// filter only components which belong to the currently selected functional group, or are global
						return ((fgUuid.equals(storageFuncGroupLoc.getUUID())) || (e.getValue() instanceof StoragePeriphsComponent))
								&& storageFuncGroupLoc.getInstances().contains(e.getValue());
				}).iterator();
		while (iterator.hasNext()) {
			Entry<@NonNull String, @NonNull AStoragePeriphsComponent> entry = iterator.next();
			AStoragePeriphsComponent config = entry.getValue();
			boolean global = false;
			if (config instanceof StoragePeriphsComponentInstance) { // FIXME TomasR v13 - Remove unused support for global
				String uuid = ((StoragePeriphsComponentInstance)config).getUUID();
				ComponentSettingView.open(viewSite, uuid, false);
			} else if (config instanceof StoragePeriphsComponent) {
				String name = ((StoragePeriphsComponent)config).getName();
				String type = ((StoragePeriphsComponent)config).getName();
				global = true;
				ComponentSettingView.open(viewSite, type, name, global, false);
			}
		}
	}

	/**
	 * Shows currently opened schematic views
	 * @param viewSite to create editors in
	 */
	public void showOpenedSchematicViews(@NonNull IViewSite viewSite) {
		@NonNull IConfigurationElement[] configurationElementsFor = Platform.getExtensionRegistry().getConfigurationElementsFor(ADDITIONAL_VIEWS_EXTENSION_ID);
		if (configurationElementsFor.length == 0) {
			return;
		}
		IConfigurationElement schematicViewHandler = null;
		for (int i = 0; i < configurationElementsFor.length; i++) { // FIXME TomasR v13 - Create generic support for opening previously opened views
			if (configurationElementsFor[0].getAttribute(EXTENSION_POINT_ID_ATTRIBUTE).startsWith(SCHEMATIC_VIEW_ID)) {
				schematicViewHandler = configurationElementsFor[i];
			}
		}
		if (schematicViewHandler == null) {
			return;
		}
		// Open new ones
		Iterator<Entry<@NonNull String, @NonNull AStoragePeriphsComponent>> iterator = schematicViews.flatEntrySet().stream()
				.filter(e -> {
					Controller controller = Controller.getInstance();
					String fgUuid = e.getKey();
					StoragePeriphsFuncGroup storageFuncGroupLoc = controller.getFunctionalGroup().getStorageFuncGroup();
					// filter only components which belong to the currently selected functional group, or are global
					return ((fgUuid.equals(storageFuncGroupLoc.getUUID())) || (e.getValue() instanceof StoragePeriphsComponent))
							&& storageFuncGroupLoc.getInstances().contains(e.getValue());
				}).iterator();
		while (iterator.hasNext()) {
			Entry<@NonNull String, @NonNull AStoragePeriphsComponent> entry = iterator.next();
			AStoragePeriphsComponent config = entry.getValue();
			if (config instanceof StoragePeriphsComponentInstance) {
				String uuid = ((StoragePeriphsComponentInstance) config).getUUID();
				try {
					APeriphsExternalViewHandler handler = (APeriphsExternalViewHandler) schematicViewHandler.createExecutableExtension(ADDITIONAL_VIEWS_EXTENSION_ATTRIBUTE_HANDLER);
					handler.openViewFor(viewSite, uuid);
				} catch (CoreException e) {
					LOGGER.log(Level.SEVERE, "[TOOL] Failed to create handler: {0}", e.getLocalizedMessage()); //$NON-NLS-1$
				}
			}
		}
	}

	/**
	 * Restores editors from XML memento
	 * @return {@code true} if nothing fails, otherwise returns {@code false}
	 */
	public boolean restoreEditors() {
		boolean result = false;
		try {
			// FIXME TomasR v13 - Consider using configuration UUID as identification instead of MEX location which is not known until saved and then can be changed
			String mexFilePath = getMexFilePath();
			if (UtilsText.isEmpty(mexFilePath)) {
				LOGGER.log(Level.FINE, "[TOOL] Empty MEX file path, not restoring editors"); //$NON-NLS-1$
				return false;
			}
			String mementoStrContent = storageHelper.loadString(EDITORS_UNDERSCORE + mexFilePath, UtilsText.EMPTY_STRING);
			if (UtilsText.EMPTY_STRING.equals(mementoStrContent)) {
				LOGGER.log(Level.FINE, "[TOOL] No editors saved for MEX file, not restoring editors"); //$NON-NLS-1$
				return false;
			}
			XMLMemento xmlMemento = XMLMemento.createReadRoot(new StringReader(mementoStrContent)); // <editors>
			// Read all entries from the root memento
			for (IMemento memento : xmlMemento.getChildren()) {
				boolean restored = restoreEditorsHandleComponentSettingsView(memento);
				if (!restored) {
					restored = restoreEditorsHandleSchematicView(memento);
				}
				if (!restored) {
					LOGGER.log(Level.WARNING, "[TOOL] Memento was not parsed by any handler: {0}", memento); //$NON-NLS-1$
				}
			}
			result = true;
		} catch (WorkbenchException e) {
			LOGGER.severe("[TOOL] XMLMemento exception occured. Detail: " + e.getMessage()); //$NON-NLS-1$
		}
		return result;
	}

	/**
	 * Tries to restore ComponentSettingsView from given memento.
	 * @param memento to be parsed
	 * @return {@code true} if the memento was successfully parsed and added, {@code false} otherwise
	 */
	private boolean restoreEditorsHandleComponentSettingsView(IMemento memento) {
		if (!memento.getType().equals(EDITOR)) {
			return false;
		}
		String funcGroupUuid = UtilsText.safeString(memento.getString(GROUP));
		IFunctionalGroup functionalGroup = Controller.getInstance().getProfile().getFunctionalGroupWithUUID(funcGroupUuid);
		if (functionalGroup == null) {
			LOGGER.info("[TOOL] Trying to restore editors: Functional group with given id was not found in current profile"); //$NON-NLS-1$
			return false;
		}
		String type = UtilsText.safeString(memento.getString(TYPE));
		String name = UtilsText.safeString(memento.getString(UUID));
		// FIXME TomasR v13 remove the global flag and parts of code from ComponentSettingsView and related classes
		Boolean globalBoolean = memento.getBoolean(GLOBAL);
		if (globalBoolean == null) {
			LOGGER.warning("[TOOL] Global is not set in XML memento"); //$NON-NLS-1$
			return false;
		}
		boolean global = globalBoolean.booleanValue();
		AStoragePeriphsComponent configToSave;
		if (global) {
			IComponentConfig configuredComponent = Controller.getInstance().getConfiguredComponent(type);
			if (configuredComponent == null) {
				LOGGER.info("[TOOL] Trying to restore component setting view of component config that is not present in current functional group"); //$NON-NLS-1$
				return false;
			}
			configToSave = configuredComponent.getStorageComponent();
		} else {
			IComponentInstanceConfig instance = functionalGroup.getInstanceByUUID(name);
			if (instance == null) {
				LOGGER.info("[TOOL] Trying to restore component setting view of component instance config that is not present in current functional group"); //$NON-NLS-1$
				return false;
			}
			configToSave = instance.getStorageComponent();
		}
		return addEditor(functionalGroup.getStorageFuncGroup(), configToSave);
	}

	/**
	 * Tries to restore schematic views from given memento.
	 * @param memento to be parsed
	 * @return {@code true} if the memento was successfully parsed and added, {@code false} otherwise
	 */
	private boolean restoreEditorsHandleSchematicView(IMemento memento) {
		if (!memento.getType().equals(SCHEMATIC)) {
			return false;
		}
		String funcGroupUuid = UtilsText.safeString(memento.getString(GROUP));
		IFunctionalGroup functionalGroup = Controller.getInstance().getProfile().getFunctionalGroupWithUUID(funcGroupUuid);
		if (functionalGroup == null) {
			LOGGER.info("[TOOL] Trying to restore editors: Functional group with given id was not found in current profile"); //$NON-NLS-1$
			return false;
		}
		String name = UtilsText.safeString(memento.getString(UUID));
		IComponentInstanceConfig instance = functionalGroup.getInstanceByUUID(name);
		if (instance == null) {
			LOGGER.info("[TOOL] Trying to restore component setting view of component instance config that is not present in current functional group"); //$NON-NLS-1$
			return false;
		}
		return addSchematic(functionalGroup.getStorageFuncGroup(), instance.getStorageComponent());
	}

	/**
	 * Saves current list of opened editors to XML memento
	 * @return {@code true} if nothing fails, otherwise returns {@code false}
	 */
	public boolean saveEditors() {
		boolean result = false;
		StringWriter writer = new StringWriter();
		try {
			XMLMemento xmlMemento = XMLMemento.createWriteRoot(EDITORS);
			// Each entry needs to be stored
			for (Entry<@NonNull String, @NonNull AStoragePeriphsComponent> editorSettings : editors.flatEntrySet()) {
				saveEditorsHandleComponentSettingsView(xmlMemento, editorSettings);
			}
			for (Entry<@NonNull String, @NonNull AStoragePeriphsComponent> editorSettings : schematicViews.flatEntrySet()) {
				saveEditorsHandleSchematicView(xmlMemento, editorSettings);
			}
			// Save and write out
			xmlMemento.save(writer);
			storageHelper.saveString(EDITORS_UNDERSCORE + getMexFilePath(), UtilsText.safeString(writer.toString()));
			result = true;
		} catch (IOException e) {
			LOGGER.info("[TOOL] File exception occured. Detail: " + e.getMessage()); //$NON-NLS-1$
		}
		return result;
	}

	/**
	 * Saves entry about open ComponentSettingsView to the memento
	 * @param xmlMemento to which the entry will be added
	 * @param editorSettings information about the entry to be saved
	 * @return {@code true} when entry is successfully saved, {@code false} otherwise
	 */
	private static boolean saveEditorsHandleComponentSettingsView(XMLMemento xmlMemento, Entry<@NonNull String, @NonNull AStoragePeriphsComponent> editorSettings) {
		String funcGroupUuid = editorSettings.getKey();
		AStoragePeriphsComponent component = editorSettings.getValue();
		// Create node and set values
		try {
			IMemento node = xmlMemento.createChild(EDITOR);
			node.putString(UUID, component.getUUID());
			String global;
			if (component instanceof StoragePeriphsComponentInstance) {
				node.putString(TYPE, ((StoragePeriphsComponentInstance) component).getType());
				global = String.valueOf(false);
			} else if (component instanceof StoragePeriphsComponent) { // FIXME TomasR v13 Remove support for global editors as they are not used anymore
				node.putString(TYPE, ((StoragePeriphsComponent) component).getName());
				global = String.valueOf(true);
			} else {
				LOGGER.log(Level.SEVERE, "[TOOL] Editor given for save is not instance or component config editor: {0}", component); //$NON-NLS-1$
				return false;
			}
			node.putString(GLOBAL, global);
			node.putString(GROUP, funcGroupUuid);
			return true;
		} catch (DOMException e) {
			LOGGER.log(Level.SEVERE, "Exception occurred during creating new XML element in memento: {0}", e.getLocalizedMessage()); //$NON-NLS-1$
			return false;
		}
	}

	/**
	 * Saves entry about open schematic views to the memento
	 * @param xmlMemento to which the entry will be added
	 * @param editorSettings information about the entry to be saved
	 * @return {@code true} when entry is successfully saved, {@code false} otherwise
	 */
	private static boolean saveEditorsHandleSchematicView(XMLMemento xmlMemento, Entry<@NonNull String, @NonNull AStoragePeriphsComponent> editorSettings) {
		String funcGroupUuid = editorSettings.getKey();
		AStoragePeriphsComponent component = editorSettings.getValue();
		// Create node and set values
		try {
			IMemento node = xmlMemento.createChild(SCHEMATIC);
			node.putString(UUID, component.getUUID());
			node.putString(GROUP, funcGroupUuid);
			return true;
		} catch (DOMException e) {
			LOGGER.log(Level.SEVERE, "Exception occurred during creating new XML element in memento: {0}", e.getLocalizedMessage()); //$NON-NLS-1$
			return false;
		}
	}

	/**
	 * Close all editors opened in UI
	 */
	public void closeOpenedEditors() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench == null) return;
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		if (window == null) return;
		IWorkbenchPage activePage = Objects.requireNonNull(workbench.getActiveWorkbenchWindow()).getActivePage();
		if (activePage == null) return;
		IViewReference[] viewReferences = activePage.getViewReferences();
		for (int index = 0; index < viewReferences.length; index++) {
			IViewReference viewRef = viewReferences[index];
			String id = viewRef.getId();
			if (id.equals(ComponentSettingView.ID)) {
				activePage.hideView(viewRef);
			}
		}
	}

	/**
	 * Close all schematic views opened in UI
	 */
	public void closeOpenedSchematicViews() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench == null) return;
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		if (window == null) return;
		IWorkbenchPage activePage = Objects.requireNonNull(workbench.getActiveWorkbenchWindow()).getActivePage();
		if (activePage == null) return;
		IViewReference[] viewReferences = activePage.getViewReferences();
		for (int index = 0; index < viewReferences.length; index++) {
			IViewReference viewRef = viewReferences[index];
			String id = viewRef.getId();
			if (id.equals(SCHEMATIC_VIEW_ID)) {
				activePage.hideView(viewRef);
			}
		}
	}

	/**
	 * Removes all entries of given group from opened ones
	 * @param group in which to close all opened editors
	 */
	public void removeEntriesOfGroup(@NonNull StoragePeriphsFuncGroup group) {
		Collection<@NonNull AStoragePeriphsComponent> editorsOfGroup = editors.get(group.getUUID());
		if (editorsOfGroup != null) {
			editorsOfGroup.clear();
			editors.remove(group.getUUID());
			schematicViews.remove(group.getUUID());
		}
	}

	/**
	 * Reopens views where their secondary ID does not correspond to their type or name
	 */
	public void reopenInvalidViews() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchPage activePage = Objects.requireNonNull(workbench.getActiveWorkbenchWindow()).getActivePage();
		if (activePage == null) return;
		IViewReference[] viewReferences = activePage.getViewReferences();
		for (int index = 0; index < viewReferences.length; index++) {
			String primaryID = viewReferences[index].getId();
			String secondaryID = UtilsText.safeString(viewReferences[index].getSecondaryId());
			if (primaryID.equals(ComponentSettingView.ID)) {
				ComponentSettingView view = (ComponentSettingView) viewReferences[index].getView(false);
				if (view == null) {
					continue;
				}
				ComponentSettingViewInput componentInput = view.componentInput;
				if (componentInput == null) {
					continue;
				}
				if (!secondaryID.startsWith(createSecondaryId(componentInput))) {
					// Close and open again
					activePage.hideView(view);
					ComponentSettingView.open(activePage, componentInput.getComponentType(), componentInput.getComponent(),
							componentInput.isGlobalConfig(), false, null);
				}
			}
		}
	}

	/**
	 * Creates secondary ID for view
	 * @param componentInput to get information from
	 * @return secondary ID string
	 */
	public static @NonNull String createSecondaryId(@NonNull ComponentSettingViewInput componentInput) {
		String identification;
		String uuid = componentInput.getUUID();
		if (UtilsText.isEmpty(uuid)) {
			identification = componentInput.getComponentType() + ComponentSettingView.SECONDARY_ID_NAME_TYPE_SEPARATOR
				+ componentInput.getComponent();
		} else {
			identification = uuid;
		}
		return identification;
	}

	/**
	 * Return set of permitted occurrence suffixes
	 * @param identificationBase which is common to all occurrences
	 * @return set of permitted suffixes
	 */
	public @NonNull Set<@NonNull Integer> getPermittedOccurrences (@NonNull String identificationBase) {
		Set<@NonNull Integer> viewOccurences = permittedOccurences.computeIfAbsent(identificationBase, (id) -> {
			HashSet<@NonNull Integer> hashSet = new HashSet<>();
			hashSet.add(Integer.valueOf(0));
			return hashSet;
		});
		assert (viewOccurences != null); // Not possible to be null
		return viewOccurences;
	}

	/**
	 * Adds one permitted occurrence to the identification base
	 * @param identificationBase of the view
	 */
	public void addPermittedOccurrence(@NonNull String identificationBase) {
		Set<@NonNull Integer> viewOccurences = permittedOccurences.computeIfAbsent(identificationBase, (id) -> {
			HashSet<@NonNull Integer> hashSet = new HashSet<>();
			hashSet.add(Integer.valueOf(0));
			return hashSet;
		});
		assert (viewOccurences != null); // Not possible to be null
		for (int i = 1; i < MAXIMAL_OCCURRENCES_OF_VIEW; i++) {
			Integer occurrence = Integer.valueOf(i);
			if (!viewOccurences.contains(occurrence)) {
				viewOccurences.add(occurrence);
				break;
			}
		}
	}

	/**
	 * Removes given permitted occurrence of the identification base
	 * @param identificationBase of the view
	 * @param occurrence of the view
	 */
	public void removePermittedOccurrence(@NonNull String identificationBase, @NonNull Integer occurrence) {
		Set<@NonNull Integer> viewOccurrences = permittedOccurences.computeIfAbsent(identificationBase, (id) -> {
			return new HashSet<>();
		});
		assert (viewOccurrences != null); // Not possible to be null
		viewOccurrences.remove(occurrence);
		if (viewOccurrences.isEmpty()) { // 0 must be always present
			viewOccurrences.add(Integer.valueOf(0));
		}
	}

	/**
	 * Restore all component editors that are saved into the current configuration
	 * @param viewSite to restore editors in
	 */
	public void restoreConfigurationEditors(@NonNull IViewSite viewSite) {
		removeState = REMOVE_DISABLED;
		closeOpenedEditors();
		clearEditors();
		removeState = REMOVE_ENABLED;
		// Run restoring
		restoreEditors();
		showOpenedEditors(viewSite);
		showOpenedSchematicViews(viewSite);
	}

	/**
	 * Opens the controls (e.g. tabs) on the path to the required setting.
	 * @param child starting point on the path to the required child
	 * @param childId the ID of the required child
	 * @param parent of the current child
	 * @param updateType type of the update to perform
	 * @return {@code true} when child was found, {@code false} otherwise
	 */
	public static boolean selectChildrenOnPath(@NonNull IChildControl child, @NonNull String childId, @NonNull UpdateType updateType) {
		List<@NonNull IChildControl> children = null;
		String myPart;
		String rest;
		int separatorIndex = childId.indexOf(SettingUtils.SEPARATOR);
		if (separatorIndex != -1) {
			myPart = childId.substring(0, separatorIndex);
			rest = childId.substring(separatorIndex + 1);
		} else { // Last part of the ID
			myPart = childId;
			rest = childId;
		}
		if (child.getChild().getName().equals(rest)) {
			return true; // Setting was found
		} else if (child instanceof ChildProvidableControlBase) {
			ChildProvidableControlBase childProvidableControlBase = (ChildProvidableControlBase) child;
			children = childProvidableControlBase.getChildren();
			if (child instanceof ArrayControl) {
				children = ((ArrayControl) child).getChildren();
			}
			IChildControl nextChild = CollectionsUtils.nullableOptionalGet(children.stream().filter(x -> x.getChild().getName().equals(myPart)).findAny());
			if (nextChild != null) {
				childProvidableControlBase.selectChildTab(nextChild);
				childProvidableControlBase.setSelectedChild(nextChild);
				childProvidableControlBase.update(updateType);
				return selectChildrenOnPath(nextChild, rest, updateType);
			}
		}
		return false;
	}

	/**
	 * Opens the controls (e.g. tabs) on the path to the required setting.
	 * @param child starting point on the path to the required child
	 * @param childId the ID of the required child
	 * @param parent of the current child
	 * @param updateType type of the update to perform
	 */
	public static void selectChildrenOnPath(@NonNull IChildControl child, @NonNull String childId,
			@Nullable ChildProvidableControlBase parent, @NonNull UpdateType updateType) {
		List<@NonNull IChildControl> children = null;
		if (child instanceof ChildProvidableControlBase) {
			ChildProvidableControlBase childProvidableControlBase = (ChildProvidableControlBase) child;
			children = childProvidableControlBase.getChildren();
			// if the current child is on the path to the required child
			if (childId.contains(child.getChild().getId())) {
				if (parent != null) {
					if (parent instanceof ArrayControl) {
						ArrayControl arrayControl = (ArrayControl) parent;
						children = arrayControl.getChildren();
					}
					parent.selectChildTab(child);
					parent.setSelectedChild(child);
					parent.update(updateType);
				}
			}
			for (IChildControl childControl : children) {
				if (childId.contains(childControl.getChild().getId())) {
					selectChildrenOnPath(childControl, childId, childProvidableControlBase, updateType);
				}
			}
		} else if (childId.equals(child.getChild().getId())) {
			// selected setting could be inside a tab, so focus on it
			if (parent != null) {
				parent.selectChildTab(child);
				parent.setSelectedChild(child);
				parent.update(updateType);
			}
		}
	}
}
