/**
 * Copyright 2017-2021 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ConfigSetConfig;
import com.nxp.swtools.periphs.model.data.XMLConstants;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing control of a config set configuration.
 * @author Juraj Ondruska
 */
public class ConfigSetControl extends ChildProvidableControlBase {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ConfigSetControl.class);
	/** content of the editor */
	private Composite contentComposite;
	/** The Quick selection control */
	private @NonNull QuickSelectionControl quickSelection;
	/** The label with the config set name, or {@code null} in case the label should be hidden */
	private @Nullable Control label;

	/**
	 * @param configSetConfig to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ConfigSetControl(@NonNull ConfigSetConfig configSetConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(configSetConfig, controlOptions, controllerWrapper);
		assert getChild().getModelData().hasQuickSelections();
		quickSelection = new QuickSelectionControl(configSetConfig, controllerWrapper);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		if (SWToolsProperties.isVerificationOn()) {
			if (getChild().getModelData().getUINameExprString() != null) {
				LOGGER.log(Level.WARNING, String.format("[DATA] %1s is used as section title from %2s; section titles are only updated on component (re-)open", //$NON-NLS-1$
						XMLConstants.LABEL_EXPR, getChild()));
			}
		}
		if (((ConfigSetConfig)child).getAllVisibleSettings().isEmpty()) {
			return;
		}
		// create default composite for config set and set layout - required if UI_BORDER_HIDDEN is used on config set
		final Composite contentCompositeLoc = new Composite(composite, SWT.NONE);
		contentComposite = contentCompositeLoc;
		GridLayoutComponents gridLayout = new GridLayoutComponents(ComponentSettingView.CONFIG_SET_COLS, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		contentCompositeLoc.setLayout(gridLayout);
		contentCompositeLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, colSpan, 1));
		super.create(contentCompositeLoc, ComponentSettingView.CONFIG_SET_COLS);
		// update layout data for label and main composite
		Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			labelControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, ComponentSettingView.CONFIG_SET_COLS, 1));
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			mainControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, ComponentSettingView.CONFIG_SET_COLS, 1));
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildProvidableControlBase#getChild()
	 */
	@Override
	public @NonNull ConfigSetConfig getChild() {
		return (ConfigSetConfig) child;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		if (!quickSelection.isEmpty()) {
			Composite labelComposite = new Composite(composite, SWT.None);
			GridLayoutComponents layout = new GridLayoutComponents(2, false);
			layout.marginHeight = 0;
			layout.marginWidth = 0;
			labelComposite.setLayout(layout);
			Control labelLoc = createLabel(labelComposite);
			quickSelection.create(labelComposite, labelLoc == null ? 2 : 1);
			return labelComposite;
		} else {
			// the quick selection won't be displayed, do not create unnecessary composite
			return createLabel(composite);
		}
	}

	/**
	 * Creates label in specified composite
	 * @param composite in which the label will be created
	 * @return the newly created label or {@code null} if there should not be label
	 */
	private Control createLabel(@NonNull Composite composite) {
		Control labelLoc = super.createLabelControl(composite);
		if (labelLoc != null) {
			setLabelInternalFontStyle(SWT.BOLD);
			Control actualLabel = labelControlInternal;
			if (actualLabel != null) {
				createErrorDecoration(actualLabel, SWT.LEFT | SWT.TOP);
			}
			labelLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false));
			label = labelLoc;
		}
		return labelLoc;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#updateLabelContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.editor.IChildControl.UpdateType)
	 */
	@Override
	protected void updateLabelContent(@NonNull Control labelCtrl, @NonNull UpdateType updateType) {
		// Add tooltip to label inside label row composite instead
		Control labelLoc = label;
		if (labelLoc != null) {
			super.updateLabelContent(labelLoc, updateType);
			String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(child));
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelLoc, toolTipText);
		}
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		super.update(updateType);
		quickSelection.update(updateType);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#fillTopOfMainControl(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	protected void fillTopOfMainControl(@NonNull Composite composite, int numColumns) {
		if (controlOptions.isLabelHidden() && !quickSelection.isEmpty()) { // if the label is hidden, create the quick selection inside of the main control
			quickSelection.create(composite, numColumns);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#dispose()
	 */
	@Override
	public void dispose() {
		if (contentComposite != null && !contentComposite.isDisposed()) {
			contentComposite.dispose();
			contentComposite = null;
		}
		quickSelection.dispose();
		super.dispose();
	}
}
